subroutine PHYrad_ecRad(dimLon,dimLat,nbLvls,                 &
      &                 yyyy,mm,dd,irhhss,                    &
      &                 rDiST ,rAlbe ,rp_RAD,rphRAD,rcdRAD,   &
      &                 emsRAD,lsmRAD,cszRAD,rLaRAD,rLoRAD,   &
      &                 rqvRAD,rqiRAD,rqwRAD,rswRAD,rqrRAD,   &
      &                 rqsRAD,rtaRAD,rtsRAD,                 &
      &                 rFIRnc,rFIRnt,rFSOnc,rFSOnt,rPLWdv,   &
      &                 rPSWdf,rPSWdr,rFLopt)

! +------------------------------------------------------------------------+
! | MAR PHYSICS                                         September 2023 MAR |
! |                                                                        |
! | PHYrad_ecRad interfaces MAR with the ECMWF solar/infrared radiative    |
! | transfer scheme ecRad 1.5.0 and runs it. It assumes said scheme has    |
! | been previously initialized with PHYrad_ecRad_init.                    |
! |                                                                        |
! | Code by J.-F. Grailet, based on previous subroutine PHYrad2CEP (main   |
! | structure) and on the ecRad offline driver.                            |
! |                                                                        |
! | Main body has been written in September and October 2022 and the first |
! | successful MAR runs with ecRad have been conducted in November 2022,   |
! | but numerous additions and adjustements were brought since then to     |
! | tune the new radiation scheme and make the most of its features, such  |
! | as spectral decomposition of shortwave radiation.                      |
! |                                                                        |
! | The subroutine has been split in two (PHYrad_ecRad_init and            |
! | PHYrad_ecRad) in September 2023 to clearly separate the operations of  |
! | initializing and running the radiation scheme while improving code     |
! | readability.                                                           |
! |                                                                        |
! | ECMWF code by Hogan et al., cf. https://github.com/ecmwf-ifs/ecrad     |
! |                                                                        |
! | General remarks:                                                       |
! | -IFS = Integrated Forecast System, a weather prediction model used     |
! |  by the ECMWF.                                                         |
! |                                                                        |
! +------------------------------------------------------------------------+

  use parkind1, only : jpim, jprb

! +--Global variables via modules
! +  ============================

  use YOMCST, only : RI0
  use YOERAD, only : NSW
  use INCMAR, only : RI0_SCALING, NAERMACC, & 
   &                 NCLOUD_PARAM, RCLOUD_PARAM_GAMMA, &
   &                 RCLOUD_PARAM_ALPHA, RCLOUD_PARAM_P, &
   &                 RCLOUD_RH_THRESHOLD_LAND, & 
   &                 RCLOUD_RH_THRESHOLD_SEA, &
   &                 BLANKET_LAYERS, &
   &                 SCALING_PQ, SCALING_PQIWP, SCALING_PQLWP, &
   &                 SCALING_PQRAIN, SCALING_PQSNOW
  
! +--Additional modules
! +  ==================
  
  ! Access to climatologies (built at set-up by PHYrad_ecRad_init)
  use radiation_climatologies, only : N_AEROSOLS_TEGEN, & 
   &                                  N_AEROSOLS_MACC, & 
   &                                  PCO2, PCH4, PN2O, PCFC11, & 
   &                                  PCFC12, PHCFC22, PCCL4, PO3, &
   &                                  PAEROSOL_OLD, PAEROSOL
  
  ! Module to handle extra ecRad output variables not used by MAR (no "only": everything is used)
  use extra_ecRad_outputs

  implicit none

! +--Input variables
! +  ===============

  integer, intent(in) :: dimLon    ! Number of longitude points (per line)
  integer, intent(in) :: dimLat    ! Number of latitude points (per column)
  integer, intent(in) :: nbLvls    ! Number of vertical levels
  
  integer, intent(in) :: yyyy      ! Year
  integer, intent(in) :: mm        ! Month
  integer, intent(in) :: dd        ! Day
  integer, intent(in) :: irhhss    ! Number of seconds in the current day
  
  real, intent(in) :: rDiST                              ! Sun-Earth distance          [UA]
  real, intent(in) :: rLaRAD(dimLon, dimLat)             ! Latitude                    [degree]
  real, intent(in) :: rLoRAD(dimLon, dimLat)             ! Longitude                   [degree]
  real, intent(in) :: rAlbe (dimLon, dimLat)             ! Surface albedo
  real, intent(in) :: emsRAD(dimLon, dimLat)             ! Surface emissivity
  real, intent(in) :: lsmRAD(dimLon, dimLat)             ! Land/sea mask (1.=land)  
  real, intent(in) :: cszRAD(dimLon, dimLat)             ! cos(solar zenithal angle)
  real, intent(in) :: rp_RAD(dimLon, dimLat, nbLvls)     ! Pressure (layer)            [Pa]
  real, intent(in) :: rphRAD(dimLon, dimLat, nbLvls + 1) ! Pressure (layer interface)  [Pa]
  real, intent(inout) :: rcdRAD(dimLon, dimLat, nbLvls)  ! Cloud Fraction (dropplets)
  real, intent(in) :: rqvRAD(dimLon, dimLat, nbLvls)     ! Vapor concentration         [kg/kg]
  real, intent(in) :: rqiRAD(dimLon, dimLat, nbLvls)     ! Ice crystals concentration  [kg/kg]
  real, intent(in) :: rqwRAD(dimLon, dimLat, nbLvls)     ! Droplet concentration       [kg/kg]
  real, intent(in) :: rswRAD(dimLon, dimLat, nbLvls)     ! Water saturation threshold  [kg/kg]
  real, intent(in) :: rqrRAD(dimLon, dimLat, nbLvls)     ! Rain drops concentration    [kg/kg]
  real, intent(in) :: rqsRAD(dimLon, dimLat, nbLvls)     ! Snowflakes concentration    [kg/kg]
  real, intent(in) :: rtaRAD(dimLon, dimLat, nbLvls)     ! Temperature (layer)         [K]
  real, intent(in) :: rtsRAD(dimLon, dimLat)             ! Temperature (surface+air)   [K]

! +--Output variables
! +  ================

  ! 3D outputs with 3rd dimension = nbLvls + 1
  real, intent(inout), allocatable :: rFIRnc(:,:,:)      ! Clear sky LW net fluxes     [W/m2]
  real, intent(inout), allocatable :: rFIRnt(:,:,:)      ! Total sky LW net fluxes     [W/m2]
  real, intent(inout), allocatable :: rFSOnc(:,:,:)      ! Clear sky SW net fluxes     [W/m2]
  real, intent(inout), allocatable :: rFSOnt(:,:,:)      ! Total sky SW net fluxes     [W/m2]
  real, intent(inout), allocatable :: rPLWdv(:,:,:)      ! Partial derivative of LW down flux
  
  ! 3D outputs with 3rd dimension = number of albedo bands
  real, intent(inout), allocatable :: rPSWdf(:,:,:)
  real, intent(inout), allocatable :: rPSWdr(:,:,:)
  
  ! 2D outputs (surface/TOA fluxes) as a single 3D array
  real, intent(inout), allocatable :: rFLopt(:,:,:)

! +--Internal variables (for use in ecRad)
! +  =====================================

! +  The next variables are tailored for one longitudinal slice (latitude by pressure levels, 2D).
! +  Indeed, ecRad is run on one longitudinal slice at a time, with a loop going through the 
! +  slices (the operations for a single slice are parallelized).
! +
! +  Remarks:
! +  -the uppercase names are meant to help the reader figure which variable is MAR input and 
! +   which variable is for ecRad.
! +  -the "(kind=jprb)" will automatically select simple or double precision depending on the 
! +   compilation parameters of ecRad (simple precision by default).
! +  -since the number of albedo bands may be controllable later via the configuration file, all 
! +   variables involving this number are allocatable.
! +  -starting from 09/02/2023, all variables with pressure levels were made allocatable to add 
! +   the "blanket" feature, which consists of adding (usually 3) additional pressure levels on 
! +   top of the MAR grid to model the stratosphere, and in particular, to improve the ozone 
! +   absorption of UV radiation by having the ozone peak in the model. This is meant to produce 
! +   better spectral fluxes and slightly reduce radiative fluxes biases in the shortwave.

  real(kind=jprb) :: PGELAM5(dimLat)
  real(kind=jprb) :: PGEMU5(dimLat)
  
  integer(kind=jpim) :: KIDIA, KFDIA, KLON, KLEV, KAER
  integer :: TIME_SEED

  real(kind=jprb) :: PRII0
  real(kind=jprb), dimension(:,:), allocatable :: PALBD
  real(kind=jprb), dimension(:,:), allocatable :: PALBP
  real(kind=jprb) :: PCCN_LAND(dimLat) ! Unused because LCCNL=.F. (for now, see ifsrrtm/yoerad.F90)
  real(kind=jprb) :: PCCN_SEA(dimLat) ! Unused because LCCNO=.F. (for now, see ifsrrtm/yoerad.F90)

  ! Pressure variables
  real(kind=jprb), dimension(:,:), allocatable :: PAPH
  real(kind=jprb), dimension(:,:), allocatable :: PAP

  ! Surface variables
  real(kind=jprb) :: PEMIS(dimLat)
  real(kind=jprb) :: PEMIW(dimLat)
  real(kind=jprb) :: PLSM(dimLat)
  real(kind=jprb) :: PMU0(dimLat)
  
  ! Water species mixing ratios
  real(kind=jprb), dimension(:,:), allocatable :: PQ
  real(kind=jprb), dimension(:,:), allocatable :: PCLFR
  real(kind=jprb), dimension(:,:), allocatable :: PQLWP ! Dropplets concentration
  real(kind=jprb), dimension(:,:), allocatable :: PQIWP
  real(kind=jprb), dimension(:,:), allocatable :: PQS
  real(kind=jprb), dimension(:,:), allocatable :: PQRAIN
  real(kind=jprb), dimension(:,:), allocatable :: PQSNOW ! Via new argument rqsRAD
  
  ! Temperature variables
  real(kind=jprb), dimension(:,:), allocatable :: PTH
  real(kind=jprb), dimension(:,:), allocatable :: PT 
  real(kind=jprb) :: PTS(dimLat)
  
  ! Outputs: main fluxes with vertical levels
  real(kind=jprb), dimension(:,:), allocatable :: PFLUX_SW
  real(kind=jprb), dimension(:,:), allocatable :: PFLUX_LW
  real(kind=jprb), dimension(:,:), allocatable :: PFLUX_SW_CLEAR
  real(kind=jprb), dimension(:,:), allocatable :: PFLUX_LW_CLEAR
  real(kind=jprb), dimension(:,:), allocatable :: PLWDERIVATIVE
  
  ! Outputs: spectral decomposition of surface solar fluxes w.r.t. albedo bands
  real(kind=jprb), dimension(:,:), allocatable :: PSWDIFFUSEBAND
  real(kind=jprb), dimension(:,:), allocatable :: PSWDIRECTBAND
  
  ! Outputs: spectral decomposition of surface solar fluxes (user-defined or defined internally)
  real(kind=jprb), dimension(:,:), allocatable :: PSW_SPECTRAL
  real(kind=jprb), dimension(:,:), allocatable :: PSW_SPECTRAL_DIR
  real(kind=jprb), dimension(:,:), allocatable :: PSW_SPECTRAL_RAW
  real(kind=jprb), dimension(:,:), allocatable :: PSW_SPECTRAL_RAW_DIR
  
  ! Remark (J.-F. Grailet): the above output variables are only generated on request by the user. 
  ! As their dimensions depend on configuration, they are allocated within RADIATION_SCHEME.
  
  ! Additional variables for storing the 3D spectral data (RADIATION_SCHEME gives 2D slices)
  real(kind=jprb), dimension(:,:,:), allocatable :: SW_SPECTRAL
  real(kind=jprb), dimension(:,:,:), allocatable :: SW_SPECTRAL_DIR
  real(kind=jprb), dimension(:,:,:), allocatable :: SW_SPECTRAL_RAW
  real(kind=jprb), dimension(:,:,:), allocatable :: SW_SPECTRAL_RAW_DIR
  
  ! Outputs: additional fluxes in 2D (no vertical levels)
  real(kind=jprb) :: PFLUX_SW_DN(dimLat)
  real(kind=jprb) :: PFLUX_LW_DN(dimLat)
  real(kind=jprb) :: PFLUX_SW_DN_CLEAR(dimLat)
  real(kind=jprb) :: PFLUX_LW_DN_CLEAR(dimLat)
  real(kind=jprb) :: PFLUX_DIR(dimLat)
  real(kind=jprb) :: PFLUX_DIR_CLEAR(dimLat)
  real(kind=jprb) :: PFLUX_DIR_INTO_SUN(dimLat)
  real(kind=jprb) :: PFLUX_UV(dimLat)
  real(kind=jprb) :: PFLUX_PAR(dimLat)
  real(kind=jprb) :: PFLUX_PAR_CLEAR(dimLat)
  real(kind=jprb) :: PFLUX_SW_DN_TOA(dimLat)
  real(kind=jprb) :: PEMIS_OUT(dimLat)
  
  ! Variable used to store the additional fluxes from above
  real(kind=jprb), dimension(:,:,:), allocatable :: PFLUX_EXTRA

! +--Additional local variables
! +  ==========================

  ! Iterators (i = longitude, j = latitude, k = level, l = aerosol, m = SW band)
  integer :: i, j, k, l, m
  
  ! Local variables to manage the blanket layers
  integer :: nbStrato
  real :: tropopause_temp1, tropopause_temp2 ! First for PTH, second for PT
  real :: strato_step1, strato_step2 ! Ditto
  
  ! Local variables used for cloud fraction parametrization
  real(kind=jprb) :: rel_humidity
  real(kind=jprb) :: q_water_total
  real(kind=jprb) :: cloud_frac_calc
  real(kind=jprb), parameter :: max_arg_exp = log(0.1e+38) ! argmax = log(rrmax) in mar.f90
  
  ! Local variable used to record only diurnal spectral data with some settings (see end)
  logical :: do_store_spectral_data

! +--Inclusions
! +  ==========

#include "radiation_scheme.intfb.h"

! +--Dimensions and others
! +  =====================

! +  J.-F. Grailet: with the exception of NSW, all the next vars could have been replaced with 
! +  their associated value in practice. They were not in case a different way of calling the 
! +  RADIATION_SCHEME subroutine was envisioned at some point (also, guarantees all code preparing 
! +  the call to RADIATION_SCHEME is in uppercase, contrary to args which are in camel case).

  KIDIA = 1        ! First column to process (DON'T CHANGE)
  KFDIA = dimLat   ! Last column to process (DON'T CHANGE)
  KLON = dimLat    ! Number of columns (longitudinal slice)
  KLEV = nbLvls    ! Number of levels (longitudinal slice)
  
  ! KLEV is increased if blanket layers are used
  nbStrato = 0
  if (allocated(BLANKET_LAYERS)) then
    nbStrato = size(BLANKET_LAYERS)
    KLEV = KLEV + nbStrato
  end if
  
  ! Number of aerosols depends on which type of aerosols climatology has been picked
  if (NAERMACC > 0) then
    KAER = N_AEROSOLS_MACC
  else
    KAER = N_AEROSOLS_TEGEN
  end if
  
  TIME_SEED = dd + (irhhss / 60) ! Time seed for McICA solver

! +--Radiation on MAR grid
! +  =====================

  ! Allocates the variables with levels (number of levels changes if blanket layers are used)
  allocate(PAPH(dimLat, nbStrato + nbLvls + 1))
  allocate(PAP(dimLat, nbStrato + nbLvls))
  
  allocate(PQ(dimLat, nbStrato + nbLvls))
  allocate(PCLFR(dimLat, nbStrato + nbLvls))
  allocate(PQLWP(dimLat, nbStrato + nbLvls))
  allocate(PQIWP(dimLat, nbStrato + nbLvls))
  allocate(PQS(dimLat, nbStrato + nbLvls))
  allocate(PQRAIN(dimLat, nbStrato + nbLvls))
  allocate(PQSNOW(dimLat, nbStrato + nbLvls))
  
  allocate(PTH(dimLat, nbStrato + nbLvls + 1))
  allocate(PT(dimLat, nbStrato + nbLvls))
  
  allocate(PFLUX_SW(dimLat, nbStrato + nbLvls + 1))
  allocate(PFLUX_LW(dimLat, nbStrato + nbLvls + 1))
  allocate(PFLUX_SW_CLEAR(dimLat, nbStrato + nbLvls + 1))
  allocate(PFLUX_LW_CLEAR(dimLat, nbStrato + nbLvls + 1))
  allocate(PLWDERIVATIVE(dimLat, nbStrato + nbLvls + 1))

  ! Allocates the variables involving albedo bands
  allocate(PALBD(dimLat, NSW))
  allocate(PALBP(dimLat, NSW))
  allocate(PSWDIFFUSEBAND(dimLat, NSW))
  allocate(PSWDIRECTBAND(dimLat, NSW))
  
  ! Allocates the full output variables (N.B.: blanket layers removed)
  allocate(rFIRnc(dimLon, dimLat, nbLvls + 1))
  allocate(rFIRnt(dimLon, dimLat, nbLvls + 1))
  allocate(rFSOnc(dimLon, dimLat, nbLvls + 1))
  allocate(rFSOnt(dimLon, dimLat, nbLvls + 1))
  allocate(rPLWdv(dimLon, dimLat, nbLvls + 1))

  allocate(rPSWdf(dimLon, dimLat, NSW))
  allocate(rPSWdr(dimLon, dimLat, NSW))

  allocate(rFLopt(dimLon, dimLat, N_EXTRA_OUTPUTS))
  
  ! Allocates the full variables for spectral decomposition (if requested)
  if (N_SW_SPECTRAL_BANDS > 0) then
    allocate(SW_SPECTRAL(dimLon, dimLat, N_SW_SPECTRAL_BANDS))
    allocate(SW_SPECTRAL_DIR(dimLon, dimLat, N_SW_SPECTRAL_BANDS))
  end if
  
  if (N_ECRAD_SW_BANDS > 0) then
    allocate(SW_SPECTRAL_RAW(dimLon, dimLat, N_ECRAD_SW_BANDS))
    allocate(SW_SPECTRAL_RAW_DIR(dimLon, dimLat, N_ECRAD_SW_BANDS))
  end if
  
  if (L_EXTRA) then
    allocate(PFLUX_EXTRA(dimLon, dimLat, N_EXTRA_OUTPUTS))
  end if
  
  ! Insolation (one value for the whole grid) (XF: rDiST is defined in Phyrad_top)
  PRII0 = (RI0 * RI0_SCALING) / (rDiST * rDiST)

  ! For each longitudinal slice
  do i=1, dimLon

    ! Setting longitude and latitude variables
    do j=1, KLON
      PGELAM5(j) = rLoRAD(i, j)
      PGEMU5(j) = SIN(rLaRAD(i, j))
    end do

! +--Radiation: global (time dependant) parameters
! +  =============================================

    ! Surface albedo
    do m=1, NSW; do j=1, KLON
      PALBD(j,m) = rAlbe(i, j)
      PALBP(j,m) = rAlbe(i, j) ! JFG: same as PALBD --> change this later ?
    end do; end do

    do j=1, KLON

      ! Surface emissivity
      PEMIS(j) = emsRAD(i, j)
      PEMIW(j) = emsRAD(i, j) ! JFG: same as PEMIS --> change this later ?

      ! Land/sea mask
      PLSM(j) = lsmRAD(i, j)

      ! Cosine (solar zenithal angle)
      PMU0(j) = cszRAD(i, j)

    end do

! +--Atmospheric thermodynamics (time and space dependant)
! +  =====================================================

    ! Final pressure level (surface)
    k = nbLvls + 1
    do j=1, KLON
      PAPH(j, nbStrato + k) = rphRAD(i, j, k)
    end do
    
    ! Directive to slightly speed up cloud fraction parametrization (Sundqvist or Xu & Randall)
    !$OMP PARALLEL DO PRIVATE(k, j, rel_humidity, q_water_total, cloud_frac_calc) SCHEDULE(RUNTIME)
    do k=1, nbLvls; do j=1, KLON

      ! Pressure distribution !XF 13/02/2020 
      PAPH(j, nbStrato + k) = rphRAD(i, j, k)
      PAP(j, nbStrato + k) = rp_RAD(i, j, k)
      
      ! Temperature distribution
      PT(j, nbStrato + k) = rtaRAD(i, j, k)
      PTH(j, nbStrato + k) = (rtaRAD(i, j, k) + rtaRAD(i, j, max(1, k - 1))) * 0.5

      ! Water species distributions
      PQ(j, nbStrato + k) = rqvRAD(i, j, k)
      PQIWP(j, nbStrato + k) = max(0., rqiRAD(i, j, k) - 1.e-9)
      PQLWP(j, nbStrato + k) = max(0., rqwRAD(i, j, k) - 1.e-9)
      PQS(j, nbStrato + k) = rswRAD(i, j, k)
      PQRAIN(j, nbStrato + k) = rqrRAD(i, j, k)
      PQSNOW(j, nbStrato + k) = rqsRAD(i, j, k)
      
      ! JFG 13/03/2023: adjustment of water species mixing ratios
      if (SCALING_PQ /= 1.) then
        PQ(j, nbStrato + k) = PQ(j, nbStrato + k) * SCALING_PQ
      end if

      if (SCALING_PQIWP /= 1.) then
        PQIWP(j, nbStrato + k) = PQIWP(j, nbStrato + k) * SCALING_PQIWP
      end if

      if (SCALING_PQLWP /= 1.) then
        PQLWP(j, nbStrato + k) = PQLWP(j, nbStrato + k) * SCALING_PQLWP
      end if

      if (SCALING_PQRAIN /= 1.) then
        PQRAIN(j, nbStrato + k) = PQRAIN(j, nbStrato + k) * SCALING_PQRAIN
      end if

      if (SCALING_PQSNOW /= 1.) then
        PQSNOW(j, nbStrato + k) = PQSNOW(j, nbStrato + k) * SCALING_PQSNOW
      end if

      ! Possible cloud fraction parametrizations (April 2023):
      ! 1) Same values as computed in the MAR (hydmic.f90)
      ! 2) ECMWF Large Scale Cloudiness, crude version (without snowflakes)
      ! 3) ECMWF Large Scale Cloudiness, full version (same formula as hydmic.f90)
      ! 4) Sundqvist '89, "Condensation and Cloud Parameterization Studies with a Mesoscale [...]"
      ! 5) Xu & Randall '96, "A Semiempirical Cloudiness Parametrization for Use in Climate Models"
      
      ! ECMWF crude (reused from PHYrad2CEP.F90)
      if (NCLOUD_PARAM == 1) then
        PCLFR(j, nbStrato + k) = (PQIWP(j, nbStrato + k) + PQLWP(j, nbStrato + k)) & 
         &                        / (RCLOUD_PARAM_GAMMA * PQS(j, nbStrato + k)) !XF
      ! ECMWF full (derived from code in hydmic.f90)
      else if (NCLOUD_PARAM == 2) then
        PCLFR(j, nbStrato + k) = (PQIWP(j, nbStrato + k) + PQLWP(j, nbStrato + k) & 
         &                        + PQSNOW(j, nbStrato + k) * 0.33 &
         &                        * (1. - min(1., exp((PT(j, nbStrato + k) - 258.15) * 0.1)))) &
         &                        / (RCLOUD_PARAM_GAMMA * PQS(j, nbStrato + k))
      ! Sundqvist '89 (added by J.-F. Grailet)
      else if (NCLOUD_PARAM == 3) then
        rel_humidity = min(1., max(PQ(j, nbStrato + k), 3.e-6) / PQS(j, nbStrato + k))
        
        cloud_frac_calc = 1 - rel_humidity
        if (PLSM(j) == 1.) then
          cloud_frac_calc = cloud_frac_calc / (1. - RCLOUD_RH_THRESHOLD_LAND)
        else
          cloud_frac_calc = cloud_frac_calc / (1. - RCLOUD_RH_THRESHOLD_SEA)
        end if
        
        PCLFR(j, nbStrato + k) = 1. - sqrt(cloud_frac_calc)
      ! Xu & Randall '96 (derived from code in hydmic.f90)
      else if (NCLOUD_PARAM == 4) then
        rel_humidity = min(1., max(PQ(j, nbStrato + k), 3.e-6) / PQS(j, nbStrato + k))
        
        q_water_total = (PQIWP(j, nbStrato + k) + PQLWP(j, nbStrato + k) & 
         &               + PQSNOW(j, nbStrato + k) * 0.33 &
         &               * (1. - min(1., exp((PT(j, nbStrato + k) - 258.15) * 0.1))))
        
        cloud_frac_calc = ((1. - rel_humidity) * PQS(j, nbStrato + k)) ** RCLOUD_PARAM_GAMMA
        cloud_frac_calc = (RCLOUD_PARAM_ALPHA * q_water_total) / max(1.0e-9, cloud_frac_calc)
        cloud_frac_calc = min(cloud_frac_calc, max_arg_exp)
        
        PCLFR(j, nbStrato + k) = (rel_humidity ** RCLOUD_PARAM_P) * (1. - exp(-cloud_frac_calc))
      ! Reuse values computed by hydmic.f90
      else
        PCLFR(j, nbStrato + k) = rcdRAD(i, j, k) !XF Droplets
      end if
      
      ! Restricts cloud fraction values to the [0.001, 1] range
      PCLFR(j, nbStrato + k) = min(1.0, PCLFR(j, nbStrato + k))
      PCLFR(j, nbStrato + k) = max(1.0e-3, PCLFR(j, nbStrato + k)) & ! No small values
       &                        * max(0., sign(1.0, rqiRAD(i, j, k) + rqwRAD(i, j, k) - 2.e-9))

      ! Sends back the recomputed cloud fraction to MAR
      rcdRAD(i, j, k) = PCLFR(j, nbStrato + k)

    end do; end do
    !$OMP END PARALLEL DO
    
    do j=1, KLON
      PTH(j, nbStrato + nbLvls + 1) = rtsRAD(i, j)
      PTS(j) = rtsRAD(i, j)
    end do

! +--Blanket layers (stratosphere)
! +  =============================

    if (nbStrato > 0) then
    
      ! Additional pressure levels
      do k=1, nbStrato; do j=1, KLON
        PAP(j, k) = BLANKET_LAYERS(k)
      end do; end do
      
      PAPH(j, 1) = 0. ! Suggested by R. J. Hogan 23/11/2023
      ! From 2 to nbStrato + 1 because final MAR half-level is set to 10. (must be replaced)
      do k=2, nbStrato + 1; do j=1, KLON
        PAPH(j, k) = (PAP(j, k - 1) + PAP(j, k)) * 0.5
      end do; end do
      
      ! Specific humidity, cloud cover and water species mixing ratios
      do k=1, nbStrato; do j=1, KLON
        PQ(j, k) = 3.1e-6 ! Paper on CKDMIP by Hogan and Matricardi, 2020 ! max(PQ(j, nbStrato+1),)
        PCLFR(j, k) = 0.
        PQLWP(j, k) = 0.
        PQIWP(j, k) = 0.
        PQS(j, k) = 0.0038 ! To simplify, for all blanket levels, mixing ratio for 0°C = 3.8 g/kg
        PQRAIN(j, k) = 0.
        PQSNOW(j, k) = 0.
      end do; end do
      
      ! Temperature is a bit trickier: the probable tropopause (i.e. first pressure level in the 
      ! column after which temperature rises) must be found, then the temperature of the first 
      ! stratospheric level (stratopause) is set as the temperature of this probable tropopause 
      ! +60°C (based on the CKDMIP paper by Hogan and Matricardi, 2020), and the additional levels 
      ! are linearly interpolated.
      
      do j=1, KLON
        
        ! Finds plausible tropopause and infers stratopause
        tropopause_temp1 = PTH(j, nbStrato + 1)
        do k=2, 4 ! To avoid going to low in the column
          if (PTH(j, nbStrato + k) < tropopause_temp1) then
            tropopause_temp1 = PTH(j, nbStrato + k)
          else
            exit
          end if
        end do
        
        tropopause_temp2 = PT(j, nbStrato + 1)
        do k=2, 4
          if (PT(j, nbStrato + k) < tropopause_temp2) then
            tropopause_temp2 = PT(j, nbStrato + k)
          else
            exit
          end if
        end do
        
        PTH(j, 1) = tropopause_temp1 + 60
        PT(j, 1) = tropopause_temp2 + 60
        
        ! Interpolates crudely the intermediate levels
        strato_step1 = (PTH(j, 1) - PTH(j, nbStrato + 1)) / nbStrato
        strato_step2 = (PT(j, 1) - PT(j, nbStrato + 1)) / nbStrato
        
        do k=2, nbStrato
          PTH(j, k) = PTH(j, k - 1) - strato_step1
          PT(j, k) = PT(j, k - 1) - strato_step2
        end do
        
      end do

    end if

! +--Solar and IR transfer through the atmosphere
! +  ============================================

! +  J.-F. Grailet remarks (October 2022):
! +  -the list of arguments mimicks the list of parameters found in drivers/radiation_scheme.F90, 
! +  -the parallelization of the workload (on one longitudinal slice) is carried out within the 
! +   RADIATION_SCHEME routine itself for the time being,
! +  -also for the time being, KIDIA is always set to 1 and KFDIA always set to KLON (full 
! +   longitudinal slice), but if necessary, this could be changed in the future.

    call RADIATION_SCHEME &
      & (KIDIA, KFDIA, KLON, KLEV, KAER, &
      &  TIME_SEED, &
      &  PRII0, &
      &  PMU0 , PTS, PALBD, PALBP, &
      &  PEMIS, PEMIW, &
      &  PCCN_LAND, PCCN_SEA, & ! Only "fills the gaps" for now (as LCCNL and LCCNO are false)
      &  PGELAM5, PGEMU5, PLSM, &
      &  PAP, PT, &
      &  PAPH, PTH, &
      &  PQ, PCO2, PCH4, PN2O, PCFC11, PCFC12, PHCFC22, PCCL4, PO3, &
      &  PCLFR, PQLWP, PQIWP, PQRAIN, PQSNOW, &
      &  PAEROSOL_OLD, PAEROSOL, &
      &  PFLUX_SW, PFLUX_LW, PFLUX_SW_CLEAR, PFLUX_LW_CLEAR, &
      &  PFLUX_SW_DN, PFLUX_LW_DN, PFLUX_SW_DN_CLEAR, PFLUX_LW_DN_CLEAR, &
      &  PFLUX_DIR, PFLUX_DIR_CLEAR, PFLUX_DIR_INTO_SUN, &
      &  PFLUX_UV, PFLUX_PAR, PFLUX_PAR_CLEAR, &
      &  PFLUX_SW_DN_TOA, PEMIS_OUT, PLWDERIVATIVE, &
      &  PSWDIFFUSEBAND, PSWDIRECTBAND, & 
      &  PSW_SPECTRAL, PSW_SPECTRAL_DIR, & ! These are allocated within (optional)
      &  PSW_SPECTRAL_RAW, PSW_SPECTRAL_RAW_DIR) ! Ditto

! +--Radiative fluxes distributions
! +  ==============================

    ! The type casting is relevant when double precision is used in ecRad
    do j=1, KLON; do k=1, nbLvls + 1
      rFIRnc(i, j, k) = real(PFLUX_LW_CLEAR(j, nbStrato + k))  ! CLEAR-SKY LW NET FLUXES
      rFIRnt(i, j, k) = real(PFLUX_LW(j, nbStrato + k))  ! TOTAL-SKY LW NET FLUXES
      rFSOnc(i, j, k) = real(PFLUX_SW_CLEAR(j, nbStrato + k))  ! TOTAL-SKY SW NET FLUXES
      rFSOnt(i, j, k) = real(PFLUX_SW(j, nbStrato + k))  ! TOTAL-SKY SW NET FLUXES
      rPLWdv(i, j, k) = real(PLWDERIVATIVE(j, nbStrato + k))
    end do; end do

! +--Spectral decomposition of surface solar fluxes w.r.t. albedo bands
! +  ==================================================================

    do j=1, KLON; do k=1, NSW
      rPSWdf(i, j, k) = real(PSWDIFFUSEBAND(j,k))
      rPSWdr(i, j, k) = real(PSWDIRECTBAND(j,k))
    end do; end do
    
! +--Extra output variables for surface/TOA fluxes
! +  =============================================

    ! Extra surface fluxes (some are used, such as the first slice)
    do j=1, KLON
      rFLopt(i, j, ID_PFLUX_SW_DN) = real(PFLUX_SW_DN(j))
      rFLopt(i, j, ID_PFLUX_LW_DN) = real(PFLUX_LW_DN(j))
      rFLopt(i, j, ID_PFLUX_SW_DN_CLEAR) = real(PFLUX_SW_DN_CLEAR(j))
      rFLopt(i, j, ID_PFLUX_LW_DN_CLEAR) = real(PFLUX_LW_DN_CLEAR(j))
      rFLopt(i, j, ID_PFLUX_DIR) = real(PFLUX_DIR(j))
      rFLopt(i, j, ID_PFLUX_DIR_CLEAR) = real(PFLUX_DIR_CLEAR(j))
      rFLopt(i, j, ID_PFLUX_DIR_INTO_SUN) = real(PFLUX_DIR_INTO_SUN(j))
      rFLopt(i, j, ID_PFLUX_UV) = real(PFLUX_UV(j))
      rFLopt(i, j, ID_PFLUX_PAR) = real(PFLUX_PAR(j))
      rFLopt(i, j, ID_PFLUX_PAR_CLEAR) = real(PFLUX_PAR_CLEAR(j))
      rFLopt(i, j, ID_PFLUX_SW_DN_TOA) = real(PFLUX_SW_DN_TOA(j))
      rFLopt(i, j, ID_PEMIS_OUT) = real(PEMIS_OUT(j))
    end do
    
    ! Extra fluxes in their initial precision for saving them in a side NetCDF file
    if (allocated(PFLUX_EXTRA)) then
      do j=1, KLON
        PFLUX_EXTRA(i, j, ID_PFLUX_SW_DN) = PFLUX_SW_DN(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_LW_DN) = PFLUX_LW_DN(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_SW_DN_CLEAR) = PFLUX_SW_DN_CLEAR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_LW_DN_CLEAR) = PFLUX_LW_DN_CLEAR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_DIR) = PFLUX_DIR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_DIR_CLEAR) = PFLUX_DIR_CLEAR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_DIR_INTO_SUN) = PFLUX_DIR_INTO_SUN(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_UV) = PFLUX_UV(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_PAR) = PFLUX_PAR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_PAR_CLEAR) = PFLUX_PAR_CLEAR(j)
        PFLUX_EXTRA(i, j, ID_PFLUX_SW_DN_TOA) = PFLUX_SW_DN_TOA(j)
        PFLUX_EXTRA(i, j, ID_PEMIS_OUT) = PEMIS_OUT(j)
      end do
    end if
    
! +--Spectral decomposition of surface solar downward fluxes (optional)
! +  ==================================================================
    
    if (allocated(PSW_SPECTRAL)) then ! Equivalent to N_SW_SPECTRAL_BANDS > 0
      do j=1, KLON; do k=1, N_SW_SPECTRAL_BANDS
        SW_SPECTRAL(i, j, k) = PSW_SPECTRAL(k, j)
        SW_SPECTRAL_DIR(i, j, k) = PSW_SPECTRAL_DIR(k, j)
      end do; end do
      deallocate(PSW_SPECTRAL)
      deallocate(PSW_SPECTRAL_DIR)
    end if
    
    if (allocated(PSW_SPECTRAL_RAW)) then
      do j=1, KLON; do k=1, N_ECRAD_SW_BANDS
        SW_SPECTRAL_RAW(i, j, k) = PSW_SPECTRAL_RAW(k, j)
        SW_SPECTRAL_RAW_DIR(i, j, k) = PSW_SPECTRAL_RAW_DIR(k, j)
      end do; end do
      deallocate(PSW_SPECTRAL_RAW)
      deallocate(PSW_SPECTRAL_RAW_DIR)
    end if
  
  end do ! End for radiation simulated on one longitudinal slice
  
  ! Shortwave radiation check (14/12/2022): to avoid storing useless data (i.e. grids filled with 
  ! null values, corresponding to night time), the content of the first level of rFLopt (optional 
  ! fluxes) is reviewed to see if there is some surface solar radiation anywhere in the grid. 
  ! Indeed, this level corresponds to the (total-sky, net) surface solar downward fluxes. If such 
  ! a level only consists in null values w.r.t. shortwave fluxes, storing spectral data may be 
  ! useless. This verification is done here because the RADIATION_SCHEME subroutine is called on a 
  ! single longitudinal slice, which may have no shortwave flux at sunrise/sunset. 
  ! 
  ! As of 05/01/2023, this check is done when L_SW_SPECTRAL_PADDING or L_SW_SPECTRAL_RAW_PADDING 
  ! are false. Otherwise, full day hourly spectral data is saved. NetCDF compression may help.
  
  do_store_spectral_data = .false.
  if (N_SW_SPECTRAL_BANDS > 0 .or. N_ECRAD_SW_BANDS > 0) then
    if (L_SW_SPECTRAL_PADDING .and. L_SW_SPECTRAL_RAW_PADDING) then
      do_store_spectral_data = .true.
    else
      do i=1, dimLon; do j=1, dimLat
        if (rFLopt(i, j, ID_PFLUX_SW_DN) /= 0.) then
          do_store_spectral_data = .true.
          exit
        end if
      end do; end do
    end if
  end if
  
  ! Outputs the spectral decomposition, if requested and needed
  if (N_SW_SPECTRAL_BANDS > 0) then
    if (L_SW_SPECTRAL_PADDING .or. do_store_spectral_data) then
      call append_sw_spectral_file(SW_SPECTRAL, SW_SPECTRAL_DIR, yyyy, mm, dd, irhhss)
    end if
  end if
  if (N_ECRAD_SW_BANDS > 0) then
    if (L_SW_SPECTRAL_RAW_PADDING .or. do_store_spectral_data) then
      call append_sw_spectral_raw_file(SW_SPECTRAL_RAW, SW_SPECTRAL_RAW_DIR, yyyy, mm, dd, irhhss)
    end if
  end if
  
  ! Frees spectral data grids, regardless of whether or not they were recorded
  if (N_SW_SPECTRAL_BANDS > 0) then
    deallocate(SW_SPECTRAL)
    deallocate(SW_SPECTRAL_DIR)
  end if
  if (N_ECRAD_SW_BANDS > 0) then
    deallocate(SW_SPECTRAL_RAW)
    deallocate(SW_SPECTRAL_RAW_DIR)
  end if
  
  ! Outputs the extra fluxes, if previously requested (PFLUX_EXTRA will be allocated)
  if (allocated(PFLUX_EXTRA)) then
    call append_extra_fluxes_file(PFLUX_EXTRA, yyyy, mm, dd, irhhss)
    
    deallocate(PFLUX_EXTRA)
  end if
  
  ! Frees the variables for ecRad whose dimensions included the number of levels
  deallocate(PAPH) 
  deallocate(PAP)
  
  deallocate(PQ)
  deallocate(PCLFR)
  deallocate(PQLWP)
  deallocate(PQIWP)
  deallocate(PQS)
  deallocate(PQRAIN)
  deallocate(PQSNOW)
  
  deallocate(PTH)
  deallocate(PT)
  
  deallocate(PFLUX_SW)
  deallocate(PFLUX_LW)
  deallocate(PFLUX_SW_CLEAR)
  deallocate(PFLUX_LW_CLEAR)
  deallocate(PLWDERIVATIVE)
  
  ! Frees the variables whose dimensions depended on albedo bands
  deallocate(PALBD)
  deallocate(PALBP)
  deallocate(PSWDIFFUSEBAND)
  deallocate(PSWDIRECTBAND)

  return
end subroutine PHYrad_ecRad
