#include "MAR_pp.def"
subroutine DYNfil_3D_mp2(f3_fil, v3_fil, e3_fil, k3_fil)
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS FILTER (3-D)                           07-09-2017-XF  MAR |
    ! |   subroutine DYNfil_3D_mp is used to Filter Horizontal Fields in 3DCode|
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT:   f3_fil(i,j,k): variable to be filtered (in surface k)       |
    ! |   ^^^^^    e3_fil(    k): value of the selectivity parameter           |
    ! |            k3_fil       : vertical dimension of the variable           |
    ! |                                                                        |
    ! |   OUTPUT:  f3_fil(i,j,k)                                               |
    ! |   ^^^^^^^                                                              |
    ! |                                                                        |
    ! |   LATERAL BOUNDARIES:                                                  |
    ! |   ^^^^^^^^^^^^^^^^^^^                                                  |
    ! |   1. The value    of the variable is fixed at the Boundary             |
    ! |   2. Filter Selectivity Parameter may be increased near the Boundary   |
    ! |                                                                (#EP)   |
    ! |                                                                        |
    ! |   REFER. : Raymond and Garder, MWR 116, Jan 1988, p209                 |
    ! |   ^^^^^^^^                                                             |
    ! +------------------------------------------------------------------------+

    use marphy
    use mardim
    use margrd
    use mar_wk

    implicit none

    integer i, j, k, m
    real f3_fil(mx, my, mz), e3_fil(mz), v3_fil(mx, my, mz)
    integer k3_fil

    real WVxy4(mx, my), WVxy6(mx, my), WVxy7(mx, my)

    real eps2(mz)
    real eps3(mz)
    real eps4(mz)
    real eps5(mz)
    real a1_fil(1:mx, mz), b1_fil(1:mx, mz), aa_fil(mz)
    real a2_fil(1:my, mz), b2_fil(1:my, mz), bb_fil(mz)

    ! +--Initialisation
    ! +  ==============

    m = mx
    m1 = mx1
    m2 = mx2
#ifdef EP
    m3 = mx - 3
    m4 = mx - 4
#endif
    mn = my
    mn1 = my1
    mn2 = my2
#ifdef EP
    mn3 = my - 3
    mn4 = my - 4
#endif

    ! +--1st Matrix Initialisation
    ! +  -------------------------

!$OMP PARALLEL do default(shared) &
!$OMP private(a1_fil,b1_fil,a2_fil,b2_fil, &
!$OMP       WTxy4,WTxy6,WTxy7,i,j,k,WVxy4,WVxy6,WVxy7)
    do k = 1, k3_fil
        a1_fil(1, k) = 0.0
        b1_fil(1, k) = 1.0
        a1_fil(mx, k) = 0.0
        b1_fil(mx, k) = 1.0
#ifdef EP
        ! Pour diminution de la sectivite du filtre vers les bords
        !   (augmentation --X 2-- parametre selectivite a chaque pas spatial)
        eps5(k) = e3_fil(k) + e3_fil(k)
        eps4(k) = eps5(k) + eps5(k)
        eps3(k) = eps4(k) + eps4(k)
        eps2(k) = eps3(k) + eps3(k)
#endif

        aa_fil(k) = 1.0 - e3_fil(k)
        bb_fil(k) = 2.0 * (1.0 + e3_fil(k))

        do i = ip11, mx1
            a1_fil(i, k) = aa_fil(k)
            b1_fil(i, k) = bb_fil(k)
        enddo
#ifdef EP
        a1_fil(2, k) = 1.0 - eps2(k)
        a1_fil(3, k) = 1.0 - eps3(k)
        a1_fil(4, k) = 1.0 - eps4(k)
        a1_fil(5, k) = 1.0 - eps5(k)
        b1_fil(2, k) = 2.0 * (1.0 + eps2(k))
        b1_fil(3, k) = 2.0 * (1.0 + eps3(k))
        b1_fil(4, k) = 2.0 * (1.0 + eps4(k))
        b1_fil(5, k) = 2.0 * (1.0 + eps5(k))
        a1_fil(m1, k) = a1_fil(2, k)
        a1_fil(m2, k) = a1_fil(3, k)
        a1_fil(m3, k) = a1_fil(4, k)
        a1_fil(m4, k) = a1_fil(5, k)
        b1_fil(m1, k) = b1_fil(2, k)
        b1_fil(m2, k) = b1_fil(3, k)
        b1_fil(m3, k) = b1_fil(4, k)
        b1_fil(m4, k) = b1_fil(5, k)
#endif

        ! +--2th Matrix Initialisation
        ! +  -------------------------

        a2_fil(1, k) = 0.0
        b2_fil(1, k) = 1.0
        a2_fil(my, k) = 0.0
        b2_fil(my, k) = 1.0

        do j = jp11, my1
            a2_fil(j, k) = aa_fil(k)
            b2_fil(j, k) = bb_fil(k)
        enddo
#ifdef EP
        a2_fil(2, k) = a1_fil(2, k)
        a2_fil(3, k) = a1_fil(3, k)
        a2_fil(4, k) = a1_fil(4, k)
        a2_fil(5, k) = a1_fil(5, k)
        b2_fil(2, k) = b1_fil(2, k)
        b2_fil(3, k) = b1_fil(3, k)
        b2_fil(4, k) = b1_fil(4, k)
        b2_fil(5, k) = b1_fil(5, k)
        a2_fil(mn1, k) = a1_fil(2, k)
        a2_fil(mn2, k) = a1_fil(3, k)
        a2_fil(mn3, k) = a1_fil(4, k)
        a2_fil(mn4, k) = a1_fil(5, k)
        b2_fil(mn1, k) = b1_fil(2, k)
        b2_fil(mn2, k) = b1_fil(3, k)
        b2_fil(mn3, k) = b1_fil(4, k)
        b2_fil(mn4, k) = b1_fil(5, k)
#endif

        !      end do

        ! +--1st Equations System
        ! +  ====================

        ! +--Gaussian Elimination Algorithm: Set Up
        ! +  --------------------------------------

        !      do k=1,k3_fil
        do j = jp11, my1

            WTxy4(1, j) = &
                f3_fil(1, jm1(j), k) + 2.0 * f3_fil(1, j, k) + f3_fil(1, jp1(j), k)
            WTxy4(mx, j) = &
                f3_fil(mx, jm1(j), k) + 2.0 * f3_fil(mx, j, k) + f3_fil(mx, jp1(j), k)

            WVxy4(1, j) = &
                v3_fil(1, jm1(j), k) + 2.0 * v3_fil(1, j, k) + v3_fil(1, jp1(j), k)
            WVxy4(mx, j) = &
                v3_fil(mx, jm1(j), k) + 2.0 * v3_fil(mx, j, k) + v3_fil(mx, jp1(j), k)

            do i = ip11, mx1
                WTxy4(i, j) = &
                    f3_fil(im1(i), jp1(j), k) + 2.0 * f3_fil(i, jp1(j), k) &
                    + f3_fil(ip1(i), jp1(j), k) + &
                    2.0 * f3_fil(im1(i), j, k) + 4.0 * f3_fil(i, j, k) &
                    + 2.0 * f3_fil(ip1(i), j, k) + &
                    f3_fil(im1(i), jm1(j), k) + 2.0 * f3_fil(i, jm1(j), k) &
                    + f3_fil(ip1(i), jm1(j), k)

                WVxy4(i, j) = &
                    v3_fil(im1(i), jp1(j), k) + 2.0 * v3_fil(i, jp1(j), k) &
                    + v3_fil(ip1(i), jp1(j), k) + &
                    2.0 * v3_fil(im1(i), j, k) + 4.0 * v3_fil(i, j, k) &
                    + 2.0 * v3_fil(ip1(i), j, k) + &
                    v3_fil(im1(i), jm1(j), k) + 2.0 * v3_fil(i, jm1(j), k) &
                    + v3_fil(ip1(i), jm1(j), k)

            enddo

        enddo
        !     end do

        ! +--Gaussian Elimination Algorithm: F-B Sweep ==> WTxy7
        ! +  ----------------------------------------------------
        ! +
        !       do k=1,k3_fil
        do j = 1, my

            ! +--Forward  Sweep

            WTxy6(1, j) = -a1_fil(1, k) / b1_fil(1, k)
            WTxy7(1, j) = WTxy4(1, j) / b1_fil(1, k)

            WVxy6(1, j) = -a1_fil(1, k) / b1_fil(1, k)
            WVxy7(1, j) = WVxy4(1, j) / b1_fil(1, k)

            do i = ip11, mx
                WTxy6(i, j) = -a1_fil(i, k) &
                              / (a1_fil(i, k) * WTxy6(i - 1, j) + b1_fil(i, k))
                WTxy7(i, j) = (WTxy4(i, j) - a1_fil(i, k) * WTxy7(i - 1, j)) &
                              / (a1_fil(i, k) * WTxy6(i - 1, j) + b1_fil(i, k))

                WVxy6(i, j) = -a1_fil(i, k) &
                              / (a1_fil(i, k) * WVxy6(i - 1, j) + b1_fil(i, k))
                WVxy7(i, j) = (WVxy4(i, j) - a1_fil(i, k) * WVxy7(i - 1, j)) &
                              / (a1_fil(i, k) * WVxy6(i - 1, j) + b1_fil(i, k))

            enddo

            ! +--Backward Sweep

            do i = mx1, 1, -1
                WTxy7(i, j) = WTxy6(i, j) * WTxy7(i + 1, j) + WTxy7(i, j)
                WVxy7(i, j) = WVxy6(i, j) * WVxy7(i + 1, j) + WVxy7(i, j)
            enddo

        enddo
        !       end do

        ! +--2th Equations System
        ! +  ====================

        ! +--Gaussian Elimination Algorithm: Set Up
        ! +  --------------------------------------

        !     do k=1,k3_fil
        do i = ip11, mx1
            WTxy4(i, 1) = f3_fil(i, 1, k)
            WTxy4(i, my) = f3_fil(i, my, k)
            WVxy4(i, 1) = v3_fil(i, 1, k)
            WVxy4(i, my) = v3_fil(i, my, k)
            do j = jp11, my1
                WTxy4(i, j) = WTxy7(i, j)
                WVxy4(i, j) = WVxy7(i, j)
            enddo
        enddo
        !     end do

        ! +--Gaussian Elimination Algorithm: F-B Sweep ==> WTxy7
        ! +  ----------------------------------------------------

        !       do k=1,k3_fil
        do i = 1, mx

            ! +--Forward  Sweep

            WTxy6(i, 1) = -a2_fil(1, k) / b2_fil(1, k)
            WTxy7(i, 1) = WTxy4(i, 1) / b2_fil(1, k)

            WVxy6(i, 1) = -a2_fil(1, k) / b2_fil(1, k)
            WVxy7(i, 1) = WVxy4(i, 1) / b2_fil(1, k)

            do j = jp11, my
                WTxy6(i, j) = -a2_fil(j, k) &
                              / (a2_fil(j, k) * WTxy6(i, j - 1) + b2_fil(j, k))
                WTxy7(i, j) = (WTxy4(i, j) - a2_fil(j, k) * WTxy7(i, j - 1)) &
                              / (a2_fil(j, k) * WTxy6(i, j - 1) + b2_fil(j, k))

                WVxy6(i, j) = -a2_fil(j, k) &
                              / (a2_fil(j, k) * WVxy6(i, j - 1) + b2_fil(j, k))
                WVxy7(i, j) = (WVxy4(i, j) - a2_fil(j, k) * WVxy7(i, j - 1)) &
                              / (a2_fil(j, k) * WVxy6(i, j - 1) + b2_fil(j, k))

            enddo

            ! +--Backward Sweep

            do j = my1, 1, -1
                WTxy7(i, j) = WTxy6(i, j) * WTxy7(i, j + 1) + WTxy7(i, j)
                WVxy7(i, j) = WVxy6(i, j) * WVxy7(i, j + 1) + WVxy7(i, j)
            enddo

        enddo
        !       end do

        ! +--Result
        ! +  ======

        !     do k=1,k3_fil
        do j = jp11, my1
            do i = ip11, mx1
                f3_fil(i, j, k) = WTxy7(i, j)
                v3_fil(i, j, k) = WVxy7(i, j)
            enddo
        enddo
    enddo
!$OMP END PARALLEL DO

    return
endsubroutine DYNfil_3D_mp2
