#include "MAR_pp.def"
subroutine dyndgz_mp(norder)
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   FAST                                    15-04-2021  MAR |
    ! |   subroutine DYNdgz includes in the Horizontal Momentum Equations      |
    ! |              the terms representing the Pressure Gradient Force (PGF)  |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT   (via common block)                                           |
    ! |   ^^^^^    brocam: Brown and Campana Time Scheme Switch                |
    ! |            itFast: Short             Time Step  Counter                |
    ! |                                                                        |
    ! |            uairDY, vairDY, pktaDY   : u, v, and P / Time Step n        |
    ! |            ubefDY, vbefDY, ddux,ddvx: u, v        / Time Step n-1, n-2 |
    ! |            uairDY: x-wind speed component                        (m/s) |
    ! |            vairDY: y-wind speed component                        (m/s) |
    ! |            pktaDY: potential temperature divided by 100.[kPa]**(R/Cp)  |
    ! |             virDY: Contribution from Air Loading               (kg/kg) |
    ! |                                                                        |
    ! |   OUTPUT  (via common block)                                           |
    ! |   ^^^^^^   uairDY, vairDY, pktaDY: u, v, and P Values / Time Step n+1  |
    ! |            gpmiDY(i,j,k) = g * z (i,j,k-1/2),  (Geopotential)  (m2/s2) |
    ! |                                                                        |
    ! |   METHOD:  1) Solves the Hydrostatic Relation:                         |
    ! |   ^^^^^^       0     =- 1 /rho -dPHI/dp                                |
    ! |                      => gives the Geopotential PHI between Sigma Levels|
    ! |            2) Solves the Contributions       :                         |
    ! |                du/dt:=         -dPHI/dx                                |
    ! |                dv/dt:=         -dPHI/dy                                |
    ! |            3) Spatial  Numerical Scheme      :                         |
    ! |                Spatial Discretisation on Arakawa A Grid                |
    ! |   norder.EQ.2  2th Order Accurate Horizontal Spatial Differencing .OR. |
    ! |   norder.NE.2  4th Order Accurate Horizontal Spatial Differencing      |
    ! |                dPHI/dx and dPHI/dy are computed on p**(R/Cp) Surfaces  |
    ! |            4) Temporal Numerical Scheme      :                         |
    ! |                Time Split (i.e. each contribution computed separately) |
    ! |                Split Time Differencing, i.e. pressure Evolution and    |
    ! |                PGF are computed on Short Time Step dtfast=dt/(ntFast+1)|
    ! |                Advection and Diffusion are Computed on a Longer One)   |
    ! |                Brown and Campana Time Scheme used over Short Time Step |
    ! |                                                                        |
    ! |   REFER.:  1) Purser and Leslie, MWR 116, 2069--2080, 1988     (A Grid)|
    ! |   ^^^^^^   2) Marchuk, Numer.Meth.in Weath.Predict.,  1974 (Time Split)|
    ! |            3) Gadd, QJRMS 104, 569--582, 1978 (Split Time Differencing)|
    ! |            4) Brown and Campana, MWR 106, 1125--1136, 1978             |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_ub
    use mar_wk
    use marvec
    
    use trackwind, only : track_wind_dyndgz, delta_u, delta_v, &
            i_dyndgz_adv, i_dyndgz_pgf, i_dyndgz_ray, &
            iloc_adv, iloc_pgf, iloc_ray, &
            itw_dgz, iloc, ntwind_dgz, c1a_tmp, ddux_tmp, ddvx_tmp, &
            ubef, vbef, dg1x, dgzx, dg1y, dgzy
    
    implicit none
    
    integer norder
    integer i, j, k, m
    
    ! +--Local  Variables
    ! +  ================
    
    ! adv_uu : Advected u-Momentum
    real, allocatable :: adv_uu(:, :, :)
    ! adv_vv : Advected v-Momentum
    real, allocatable :: adv_vv(:, :, :)
    
    real bca, bcb, c1a, c1a_adv, c1a_pgf, ddux, ddvx, fraCLS, sigCLS
    real Raylei
    
    ! +--DATA
    ! +  ====
    ! +...Parameters of the Brown-Campana (1978, MWR, p.1125) scheme
    ! +   WARNING : scheme is unstable for bca maximum value (0.25)
    data bca/0.245e0/, bcb/0.510e0/
    
    ! +--Allocate
    allocate(adv_uu(mx, my, mz))
    allocate(adv_vv(mx, my, mz))
    
    ! +--Contributions from Momentum Advection
    ! +  =====================================
    
    ! +  **********
    call DYNadv_dLF_mp(norder, uairDY, vairDY, adv_uu, adv_vv)
    ! +  **********
    
    ! +--Integration of the Hydrostatic Equation
    ! +  =======================================
    
    ! +--EXNER Potential
    ! +  ---------------
    
    if(brocam) then
        do j = 1, my
            do i = 1, mx
                WKxy4(i, j) = exp(cap * log(pstDYn(i, j) + ptopDY))
                WKxy1(i, j) = cp * WKxy4(i, j)
            enddo
        enddo
    else
        do j = 1, my
            do i = 1, mx
                WKxy4(i, j) = exp(cap * log(pstDY(i, j) + ptopDY))
                WKxy1(i, j) = cp * WKxy4(i, j)
            enddo
        enddo
    
    endif
    
    ! +--Surface Contribution to Exner Function
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    !   fraCLS    : CLS Fraction where Temperature~Surface Temperature
    ! 0<fraCLS<1    and generally close to zero
    !   fraCLS    = 0.5  ==> linear variation of potential Temperature
    !    is assumed beween levels k=mz and k=mzz
    fraCLS = 0.0d+0
#ifdef IL
    fraCLS = 0.5d+0
#endif
    ! +
    sigCLS = (1.0d+0 - fraCLS) + fraCLS * sigma(mz)
    
    if(brocam) then
        do j = 1, my
            do i = 1, mx
                WKxy5(i, j) = cp * exp(cap * log(pstDYn(i, j) * sigCLS + ptopDY))
            enddo
        enddo
    else
        do j = 1, my
            do i = 1, mx
                WKxy5(i, j) = cp * exp(cap * log(pstDY(i, j) * sigCLS + ptopDY))
            enddo
        enddo
    endif
    
    ! +--Atmospheric Contribution to Exner Function
    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    !$OMP PARALLEL do private (i,j,k)
    do j = 1, my
        do k = 1, mz
            if(k >= mz .and. ptopDY <= 0.0) then
                ! do j=1,my
                do i = 1, mx
                    WKxy3(i, j) = 0.0
                enddo
                ! end do
            else
                if(brocam) then
                    ! do j=1,my
                    do i = 1, mx
                        WKxy3(i, j) = exp(cap * log(pstDYn(i, j) * sigmid(mzz - k) + ptopDY))
                    enddo
                    ! end do
                else
                    ! do j=1,my
                    do i = 1, mx
                        WKxy3(i, j) = exp(cap * log(pstDY(i, j) * sigmid(mzz - k) + ptopDY))
                    enddo
                    ! end do
                endif
            endif
            
            ! do j=1,my
            do i = 1, mx
                WKxy2(i, j) = cp * WKxy3(i, j)
                ! WKxyz4 : p ** (R/Cp)
                WKxyz4(i, j, mzz - k) = WKxy3(i, j)
            enddo
            ! end do
            
            ! +--GEO---Potential (Mid Layer k-1/2)
            ! +  ---------------
            
            if(k == 1) then
                ! +--Of the Surface Layer
                ! +  ~~~~~~~~~~~~~~~~~~~~
                ! do j=1,my
                do i = 1, mx
                    ! REMARK : It is assumed that the Geopotential Difference
                    !          in Lower Layer depends only on pktaDY at 1st Sigma Lev.
                    ! gpmiDY(mz) = gplvDY(mzz) + Cp * Delta[P**(R/Cp)] * [Theta/P0**(R/Cp)]
                    !            = gpmiDY_surf + Cp * Delta T
                    gpmiDY(i, j, mz) = gplvDY(i, j, mzz) &
                            + ((WKxy5(i, j) - WKxy2(i, j)) * pktaDY(i, j, mz) &
                                    + (WKxy1(i, j) - WKxy5(i, j)) * pktaDY(i, j, mzz)) &
                                    * (1.0 + virDY(i, j, mz))
                enddo
                ! end do
            else
                ! +--Above the Surface Layer
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~
                ! do j=1,my
                do i = 1, mx
                    gpmiDY(i, j, mzz - k) = gpmiDY(i, j, mzz + 1 - k) &
                            + (WKxy1(i, j) - WKxy2(i, j)) * pktaDY(i, j, mzz - k) &
                                    * (1.0 + virDY(i, j, mzz - k))
                enddo
                ! end do
            endif
            ! do j=1,my
            do i = 1, mx
                WKxy1(i, j) = WKxy2(i, j)
            enddo
            ! end do
        enddo
    enddo
    !$OMP END PARALLEL DO
    
    ! ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    
    ! +--Update of u and v at the Lateral Boundaries
    ! +  ===========================================
    
    !$OMP PARALLEL do private(i,j,k,c1a,c1a_adv,c1a_pgf,ddux,ddvx)
    do k = 1, mz
        do j = 1, my
            ubefDY(1, j, k) = uairDY(1, j, k)
            ubefDY(mx, j, k) = uairDY(mx, j, k)
            vbefDY(1, j, k) = vairDY(1, j, k)
            vbefDY(mx, j, k) = vairDY(mx, j, k)
        enddo
        !       end do
        if(mmy > 1) then
            !       do k=1,mz
            do i = 1, mx
                ubefDY(i, 1, k) = uairDY(i, 1, k)
                ubefDY(i, my, k) = uairDY(i, my, k)
                vbefDY(i, 1, k) = vairDY(i, 1, k)
                vbefDY(i, my, k) = vairDY(i, my, k)
            enddo
            !       end do
        endif
        
        ! ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        
        ! +--Mesoscale Geopotential Gradient
        ! +  ===============================
        
        !     do k=1,mz
        
        ! +--For Hydrostatic Contribution
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if (k==mz) then
            do j = 1, my
                do i = 1, mx
                    WKxyz7(i, j, k) = WKxy4(i, j)
                    WKxyz1(i, j, k) = gplvDY(i, j, mzz)
                enddo
            enddo
        else
            do j = 1, my
                do i = 1, mx
                    WKxyz7(i, j, k) = WKxyz4(i, j, k + 1)
                    WKxyz1(i, j, k) = gpmiDY(i, j, k + 1)
                enddo
            enddo
        end if
        
        do j = 1, my
            do i = 1, mx
                ! WKxyz8 = P**(R/Cp)(k-1/2) - P**(R/Cp)(k+1/2)
                WKxyz8(i, j, k) = WKxyz4(i, j, k) - WKxyz7(i, j, k)
            enddo
        enddo
        
        ! +--For Hydrostatic Contribution
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        ! do j = jp11, my1
        !     do i = ip11, mx1 ! i = 1, mx
        do j = 1, my
            do i = 1, mx
                ! WKxyz2 = ( P**(R/Cp)(k-1/2) * gpmiDY(k + 1/2) - P**(R/Cp)(k+1/2) * gpmiDY(i, j, k-1/2) ) / ( P**(R/Cp)(k-1/2) - P**(R/Cp)(k+1/2) )
                WKxyz2(i, j, k) = (WKxyz4(i, j, k) * WKxyz1(i, j, k) - WKxyz7(i, j, k) * gpmiDY(i, j, k)) / WKxyz8(i, j, k)
                ! WKxyz3 = ( gpmiDY(i, j, k-1/2) - gpmiDY(k + 1/2) ) / ( P**(R/Cp)(k-1/2) - P**(R/Cp)(k+1/2) ) (later : *  P**(R/Cp)(k))
                WKxyz3(i, j, k) = (gpmiDY(i, j, k) - WKxyz1(i, j, k)) / WKxyz8(i, j, k)
                ! P2 = P**(R/Cp)(k-1/2) | P1 = P**(R/Cp)(k+1/2) | Phi2 = gpmiDY(i, j, k-1/2) | Phi2 = gpmiDY(i, j, k+1/2)
                ! Phi = (Phi1 + Phi2) / 2.
                ! (P2 - P1) * (Phi1 + Phi2) = P2 * Phi1 + P2 * Phi2 - P1 * Phi1 - P1 * Phi2
                !                           = (P2 * Ph1 - P1 * Phi2) + (P2 * Phi2 - P1 * Phi1)
                !                           = (P2 * Ph1 - P1 * Phi2) +  P * (Phi2 - Phi1)
                !                           =   WKxyz2 * (P2 - P1)   +  P * WKxyz3 * (P2 - P1)
                !                           =   (P2 - P1) * 2 * Phi
            enddo
        enddo
        
        ! +--Gradient following x
        ! +  --------------------
        if(norder == 2) then
            do i = 1, mx
                do j = jp11, my1
                    WKxyz5(i, j, k) = (WKxyz2(ip1(i), j, k) - WKxyz2(im1(i), j, k)) * dxinv3(i, j)
                    WKxyz6(i, j, k) = (WKxyz3(ip1(i), j, k) - WKxyz3(im1(i), j, k)) * dxinv3(i, j)
                enddo
            enddo
        else
            do i = 1, mx
                do j = jp11, my1
                    WKxyz5(i, j, k) = &
                            fac43 * (WKxyz2(ip1(i), j, k) - WKxyz2(im1(i), j, k) &
                                    - 0.125 * (WKxyz2(ip2(i), j, k) - WKxyz2(im2(i), j, k))) * dxinv3(i, j)
                    WKxyz6(i, j, k) = &
                            fac43 * (WKxyz3(ip1(i), j, k) - WKxyz3(im1(i), j, k) &
                                    - 0.125 * (WKxyz3(ip2(i), j, k) - WKxyz3(im2(i), j, k))) * dxinv3(i, j)
                enddo
            enddo
        endif
        
        ! +--Contribution to u Wind Speed Component
        ! +  --------------------------------------
        if(itFast == 1) then
            ! +- First Step
            ! +  ~~~~~~~~~~
            do j = jp11, my1
                do i = ip11, mx1
                    c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                    c1a_adv = adv_uu(i, j, k)
                    c1a = c1a_pgf + c1a_adv
                    
                    ubefDY(i, j, k) = uairDY(i, j, k) + c1a * dtfast
                    uairDY(i, j, k) = uairDY(i, j, k) + c1a * dtfast * 2.
                    
                    dg1xDY(i, j, k) = c1a ! dudt_tm2
                    dgzxDY(i, j, k) = c1a ! dudt_tm1
                    
                    if(track_wind_dyndgz) then
                        c1a_tmp(iloc_adv) = c1a_adv
                        c1a_tmp(iloc_pgf) = c1a_pgf
                        do iloc = 1, 2
                            ubef(i, j, k, iloc) = delta_u(i, j, k, itw_dgz(iloc)) + c1a_tmp(iloc) * dtfast
                            delta_u(i, j, k, itw_dgz(iloc)) = delta_u(i, j, k, itw_dgz(iloc)) + &
                                    c1a_tmp(iloc) * dtfast * 2.
                            dg1x(i, j, k, iloc) = c1a_tmp(iloc)
                            dgzx(i, j, k, iloc) = c1a_tmp(iloc)
                        end do
                    end if
                enddo
            enddo
        else
            ! +- Next  Step
            ! +  ~~~~~~~~~~
            if(itFast <= ntFast) then
                if(brocam) then
                    do j = jp11, my1
                        do i = ip11, mx1
                            c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                            c1a_adv = adv_uu(i, j, k)
                            c1a = c1a_pgf + c1a_adv
                            ddux = ubefDY(i, j, k)
                            ubefDY(i, j, k) = uairDY(i, j, k)
                            uairDY(i, j, k) = ddux + 2.0 * dtfast &
                                    * (bcb * dgzxDY(i, j, k) &
                                            + bca * (c1a + dg1xDY(i, j, k)))
                            ! +...            U (n+1)       =  U(n-1) + 2     Dt  (Du/Dt)
#ifdef rt
                            ! +- Robert Time Filter
                            ! +  ~~~~~~~~~~~~~~~~~~
                            ubefDY(i, j, k) = ubefDY(i, j, k) &
                                    + Robert * (0.5 * (uairDY(i, j, k) + ddux) - ubefDY(i, j, k))
#endif
                            
                            dg1xDY(i, j, k) = dgzxDY(i, j, k) ! dudt_tm2 = dudt_tm1
                            dgzxDY(i, j, k) = c1a ! dudt_tm1 = dudt_t
                            if(track_wind_dyndgz) then
                                c1a_tmp(iloc_adv) = c1a_adv
                                c1a_tmp(iloc_pgf) = c1a_pgf
                                do iloc = 1, 2
                                    ddux_tmp(iloc) = ubef(i, j, k, iloc)
                                    ubef(i, j, k, iloc) = delta_u(i, j, k, itw_dgz(iloc))
                                    delta_u(i, j, k, itw_dgz(iloc)) = ddux_tmp(iloc) + 2.0 * dtfast &
                                            * (bcb * dgzx(i, j, k, iloc) &
                                                    + bca * (c1a_tmp(iloc) + dg1x(i, j, k, iloc)))
                                    dg1x(i, j, k, iloc) = dgzx(i, j, k, iloc)
                                    dgzx(i, j, k, iloc) = c1a_tmp(iloc)
                                end do
                            end if
                        enddo
                    enddo
                else
                    do j = jp11, my1
                        do i = ip11, mx1
                            c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                            c1a_adv = adv_uu(i, j, k)
                            c1a = c1a_pgf + c1a_adv
                            ddux = ubefDY(i, j, k)
                            ubefDY(i, j, k) = uairDY(i, j, k)
                            uairDY(i, j, k) = ddux + 2.0 * dtfast * c1a
                            ! +...            U (n+1)       =  U(n-1) + 2     Dt     (Du/Dt)
                            if(track_wind_dyndgz) then
                                c1a_tmp(iloc_adv) = c1a_adv
                                c1a_tmp(iloc_pgf) = c1a_pgf
                                do iloc = 1, 2
                                    ddux_tmp(iloc) = ubef(i, j, k, iloc)
                                    ubef(i, j, k, iloc) = delta_u(i, j, k, itw_dgz(iloc))
                                    delta_u(i, j, k, itw_dgz(iloc)) = ddux_tmp(iloc) + 2.0 * dtfast * c1a_tmp(iloc)
                                end do
                            end if
                        enddo
                    enddo
                endif
            else
                ! +- Last  Step
                ! +  ~~~~~~~~~~
                if(brocam) then
                    do j = jp11, my1
                        do i = ip11, mx1
                            c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                            c1a_adv = adv_uu(i, j, k)
                            c1a = c1a_pgf + c1a_adv
                            ddux = ubefDY(i, j, k)
                            ubefDY(i, j, k) = uairDY(i, j, k)
                            uairDY(i, j, k) = ddux + dtfast &
                                    * (bcb * dgzxDY(i, j, k) &
                                            + bca * (c1a + dg1xDY(i, j, k)))
                            ! +...            U (n+1)       =  U(n)   +    Dt     (Du/Dt)'
                            ! +               Leapfrog-Backward (e.g. Haltiner and Williams, p.152)
                            
                            dg1xDY(i, j, k) = dgzxDY(i, j, k)
                            dgzxDY(i, j, k) = c1a
                            
                            if(track_wind_dyndgz) then
                                c1a_tmp(iloc_adv) = c1a_adv
                                c1a_tmp(iloc_pgf) = c1a_pgf
                                do iloc = 1, 2
                                    ddux_tmp(iloc) = ubef(i, j, k, iloc)
                                    ubef(i, j, k, iloc) = delta_u(i, j, k, itw_dgz(iloc))
                                    delta_u(i, j, k, itw_dgz(iloc)) = ddux_tmp(iloc) + dtfast &
                                            * (bcb * dgzx(i, j, k, iloc) &
                                                    + bca * (c1a_tmp(iloc) + dg1x(i, j, k, iloc)))
                                    dg1x(i, j, k, iloc) = dgzx(i, j, k, iloc)
                                    dgzx(i, j, k, iloc) = c1a_tmp(iloc)
                                end do
                            end if
                        enddo
                    enddo
                else
                    do j = jp11, my1
                        do i = ip11, mx1
                            c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                            c1a_adv = adv_uu(i, j, k)
                            c1a = c1a_pgf + c1a_adv
                            ddux = ubefDY(i, j, k)
                            ubefDY(i, j, k) = uairDY(i, j, k)
                            uairDY(i, j, k) = ddux + dtfast * c1a
                            ! +...            U (n+1)       =  U(n)   +    Dt     (Du/Dt)'
                            ! +               Leapfrog-Backward (e.g. Haltiner and Williams, p.152)
                            if(track_wind_dyndgz) then
                                c1a_tmp(iloc_adv) = c1a_adv
                                c1a_tmp(iloc_pgf) = c1a_pgf
                                do iloc = 1, 2
                                    ddux_tmp(iloc) = ubef(i, j, k, iloc)
                                    ubef(i, j, k, iloc) = delta_u(i, j, k, itw_dgz(iloc))
                                    delta_u(i, j, k, itw_dgz(iloc)) = ddux_tmp(iloc) + dtfast * c1a_tmp(iloc)
                                end do
                            end if
                        enddo
                    enddo
                endif
            endif
        endif
        
        ! +- Gradient following y
        ! +  --------------------
        
        if(mmy > 1) then
            ! do j = 1, my
            !     WKxyz2(1, j, k) = 0.
            !     WKxyz2(mx, j, k) = 0.
            !     WKxyz3(1, j, k) = 0.
            !     WKxyz3(mx, j, k) = 0.
            ! enddo
            ! do i = ip11, mx1
            !     WKxyz2(i, 1, k) = (WKxyz4(i, 1, k) * WKxyz1(i, 1, k) - WKxyz7(i, 1, k) * gpmiDY(i, 1, k)) / WKxyz8(i, 1, k)
            !     WKxyz3(i, 1, k) = (gpmiDY(i, 1, k) - WKxyz1(i, 1, k)) / WKxyz8(i, 1, k)
            !     WKxyz2(i, my, k) = (WKxyz4(i, my, k) * WKxyz1(i, my, k) - WKxyz7(i, my, k) * gpmiDY(i, my, k)) / WKxyz8(i, my, k)
            !     WKxyz3(i, my, k) = (gpmiDY(i, my, k) - WKxyz1(i, my, k)) / WKxyz8(i, my, k)
            ! end do
            ! do j = jp11, my1
            ! do i = 1, mx
            
            ! +--For Hydrostatic Contribution
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            ! do i = ip11, mx1
            !     do j = 1, my
            !         WKxyz2(i, j, k) = (WKxyz4(i, j, k) * WKxyz1(i, j, k) &
            !                 - WKxyz7(i, j, k) * gpmiDY(i, j, k)) / WKxyz8(i, j, k)
            !         WKxyz3(i, j, k) = (gpmiDY(i, j, k) - WKxyz1(i, j, k)) / WKxyz8(i, j, k)
            !     enddo
            ! enddo
            
            if(norder == 2) then
                do j = 1, my
                    do i = ip11, mx1
                        WKxyz5(i, j, k) = (WKxyz2(i, jp1(j), k) - WKxyz2(i, jm1(j), k)) * dyinv3(i, j)
                        WKxyz6(i, j, k) = (WKxyz3(i, jp1(j), k) - WKxyz3(i, jm1(j), k)) * dyinv3(i, j)
                    enddo
                enddo
            else
                do j = 1, my
                    do i = ip11, mx1
                        WKxyz5(i, j, k) = fac43 * (WKxyz2(i, jp1(j), k) - WKxyz2(i, jm1(j), k) &
                                - 0.125 * (WKxyz2(i, jp2(j), k) - WKxyz2(i, jm2(j), k))) * dyinv3(i, j)
                        WKxyz6(i, j, k) = fac43 * (WKxyz3(i, jp1(j), k) - WKxyz3(i, jm1(j), k) &
                                - 0.125 * (WKxyz3(i, jp2(j), k) - WKxyz3(i, jm2(j), k))) * dyinv3(i, j)
                    enddo
                enddo
            endif
            
            ! +--Contribution to v Wind Speed Component
            ! +  --------------------------------------
            if(itFast == 1) then
                ! +- First Step
                ! +  ~~~~~~~~~~
                do j = jp11, my1
                    do i = ip11, mx1
                        c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                        c1a_adv = adv_vv(i, j, k)
                        c1a = c1a_pgf + c1a_adv
                        vbefDY(i, j, k) = vairDY(i, j, k) + c1a * dtfast
                        vairDY(i, j, k) = vairDY(i, j, k) + c1a * dtfast * 2.
                        dg1yDY(i, j, k) = c1a
                        dgzyDY(i, j, k) = c1a
                        
                        if(track_wind_dyndgz) then
                            c1a_tmp(iloc_adv) = c1a_adv
                            c1a_tmp(iloc_pgf) = c1a_pgf
                            do iloc = 1, 2
                                vbef(i, j, k, iloc) = delta_v(i, j, k, itw_dgz(iloc)) + c1a_tmp(iloc) * dtfast
                                delta_v(i, j, k, itw_dgz(iloc)) = delta_v(i, j, k, itw_dgz(iloc)) + &
                                        c1a_tmp(iloc) * dtfast * 2.
                                dg1y(i, j, k, iloc) = c1a_tmp(iloc)
                                dgzy(i, j, k, iloc) = c1a_tmp(iloc)
                            end do
                        end if
                    enddo
                enddo
            else
                ! +- Next  Step
                ! +  ~~~~~~~~~~
                if(itFast <= ntFast) then
                    if(brocam) then
                        do j = jp11, my1
                            do i = ip11, mx1
                                c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                                c1a_adv = adv_vv(i, j, k)
                                c1a = c1a_pgf + c1a_adv
                                
                                ddvx = vbefDY(i, j, k)
                                vbefDY(i, j, k) = vairDY(i, j, k)
                                vairDY(i, j, k) = ddvx + 2.0 * dtfast &
                                        * (bcb * dgzyDY(i, j, k) &
                                                + bca * (c1a + dg1yDY(i, j, k)))
                                ! +               V (n+1)       = V(n-1)  +   2    Dt  (Dv/Dt)
#ifdef rt
                                ! +- Robert Time Filter
                                ! +  ~~~~~~~~~~~~~~~~~~
                                vbefDY(i, j, k) = vbefDY(i, j, k) &
                                        + Robert * (0.5 * (vairDY(i, j, k) + ddvx) - vbefDY(i, j, k))
#endif
                                dg1yDY(i, j, k) = dgzyDY(i, j, k)
                                dgzyDY(i, j, k) = c1a
                                if(track_wind_dyndgz) then
                                    c1a_tmp(iloc_adv) = c1a_adv
                                    c1a_tmp(iloc_pgf) = c1a_pgf
                                    do iloc = 1, 2
                                        ddvx_tmp(iloc) = vbef(i, j, k, iloc)
                                        vbef(i, j, k, iloc) = delta_v(i, j, k, itw_dgz(iloc))
                                        delta_v(i, j, k, itw_dgz(iloc)) = ddvx_tmp(iloc) + 2.0 * dtfast &
                                                * (bcb * dgzy(i, j, k, iloc) &
                                                        + bca * (c1a_tmp(iloc) + dg1y(i, j, k, iloc)))
                                        dg1y(i, j, k, iloc) = dgzy(i, j, k, iloc)
                                        dgzy(i, j, k, iloc) = c1a_tmp(iloc)
                                    end do
                                end if
                            enddo
                        enddo
                    else
                        do j = jp11, my1
                            do i = ip11, mx1
                                c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                                c1a_adv = adv_vv(i, j, k)
                                c1a = c1a_pgf + c1a_adv
                                ddvx = vbefDY(i, j, k)
                                vbefDY(i, j, k) = vairDY(i, j, k)
                                vairDY(i, j, k) = ddvx + 2.0 * dtfast * c1a
                                ! +               V (n+1)       = V(n-1)  + 2   Dt   (Dv/Dt)
                                if(track_wind_dyndgz) then
                                    c1a_tmp(iloc_adv) = c1a_adv
                                    c1a_tmp(iloc_pgf) = c1a_pgf
                                    do iloc = 1, 2
                                        ddvx_tmp(iloc) = vbef(i, j, k, iloc)
                                        vbef(i, j, k, iloc) = delta_v(i, j, k, itw_dgz(iloc))
                                        delta_v(i, j, k, itw_dgz(iloc)) = ddvx_tmp(iloc) + 2.0 * dtfast * c1a_tmp(iloc)
                                    end do
                                end if
                            enddo
                        enddo
                    endif
                else
                    ! +- Last  Step
                    ! +  ~~~~~~~~~~
                    if(brocam) then
                        do j = jp11, my1
                            do i = ip11, mx1
                                c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                                c1a_adv = adv_vv(i, j, k)
                                c1a = c1a_pgf + c1a_adv
                                ddvx = vbefDY(i, j, k)
                                vbefDY(i, j, k) = vairDY(i, j, k)
                                vairDY(i, j, k) = ddvx + dtfast &
                                        * (bcb * dgzyDY(i, j, k) &
                                                + bca * (c1a + dg1yDY(i, j, k)))
                                ! +...            V (n+1)       =  V(n)   +   Dt (Dv/Dt)'
                                ! +               Leapfrog-Backward (e.g. Haltiner and Williams, p.152)
                                
                                dg1yDY(i, j, k) = dgzyDY(i, j, k)
                                dgzyDY(i, j, k) = c1a
                                if(track_wind_dyndgz) then
                                    c1a_tmp(iloc_adv) = c1a_adv
                                    c1a_tmp(iloc_pgf) = c1a_pgf
                                    do iloc = 1, 2
                                        ddvx_tmp(iloc) = vbef(i, j, k, iloc)
                                        vbef(i, j, k, iloc) = delta_v(i, j, k, itw_dgz(iloc))
                                        delta_v(i, j, k, itw_dgz(iloc)) = ddvx_tmp(iloc) + dtfast &
                                                * (bcb * dgzy(i, j, k, iloc) &
                                                        + bca * (c1a_tmp(iloc) + dg1y(i, j, k, iloc)))
                                        dg1y(i, j, k, iloc) = dgzy(i, j, k, iloc)
                                        dgzy(i, j, k, iloc) = c1a_tmp(iloc)
                                    end do
                                end if
                            enddo
                        enddo
                    else
                        do j = jp11, my1
                            do i = ip11, mx1
                                c1a_pgf = -(WKxyz5(i, j, k) + pkDY(i, j, k) * WKxyz6(i, j, k))
                                c1a_adv = adv_vv(i, j, k)
                                c1a = c1a_pgf + c1a_adv
                                ddvx = vbefDY(i, j, k)
                                vbefDY(i, j, k) = vairDY(i, j, k)
                                vairDY(i, j, k) = ddvx + dtfast * c1a
                                ! +...            V (n+1)       =  V(n)   +    Dt (Dv/Dt)'
                                ! +               Leapfrog-Backward (e.g. Haltiner and Williams, p.152)
                                if(track_wind_dyndgz) then
                                    c1a_tmp(iloc_adv) = c1a_adv
                                    c1a_tmp(iloc_pgf) = c1a_pgf
                                    do iloc = 1, 2
                                        ddvx_tmp(iloc) = vbef(i, j, k, iloc)
                                        vbef(i, j, k, iloc) = delta_v(i, j, k, itw_dgz(iloc))
                                        delta_v(i, j, k, itw_dgz(iloc)) = ddvx_tmp(iloc) + dtfast * c1a_tmp(iloc)
                                    end do
                                end if
                            enddo
                        enddo
                    endif
                endif
            endif
        endif
        
        ! +--Rayleigh Friction (Ref. ARPS 4.0 User's Guide, para 6.4.3 p.152)
        ! +  =================
        
        if(k <= mzabso) then
            do j = 1, my
                do i = 1, mx
                    if(track_wind_dyndgz) then
                        ubef(i, j, k, iloc_ray) = uairDY(i, j, k)
                        vbef(i, j, k, iloc_ray) = vairDY(i, j, k)
                    end if
                    uairDY(i, j, k) = (uairDY(i, j, k) + Ray_UB(k) * dtFast * uairUB(i, j, k)) &
                            / (1.0 + Ray_UB(k) * dtFast)
                    vairDY(i, j, k) = (vairDY(i, j, k) + Ray_UB(k) * dtFast * vairUB(i, j, k)) &
                            / (1.0 + Ray_UB(k) * dtFast)
                    if(track_wind_dyndgz) then
                        delta_u(i, j, k, itw_dgz(iloc)) = delta_u(i, j, k, itw_dgz(iloc)) + &
                                uairDY(i, j, k) - ubef(i, j, k, iloc_ray)
                        delta_v(i, j, k, itw_dgz(iloc)) = delta_v(i, j, k, itw_dgz(iloc)) + &
                                vairDY(i, j, k) - vbef(i, j, k, iloc_ray)
                    end if
                enddo
            enddo
        endif
    enddo
    !$OMP END PARALLEL DO
    deallocate (adv_uu)
    deallocate (adv_vv)
    return
end
