#include "MAR_pp.def"
subroutine DYNadv_ver
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   SLOW                                    18-09-2001  MAR |
    ! |   subroutine DYNadv_ver  includes the Vertical  Advection Contribution |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT   (via common block)                                           |
    ! |   ^^^^^    iterun: Run   Iteration                        Counter      |
    ! |            uairDY, vairDY, pktaDY Values    / Time Step n              |
    ! |            uairDY  : x-wind speed component                     (m/s)  |
    ! |            vairDY  : y-wind speed component                     (m/s)  |
    ! |            pktaDY: potential temperature divided by 100.[kPa]**(R/Cp)  |
    ! |                                                                        |
    ! |   OUTPUT  (via common block)                                           |
    ! |   ^^^^^^   uairDY, vairDY, pktaDY Values    / Time Step n+1            |
    ! |                                                                        |
    ! |   METHOD:  Unstaggered Grid: 1st Accurate in Space Upstream Scheme     |
    ! |   ^^^^^^^^   Staggered Grid: 2nd Accurate in Space                     |
    ! |                                                                        |
    ! | # OPTIONS: #VA: Vertical Average preferred in Centered Conserv Scheme  |
    ! | # ^^^^^^^^ #NS: NO   Slip Surface BC used  in Centered Conserv Scheme  |
    ! | #          #UR: Upper Radiating Condition (to be corrected, DO'NT USE) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+

    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_wk
#ifdef CA
    use mar_ca
#endif

    implicit none

#ifdef UR
    real uairUP, vairUP, pktaUP
    common / DYNadv_ver_var / uairUP(mx, my, 0:2), &
        vairUP(mx, my, 0:2), &
        pktaUP(mx, my, 0:2)
#endif
#ifdef WA
    integer nadvrd
    common / DYNadv_ver_loc / nadvrd
#endif
    logical centrL
#ifdef ZU
    logical adv3rd
    real gat(mx, my, mz), ga0(mx, my)
    data adv3rd/.true./
#endif

    ! +--Local  Variables
    ! +  ================

    integer i, j, k, m
    integer itimax, ntimax, nt__UP, nt, kk
    real cflmax, cflsig, faccfl, dt__UP, dt_sig, dsgm
    real uair_0, uair_c, uair_1, uair_2, uair_d
    real vair_0, vair_c, vair_1, vair_2, vair_d
    real pkta_0, pkta_c, pkta_1, pkta_2, pkta_d
    real old__u, old__v, old__t

    ! +--DATA
    ! +  ====

    data centrL/.true./
#ifdef UP
    centrL = .false.
#endif

    ! +--Initialization of the Upper Radiating Boundary Condition
    ! +  ========================================================
#ifdef UR
    if(iterun == 1) then
        do k = 0, 2
            kk = max(1, k)
            do j = jp11, my1
                do i = ip11, mx1
                    uairUP(i, j, k) = uairDY(i, j, kk)
                    vairUP(i, j, k) = vairDY(i, j, kk)
                    pktaUP(i, j, k) = pktaDY(i, j, kk)
                enddo
            enddo
        enddo
    endif
#endif

    ! +--Slip condition for Mountain Wave Experiments
    ! +  ============================================
#ifdef OM
    do j = jp11, my1
        do i = ip11, mx1
            psigDY(i, j, mz) = 0.0
        enddo
    enddo
#endif

    ! +--First and Second Order Schemes
    ! +  ==============================
#ifdef ZU
    if(.not. adv3rd) then
#endif

        ! +--Courant Number
        ! +  --------------

        cflmax = 0.0

        ! +--Centered second Order Scheme on a       staggered Grid
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(centrL .and. staggr) then

#ifdef WA
            write(6, 6001) iterun
6001        format(i6, ' 6001       centrL .and.       staggr /CFL Number')
#endif

            do k = 1, mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz7(i, j, k) = dt * psigDY(i, j, k) &
                                          / (pstDYn(i, j) * dsigm1(k) * 2.0)
                        cflsig = abs(WKxyz7(i, j, k) + WKxyz7(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo
            enddo

            do j = jp11, my1
                do i = ip11, mx1
                    WKxyz8(i, j, 1) = 0.00
#ifdef UR
                    WKxyz8(i, j, 1) = dt * psigDY(i, j, 1) * 0.33 &
                                      / (pstDYn(i, j) * dsigm1(1) * 2.)
                    ! WKxyz8(i,j,1)<--"psigDY(i,j,0)"
                    cflsig = abs(WKxyz8(i, j, k) + WKxyz8(i, j, k))
                    cflmax = max(cflsig, cflmax)
#endif
                enddo
            enddo

            do k = kp1(1), mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz8(i, j, k) = dt * psigDY(i, j, km1(k)) &
                                          / (pstDYn(i, j) * dsigm1(k) * 2.)
                        cflsig = abs(WKxyz8(i, j, k) + WKxyz8(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo
            enddo

        else

            ! +--Upstream first  Order Scheme on a       staggered Grid
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(staggr) then

#ifdef WA
                write(6, 6002) iterun
6002            format(i6, ' 6002 .not. centrL .and.       staggr /Wind Speed')
#endif

                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz8(i, j, k) = (psigDY(i, j, k - 1) * dsig_1(k - 1) &
                                               + psigDY(i, j, k) * dsig_1(k)) &
                                              / (dsig_1(k - 1) + dsig_1(k))
                        enddo
                    enddo
                enddo

                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz8(i, j, 1) = psigDY(i, j, 1) * dsig_1(1) &
                                          / (dsig_1(0) + dsig_1(1))
                    enddo
                enddo

                ! +--Upstream first  Order Scheme on a  non  staggered Grid
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else

#ifdef WA
                write(6, 6003) iterun
6003            format(i6, ' 6003 (.not.)centrL.and. .not. staggr /Wind Speed')
#endif

                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz8(i, j, k) = psigDY(i, j, k)
                        enddo
                    enddo
                enddo

            endif

            ! +--Centered second Order Scheme on a  non  staggered Grid
            ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            if(centrL) then

#ifdef WA
                write(6, 6004) iterun
6004            format(i6, ' 6004        centrL.and. .not. staggr /CFL Number')
#endif

                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz7(i, j, k) = dt * WKxyz8(i, j, k) &
                                              / (pstDYn(i, j) * dsigm1(k) * 2.)
                            cflsig = abs(WKxyz7(i, j, k))
                            cflmax = max(cflsig, cflmax)
                        enddo
                    enddo
                enddo

                ! +--Upstream first  Order Scheme on a (non) staggered Grid
                ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            else

#ifdef WA
                write(6, 6005) iterun
6005            format(i6, ' 6005  .not. centrL.and.(.not.)staggr /CFL Number')
#endif

                do k = 1, mmz1
                    do j = jp11, my1
                        do i = ip11, mx1
                            if(WKxyz8(i, j, k) > 0.0) then
                                WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k - 1))
                            else
                                WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k))
                            endif
                            cflsig = abs(WKxyz7(i, j, k))
                            cflmax = max(cflsig, cflmax)
                        enddo
                    enddo
                enddo

                k = mz
                do j = jp11, my1
                    do i = ip11, mx1
                        if(WKxyz8(i, j, k) > 0.0) then
                            WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k - 1))
                        else
                            WKxyz7(i, j, k) = -dt * WKxyz8(i, j, k) / (pstDYn(i, j) * dsig_1(k))
                        endif
                        cflsig = abs(WKxyz7(i, j, k))
                        cflmax = max(cflsig, cflmax)
                    enddo
                enddo

                do j = 1, my
                    do i = 1, mx
                        WKxyz7(i, j, 1) = 0.0
                    enddo
                enddo

                ! +--Work Array Reset
                ! +  ~~~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = 1, my
                        do i = 1, mx
                            WKxyz8(i, j, k) = 0.0
                        enddo
                    enddo
                enddo

            endif

        endif

        ! +--Set Up    of the Local Split Time Differencing
        ! +  ----------------------------------------------

        cflmax = 2.0 * cflmax
        ! +...      restricted CFL Criterion

        ntimax = cflmax
        if(centrL) then
            ntimax = max(2, ntimax)
#ifdef WA
            write(6, 6006) ntimax
6006        format(i6, ' 6006        centrL.and.(.not.)staggr /Nb Iterat.')
#endif
        else
            ntimax = max(1, ntimax)
#ifdef WA
            write(6, 6007) ntimax
6007        format(i6, ' 6007  .not. centrL.and.(.not.)staggr /Nb Iterat.')
#endif
        endif

        ! +--Update of CFL Number
        ! +  ~~~~~~~~~~~~~~~~~~~~
        if(ntimax > 1) then
            faccfl = 1.0 / ntimax
            do k = 1, mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz7(i, j, k) = WKxyz7(i, j, k) * faccfl
                        WKxyz8(i, j, k) = WKxyz8(i, j, k) * faccfl
                    enddo
                enddo
            enddo
        endif

        ! +--OUTPUT for Verification
        ! +  ~~~~~~~~~~~~~~~~~~~~~~~
#ifdef WA
        nadvrd = nadvrd + 1
        write(6, 6000) nadvrd, cflmax, ntimax
6000    format(i6, ' CFLmax ', 3x, ' ', 3x, '  =', f7.4, &
               6x, ' ntimax ', 8x, '   =', i4)
#endif

        ! +--2nd Order Centered Energy conserving:  Local Split Time Differencing
        ! +  --------- (Haltiner & Williams 1980 7.2.2, (7-47b) p.220) ----------
        ! +            -----------------------------------------------

        if(centrL) then

            if(staggr) then

#ifdef WA
                write(6, 6008)
6008            format(6x, ' 6008        centrL.and.       staggr /A Contrib.')
#endif

                do itimax = 1, ntimax

                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then

                        do j = jp11, my1

                            ! +--Vertical Differences

                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = uairDY(i, j, k)
                                WKxzq(i, k) = vairDY(i, j, k)
                                WKxzx(i, k) = pktaDY(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (uairDY(i, j, k) * dsigm1(k) * 2.0 &
                                               + uairDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (vairDY(i, j, k) * dsigm1(k) * 2.0 &
                                               + vairDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (pktaDY(i, j, k) * dsigm1(k) * 2.0 &
                                               + pktaDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                uair_0 = WKxzp(i, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                WKxza(i, k) = (WKxzp(i, k) - uair_0)
                                vair_0 = WKxzq(i, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                WKxzb(i, k) = (WKxzq(i, k) - vair_0)
                                pkta_0 = WKxzx(i, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif
                                WKxzc(i, k) = (WKxzx(i, k) - pkta_0)
                            enddo

                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = uairDY(i, j, k)
                                    WKxzq(i, k) = vairDY(i, j, k)
                                    WKxzx(i, k) = pktaDY(i, j, k)
#ifdef VA
                                    WKxzp(i, k) = (uairDY(i, j, k - 1) * dsigm1(k - 1) &
                                                   + uairDY(i, j, k) * dsigm1(k) * 2.0 &
                                                   + uairDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (vairDY(i, j, k - 1) * dsigm1(k - 1) &
                                                   + vairDY(i, j, k) * dsigm1(k) * 2.0 &
                                                   + vairDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (pktaDY(i, j, k - 1) * dsigm1(k - 1) &
                                                   + pktaDY(i, j, k) * dsigm1(k) * 2.0 &
                                                   + pktaDY(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo

                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = uairDY(i, j, k)
                                WKxzq(i, k) = vairDY(i, j, k)
                                WKxzx(i, k) = pktaDY(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (uairDY(i, j, k - 1) * dsigm1(k - 1) &
                                               + uairDY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (vairDY(i, j, k - 1) * dsigm1(k - 1) &
                                               + vairDY(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (pktaDY(i, j, k - 1) * dsigm1(k - 1) &
                                               + pktaDY(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo

                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (pktaDY(i, j, k) - WKxzx(i, k - 1))
#endif
                            enddo

                            ! +--Advection Contribution

                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxza(i, k)
                                    WKxyz1(i, j, k) = uairDY(i, j, k) - WKxzd(i, k)
                                    WKxyz4(i, j, k) = uairDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))

                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxzb(i, k)
                                    WKxyz2(i, j, k) = vairDY(i, j, k) - WKxzd(i, k)
                                    WKxyz5(i, j, k) = vairDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))

                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxzc(i, k)
                                    WKxyz3(i, j, k) = pktaDY(i, j, k) - WKxzd(i, k)
                                    WKxyz6(i, j, k) = pktaDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                              + WKxyz8(i, j, k) * WKxza(i, k)
                                WKxyz1(i, j, k) = uairDY(i, j, k) - WKxzd(i, k)
                                WKxyz4(i, j, k) = uairDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))

                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                              + WKxyz8(i, j, k) * WKxzb(i, k)
                                WKxyz2(i, j, k) = vairDY(i, j, k) - WKxzd(i, k)
                                WKxyz5(i, j, k) = vairDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))

                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                              + WKxyz8(i, j, k) * WKxzc(i, k)
                                WKxyz3(i, j, k) = pktaDY(i, j, k) - WKxzd(i, k)
                                WKxyz6(i, j, k) = pktaDY(i, j, k) - (WKxzd(i, k) + WKxzd(i, k))
                            enddo

                        enddo

                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then

                        ! +--Vertical Differences

                        do j = jp11, my1

                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                uair_0 = WKxzp(i, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                WKxza(i, k) = (WKxzp(i, k) - uair_0)
                                vair_0 = WKxzq(i, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                WKxzb(i, k) = (WKxzq(i, k) - vair_0)
                                pkta_0 = WKxzx(i, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif
                                WKxzc(i, k) = (WKxzx(i, k) - pkta_0)
                            enddo

                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo

                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz4(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz5(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz6(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo

                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0
                                WKxy2(i, j) = 0.0
                                WKxy3(i, j) = 0.0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (pktaDY(i, j, k) - WKxzx(i, k - 1))
#endif
                            enddo

                            ! +--Advection Contribution

                            do k = 1, mmz1
                                do i = ip11, mx1
                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxza(i, k)
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))

                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxzb(i, k)
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))

                                    WKxzd(i, k) = WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                                  + WKxyz8(i, j, k) * WKxzc(i, k)
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy1(i, j) &
                                              + WKxyz8(i, j, k) * WKxza(i, k)
                                old__u = WKxyz1(i, j, k)
                                WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                WKxyz4(i, j, k) = old__u - (WKxzd(i, k) + WKxzd(i, k))

                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy2(i, j) &
                                              + WKxyz8(i, j, k) * WKxzb(i, k)
                                old__v = WKxyz2(i, j, k)
                                WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                WKxyz5(i, j, k) = old__v - (WKxzd(i, k) + WKxzd(i, k))

                                WKxzd(i, k) = WKxyz7(i, j, k) * WKxy3(i, j) &
                                              + WKxyz8(i, j, k) * WKxzc(i, k)
                                old__t = WKxyz3(i, j, k)
                                WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                WKxyz6(i, j, k) = old__t - (WKxzd(i, k) + WKxzd(i, k))
                            enddo

                        enddo

                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else

                        do j = jp11, my1

                            ! +--Vertical Differences

                            k = 1
                            dsgm = 2.0 * dsigm1(k) + dsigm1(k + 1)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                               + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                uair_0 = WKxzp(i, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                WKxza(i, k) = (WKxzp(i, k) - uair_0)
                                vair_0 = WKxzq(i, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                WKxzb(i, k) = (WKxzq(i, k) - vair_0)
                                pkta_0 = WKxzx(i, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif
                                WKxzc(i, k) = (WKxzx(i, k) - pkta_0)
                            enddo

                            do k = kp1(1), mmz1
                                dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k) + dsigm1(k + 1)
                                do i = ip11, mx1
                                    WKxzp(i, k) = WKxyz4(i, j, k)
                                    WKxzq(i, k) = WKxyz5(i, j, k)
                                    WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                    WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz4(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz4(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz5(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz5(i, j, k + 1) * dsigm1(k + 1)) / dsgm
                                    WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                                   + WKxyz6(i, j, k) * dsigm1(k) * 2.0 &
                                                   + WKxyz6(i, j, k + 1) * dsigm1(k + 1)) / dsgm
#endif
                                enddo
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                    WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                    WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                                enddo
                            enddo

                            k = mmz
                            dsgm = dsigm1(k - 1) + 2.0 * dsigm1(k)
                            do i = ip11, mx1
                                WKxzp(i, k) = WKxyz4(i, j, k)
                                WKxzq(i, k) = WKxyz5(i, j, k)
                                WKxzx(i, k) = WKxyz6(i, j, k)
#ifdef VA
                                WKxzp(i, k) = (WKxyz4(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz4(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzq(i, k) = (WKxyz5(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz5(i, j, k) * dsigm1(k) * 2.0) / dsgm
                                WKxzx(i, k) = (WKxyz6(i, j, k - 1) * dsigm1(k - 1) &
                                               + WKxyz6(i, j, k) * dsigm1(k) * 2.0) / dsgm
#endif
                            enddo

                            do i = ip11, mx1
                                WKxza(i, k) = (WKxzp(i, k) - WKxzp(i, k - 1))
                                WKxzb(i, k) = (WKxzq(i, k) - WKxzq(i, k - 1))
                                WKxzc(i, k) = (WKxzx(i, k) - WKxzx(i, k - 1))
                            enddo

                            k = mzz
                            do i = ip11, mx1
                                WKxy1(i, j) = 0.0d+0
                                WKxy2(i, j) = 0.0d+0
                                WKxy3(i, j) = 0.0d+0
#ifdef NS
                                WKxy1(i, j) = -WKxzp(i, k - 1)
                                WKxy2(i, j) = -WKxzq(i, k - 1)
                                WKxy3(i, j) = (pktaDY(i, j, k) - WKxzx(i, k - 1))
#endif
                            enddo

                            ! +--Wind     Advection

                            do k = 1, mmz1
                                do i = ip11, mx1
                                    uairDY(i, j, k) = WKxyz1(i, j, k) &
                                                      - (WKxyz7(i, j, k) * WKxza(i, k + 1) &
                                                         + WKxyz8(i, j, k) * WKxza(i, k))
                                    vairDY(i, j, k) = WKxyz2(i, j, k) &
                                                      - (WKxyz7(i, j, k) * WKxzb(i, k + 1) &
                                                         + WKxyz8(i, j, k) * WKxzb(i, k))
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                uairDY(i, j, k) = WKxyz1(i, j, k) &
                                                  - (WKxyz7(i, j, k) * WKxy1(i, j) &
                                                     + WKxyz8(i, j, k) * WKxza(i, k))
                                vairDY(i, j, k) = WKxyz2(i, j, k) &
                                                  - (WKxyz7(i, j, k) * WKxy2(i, j) &
                                                     + WKxyz8(i, j, k) * WKxzb(i, k))
                            enddo

                            ! +--Pot.Temp.Advect.avoids double Counting in case of convective Adjustment

                            do k = 1, mmz1
                                do i = ip11, mx1
#ifdef cA
                                    if(adj_CA(i, j) == 0) then
#endif
                                        pktaDY(i, j, k) = WKxyz3(i, j, k) &
                                                          - (WKxyz7(i, j, k) * WKxzc(i, k + 1) &
                                                             + WKxyz8(i, j, k) * WKxzc(i, k))
#ifdef cA
                                    endif
#endif
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
#ifdef cA
                                if(adj_CA(i, j) == 0) then
#endif
                                    pktaDY(i, j, k) = WKxyz3(i, j, k) &
                                                      - (WKxyz7(i, j, k) * WKxy3(i, j) &
                                                         + WKxyz8(i, j, k) * WKxzc(i, k))
#ifdef cA
                                endif
#endif
                            enddo

                        enddo

                    endif

                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------

                enddo

                ! +--2nd Order Centered Leap-Frog Backward: Local Split Time Differencing
                ! +  --------------------------------------------------------------------

            else

#ifdef WA
                write(6, 6009)
6009            format(6x, ' 6009        centrL.and. .not. staggr /A Contrib.')
#endif

                do itimax = 1, ntimax

                    ! +--First        internal Time Step
                    ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    if(itimax == 1) then

                        do j = jp11, my1

                            ! +--Advection Increment

                            k = 1
                            do i = ip11, mx1
                                uair_0 = uairDY(i, j, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                WKxza(i, k) = (uairDY(i, j, k + 1) - uair_0) &
                                              * WKxyz7(i, j, k)
                                vair_0 = vairDY(i, j, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                WKxzb(i, k) = (vairDY(i, j, k + 1) - vairDY(i, j, k)) &
                                              * WKxyz7(i, j, k)
                                pkta_0 = pktaDY(i, j, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif
                                WKxzc(i, k) = (pktaDY(i, j, k + 1) - pktaDY(i, j, k)) &
                                              * WKxyz7(i, j, k)
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (uairDY(i, j, k + 1) - uairDY(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (vairDY(i, j, k + 1) - vairDY(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (pktaDY(i, j, k + 1) - pktaDY(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -uairDY(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzb(i, k) = -vairDY(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzc(i, k) = (pktaDY(i, j, k + 1) - pktaDY(i, j, k - 1)) &
                                              * WKxyz7(i, j, k)
                            enddo

                            ! +--Advection Contribution

                            do k = 1, mmz
                                do i = ip11, mx1
                                    WKxyz1(i, j, k) = uairDY(i, j, k) - WKxza(i, k)
                                    WKxyz4(i, j, k) = uairDY(i, j, k) - (WKxza(i, k) + WKxza(i, k))
                                    WKxyz2(i, j, k) = vairDY(i, j, k) - WKxzb(i, k)
                                    WKxyz5(i, j, k) = vairDY(i, j, k) - (WKxzb(i, k) + WKxzb(i, k))
                                    WKxyz3(i, j, k) = pktaDY(i, j, k) - WKxzc(i, k)
                                    WKxyz6(i, j, k) = pktaDY(i, j, k) - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo
                        enddo

                        ! +--Intermediary internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else if(itimax < ntimax) then

                        ! +--Advection Increment

                        do j = jp11, my1

                            k = 1
                            do i = ip11, mx1
                                uair_0 = WKxyz4(i, j, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                vair_0 = WKxyz5(i, j, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                pkta_0 = WKxyz6(i, j, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif

                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - uair_0) &
                                              * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - vair_0) &
                                              * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - pkta_0) &
                                              * WKxyz7(i, j, k)
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzc(i, k) = (pktaDY(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                              * WKxyz7(i, j, k)
                            enddo

                            ! +--Advection Contribution

                            do k = 1, mmz
                                do i = ip11, mx1
                                    old__u = WKxyz1(i, j, k)
                                    WKxyz1(i, j, k) = WKxyz4(i, j, k)
                                    WKxyz4(i, j, k) = old__u - (WKxza(i, k) + WKxza(i, k))
                                    old__v = WKxyz2(i, j, k)
                                    WKxyz2(i, j, k) = WKxyz5(i, j, k)
                                    WKxyz5(i, j, k) = old__v - (WKxzb(i, k) + WKxzb(i, k))
                                    old__t = WKxyz3(i, j, k)
                                    WKxyz3(i, j, k) = WKxyz6(i, j, k)
                                    WKxyz6(i, j, k) = old__t - (WKxzc(i, k) + WKxzc(i, k))
                                enddo
                            enddo

                        enddo

                        ! +--Last         internal Time Step
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                    else

                        do j = jp11, my1

                            ! +--Advection Increment

                            k = 1
                            do i = ip11, mx1
                                uair_0 = WKxyz4(i, j, k)
#ifdef UR
                                uair_0 = uairUP(i, j, 0)
#endif
                                vair_0 = WKxyz5(i, j, k)
#ifdef UR
                                vair_0 = vairUP(i, j, 0)
#endif
                                pkta_0 = WKxyz6(i, j, k)
#ifdef UR
                                pkta_0 = pktaUP(i, j, 0)
#endif

                                WKxza(i, k) = (WKxyz4(i, j, k + 1) - uair_0) &
                                              * WKxyz7(i, j, k)
                                WKxzb(i, k) = (WKxyz5(i, j, k + 1) - vair_0) &
                                              * WKxyz7(i, j, k)
                                WKxzc(i, k) = (WKxyz6(i, j, k + 1) - pkta_0) &
                                              * WKxyz7(i, j, k)
                            enddo

                            do k = kp1(1), mmz1
                                do i = ip11, mx1
                                    WKxza(i, k) = (WKxyz4(i, j, k + 1) - WKxyz4(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzb(i, k) = (WKxyz5(i, j, k + 1) - WKxyz5(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                    WKxzc(i, k) = (WKxyz6(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                                  * WKxyz7(i, j, k)
                                enddo
                            enddo

                            k = mmz
                            do i = ip11, mx1
                                WKxza(i, k) = -WKxyz4(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzb(i, k) = -WKxyz5(i, j, k - 1) &
                                              * WKxyz7(i, j, k)
                                WKxzc(i, k) = (pktaDY(i, j, k + 1) - WKxyz6(i, j, k - 1)) &
                                              * WKxyz7(i, j, k)
                            enddo

                            ! +--Wind     Advection

                            do k = 1, mmz
                                do i = ip11, mx1
                                    uairDY(i, j, k) = WKxyz1(i, j, k) - WKxza(i, k)
                                    vairDY(i, j, k) = WKxyz2(i, j, k) - WKxzb(i, k)
                                enddo

                                ! +--Pot.Temp.Advect.avoids double Counting in case of convective Adjustment

                                do i = ip11, mx1
#ifdef cA
                                    if(adj_CA(i, j) == 0) then
#endif
                                        pktaDY(i, j, k) = WKxyz3(i, j, k) - WKxzc(i, k)
#ifdef cA
                                    endif
#endif
                                enddo
                            enddo

                        enddo

                    endif

                    ! +--End of the                             Local Split Time Differencing
                    ! +  --------------------------------------------------------------------

                enddo

            endif

            ! +--First  Order Upstream Scheme:          Local Split Time Differencing
            ! +  --------------------------------------------------------------------

        else

#ifdef WA
            write(6, 6010)
6010        format(6x, ' 6010  .not. centrL.and.(.not.)staggr /A Contrib.')
#endif

            do itimax = 1, ntimax

                ! +--Auxiliary Variables
                ! +  ~~~~~~~~~~~~~~~~~~~
#ifdef WA
                write(6, 6011) itimax, WKxyz1(imez, jmez, mz1), WKxyz1(imez, jmez, mz) &
                    , uairDY(imez, jmez, mz1), uairDY(imez, jmez, mz)
6011            format(6x, ' 6011  .not. centrL.and.(.not.)staggr /A Contrib.', &
                       4f9.6)
#endif

                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = uairDY(i, j, k)
                            WKxyz2(i, j, k) = vairDY(i, j, k)
                            WKxyz3(i, j, k) = pktaDY(i, j, k)
                        enddo
                    enddo
                enddo

                ! +--Vertical Differences
                ! +  ~~~~~~~~~~~~~~~~~~~~
                k = 1
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz4(i, j, k) = 0.0d+0
                        WKxyz5(i, j, k) = 0.0d+0
                        WKxyz6(i, j, k) = 0.0d+0
#ifdef UR
                        WKxyz4(i, j, k) = WKxyz1(i, j, k) - uairUP(i, j, 0)
                        WKxyz5(i, j, k) = WKxyz2(i, j, k) - vairUP(i, j, 0)
                        WKxyz6(i, j, k) = WKxyz3(i, j, k) - pktaUP(i, j, 0)
#endif
                    enddo
                enddo

                do k = kp1(1), mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz4(i, j, k) = WKxyz1(i, j, k) - WKxyz1(i, j, k - 1)
                            WKxyz5(i, j, k) = WKxyz2(i, j, k) - WKxyz2(i, j, k - 1)
                            WKxyz6(i, j, k) = WKxyz3(i, j, k) - WKxyz3(i, j, k - 1)
                        enddo
                    enddo
                enddo

                k = mzz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxy1(i, j) = -WKxyz1(i, j, k - 1)
                        WKxy2(i, j) = -WKxyz2(i, j, k - 1)
                        WKxy3(i, j) = pktaDY(i, j, k) - WKxyz3(i, j, k - 1)
                    enddo
                enddo
                ! +
                ! +--Advection Contribution
                ! +  ~~~~~~~~~~~~~~~~~~~~~~
                do k = 1, mmz1
                    do j = jp11, my1
                        do i = ip11, mx1
                            WKxyz1(i, j, k) = uairDY(i, j, k) &
                                              + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                              + max(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k + 1)
                            WKxyz2(i, j, k) = vairDY(i, j, k) &
                                              + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                              + max(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k + 1)
                            WKxyz3(i, j, k) = pktaDY(i, j, k) &
                                              + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                              + max(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k + 1)
                        enddo
                    enddo
                enddo

                k = mz
                do j = jp11, my1
                    do i = ip11, mx1
                        WKxyz1(i, j, k) = uairDY(i, j, k) &
                                          + min(zero, WKxyz7(i, j, k)) * WKxyz4(i, j, k) &
                                          + max(zero, WKxyz7(i, j, k)) * WKxy1(i, j)
                        WKxyz2(i, j, k) = vairDY(i, j, k) &
                                          + min(zero, WKxyz7(i, j, k)) * WKxyz5(i, j, k) &
                                          + max(zero, WKxyz7(i, j, k)) * WKxy2(i, j)
                        WKxyz3(i, j, k) = pktaDY(i, j, k) &
                                          + min(zero, WKxyz7(i, j, k)) * WKxyz6(i, j, k) &
                                          + max(zero, WKxyz7(i, j, k)) * WKxy3(i, j)
                    enddo
                enddo

                ! +--Wind    Update
                ! +  ~~~~~~~~~~~~~~
                do k = 1, mz
                    do j = jp11, my1
                        do i = ip11, mx1
                            uairDY(i, j, k) = WKxyz1(i, j, k)
                            vairDY(i, j, k) = WKxyz2(i, j, k)
                        enddo

                        ! +--Pot.Temp.Update avoids double Counting in case of convective Adjustment
                        ! +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        do i = ip11, mx1
#ifdef cA
                            if(adj_CA(i, j) == 0) then
#endif
                                pktaDY(i, j, k) = WKxyz3(i, j, k)
#ifdef cA
                            endif
#endif
                        enddo
                    enddo
                enddo

                ! +--End of the                             Local Split Time Differencing
                ! +  --------------------------------------------------------------------
#ifdef WA
                write(6, 6012) itimax, WKxyz1(imez, jmez, mz1), WKxyz1(imez, jmez, mz) &
                    , uairDY(imez, jmez, mz1), uairDY(imez, jmez, mz)
6012            format(6x, ' 6012  .not. centrL.and.(.not.)staggr /A Contrib.', &
                       4f9.6)
#endif
            enddo

        endif

#ifdef UR
        ! Upper Radiating Boundary Condition
        ! ----------------------------------
        do j = jp11, my1
            do i = ip11, mx1
                uair_c = zero
                uair_1 = (uairDY(i, j, 1) + uairUP(i, j, 1)) * 0.5
                uair_2 = (uairDY(i, j, 2) + uairUP(i, j, 2)) * 0.5
                uair_d = uair_2 - uair_1
                if(uair_d /= zero) &
                    uair_c = -(uairDY(i, j, 1) - uairUP(i, j, 1)) * dsig_1(1) / (dt * uair_d)
                if(uair_c < zero) then
                    dt__UP = -sigma(1) / uair_c
                    nt__UP = dt / dt__UP
                    nt__UP = max(1, nt__UP)
                    dt__UP = dt / nt__UP
                    dt_sig = dt__UP * uair_c / sigma(1)
                    do nt = 1, nt__UP
                        uairUP(i, j, 0) = uairUP(i, j, 0) &
                                          - (uair_1 - uairUP(i, j, 0)) * dt_sig
                    enddo
                endif
                do k = 1, 2
                    uairUP(i, j, k) = uairDY(i, j, k)
                enddo
                vair_c = zero
                vair_1 = (vairDY(i, j, 1) + vairUP(i, j, 1)) * 0.5
                vair_2 = (vairDY(i, j, 2) + vairUP(i, j, 2)) * 0.5
                vair_d = vair_2 - vair_1
                if(vair_d /= zero) &
                    vair_c = -(vairDY(i, j, 1) - vairUP(i, j, 1)) * dsig_1(1) / (dt * vair_d)
                if(vair_c < zero) then
                    dt__UP = -sigma(1) / vair_c
                    nt__UP = dt / dt__UP
                    nt__UP = max(1, nt__UP)
                    dt__UP = dt / nt__UP
                    dt_sig = dt__UP * vair_c / sigma(1)
                    do nt = 1, nt__UP
                        vairUP(i, j, 0) = vairUP(i, j, 0) &
                                          - (vair_1 - vairUP(i, j, 0)) * dt_sig
                    enddo
                endif
                do k = 1, 2
                    vairUP(i, j, k) = vairDY(i, j, k)
                enddo
                pkta_c = zero
                pkta_1 = (pktaDY(i, j, 1) + pktaUP(i, j, 1)) * 0.5
                pkta_2 = (pktaDY(i, j, 2) + pktaUP(i, j, 2)) * 0.5
                pkta_d = pkta_2 - pkta_1
                if(pkta_d /= zero) &
                    pkta_c = -(pktaDY(i, j, 1) - pktaUP(i, j, 1)) * dsig_1(1) / (dt * pkta_d)
                if(pkta_c < zero) then
                    dt__UP = -sigma(1) / pkta_c
                    nt__UP = dt / dt__UP
                    nt__UP = max(1, nt__UP)
                    dt__UP = dt / nt__UP
                    dt_sig = dt__UP * pkta_c / sigma(1)
                    do nt = 1, nt__UP
                        pktaUP(i, j, 0) = pktaUP(i, j, 0) &
                                          - (pkta_1 - pktaUP(i, j, 0)) * dt_sig
                    enddo
                endif
                do k = 1, 2
                    pktaUP(i, j, k) = pktaDY(i, j, k)
                enddo
            enddo
        enddo
#endif

        ! +--Work Arrays Reset
        ! +  -----------------

        do j = 1, my
            do i = 1, mx
                WKxy1(i, j) = 0.0
                WKxy2(i, j) = 0.0
                WKxy3(i, j) = 0.0
            enddo
        enddo

        do k = 1, mz
            do i = 1, mx
                WKxza(i, k) = 0.0
                WKxzb(i, k) = 0.0
                WKxzc(i, k) = 0.0
                WKxzd(i, k) = 0.0
                WKxzp(i, k) = 0.0
                WKxzq(i, k) = 0.0
                WKxzx(i, k) = 0.0
            enddo
        enddo

        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    WKxyz1(i, j, k) = 0.0
                    WKxyz2(i, j, k) = 0.0
                    WKxyz3(i, j, k) = 0.0
                    WKxyz4(i, j, k) = 0.0
                    WKxyz5(i, j, k) = 0.0
                    WKxyz6(i, j, k) = 0.0
                    WKxyz7(i, j, k) = 0.0
                    WKxyz8(i, j, k) = 0.0
                enddo
            enddo
        enddo

        ! +--Third Order Vertical Scheme
        ! +  ===========================
#ifdef ZU
    else
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = 0.0
#endif
#ifdef ZO
                ga0(i, j) = uairDY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = uairDY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    uairDY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
#endif
#ifdef ZO
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = vairDY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = vairDY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    vairDY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
        do j = jp11, my1
            do i = ip11, mx1
                ga0(i, j) = pktaDY(i, j, mzz)
#endif
#ifdef ZO
                ga0(i, j) = pktaDY(i, j, mz)
#endif
#ifdef ZU
            enddo
        enddo
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    gat(i, j, k) = pktaDY(i, j, k)
                enddo
            enddo
        enddo
        ! +    ****************
        call DYNadv_cubv(gat, ga0)
        ! +    ****************
        do k = 1, mz
            do j = jp11, my1
                do i = ip11, mx1
                    pktaDY(i, j, k) = gat(i, j, k)
                enddo
            enddo
        enddo
    endif
#endif
    return
endsubroutine DYNadv_ver
