#include "MAR_pp.def"
subroutine DYNadv_LFB_2p(iswater, qqmass, niSlow, nordAV, ffx)
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   SLOW                                    09-12-2022  MAR |
    ! |   subroutine DYNadv_LFB_2s solves Advection (LeapFrog Backward Scheme) |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT/        nnSlow          : Time      Step Counter Maximum Value |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   INPUT/   (via common block)                                          |
    ! |   ^^^^^^        qqmass          : Mass Conservation Switch (+def.var.) |
    ! |                 iterun          : Long Time Step Counter               |
    ! |                 dt              : Time Step                            |
    ! |                 opstDY(mx,my)   : MASS, Time Step n                    |
    ! |                 pstDYn(mx,my)   : MASS, Time Step n+1                  |
    ! |                 uairDY(mx,my,mz): Advection Vector: x-----Direction    |
    ! |                 vairDY(mx,my,mz): Advection Vector: y-----Direction    |
    ! |                 psigDY(mx,my,mz): Advection Vector: sigma-Direction    |
    ! |                                                                        |
    ! |   INPUT/       ffx(mx,my,mz+1): Advected  Variable                     |
    ! |   OUTPUT                                                               |
    ! |   ^^^^^^                                                               |
    ! |                                                                        |
    ! |   METHOD:  2th order accurate Time       Scheme (leapfrog backw.) .and.|
    ! |   ^^^^^^  (2th order accurate Horizontal Scheme on Arakawa A grid .OR. |
    ! |            4th order accurate Horizontal Scheme on Arakawa A grid     )|
    ! |            2th order          Vertical   Scheme                        |
    ! |                                                                        |
    ! |            Robert Time Filter (for minimizing the computational mode)  |
    ! |                                                                        |
    ! |   CAUTION: This routine must be used                                   |
    ! |   ^^^^^^^  with a positive  definite restoring Procedure               |
    ! |            for    positive  definite Variables                         |
    ! |           (Such a Procedure is setup after digital filtering in MAR)   |
    ! |                                                                        |
    ! |   REFER.:  Use of  A grid: Purser   & Leslie,   1988, MWR 116, p.2069  |
    ! |   ^^^^^^   Time    Scheme: Haltiner & Williams, 1980, 5-2,     p.152   |
    ! |            Spatial Scheme: Haltiner & Williams, 1980, 5-6-5,   p.135   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use marqqm
    use mar_wk
    use trackwater, only : track_water_dynadv, jtw, &
            delta_qv, j_dynadv_hor, j_dynadv_ver, j_dynadv_sav, &
            dqp1_h, dqp1_v, dqm1_h, dqm1_v, delta_qv_tmp
    
    implicit none
    
    ! iswater : check if water to activate track_water
    logical, intent(in) :: iswater
    ! qqmass: mass Conservation Switch
    logical, intent(in) :: qqmass
    ! niSlow : Time Step Counter Initial Value
    integer, intent(in) :: niSlow
    integer, intent(in) :: nordAV
    ! ffx : Advected  Variable (t=n)
    real, intent(inout) :: ffx(mx, my, mzz)
    
    integer i, j, k, m
    
    
    ! +--Local  Variables
    ! +  ================
    ! daSlow : Initialization Switch
    logical daSlow
    common / DYNadv_LFB_2s_log / daSlow
    ! dtSlow : Time Step
    real dtSlow
    common / DYNadv_LFB_2s_rea / dtSlow
    real rtSlow
    ! itSlow : Time Step Counter
    integer itSlow
    integer nnSlow, n0Slow
    common / DYNadv_LFB_2s_int / nnSlow
    ! ffm1 : Advected  Variable (t=n-1)
    real ffm1
    ! dff : Variable  Increment
    real dff, dff_h, dff_v, flag
    
    integer kk
    real summ, sumn, summ_tmp
    
    ! ffo : Advected  Variable (t=n)
    ! ffp1 : Advected  Variable
    real, allocatable :: ffp1(:, :, :)
    real, allocatable :: ffo(:, :, :)
    
    logical LFBord, FLhost
    data LFBord/.false./
    data FLhost/.true./
    
    allocate (ffp1(mx, my, mzz))
    allocate (ffo(mx, my, mzz))
    
    ! +--Start the Leapfrog Backward Scheme
    ! +  ==================================
    
    if(.not. qqmass) then
        if(.not. daSlow) then
            daSlow = .true.
            ! n0Slow : previous Nb of  Time Steps
            n0Slow = niSlow
        else
            ! n0Slow : previous Nb of  Time Steps
            n0Slow = nnSlow
        endif
        ! rtSlow : minimum acceptable Nb of Time Steps in the leap-frog Backward Scheme
        rtSlow = CFLzDY
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    rtSlow = max(rtSlow, abs(uairDY(i, j, k)) * 2.0 * dt / dx)
                    rtSlow = max(rtSlow, abs(vairDY(i, j, k)) * 2.0 * dt / dx)
                enddo
            enddo
        enddo
        nnSlow = rtSlow + 0.5
        nnSlow = max(nnSlow, ntFast)
        if(mod(nnSlow, 2) == 0) nnSlow = nnSlow + 1
        dtSlow = dt / (nnSlow + 1)
        
        if(nnSlow /= n0Slow) then
            write(6, 6000) nnSlow, dtSlow
            6000        format(/, 'Advection Leap-Frog Backward Properties', &
                    /, ' ntSlow = ', i6, &
                    /, ' dtSlow = ', f9.2, /, 1x)
        endif
    endif
    
    ! +--Mass Conservation
    ! +  =================
    ffo = ffx
    ! +--Start Leap-Frog Backward
    ! +  ========================
    sumn = 0.
    summ = 0.
    
    !$OMP PARALLEL DO REDUCTION(+: sumn,summ) &
    !$OMP private (i,j,k,kk,dff,ffm1,flag,dff_v,dff_h,itSlow,summ_tmp,WTxy1,WTxy2) &
    !$OMP schedule(dynamic)
    do k = 1, mz
        
        flag = 0
        do j = jp11, my1
            do i = ip11, mx1
                if(abs(ffo(i, j, k)) >   1.5 * eps9) flag = 1
            enddo
        enddo
        
        if(flag == 1) then
            
            ! +--Vertical Differences
            ! +  ~~~~~~~~~~~~~~~~~~~~
            do j = jp11, my1
                do i = ip11, mx1
                    if(k == 1) then
                        WTxy1(i, j) = ffo(i, j, k) - ffo(i, j, kp1(k))
                        !CAa WTxyz3 = diffz_ffo * wsig / dsigm1 = mass flux
                        WTxyz3(i, j, k) = WTxy1(i, j) * wsigDY(i, j, k) * 0.5 / dsigm1(1)
                    else
                        WTxy1(i, j) = ffo(i, j, k) - ffo(i, j, kp1(k))
                        kk = km1(k)
                        WTxy2(i, j) = ffo(i, j, kk) - ffo(i, j, k)
                        !CAa WTxyz3 = diffz_ffo * wsig / dsigm1 = mass flux
                        WTxyz3(i, j, k) = (WTxy1(i, j) * wsigDY(i, j, k) &
                                + WTxy2(i, j) * wsigDY(i, j, kk)) * 0.5 / dsigm1(k)
                    endif
                enddo
            enddo
            
            do itSlow = 1, nnSlow + 1
                ! +--Mass Divergence
                ! +  ===============
                
                ! +--2th centered Differences  / sigma-Direction  / Energy conserving
                ! +  --- (Haltiner and Williams, 1980, 7.2.2, Eqn. (7-47b) p.220) ---
                ! +      --------------------------------------------------------
                
                ! +--Mass Update (Leapfrog-Backward)
                ! +  ===============================
                ! https://en.wikipedia.org/wiki/Finite_difference_coefficient
                !CAa                                                    -2    -1  0    1      2
                !CAa First derivative, central difference, accuracy=4 : 1/12 −2/3 0   2/3   −1/12
                !CAa                                             2/3 * (1/8   -1. 0    1.   -1/8 )
                !CAa dxinv3 = 1. / (2 dx)
                !CAa dtSlow = dt / (1 + nnSlow) (at least 2 1/2 time steps)
                if(itSlow == 1) then
                    if(nordAV == 2) then
                        ! do k = 1, mz
                        do j = jp11, my1
                            do i = ip11, mx1
                                dff_h = uairDY(i, j, k) * dxinv3(i, j) * (ffx(im1(i), j, k) - ffx(ip1(i), j, k)) + &
                                        vairDY(i, j, k) * dyinv3(i, j) * (ffx(i, jm1(j), k) - ffx(i, jp1(j), k))
                                dff_v = WTxyz3(i, j, k)
                                dff = (dff_h + dff_v) * dtSlow
                                ffx(i, j, k) = ffx(i, j, k) + dff
                                ffp1(i, j, k) = ffx(i, j, k) + dff + dff
                                if(iswater.and.track_water_dynadv) then
                                    dff_h = dff_h * dtSlow
                                    dff_v = dff_v * dtSlow
                                    delta_qv(i, j, k, j_dynadv_hor) = delta_qv(i, j, k, j_dynadv_hor) + dff_h
                                    delta_qv(i, j, k, j_dynadv_ver) = delta_qv(i, j, k, j_dynadv_ver) + dff_v
                                    dqp1_h(i, j, k) = delta_qv(i, j, k, j_dynadv_hor) + dff_h + dff_h
                                    dqp1_v(i, j, k) = delta_qv(i, j, k, j_dynadv_ver) + dff_v + dff_v
                                end if
                            enddo
                        enddo
                        ! end do
                    else
                        ! do k = 1, mz
                        do j = jp11, my1
                            do i = ip11, mx1
                                dff_h = uairDY(i, j, k) * dxinv3(i, j) * fac43 * (&
                                        0.125 * (ffx(ip2(i), j, k) - ffx(im2(i), j, k)) &
                                                - ffx(ip1(i), j, k) + ffx(im1(i), j, k)) + &
                                        vairDY(i, j, k) * dyinv3(i, j) * fac43 * (&
                                                0.125 * (ffx(i, jp2(j), k) - ffx(i, jm2(j), k)) &
                                                        - ffx(i, jp1(j), k) + ffx(i, jm1(j), k))
                                dff_v = WTxyz3(i, j, k)
                                dff = (dff_h + dff_v) * dtSlow
                                ffx(i, j, k) = ffx(i, j, k) + dff
                                ffp1(i, j, k) = ffx(i, j, k) + dff + dff
                                if(iswater.and.track_water_dynadv) then
                                    dff_h = dff_h * dtSlow
                                    dff_v = dff_v * dtSlow
                                    delta_qv(i, j, k, j_dynadv_hor) = delta_qv(i, j, k, j_dynadv_hor) + dff_h
                                    delta_qv(i, j, k, j_dynadv_ver) = delta_qv(i, j, k, j_dynadv_ver) + dff_v
                                    dqp1_h(i, j, k) = delta_qv(i, j, k, j_dynadv_hor) + dff_h + dff_h
                                    dqp1_v(i, j, k) = delta_qv(i, j, k, j_dynadv_ver) + dff_v + dff_v
                                end if
                            enddo
                        enddo
                        ! end do
                    endif
                else
                    if(itSlow <= nnSlow) then
                        if(nordAV == 2) then
                            ! do k = 1, mz
                            do j = jp11, my1
                                do i = ip11, mx1
                                    dff_h = uairDY(i, j, k) * dxinv3(i, j) * (ffx(im1(i), j, k) - ffx(ip1(i), j, k)) + &
                                            vairDY(i, j, k) * dyinv3(i, j) * (ffx(i, jp1(j), k) - ffx(i, jp1(j), k))
                                    dff_v = WTxyz3(i, j, k)
                                    dff = (dff_h + dff_v) * dtSlow
                                    ffm1 = ffx(i, j, k)
                                    ffx(i, j, k) = ffp1(i, j, k)
                                    ffp1(i, j, k) = ffm1 + dff + dff
                                    if(iswater.and.track_water_dynadv) then
                                        dff_h = dff_h * dtSlow
                                        dff_v = dff_v * dtSlow
                                        dqm1_h(i, j, k) = delta_qv(i, j, k, j_dynadv_hor)
                                        dqm1_v(i, j, k) = delta_qv(i, j, k, j_dynadv_ver)
                                        delta_qv(i, j, k, j_dynadv_hor) = dqp1_h(i, j, k)
                                        delta_qv(i, j, k, j_dynadv_ver) = dqp1_v(i, j, k)
                                        dqp1_h(i, j, k) = dqm1_h(i, j, k) + dff_h + dff_h
                                        dqp1_v(i, j, k) = dqm1_v(i, j, k) + dff_v + dff_v
                                    end if
#ifdef rt
                                    ! +--Robert Time Filter
                                    ! +  ~~~~~~~~~~~~~~~~~~
                                    ffx(i, j, k) = ffx(i, j, k) + &
                                            Robert * (0.5 * (ffp1(i, j, k) + ffm1) - ffx(i, j, k))
#endif
                                enddo
                            enddo
                            ! end do
                        else
                            ! do k = 1, mz
                            do j = jp11, my1
                                do i = ip11, mx1
                                    dff_h = uairDY(i, j, k) * dxinv3(i, j) * fac43 * (&
                                            0.125 * (ffx(ip2(i), j, k) - ffx(im2(i), j, k)) &
                                                    - ffx(ip1(i), j, k) + ffx(im1(i), j, k)) &
                                            + vairDY(i, j, k) * dyinv3(i, j) * fac43 * (&
                                                    0.125 * (ffx(i, jp2(j), k) - ffx(i, jm2(j), k)) &
                                                            - ffx(i, jp1(j), k) + ffx(i, jm1(j), k))
                                    dff_v = WTxyz3(i, j, k)
                                    dff = (dff_h + dff_v) * dtSlow
                                    ffm1 = ffx(i, j, k)
                                    ffx(i, j, k) = ffp1(i, j, k)
                                    ffp1(i, j, k) = ffm1 + dff + dff
                                    if(iswater.and.track_water_dynadv) then
                                        dff_h = dff_h * dtSlow
                                        dff_v = dff_v * dtSlow
                                        dqm1_h(i, j, k) = delta_qv(i, j, k, j_dynadv_hor)
                                        dqm1_v(i, j, k) = delta_qv(i, j, k, j_dynadv_ver)
                                        delta_qv(i, j, k, j_dynadv_hor) = dqp1_h(i, j, k)
                                        delta_qv(i, j, k, j_dynadv_ver) = dqp1_v(i, j, k)
                                        dqp1_h(i, j, k) = dqm1_h(i, j, k) + dff_h + dff_h
                                        dqp1_v(i, j, k) = dqm1_v(i, j, k) + dff_v + dff_v
                                    end if
#ifdef rt
                                    ! +--Robert Time Filter
                                    ! +  ~~~~~~~~~~~~~~~~~~
                                    ffx(i, j, k) = ffx(i, j, k) + &
                                            Robert * (0.5 * (ffp1(i, j, k) + ffm1) - ffx(i, j, k))
#endif
                                enddo
                            enddo
                            ! end do
                        endif
                    else
                        if(nordAV == 2) then
                            ! do k = 1, mz
                            do j = jp11, my1
                                do i = ip11, mx1
                                    dff_h = uairDY(i, j, k) * dxinv3(i, j) * &
                                            (ffx(im1(i), j, k) - ffx(ip1(i), j, k)) + &
                                            vairDY(i, j, k) * dyinv3(i, j) * &
                                                    (ffx(i, jm1(j), k) - ffx(i, jp1(j), k))
                                    dff_v = WTxyz3(i, j, k)
                                    dff = (dff_h + dff_v) * dtSlow
                                    ffx(i, j, k) = ffx(i, j, k) + dff
                                    if(iswater.and.track_water_dynadv) then
                                        dff_h = dff_h * dtSlow
                                        dff_v = dff_v * dtSlow
                                        delta_qv(i, j, k, j_dynadv_hor) = delta_qv(i, j, k, j_dynadv_hor) + dff_h
                                        delta_qv(i, j, k, j_dynadv_ver) = delta_qv(i, j, k, j_dynadv_ver) + dff_v
                                    end if
                                enddo
                            enddo
                            ! end do
                        else
                            ! do k = 1, mz
                            do j = jp11, my1
                                do i = ip11, mx1
                                    dff_h = uairDY(i, j, k) * dxinv3(i, j) * fac43 * (&
                                            0.125 * (ffx(ip2(i), j, k) - ffx(im2(i), j, k)) &
                                                    - ffx(ip1(i), j, k) + ffx(im1(i), j, k)) &
                                            + vairDY(i, j, k) * dyinv3(i, j) * fac43 * (&
                                                    0.125 * (ffx(i, jp2(j), k) - ffx(i, jm2(j), k)) &
                                                            - ffx(i, jp1(j), k) + ffx(i, jm1(j), k))
                                    dff_v = WTxyz3(i, j, k)
                                    dff = (dff_h + dff_v) * dtSlow
                                    ffx(i, j, k) = ffx(i, j, k) + dff
                                    if(iswater.and.track_water_dynadv) then
                                        dff_h = dff_h * dtSlow
                                        dff_v = dff_v * dtSlow
                                        delta_qv(i, j, k, j_dynadv_hor) = delta_qv(i, j, k, j_dynadv_hor) + dff_h
                                        delta_qv(i, j, k, j_dynadv_ver) = delta_qv(i, j, k, j_dynadv_ver) + dff_v
                                    end if
                                enddo
                            enddo
                            ! end do
                        endif
                        ! +*** Leapfrog-Backward (e.g. Haltiner and Williams, p.152)
                    endif
                endif
            enddo
            
            if(qqmass) then
                ! +--Mass Conservation
                ! +  =================
                do j = 1, my
                    do i = 1, mx
                        if(iswater.and.track_water_dynadv) then
                            delta_qv_tmp = ffx(i, j, k)
                        end if
                        ffx(i, j, k) = max(zero, ffx(i, j, k))
                        if(iswater.and.track_water_dynadv) then
                            delta_qv(i, j, k, j_dynadv_sav) = delta_qv(i, j, k, j_dynadv_sav) + ffx(i, j, k) - delta_qv_tmp
                        end if
                        ffo(i, j, k) = max(zero, ffo(i, j, k))
                        sumn = sumn + pstDYn(i, j) * dsigm1(k) * ffx(i, j, k)
                        summ = summ + opstDY(i, j) * dsigm1(k) * ffo(i, j, k)
                    enddo
                enddo
                
                !CAa fluxes at borders : difference between i = 1 and mx
                ! ------------------------------------------------------
                do j = 1, my
                    ! MARv<3
                    ! dtx = dt/dx
                    summ_tmp = dsigm1(k) * dtx * dsigm1(k) &
                            * (opstDY(1, j) * ffo(1, j, k) * uairDY(1, j, k) &
                                    - opstDY(mx, j) * ffo(mx, j, k) * uairDY(mx, j, k))
#ifdef AC
                    ! to uncomment if not enough precipitation
                    !CAa : debug line above
                    summ_tmp = summ_tmp / dsigm1(k)
#endif
#ifdef EU
                    ! to uncomment if not enough precipitation
                    summ_tmp = summ_tmp / dsigm1(k)
#endif
                    summ = summ + summ_tmp
                enddo
                !CAa fluxes at borders : difference between j = 1 and my
                ! ------------------------------------------------------
                do i = 1, mx
                    ! MARv<3
                    ! dtx = dt/dx
                    summ_tmp = dsigm1(k) * dtx * dsigm1(k) &
                            * (opstDY(i, 1) * ffo(i, 1, k) * vairDY(i, 1, k) &
                                    - opstDY(i, my) * ffo(i, my, k) * vairDY(i, my, k))
#ifdef AC
                    ! to uncomment if not enough precipitation
                    !CAa debug line above
                    summ_tmp = summ_tmp / dsigm1(k)
#endif
#ifdef EU
                    ! to uncomment if not enough precipitation
                    summ_tmp = summ_tmp / dsigm1(k)
#endif
                    summ = summ + summ_tmp
                enddo
            endif
        endif
    enddo
    !$OMP END PARALLEL DO
    
    if(sumn > zero .and. qqmass) then
        summ = summ / sumn
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    if(iswater.and.track_water_dynadv) then
                        delta_qv_tmp = ffx(i, j, k)
                    end if
                    ffx(i, j, k) = summ * ffx(i, j, k)
                    if(iswater.and.track_water_dynadv) then
                        delta_qv(i, j, k, j_dynadv_sav) = delta_qv(i, j, k, j_dynadv_sav) + ffx(i, j, k) - delta_qv_tmp
                    end if
                enddo
            enddo
        enddo
    endif
    
    deallocate (ffp1)
    deallocate (ffo)
    
    return
endsubroutine DYNadv_LFB_2p
