#include "MAR_pp.def"
subroutine DYNadv_LFB(norder)
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   SLOW                                    08-12-2022  MAR |
    ! |   subroutine DYNadv_LFB manages Leap-Frog Backward   Advection Scheme  |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT/  (via common block)                                           |
    ! |   ^^^^^^        iterun          : Long Time Step Counter               |
    ! |                 ntFast          : Time      Step Counter Maximum Value |
    ! |                 micphy          : Hydrometeors   Switch                |
    ! |                                                                        |
    ! |   INPUT/  (via common block)                                           |
    ! |   OUTPUT        pktaDY(mx,my,mzz) Potent. Temperat. / p_0**kappa       |
    ! |   ^^^^^^          qvDY(mx,my,mz): Water Vapor  Concentration   [kg/kg] |
    ! |                 ccniHY(mx,my,mz): Ice crystals Number              [-] |
    ! |                   qiHY(mx,my,mz): Ice crystals Concentration   [kg/kg] |
    ! |                   qsHY(mx,my,mz): Snow  Flakes Concentration   [kg/kg] |
    ! |                   qwHY(mx,my,mz): Cloud Dropl. Concentration   [kg/kg] |
    ! |                   qrHY(mx,my,mz): Rain  Drops  Concentration   [kg/kg] |
    ! |   SEE DYNdgz:   uairDY(mx,my,mz): Wind  Speed  x-Direction       [m/s] |
    ! |                 vairDY(mx,my,mz): Wind  Speed  y-Direction       [m/s] |
    ! |                                                                        |
    ! |   METHOD:  2th order accurate Time       Scheme (leapfrog backw.) .and.|
    ! |   ^^^^^^  (2th order accurate Horizontal Scheme on Arakawa A grid .OR. |
    ! |            4th order accurate Horizontal Scheme on Arakawa A grid     )|
    ! |            2th order          Vertical   Scheme                        |
    ! |                                                                        |
    ! |   CAUTION: This routine must be used                                   |
    ! |   ^^^^^^^  with a positive  definite restoring Procedure               |
    ! |            for    positive  definite Variables                         |
    ! |           (Such a Procedure is setup after digital filtering in MAR)   |
    ! |                                                                        |
    ! |   REFER.:  Use of  A grid: Purser   & Leslie,   1988, MWR 116, p.2069  |
    ! |   ^^^^^^   Time    Scheme: Haltiner & Williams, 1980, 5-2,     p.152   |
    ! |            Spatial Scheme: Haltiner & Williams, 1980, 5-6-5,   p.135   |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_dy
    use mar_sl
    use mar_hy
#ifdef TC
    use mar_tc
#endif
#ifdef iso
    use mariso, only : wiso, niso, Rdefault, &
            qvDY_iso, qvapSL_iso, qiHY_iso, &
            qsHY_iso, qwHY_iso, qrHY_iso, qsrfHY_iso
#endif
    
    implicit none
    
    integer, intent(in) :: norder
    ! +--Local  Variables
    ! +  ================
    ! iswater : to activate track_water
    logical :: iswater
    ! qqmass : mass conservation switch
    logical :: qqmass
    integer i, j, k, m
    ! ntSlow = Time Step Counter Maximum Value
    integer ntSlow
    integer n
    ! ff : Advected Variable
    real ff(mx, my, mzz)
#ifdef iso
    ! ff_Riso : isotopic ratio of Advected Variable
    real ff_Riso(niso, mx, my, mzz)
#endif
    
    ! +--DATA
    ! +  ====
    ntSlow = ntFast
    
    ! +--Advection of x-Momentum (uairDY)
    ! +  ================================
    qqmass = .false.
    !      do k=1,mz
    !        do j=1,my
    !        do i=1,mx
    !            ff(i,j,k) =  uairDY(i,j,k)
    !        end do
    !        end do
    !      end do
    !           k=  mzz
    !        do j=1,my
    !        do i=1,mx
    !            ff(i,j,k) =  0.
    !        end do
    !        end do
    !c #NV if (no_vec) then
    !C +          *************
    !c #NV   call DYNadv_LFB_2s(ntSlow,norder,ff)
    !C +          *************
    !c #NV else
    !C +                *************
    !        call       DYNadv_LFB_2v(ntSlow,norder,ff)
    !C +                *************
    !c #NV end if
    !      do k=1,mz
    !        do j=1,my
    !        do i=1,mx
    !          uairDY(i,j,k) =   ff(i,j,k)
    !        end do
    !        end do
    !      end do
    
    !C +--Advection of y-Momentum (vairDY)
    !C +  ================================
    !      do k=1,mz
    !        do j=1,my
    !        do i=1,mx
    !            ff(i,j,k) =  vairDY(i,j,k)
    !        end do
    !        end do
    !      end do
    !           k=  mzz
    !        do j=1,my
    !        do i=1,mx
    !            ff(i,j,k) =  0.
    !        end do
    !        end do
    
    !c #NV if (no_vec) then
    !C +                *************
    !c #NV   call       DYNadv_LFB_2s(ntSlow,norder,ff)
    !C +                *************
    !c #NV else
    !C +                *************
    !        call       DYNadv_LFB_2v(ntSlow,norder,ff)
    !C +                *************
    !c #NV end if
    !      do k=1,mz
    !        do j=1,my
    !        do i=1,mx
    !          vairDY(i,j,k) =   ff(i,j,k)
    !        end do
    !        end do
    !      end do
    
    ! +--Advection of Heat (pktaDY)
    ! +  ==========================
    qqmass = .false.
    ! CAa norder is an input of the routine, already equal to 4
    ! CAa cannot be assigned here because of intent(in)
    ! CAa todo : check with XF if norder should be let as an option or not
    ! norder = 4
    do k = 1, mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = pktaDY(i, j, k)
            enddo
        enddo
    enddo
    !CAa ff = pkta -> not water
    iswater = .false.
    !CAa : At first call of DYNadv_LFB_2p, qqmass = .false. -> setup of dtSlow
    ! +  *************
    call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
    ! +  *************
    
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                pktaDY(i, j, k) = ff(i, j, k)
            enddo
        enddo
    enddo
    
    ! +--Advection of Water Vapor (qvDY)
    ! +  ===============================
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = qvDY(i, j, k)
            enddo
        enddo
    enddo
    k = mzz
    do j = 1, my
        do i = 1, mx
            ff(i, j, k) = qvapSL(i, j)
        enddo
    enddo

#ifdef iso
    ! Compute isotopic ratio *before* advection of water vapor
    ! Advection of water vapor isotopes, based on LMDZ : dyn3dmem/vlsplt_loc.F/vlz_loc
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    call Riso_from_qiso(wiso, qvDY_iso(wiso, i, j, k), qvDY(i, j, k), ff_Riso(wiso, i, j, k))
                enddo
            enddo
        enddo
    enddo
    ! k = mzz
    do j = 1, my
        do i = 1, mx
            do wiso = 1, niso
                call Riso_from_qiso(wiso, qvapSL_iso(wiso, i, j), qvapSL(i, j), ff_Riso(wiso, i, j, mzz))
            enddo
        enddo
    enddo
#endif
    
    !CAa Mass conservation of water vapor on the full domain (xyz)
    qqmass = .true.
    !CAa ff = qvDY -> track water
    iswater = .true.
    ! +  *************
    call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
    ! +  *************
    
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                qvDY(i, j, k) = ff(i, j, k)
            enddo
        enddo
    enddo

#ifdef iso
    ! qqmass = .false. because mass conservation is on qvDY
    qqmass = .false.
    do wiso = 1, niso
        ! +  *************
        call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff_Riso(wiso, :, :, :))
        ! +  *************
    enddo
    do k = 1, mz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    qvDY_iso(wiso, i, j, k) = ff_Riso(wiso, i, j, k) * qvDY(i, j, k)
                enddo
            enddo
        enddo
    enddo
#endif
    
    if(micphy) then
        ! +--Advection of Ice Crystals Nb (ccniHY)
        ! +  =====================================
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = ccniHY(i, j, k)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = 0.
            enddo
        enddo
        
        !CAa ff = ccniHY -> do not track water
        iswater = .false.
        ! +  ****************
        call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
        ! +  ****************
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ccniHY(i, j, k) = ff(i, j, k)
                enddo
            enddo
        enddo
        
        ! +--Advection of Cloud Crystals (qiHY)
        ! +  ==================================
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = qiHY(i, j, k)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = 0.
            enddo
        enddo

#ifdef iso
        ! Compute isotopic ratio *before* advection of cloud ice
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        call Riso_from_qiso(wiso, qiHY_iso(wiso, i, j, k), qiHY(i, j, k), ff_Riso(wiso, i, j, k))
                    enddo
                enddo
            enddo
        enddo
        ! k = mzz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    ff_Riso(wiso, i, j, mzz) = Rdefault(wiso)
                enddo
            enddo
        enddo
#endif
        !CAa Mass conservation on the full domain (xyz)
        qqmass = .true.
        !CAa ff = qiHY -> do not track water
        iswater = .false.
        ! +  ****************
        call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
        ! +  ****************
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    qiHY(i, j, k) = ff(i, j, k)
                enddo
            enddo
        enddo
#ifdef iso
        ! qqmass = .false. because mass conservation is on qiHY
        qqmass = .false.
        do wiso = 1, niso
            ! +  *************
            call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff_Riso(wiso, :, :, :))
            ! +  *************
        enddo
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        qiHY_iso(wiso, i, j, k) = ff_Riso(wiso, i, j, k) * qiHY(i, j, k)
                    enddo
                enddo
            enddo
        enddo
#endif
        ! +--Advection of Snow Flakes (qsHY)
        ! +  ===============================
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = qsHY(i, j, k)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = qsrfHY(i, j)
            enddo
        enddo

#ifdef iso
        ! Compute isotopic ratio *before* advection of snow flakes
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        call Riso_from_qiso(wiso, qsHY_iso(wiso, i, j, k), qsHY(i, j, k), ff_Riso(wiso, i, j, k))
                    enddo
                enddo
            enddo
        enddo
        ! k = mzz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    ! todo : track qsrfHY -> qsrfHY_iso
                    call Riso_from_qiso(wiso, qsrfHY_iso(wiso, i, j), qsrfHY(i, j), ff_Riso(wiso, i, j, mzz))
                enddo
            enddo
        enddo
#endif
        !CAa Mass conservation of water vapor on the full domain (xyz)
        qqmass = .true.
        !CAa ff = qsHY -> do not track water
        iswater = .false.
        ! +  ****************
        call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
        ! +  ****************
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    qsHY(i, j, k) = ff(i, j, k)
                enddo
            enddo
        enddo
#ifdef iso
        ! qqmass = .false. because mass conservation is on qsHY
        qqmass = .false.
        do wiso = 1, niso
            ! +  *************
            call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff_Riso(wiso, :, :, :))
            ! +  *************
        enddo
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        qsHY_iso(wiso, i, j, k) = ff_Riso(wiso, i, j, k) * qsHY(i, j, k)
                    enddo
                enddo
            enddo
        enddo
#endif
        ! +--Advection of Cloud Dropplets (qwHY)
        ! +  ===================================
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = qwHY(i, j, k)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = 0.
            enddo
        enddo
#ifdef iso
        ! Compute isotopic ratio *before* advection of cloud ice
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        call Riso_from_qiso(wiso, qwHY_iso(wiso, i, j, k), qwHY(i, j, k), ff_Riso(wiso, i, j, k))
                    enddo
                enddo
            enddo
        enddo
        ! k = mzz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    ff_Riso(wiso, i, j, mzz) = Rdefault(wiso)
                enddo
            enddo
        enddo
#endif
        !CAa Mass conservation on the full domain (xyz)
        qqmass = .true.
        !CAa ff = qwHY -> do not track water
        iswater = .false.
        ! +  *************
        call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
        ! +  *************
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    qwHY(i, j, k) = ff(i, j, k)
                enddo
            enddo
        enddo
#ifdef iso
        ! qqmass = .false. because mass conservation is on qwHY
        qqmass = .false.
        do wiso = 1, niso
            ! +  *************
            call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff_Riso(wiso, :, :, :))
            ! +  *************
        enddo
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        qwHY_iso(wiso, i, j, k) = ff_Riso(wiso, i, j, k) * qwHY(i, j, k)
                    enddo
                enddo
            enddo
        enddo
#endif
        ! +--Advection of Rain Drops (qrHY)
        ! +  ==============================
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = qrHY(i, j, k)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = 0.
            enddo
        enddo
#ifdef iso
        ! Compute isotopic ratio *before* advection of cloud ice
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        call Riso_from_qiso(wiso, qrHY_iso(wiso, i, j, k), qrHY(i, j, k), ff_Riso(wiso, i, j, k))
                    enddo
                enddo
            enddo
        enddo
        ! k = mzz
        do j = 1, my
            do i = 1, mx
                do wiso = 1, niso
                    ff_Riso(wiso, i, j, mzz) = Rdefault(wiso)
                enddo
            enddo
        enddo
#endif
        !CAa Mass conservation on the full domain (xyz)
        qqmass = .true.
        !CAa ff = qrHY -> do not track water
        iswater = .false.
        ! +  ****************
        call DYNadv_LFB_2p(iswater, qqmass, ntSlow, norder, ff)
        ! +  ****************
        
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    qrHY(i, j, k) = ff(i, j, k)
                enddo
            enddo
        enddo
#ifdef iso
        ! qqmass = .false. because mass conservation is on qwHY
        qqmass = .false.
        do wiso = 1, niso
            ! +  *************
            call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff_Riso(wiso, :, :, :))
            ! +  *************
        enddo
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    do wiso = 1, niso
                        qrHY_iso(wiso, i, j, k) = ff_Riso(wiso, i, j, k) * qrHY(i, j, k)
                    enddo
                enddo
            enddo
        enddo
#endif
    endif

#ifdef TC
    ! +--Advection of Tracers (qxTC)
    ! +  ===========================
    do n = 1, ntrac
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    ff(i, j, k) = qxTC(i, j, k, n)
                enddo
            enddo
        enddo
        k = mzz
        do j = 1, my
            do i = 1, mx
                ff(i, j, k) = 0.
            enddo
        enddo
        if(no_vec) then
            if(openmp) then
                ! +  ****************
                call DYNadv_LFB_2p(qqmass, ntSlow, norder, ff)
                ! +  ****************
            else
                ! +  *************
                call DYNadv_LFB_2s(ntSlow, norder, ff)
                ! +  *************
            endif
        else
            ! +  *************
            !CAa warning, todo : DYNadv_LFB_2v does not exist
            call DYNadv_LFB_2v(ntSlow, norder, ff)
            ! +  *************
        endif
        do k = 1, mz
            do j = 1, my
                do i = 1, mx
                    qxTC(i, j, k, n) = ff(i, j, k)
                enddo
            enddo
        enddo
    enddo
#endif
    
    return
endsubroutine DYNadv_LFB
