#include "MAR_pp.def"
subroutine DYNadv_hor(qqmass, ff, fp0, fp1, fu, fv)
    ! +
    ! +------------------------------------------------------------------------+
    ! | MAR DYNAMICS   SLOW                                    27-09-2001  MAR |
    ! |   subroutine DYNadv_hor includes the Horizontal Advection Contribution |
    ! |              solved by using a Cubic Spline Technique                  |
    ! +------------------------------------------------------------------------+
    ! |                                                                        |
    ! |   INPUT:   fp0: mass                                                   |
    ! |   ^^^^^    fu:  advection velocity, x-direction, (e.g., uairDY)        |
    ! |            fv:  advection velocity, y-direction, (e.g., vairDY)        |
    ! |                                                                        |
    ! |            iterun: Iteration            Index                          |
    ! |            openLB: Zero-Gradient  LBC   Switch                         |
    ! |            FirstC: First Call at time=t Switch                         |
    ! |            qqmass: Mass  Conservation   Switch                         |
    ! |                                                                        |
    ! |   INPUT &/ ff: advected  variable, which may be:                       |
    ! |   OUTPUT : uairDY,vairDY, pktaDY, qvDY, qwHY,qrHY, qiHY,ccniHY,qsHY    |
    ! |   ^^^^^^^^ uairDY  : x-wind speed component                     (m/s)  |
    ! |            vairDY  : y-wind speed component                     (m/s)  |
    ! |            pktaDY: Potential Temperature divided by 100.[kPa]**(R/Cp)  |
    ! |              qvDY: Air specific Humidity                      (kg/kg)  |
    ! |            ****HY: Hydrometeor  Concentration                 (kg/kg)  |
    ! |                                                                        |
    ! |   METHOD : The following Contributions may be taken into account:      |
    ! |   ^^^^^^^^     du/dt:=-udu/dx -vdu/dy                                  |
    ! |                dv/dt:=-udv/dx -vdv/dy                                  |
    ! |                dP/dt:=-udP/dx -vdP/dy  (Potential Temperature)         |
    ! |                dq/dt:=-udq/dx -vdq/dy  (Water     Species)             |
    ! |            Correction for Mass Conservation  (qqmass = .true.)         |
    ! |            is based on the assumption that the meteorological fields   |
    ! |            at Lateral Boundaries (LB) change only through relaxation   |
    ! |            of MAR Fields towards Large Scale Meteorological Fields,    |
    ! |            ==> Total Mass remains constant during "inner" Advection    |
    ! |                (i1,i2) = (1,mx)   ==>    correction operated at LB's   |
    ! |                          (   correction slightly inconsistent     )    |
    ! |                (i1,i2) = (2,mx-1) ==> no correction operated at LB's   |
    ! |                          (LB relaxation slightly badly conditioned)    |
    ! |            Inclusion of Mass Flux at the LB's causes a conflict        |
    ! |                                                      with LBC scheme   |
    ! |            This is verified by the onset of spurious waves   at LB's   |
    ! |                                                                        |
    ! |   REFER. : Alpert, thesis, 1980                                        |
    ! |   ^^^^^^^^ Pielke, Mesoscale Meteorological Modeling, 297--307, 1984   |
    ! |           (Seibert and Morariu, JAM, p.118, 1991)                      |
    ! |                                                                        |
    ! | # OPTIONS: #MC (Mass   Correction)             performed               |
    ! | # ^^^^^^^^ #MD (Mass   Difference) Correction  preferred               |
    ! |                                                                        |
    ! +------------------------------------------------------------------------+
    ! +
    use marctr
    use marphy
    use mardim
    use margrd
    use mar_hy
    use mar_CU
    use mar_lb
    use mar_wk
    ! +
    implicit none

    ! qqmass : mass conservation switch
    logical, intent(in) :: qqmass

    integer i, j, k, m
    real fp0(mx, my)
    real fp1(mx, my)
    real ff(mx, my, mz)
    real fu(mx, my, mz)
    real fv(mx, my, mz)
    ! +
    ! +
    ! +--Local  Variables
    ! +  ================
    ! +
    integer iunPos, junPos, iindex, jindex
    ! +
    integer i1_adh, i2_adh, j1_adh, j2_adh, k_pdim
    parameter(i1_adh=1, i2_adh=mx, j1_adh=1, j2_adh=my)
    parameter(k_pdim=mz)
#ifdef QB
    common / DYNadv_hor_loc / i1_adh, i2_adh, j1_adh, j2_adh
#endif
#ifdef MC
    real sumMav
#endif
    real sumMx
    common / DYNadv_horrloc / sumMx(mz)
    ! +
    real dff
    real f0(mx, my, mz)
    real sumM0(mz), sumM1(mz)
    real sumP0(mz), sumP1(mz)
    real sumF0(mz)
    real rsum, rsumd, rsumds, rsumda, FlwPos
    ! +
#ifdef SP
    logical log_xx, log_yy
#endif
    ! +
    logical qqflux
    logical NestOK
    ! +
    ! +
    ! +--DATA
    ! +  ====
    ! +
    data qqflux/.false./
    data NestOK/.true./
    ! +
    ! +
    ! ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ! +
    ! +
    ! +--Conservation: Mass
    ! +  ==================
    ! +
    if(qqmass) then
        if(FirstC) then
#ifdef MC
            ! +
            ! +--Interior of the Model Domain
            ! +  ----------------------------
            ! +
            if(qqflux) then
                do k = 1, k_pdim
                    sumM0(k) = 0.0
                    sumM1(k) = 0.0
                    do j = j1_adh, j2_adh
                        do i = i1_adh, i2_adh
                            sumM0(k) = sumM0(k) + fp0(i, j)
                            sumM1(k) = sumM1(k) + fp1(i, j)
                        enddo
                    enddo
                enddo
            else
                do k = 1, k_pdim
                    sumM0(k) = 0.0
                    sumM1(k) = 0.0
                    do j = 1, my
                        do i = 1, mx
                            sumM0(k) = sumM0(k) + fp0(i, j)
                            sumM1(k) = sumM1(k) + fp1(i, j)
                        enddo
                    enddo
                enddo
            endif
            ! +
            ! +
            ! +--Mass Inflow / x-Lateral Boundaries
            ! +  ----------------------------------
            ! +
            if(mmx > 1 .and. qqflux) then
                ! +
                ! +--"x-small" Boundary
                ! +  ~~~~~~~~~~~~~~~~~~
                i = i1_adh
                do k = 1, k_pdim
                    do j = j1_adh, j2_adh
                        sumM0(k) = sumM0(k) + fp0(i, j) * fu(i, j, k) * dtx
                    enddo
                    ! +
                    ! +--"x-large" Boundary
                    ! +  ~~~~~~~~~~~~~~~~~~
                    i = i2_adh
                    do j = j1_adh, j2_adh
                        sumM0(k) = sumM0(k) - fp0(i, j) * fu(i, j, k) * dtx
                    enddo
                enddo
            endif
            ! +
            ! +
            ! +--Mass Inflow / y-Lateral Boundaries
            ! +  ----------------------------------
            ! +
            if(mmy > 1 .and. qqflux) then
                ! +
                ! +--"y-small" Boundary
                ! +  ~~~~~~~~~~~~~~~~~~
                j = j1_adh
                do k = 1, k_pdim
                    do i = i1_adh, i2_adh
                        sumM0(k) = sumM0(k) + fp0(i, j) * fv(i, j, k) * dtx
                    enddo
                    ! +
                    ! +--"y-large" Boundary
                    ! +  ~~~~~~~~~~~~~~~~~~
                    j = j2_adh
                    do i = i1_adh, i2_adh
                        sumM0(k) = sumM0(k) - fp0(i, j) * fv(i, j, k) * dtx
                    enddo
                enddo
            endif
            sumMav = 0.
#endif
            do k = 1, k_pdim
                sumMx(k) = 1.
#ifdef MC
                sumMx(k) = sumM0(k) / sumM1(k)
                sumMav = sumMav + sumMx(k) * dsigm1(k)
#endif
            enddo
#ifdef MC
            do j = j1_adh, j2_adh
                do i = i1_adh, i2_adh
                    fp1(i, j) = fp1(i, j) * sumMav
                enddo
            enddo
#endif
        endif
        ! +
        ! +
        ! +--Conservation: Property
        ! +  ======================
        ! +
        ! +--Interior of the Model Domain
        ! +  ----------------------------
        ! +
        if(qqflux) then
            do k = 1, k_pdim
                sumP0(k) = 0.0
                sumF0(k) = 0.0
                do j = j1_adh, j2_adh
                    do i = i1_adh, i2_adh
                        f0(i, j, k) = ff(i, j, k) * fp0(i, j)
                        sumP0(k) = sumP0(k) + f0(i, j, k)
                    enddo
                enddo
            enddo
        else
            do k = 1, k_pdim
                sumP0(k) = 0.0
                do j = 1, my
                    do i = 1, mx
                        f0(i, j, k) = ff(i, j, k) * fp0(i, j)
                        sumP0(k) = sumP0(k) + f0(i, j, k)
                    enddo
                enddo
            enddo
        endif
        ! +
        ! +
        ! +--Mass Inflow / x-Lateral Boundaries
        ! +  ----------------------------------
        ! +
        if(mmx > 1 .and. qqflux) then
            do k = 1, k_pdim
                i = i1_adh
                do j = j1_adh, j2_adh
                    sumF0(k) = sumF0(k) &
                               + ff(i, j, k) * fp0(i, j) * fu(i, j, k) * dtx
                enddo
                i = i2_adh
                do j = j1_adh, j2_adh
                    sumF0(k) = sumF0(k) &
                               - ff(i, j, k) * fp0(i, j) * fu(i, j, k) * dtx
                enddo
            enddo
        endif
        ! +
        ! +
        ! +--Mass Inflow / y-Lateral Boundaries
        ! +  ----------------------------------
        ! +
        if(mmy > 1 .and. qqflux) then
            do k = 1, k_pdim
                j = j1_adh
                do i = i1_adh, i2_adh
                    sumF0(k) = sumF0(k) &
                               + ff(i, j, k) * fp0(i, j) * fv(i, j, k) * dtx
                enddo
                j = j2_adh
                do i = i1_adh, i2_adh
                    sumF0(k) = sumF0(k) &
                               - ff(i, j, k) * fp0(i, j) * fv(i, j, k) * dtx
                enddo
            enddo
        endif
        if(qqflux) then
            do k = 1, k_pdim
                sumP0(k) = sumP0(k) + sumF0(k)
            enddo
        endif
        ! +
        ! +
        ! +--Positive Definiteness Condition
        ! +  -------------------------------
        ! +
        do k = 1, k_pdim
            sumP0(k) = max(sumP0(k), zero)
        enddo
        ! +
    endif
    ! +
    ! +
    ! +--Time Splitting (Alternate Direction)
    ! +  ====================================
    ! +
#ifdef SP
    log_xx = .false.
    log_yy = .false.
#endif
    ! +
300 continue
#ifdef SP
    if(mod(itexpe, 2) == 0 .and. .not. log_yy) go to 301
#endif
    ! +
    ! +
    ! +--Advection Contribution following x
    ! +  ==================================
    ! +
    if(mmx > 1) then
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz1(i, j, k) = fu(i, j, k) * dtx
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, LBC
        ! +  ---------------------------
        ! +
        do k = 1, k_pdim
            do j = jp11, my1
                WKxyz2(1, j, k) = 0.0
                WKxyz2(mx, j, k) = 0.0
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, Forward  Sweep
        ! +  --------------------------------------
        ! +
        do i = ip11, mx1
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz2(i, j, k) = (3.0 * (ff(i + 1, j, k) - ff(i - 1, j, k)) / dx &
                                       - WKxyz2(i - 1, j, k)) / CUspxh(i)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, Backward Sweep
        ! +  --------------------------------------
        ! +
        do i = mx2, ip11, -1
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz2(i, j, k) = CUspxb(i) * WKxyz2(i + 1, j, k) + WKxyz2(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Difference
        ! +  ----------------------
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz2(i, j, k) = WKxyz2(i, j, k) * dx
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Interpolated Variable
        ! +  ---------------------
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz3(i, j, k) = WKxyz1(i, j, k) * WKxyz1(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Direction of Advection
        ! +  ----------------------
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz4(i, j, k) = sign(unun, WKxyz1(i, j, k))
                    iunPos = WKxyz4(i, j, k)
                    iindex = max(1, i - iunPos)
                    iindex = min(mx, iindex)
                    WKxyz5(i, j, k) = WKxyz2(iindex, j, k)
                    WKxyz6(i, j, k) = WKxyz4(i, j, k) * (ff(iindex, j, k) - ff(i, j, k))
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Advection
        ! +  ---------
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz7(i, j, k) = WKxyz6(i, j, k) + WKxyz6(i, j, k) + WKxyz2(i, j, k)
                    WKxyz8(i, j, k) = WKxyz7(i, j, k) + WKxyz6(i, j, k) + WKxyz2(i, j, k)
                    WKxyz6(i, j, k) = ff(i, j, k) - WKxyz1(i, j, k) * WKxyz2(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    WKxyz6(i, j, k) = ff(i, j, k) - WKxyz1(i, j, k) * WKxyz2(i, j, k) &
                                      + WKxyz4(i, j, k) * WKxyz3(i, j, k) * (WKxyz8(i, j, k) + WKxyz5(i, j, k)) &
                                      - WKxyz1(i, j, k) * WKxyz3(i, j, k) * (WKxyz7(i, j, k) + WKxyz5(i, j, k))
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Inflow LBC
        ! +  ----------
        ! +
        if(qqflux) then
            ! +
            ! +--Large Scale Contribution over dt (dff, to be implemented)
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            if(NestOK) then
                dff = 0.
                ! +
                ! +--Host Model Solution is preferred ("outer" solution if inflow included)
                ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                do k = 1, k_pdim
                    do j = 1, my
                        WKxyz6(1, j, k) = ff(1, j, k) + dff
                        WKxyz6(mx, j, k) = ff(mx, j, k) + dff
                    enddo
                enddo
                ! +
                ! +--MAR        Solution is preferred ("inner" solution if inflow included)
                ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            else
                do k = 1, k_pdim
                    do j = 1, my
                        FlwPos = max(zero, sign(unun, WKxyz4(1, j, k)))
                        WKxyz6(1, j, k) = WKxyz6(1, j, k) * (unun - FlwPos) &
                                          + ff(1, j, k) * FlwPos
                        FlwPos = max(zero, sign(unun, WKxyz4(mx, j, k)))
                        WKxyz6(mx, j, k) = WKxyz6(mx, j, k) * FlwPos &
                                           + ff(mx, j, k) * (unun - FlwPos)
                    enddo
                enddo
            endif
            ! +
            ! +--Host Model Solution is preferred
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        else
            do k = 1, k_pdim
                do j = 1, my
                    WKxyz6(1, j, k) = ff(1, j, k)
                    WKxyz6(mx, j, k) = ff(mx, j, k)
                enddo
            enddo
        endif
        ! +
        ! +
        ! +--Finalisation
        ! +  ------------
        ! +
        do i = 1, mx
            do k = 1, k_pdim
                do j = jp11, my1
                    ff(i, j, k) = WKxyz6(i, j, k)
                    WKxyz1(i, j, k) = 0.0
                    WKxyz2(i, j, k) = 0.0
                    WKxyz3(i, j, k) = 0.0
                    WKxyz4(i, j, k) = 0.0
                    WKxyz5(i, j, k) = 0.0
                    WKxyz6(i, j, k) = 0.0
                    WKxyz7(i, j, k) = 0.0
                    WKxyz8(i, j, k) = 0.0
                enddo
            enddo
        enddo
    endif
    ! +
#ifdef SP
    log_xx = .true.
#endif
    ! +
    ! +
    ! ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ! +
    ! +
    ! +--Time Splitting (Alternate Direction)
    ! +  ====================================
    ! +
301 continue
#ifdef SP
    if(log_yy) go to 302
#endif
    ! +
    ! +
    ! +--Advection Contribution following y
    ! +  ==================================
    ! +
    if(mmy > 1) then
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz1(i, j, k) = fv(i, j, k) * dtx
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, LBC
        ! +  ---------------------------
        ! +
        do k = 1, k_pdim
            do i = ip11, mx1
                WKxyz2(1, 1, k) = 0.0
                WKxyz2(1, my, k) = 0.0
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, Forward  Sweep
        ! +  --------------------------------------
        ! +
        do j = jp11, my1
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz2(i, j, k) = (3.0 * (ff(i, j + 1, k) - ff(i, j - 1, k)) / dx &
                                       - WKxyz2(i, j - 1, k)) / CUspyh(j)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Derivative, Backward Sweep
        ! +  --------------------------------------
        ! +
        do j = my2, jp11, -1
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz2(i, j, k) = CUspyb(j) * WKxyz2(i, j + 1, k) + WKxyz2(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--First Order Difference
        ! +  ----------------------
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz2(i, j, k) = WKxyz2(i, j, k) * dx
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Interpolated Variable
        ! +  ---------------------
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz3(i, j, k) = WKxyz1(i, j, k) * WKxyz1(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Direction of Advection
        ! +  ----------------------
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz4(i, j, k) = sign(unun, WKxyz1(i, j, k))
                    junPos = WKxyz4(i, j, k)
                    jindex = max(1, j - junPos)
                    jindex = min(my, jindex)
                    WKxyz5(i, j, k) = WKxyz2(i, jindex, k)
                    WKxyz6(i, j, k) = WKxyz4(i, j, k) * (ff(i, jindex, k) - ff(i, j, k))
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Advection
        ! +  ---------
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    WKxyz7(i, j, k) = WKxyz6(i, j, k) + WKxyz6(i, j, k) + WKxyz2(i, j, k)
                    WKxyz8(i, j, k) = WKxyz7(i, j, k) + WKxyz6(i, j, k) + WKxyz2(i, j, k)
                    WKxyz6(i, j, k) = ff(i, j, k) - WKxyz1(i, j, k) * WKxyz2(i, j, k)
                enddo
            enddo
        enddo
        ! +
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    ff(i, j, k) = ff(i, j, k) - WKxyz1(i, j, k) * WKxyz2(i, j, k) &
                                  + WKxyz4(i, j, k) * WKxyz3(i, j, k) * (WKxyz8(i, j, k) + WKxyz5(i, j, k)) &
                                  - WKxyz1(i, j, k) * WKxyz3(i, j, k) * (WKxyz7(i, j, k) + WKxyz5(i, j, k))
                enddo
            enddo
        enddo
        ! +
        ! +
        ! +--Inflow LBC
        ! +  ----------
        ! +
        if(qqflux) then
            ! +
            ! +--Large Scale Contribution over dt (dff, to be implemented)
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            if(NestOK) then
                dff = 0.
                ! +
                ! +--Host Model Solution is preferred ("outer" solution if inflow included)
                ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                do k = 1, k_pdim
                    do i = 1, mx
                        WKxyz6(i, 1, k) = ff(i, 1, k)
                        WKxyz6(i, my, k) = ff(i, my, k)
                    enddo
                enddo
                ! +
                ! +--Nested Model Solution is preferred
                ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                ! +
                ! +--MAR        Solution is preferred ("inner" solution if inflow included)
                ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            else
                do k = 1, k_pdim
                    do i = 1, mx
                        FlwPos = max(zero, sign(unun, WKxyz4(i, 1, k)))
                        WKxyz6(i, 1, k) = WKxyz6(i, 1, k) * (unun - FlwPos) &
                                          + ff(i, 1, k) * FlwPos
                        FlwPos = max(zero, sign(unun, WKxyz4(i, my, k)))
                        WKxyz6(i, my, k) = WKxyz6(i, my, k) * FlwPos &
                                           + ff(i, my, k) * (unun - FlwPos)
                    enddo
                enddo
            endif
            ! +
            ! +--Host Model Solution is preferred
            ! +  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
        else
            do k = 1, k_pdim
                do i = 1, mx
                    WKxyz6(i, 1, k) = ff(i, 1, k)
                    WKxyz6(i, my, k) = ff(i, my, k)
                enddo
            enddo
        endif
        ! +
        ! +
        ! +--Finalisation
        ! +  ------------
        ! +
        do j = 1, my
            do k = 1, k_pdim
                do i = ip11, mx1
                    ff(i, j, k) = WKxyz6(i, j, k)
                    WKxyz1(i, j, k) = 0.0
                    WKxyz2(i, j, k) = 0.0
                    WKxyz3(i, j, k) = 0.0
                    WKxyz4(i, j, k) = 0.0
                    WKxyz5(i, j, k) = 0.0
                    WKxyz6(i, j, k) = 0.0
                    WKxyz7(i, j, k) = 0.0
                    WKxyz8(i, j, k) = 0.0
                enddo
            enddo
        enddo
    endif
    ! +
#ifdef SP
    log_yy = .true.
    if(.not. log_xx) go to 300
#endif
302 continue
    ! +
    ! +
    ! +--Conservation
    ! +  ============
    ! +
    if(qqmass) then
        ! +
        ! +
        ! +--Fluxes at the Lateral Boundaries      through Advection
        ! +  -------------------------------------------------------
        ! +
        if(qqflux) then
            do k = 1, k_pdim
                sumP1(k) = 0.0
                do j = j1_adh, j2_adh
                    do i = i1_adh, i2_adh
                        WKxy1(i, j) = ff(i, j, k) * fp1(i, j)
                        sumP1(k) = sumP1(k) + WKxy1(i, j)
                    enddo
                enddo
                ! +
                rsumd = sumP1(k)
#ifdef MD
                rsumd = sumP1(k) - sumP0(k)
#endif
                rsumds = sign(unun, rsumd)
                rsumda = abs(rsumd)
                rsumd = max(eps9, rsumda) * rsumds
                rsum = sumP0(k) / rsumd
#ifdef MD
                rsum = sumF0(k) / rsumd
#endif
                do j = j1_adh, j2_adh
                    do i = i1_adh, i2_adh
                        ff(i, j, k) = ff(i, j, k) * rsum
#ifdef MD
                        ff(i, j, k) = (f0(i, j, k) &
                                       + (WKxy1(i, j) - f0(i, j, k)) * rsum) &
                                      / fp1(i, j)
#endif
                    enddo
                enddo
                ! +
            enddo
            ! +
            do j = 1, my
                do i = 1, mx
                    WKxy1(i, j) = 0.0
                enddo
            enddo
            ! +
            ! +
            ! +--Fluxes at the Lateral Boundaries only through the nudging Procedure
            ! +  -------------------------------------------------------------------
            ! +
        else
            ! +
            do k = 1, k_pdim
                sumP1(k) = 0.0
                do j = 1, my
                    do i = 1, mx
                        sumP1(k) = sumP1(k) + ff(i, j, k) * fp1(i, j)
                    enddo
                enddo
                ! +
                rsumd = sumP1(k) * sumMx(k)
                rsumds = sign(unun, rsumd)
                rsumda = abs(rsumd)
                rsumd = max(eps9, rsumda) * rsumds
                rsum = sumP0(k) / rsumd
                do j = 1, my
                    do i = 1, mx
                        ff(i, j, k) = ff(i, j, k) * rsum
                    enddo
                enddo
                ! +
            enddo
        endif
        ! +
    endif
    ! +
    ! +
    ! ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ! +
    ! +
    return
end
