MODULE MODD_CONVPAR_SHAL
!     ########################
!
!!****  *MODD_CONVPAR_SHAL* - Declaration of convection constants
!!
!!    PURPOSE
!!    -------
!!      The purpose of this declarative module is to declare  the
!!      constants in the deep convection parameterization.
!!
!!
!!**  IMPLICIT ARGUMENTS
!!    ------------------
!!      None
!!
!!    REFERENCE
!!    ---------
!!      Book2 of documentation of Meso-NH (MODD_CONVPAR_SHAL)
!!
!!    AUTHOR
!!    ------
!!      P. Bechtold   *Laboratoire d'Aerologie*
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  04/10/98
!!      E. Bazile   05/05/09
!-------------------------------------------------------------------------------
!
!*       0.   DECLARATIONS
!             ------------
!
    implicit none
!
    REAL, SAVE :: XA25        ! 25 km x 25 km reference grid area
!
    REAL, SAVE :: XCRAD       ! cloud radius
    REAL, SAVE :: XCTIME_SHAL ! convective adjustment time
    REAL, SAVE :: XCDEPTH     ! minimum necessary cloud depth
    REAL, SAVE :: XCDEPTH_D   ! maximum allowed cloud thickness
    REAL, SAVE :: XDTPERT     ! add small Temp perturb. at LCL
    REAL, SAVE :: XATPERT     ! Parameter for temp Perturb
    REAL, SAVE :: XBTPERT     ! Parameter for temp Perturb
    ! (XATPERT* TKE/Cp + XBTPERT) * XDTPERT
    REAL, SAVE :: XENTR       ! entrainment constant (m/Pa) = 0.2 (m)
!
    REAL, SAVE :: XZLCL       ! maximum allowed allowed height
    ! difference between departure level and surface
    REAL, SAVE :: XZPBL       ! minimum mixed layer depth to sustain convection
    REAL, SAVE :: XWTRIG      ! constant in vertical velocity trigger
!
!
    REAL, SAVE :: XNHGAM      ! accounts for non-hydrost. pressure
    ! in buoyancy term of w equation
    ! = 2 / (1+gamma)
    REAL, SAVE :: XTFRZ1      ! begin of freezing interval
    REAL, SAVE :: XTFRZ2      ! end of freezing interval
!
!
    REAL, SAVE :: XSTABT      ! factor to assure stability in  fractional time
    ! integration, routine CONVECT_CLOSURE
    REAL, SAVE :: XSTABC      ! factor to assure stability in CAPE adjustment,
    !  routine CONVECT_CLOSURE
    REAL, SAVE :: XAW, XBW     ! Parameters for WLCL = XAW * W + XBW
    LOGICAL, SAVE :: LLSMOOTH ! Default=TRUE but not necessary
!$OMP threadprivate(XA25,XCRAD,XCTIME_SHAL,XCDEPTH,XCDEPTH_D,XDTPERT,XATPERT, &
!$OMP XENTR,XZLCL,XZPBL,XWTRIG,XNHGAM,XTFRZ1,XTFRZ2,XSTABT,XSTABC, &
!$OMP XAW,XBW ,LLSMOOTH)
ENDMODULE MODD_CONVPAR_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

MODULE MODD_CONVPAR
!     ###################
!
!!****  *MODD_CONVPAR* - Declaration of convection constants
!!
!!    PURPOSE
!!    -------
!      The purpose of this declarative module is to declare  the
!      constants in the deep convection parameterization.
!
!!
!!**  IMPLICIT ARGUMENTS
!!    ------------------
!!      None
!!
!!    REFERENCE
!!    ---------
!!      Book2 of documentation of Meso-NH (MODD_CONVPAR)
!!
!!    AUTHOR
!!    ------
!!      P. Bechtold   *Laboratoire d'Aerologie*
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  15/11/96
!-------------------------------------------------------------------------------
!
!*       0.   DECLARATIONS
!             ------------
!
    implicit none
!
    REAL, SAVE :: XA25        ! 25 km x 25 km reference grid area
!
    REAL, SAVE :: XCRAD       ! cloud radius
    REAL, SAVE :: XCDEPTH     ! minimum necessary cloud depth
    REAL, SAVE :: XENTR       ! entrainment constant (m/Pa) = 0.2 (m)
!
    REAL, SAVE :: XZLCL       ! maximum allowed allowed height
    ! difference between departure level and surface
    REAL, SAVE :: XZPBL       ! minimum mixed layer depth to sustain convection
    REAL, SAVE :: XWTRIG      ! constant in vertical velocity trigger
!
!
    REAL, SAVE :: XNHGAM      ! accounts for non-hydrost. pressure
    ! in buoyancy term of w equation
    ! = 2 / (1+gamma)
    REAL, SAVE :: XTFRZ1      ! begin of freezing interval
    REAL, SAVE :: XTFRZ2      ! end of freezing interval
!
    REAL, SAVE :: XRHDBC      ! relative humidity below cloud in downdraft
!
    REAL, SAVE :: XRCONV      ! constant in precipitation conversion
    REAL, SAVE :: XSTABT      ! factor to assure stability in  fractional time
    ! integration, routine CONVECT_CLOSURE
    REAL, SAVE :: XSTABC      ! factor to assure stability in CAPE adjustment,
    !  routine CONVECT_CLOSURE
    REAL, SAVE :: XUSRDPTH    ! pressure thickness used to compute updraft
    ! moisture supply rate for downdraft
    REAL, SAVE :: XMELDPTH    ! layer (Pa) through which precipitation melt is
    ! allowed below  melting level
    REAL, SAVE :: XUVDP       ! constant for pressure perturb in momentum transport
!$OMP threadprivate(XA25,XCRAD,XCDEPTH,XENTR,XZLCL,XZPBL,XWTRIG,XNHGAM,XTFRZ1, &
!$OMP XTFRZ2,XRHDBC,XRCONV,XSTABT,XSTABC,XUSRDPTH,XMELDPTH,XUVDP)
ENDMODULE MODD_CONVPAR

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!       #####################
MODULE MODI_INI_CONVPAR_E1
!       #####################
!
    INTERFACE
!
        subroutine INI_CONVPAR_E1
        ENDsubroutine INI_CONVPAR_E1
!
    ENDINTERFACE
!
ENDMODULE MODI_INI_CONVPAR_E1
!
!
!     #########################
subroutine INI_CONVPAR_E1
!     #########################
!
!!****  *INI_CONVPAR * - routine to initialize the convective constants modules
!!                       with modifications for ensemble run.
!!
!!    PURPOSE
!!    -------
!       The purpose of this routine is to initialize  the constants
!     stored in  modules MODD_CONVPAR, MODD_CST, MODD_CONVPAREXT.
!
!
!!**  METHOD
!!    ------
!!      The deep convection constants are set to their numerical values
!!
!!
!!    EXTERNAL
!!    --------
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CONVPAR   : contains deep convection constants
!!
!!    REFERENCE
!!    ---------
!!      Book2 of the documentation (module MODD_CONVPAR, routine INI_CONVPAR)
!!
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  15/04/98 adapted for ARPEGE
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAR
!
    implicit none
!
!-------------------------------------------------------------------------------
!
!*       1.    Set the thermodynamical and numerical constants for
!              the deep convection parameterization
!              ---------------------------------------------------
!
!
    XA25 = 100.E6    ! 25 km x 25 km reference grid area
!
    XCRAD = 500.     ! cloud radius
    XCDEPTH = 3.E3      ! minimum necessary cloud depth
    XENTR = 0.03      ! entrainment constant (m/Pa) = 0.2 (m)
!
    XZLCL = 3.5E3     ! maximum allowed allowed height
    ! difference between the surface and the LCL
    XZPBL = 60.E2     ! minimum mixed layer depth to sustain convection
    XWTRIG = 6.00      ! constant in vertical velocity trigger
!
!
    XNHGAM = 1.3333    ! accounts for non-hydrost. pressure
    ! in buoyancy term of w equation
    ! = 2 / (1+gamma)
    XTFRZ1 = 268.16    ! begin of freezing interval
    XTFRZ2 = 248.16    ! end of freezing interval
!
    XRHDBC = 0.9       ! relative humidity below cloud in downdraft

    XRCONV = 0.015     ! constant in precipitation conversion
    XSTABT = 0.75      ! factor to assure stability in  fractional time
    ! integration, routine CONVECT_CLOSURE
    XSTABC = 0.95      ! factor to assure stability in CAPE adjustment,
    !  routine CONVECT_CLOSURE
    XUSRDPTH = 165.E2    ! pressure thickness used to compute updraft
    ! moisture supply rate for downdraft
    XMELDPTH = 200.E2    ! layer (Pa) through which precipitation melt is
    ! allowed below downdraft
    XUVDP = 0.7       ! constant for pressure perturb in momentum transport
!
!
ENDsubroutine INI_CONVPAR_E1

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 init 2006/05/18 13:07:25
!-----------------------------------------------------------------
!       #####################
MODULE MODI_INI_CONVPAR
!       #####################
!
    INTERFACE
!
        subroutine INI_CONVPAR
        ENDsubroutine INI_CONVPAR
!
    ENDINTERFACE
!
ENDMODULE MODI_INI_CONVPAR
!
!
!
!     ######################
subroutine INI_CONVPAR
!     ######################
!
!!****  *INI_CONVPAR * - routine to initialize the constants modules
!!
!!    PURPOSE
!!    -------
!       The purpose of this routine is to initialize  the constants
!     stored in  modules MODD_CONVPAR, MODD_CST, MODD_CONVPAREXT.
!
!
!!**  METHOD
!!    ------
!!      The deep convection constants are set to their numerical values
!!
!!
!!    EXTERNAL
!!    --------
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CONVPAR   : contains deep convection constants
!!
!!    REFERENCE
!!    ---------
!!      Book2 of the documentation (module MODD_CONVPAR, routine INI_CONVPAR)
!!
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  15/04/98 adapted for ARPEGE
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAR
!
    implicit none
!
!-------------------------------------------------------------------------------
!
!*       1.    Set the thermodynamical and numerical constants for
!              the deep convection parameterization
!              ---------------------------------------------------
!
!
    XA25 = 625.E6    ! 25 km x 25 km reference grid area
!
    XCRAD = 1500.     ! cloud radius
    XCDEPTH = 2.5E3      ! minimum necessary cloud depth
    XENTR = 0.03      ! entrainment constant (m/Pa) = 0.2 (m)
!
    XZLCL = 3.5E3     ! maximum allowed allowed height
    ! difference between the surface and the LCL
    XZPBL = 60.E2     ! minimum mixed layer depth to sustain convection
    XWTRIG = 6.00      ! constant in vertical velocity trigger
!
!
    XNHGAM = 1.3333    ! accounts for non-hydrost. pressure
    ! in buoyancy term of w equation
    ! = 2 / (1+gamma)
    XTFRZ1 = 268.16    ! begin of freezing interval
    XTFRZ2 = 248.16    ! end of freezing interval
!
    XRHDBC = 0.9       ! relative humidity below cloud in downdraft

    XRCONV = 0.015     ! constant in precipitation conversion
    XSTABT = 0.75      ! factor to assure stability in  fractional time
    ! integration, routine CONVECT_CLOSURE
    XSTABC = 0.95      ! factor to assure stability in CAPE adjustment,
    !  routine CONVECT_CLOSURE
    XUSRDPTH = 165.E2    ! pressure thickness used to compute updraft
    ! moisture supply rate for downdraft
    XMELDPTH = 100.E2    ! layer (Pa) through which precipitation melt is
    ! allowed below downdraft
    XUVDP = 0.7       ! constant for pressure perturb in momentum transport
!
!
ENDsubroutine INI_CONVPAR

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 init 2006/05/18 13:07:25
!-----------------------------------------------------------------
!       #####################
MODULE MODI_INI_CONVPAR_SHAL
!       #####################
!
    INTERFACE
!
        subroutine INI_CONVPAR_SHAL
        ENDsubroutine INI_CONVPAR_SHAL
!
    ENDINTERFACE
!
ENDMODULE MODI_INI_CONVPAR_SHAL
!
!
!     ###########################
subroutine INI_CONVPAR_SHAL
!     ###########################
!
!!****  *INI_CONVPAR * - routine to initialize the constants modules
!!
!!    PURPOSE
!!    -------
!!       The purpose of this routine is to initialize  the constants
!!     stored in  modules MODD_CONVPAR_SHAL
!!
!!
!!**  METHOD
!!    ------
!!      The shallow convection constants are set to their numerical values
!!
!!
!!    EXTERNAL
!!    --------
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CONVPAR_SHAL   : contains deep convection constants
!!
!!    REFERENCE
!!    ---------
!!      Book2 of the documentation (module MODD_CONVPAR_SHAL, routine INI_CONVPAR)
!!
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  15/04/98 adapted for ARPEGE
!!                  05/05/09 E. Bazile
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAR_SHAL
!
    implicit none
!
!-------------------------------------------------------------------------------
!
!*       1.    Set the thermodynamical and numerical constants for
!              the deep convection parameterization
!              ---------------------------------------------------
!
!
    XA25 = 625.E6    ! 25 km x 25 km reference grid area
!
    XCRAD = 50.    ! cloud radius
    XCTIME_SHAL = 10800. ! convective adjustment time
    XCDEPTH = 0.5E3  ! minimum necessary shallow cloud depth
    XCDEPTH_D = 2.5E3  ! maximum allowed shallow cloud depth
    XDTPERT = .2     ! add small Temp perturbation at LCL
    XATPERT = 0.     ! 0.=original scheme , recommended = 1000.
    XBTPERT = 1.     ! 1.=original scheme , recommended = 0.
!
    XENTR = 0.02      ! entrainment constant (m/Pa) = 0.2 (m)
!
    XZLCL = 0.5E3     ! maximum allowed allowed height
    ! difference between the DPL and the surface
    XZPBL = 40.E2     ! minimum mixed layer depth to sustain convection
!
!
    XNHGAM = 1.3333    ! accounts for non-hydrost. pressure
    ! in buoyancy term of w equation
    ! = 2 / (1+gamma)
    XTFRZ1 = 268.16    ! begin of freezing interval
    XTFRZ2 = 248.16    ! end of freezing interval
!

    XSTABT = 0.75      ! factor to assure stability in  fractional time
    ! integration, routine CONVECT_CLOSURE
    XSTABC = 0.95      ! factor to assure stability in CAPE adjustment,
    !  routine CONVECT_CLOSURE
    XAW = 0.        ! 0.= Original scheme , 1 = recommended
    XBW = 1.        ! 1.= Original scheme , 0 = recommended
    LLSMOOTH = .true.
!
!
ENDsubroutine INI_CONVPAR_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!     ######spl
subroutine CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                      PDTCONV, ODEEP, OSHAL, OREFRESH_ALL, ODOWN, KICE, &
                      OSETTADJ, PTADJD, PTADJS, &
                      KENSM, &
                      PPABS, PZZ, PDXDY, &
                      PT, PRV, PRC, PRI, PU, PV, PW, &
                      KCOUNT, PTTEN, PRVTEN, PRCTEN, PRITEN, &
                      PPRTEN, PPRSTEN, &
                      PUMF, PDMF, PPRLFLX, PPRSFLX, PCAPE, KCLTOP, KCLBAS, &
                      OCHTRANS, KCH1, PCH1, PCH1TEN, &
                      SeBi_TKE)
!   ############################################################################
!
!!**** Interface routine to the fast MNH convection code developed for ECMWF/ARPEGE IFS
!!     having a structure typical for operational routines
!!
!!
!!    PURPOSE
!!    -------
!!      The routine interfaces the MNH convection code as developed for operational
!!      forecast models like ECMWF, ARPEGE or HIRLAM with the typical MNH array structure
!!      Calls the deep and/or shallow convection routine
!!
!!
!!**  METHOD
!!    ------
!!     Returns one tendency for shallow+deep convection but each part can
!!     be activated/desactivated separately.
!!     For deep convection one can enable up to 3 additional ensemble members
!!     - this substantially improves the smoothness of the scheme and
!!       allows for runs with different cloud radii (entrainment rates) and
!!       reduces the arbitrariness inherent to convective trigger condition
!!
!!
!!
!!    EXTERNAL
!!    --------
!!    CONVECT_DEEP
!!    CONVECT_SHALLOW
!!    INI_CONVPAR, INI_CONVPAR1
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    11/12/98
!!      Modif  11/04/O2 allow for ensemble of deep updrafts/downdrafts
!!
!!    REFERENCE
!!    ---------
!!    Bechtold et al., 2001, Quart. J. Roy. Meteor. Soc., Vol 127, pp 869-886:
!!           A mass flux convection scheme for regional and global models.
!!
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
    INTEGER, INTENT(IN) :: KIDIA    ! value of the first point in x
    INTEGER, INTENT(IN) :: KFDIA    ! value of the last point in x
    INTEGER, INTENT(IN) :: KBDIA    ! vertical  computations start at
!                                                ! KBDIA that is at least 1
    INTEGER, INTENT(IN) :: KTDIA    ! vertical computations can be
    ! limited to KLEV + 1 - KTDIA
    ! default=1
    REAL, INTENT(IN) :: PDTCONV  ! Interval of time between two
    ! calls of the deep convection
    ! scheme
    LOGICAL, INTENT(IN) :: ODEEP    ! switch for deep convection
    LOGICAL, INTENT(IN) :: OSHAL    ! switch for shallow convection
    LOGICAL, INTENT(IN) :: OREFRESH_ALL ! refresh or not all
    ! tendencies  at every call
    LOGICAL, INTENT(IN) :: ODOWN    ! take or not convective
    ! downdrafts into account
    INTEGER, INTENT(IN) :: KICE     ! flag for ice ( 1 = yes,
    !                0 = no ice )
    INTEGER, INTENT(IN) :: KENSM    ! number of additional deep convection calls
    ! for ensemble (presently limited to 3)
    ! KENSM=0 corresponds to base run with
    ! 1 deep and 1 shallow call
    LOGICAL, INTENT(IN) :: OSETTADJ ! logical to set convective
    ! adjustment time by user
    REAL, INTENT(IN) :: PTADJD   ! user defined deep adjustment time (s)
    REAL, INTENT(IN) :: PTADJS   ! user defined shal. adjustment time (s)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PT     ! grid scale T at time t  (K)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV    ! grid scale water vapor  (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC    ! grid scale r_c  (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI    ! grid scale r_i  (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PU     ! grid scale horiz. wind u (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PV     ! grid scale horiz. wind v (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PW     ! grid scale vertical velocity (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPABS  ! grid scale pressure (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZZ    ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m2)

!
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCOUNT   ! convective counter(recompute
    ! tendency or keep it
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PTTEN  ! convective temperat. tendency (K/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRVTEN ! convective r_v tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRCTEN ! convective r_c tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRITEN ! convective r_i tendency (1/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PPRTEN ! total surf precipitation tendency (m/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PPRSTEN! solid surf precipitation tendency (m/s)
!
! Chemical Tracers:
    LOGICAL, INTENT(IN)        :: OCHTRANS ! flag to compute convective
    ! transport for chemical tracer
    INTEGER, INTENT(IN)        :: KCH1     ! number of species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(IN)   :: PCH1     ! grid scale chemical species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(INOUT):: PCH1TEN  ! chemical convective tendency
    ! (1/s)
!
! Diagnostic variables:
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF   ! updraft mass flux   (kg/s m2)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDMF   ! downdraft mass flux (kg/s m2)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PPRLFLX! liquid precip flux  (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PPRSFLX! solid precip flux   (m/s)
    REAL, DIMENSION(KLON), INTENT(INOUT)    :: PCAPE  ! CAPE (J/kg)
    INTEGER, DIMENSION(KLON), INTENT(INOUT)    :: KCLTOP ! cloud top level (number of model level)
    INTEGER, DIMENSION(KLON), INTENT(INOUT)    :: KCLBAS ! cloud base level(number of model level)
    ! they are given a value of
    ! 0 if no convection
!
!*       0.2   Declarations of local variables :
!
    INTEGER  :: JI, JK, JN  ! loop index
!
    REAL, DIMENSION(KLON)               :: ZTIMEC, ZPRLTEN
!
! special for shallow convection
    REAL, DIMENSION(:, :), ALLOCATABLE   :: ZTTENS, ZRVTENS, ZRCTENS, ZRITENS, &
                                            ZUMFS
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZCH1TENS
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ICLBASS, ICLTOPS
!
!*       0.3   Declarations of additional Ensemble fields:
!
    integer                 :: KENS     ! number of allowed additional deep convection calls
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZTTENE   ! convective temperat. tendency (K/s)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZRVTENE  ! convective r_v tendency (1/s)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZRCTENE  ! convective r_c tendency (1/s)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZRITENE  ! convective r_i tendency (1/s)
    REAL, DIMENSION(:, :), ALLOCATABLE :: ZPRLTENE ! liquid surf precipitation tendency (m/s)
    REAL, DIMENSION(:, :), ALLOCATABLE :: ZPRSTENE ! solid surf precipitation tendency (m/s)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZUMFE    ! updraft mass flux   (kg/s m2)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZDMFE    ! downdraft mass flux (kg/s m2)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZPRLFLXE ! liquid precip flux  (m/s)
    REAL, DIMENSION(:, :, :), ALLOCATABLE :: ZPRSFLXE ! solid precip flux   (m/s)
    REAL, DIMENSION(:, :, :, :), ALLOCATABLE:: ZCH1TENE ! chemical convective tendency
    INTEGER, DIMENSION(:, :), ALLOCATABLE :: ICLTOPE  ! cloud top level (number of model level)
    INTEGER, DIMENSION(:, :), ALLOCATABLE :: ICLBASE  ! cloud base level(number of model level)
    REAL, DIMENSION(:), ALLOCATABLE :: ZEDUMMY  ! field not to be recomputed by ensemble
    INTEGER, DIMENSION(:), ALLOCATABLE :: IEDUMMY  ! field not to be recomputed by ensemble
    REAL, DIMENSION(:), ALLOCATABLE :: ZWEIGHT  ! weighting factor for ensemble members
    real                 :: ZSUM     ! sum of weighting factors

    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: SeBi_TKE   ! TKE
!
!-------------------------------------------------------------------------------
!
!
!*       0.5  Allocate 2D (horizontal, vertical) arrays and additional ensemble arrays
!             ------------------------------------------------------------------------
!
    ALLOCATE(ZTTENS(KLON, KLEV)); ALLOCATE(ZRVTENS(KLON, KLEV))
    ALLOCATE(ZRCTENS(KLON, KLEV)); ALLOCATE(ZRITENS(KLON, KLEV))
    ALLOCATE(ZCH1TENS(KLON, KLEV, KCH1))
    ALLOCATE(ZUMFS(KLON, KLEV))
    ALLOCATE(ICLBASS(KLON)); ALLOCATE(ICLTOPS(KLON))
!
    KCLTOP(:) = 1 ! set default value when no convection
    KCLBAS(:) = 1 ! can be changed  depending on user
    ICLTOPS(:) = 1
    ICLBASS(:) = 1
!
    KENS = MIN(KENSM, 3)
    if(KENS > 0) then
        ALLOCATE(ZTTENE(KLON, KLEV, KENS))
        ALLOCATE(ZRVTENE(KLON, KLEV, KENS))
        ALLOCATE(ZRCTENE(KLON, KLEV, KENS))
        ALLOCATE(ZRITENE(KLON, KLEV, KENS))
        ALLOCATE(ZUMFE(KLON, KLEV, KENS))
        ALLOCATE(ZDMFE(KLON, KLEV, KENS))
        ALLOCATE(ZCH1TENE(KLON, KLEV, KCH1, KENS))
        ALLOCATE(ZPRLFLXE(KLON, KLEV, KENS))
        ALLOCATE(ZPRSFLXE(KLON, KLEV, KENS))
        ALLOCATE(ZPRLTENE(KLON, KENS))
        ALLOCATE(ZPRSTENE(KLON, KENS))
        ALLOCATE(ICLTOPE(KLON, KENS))
        ALLOCATE(ICLBASE(KLON, KENS))
        ALLOCATE(ZEDUMMY(KLON))
        ALLOCATE(IEDUMMY(KLON))
        ALLOCATE(ZWEIGHT(KENS))
    endif
!
!*       4.a  Call deep convection routine
!             ----------------------------
!
    if(ODEEP) then
!
! 1. Base version
!
        call INI_CONVPAR
!
        if(OSETTADJ) ZTIMEC(:) = PTADJD

!

!print*, "PRV beg",MAXVAL(PTTEN),MAXVAL(PRVTEN),MAXVAL(PRCTEN),MAXVAL(PRITEN)
        call DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                             PDTCONV, KICE, OREFRESH_ALL, ODOWN, OSETTADJ, &
                             PPABS, PZZ, PDXDY, ZTIMEC, &
                             PT, PRV, PRC, PRI, PU, PV, PW, &
                             KCOUNT, PTTEN, PRVTEN, PRCTEN, PRITEN, &
                             ZPRLTEN, PPRSTEN, &
                             KCLTOP, KCLBAS, PPRLFLX, PPRSFLX, &
                             PUMF, PDMF, PCAPE, &
                             OCHTRANS, KCH1, PCH1, PCH1TEN)
!
!  2. Additional Ensemble members
!
        if(KENS > 0) then
!
            call INI_CONVPAR_E1
!
!* first member - changes in MODD_CONVPAR (cloud radius of 500 m or so)
!                                          specified in INI_CONVPAR1
!
            call DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                 PDTCONV, KICE, OREFRESH_ALL, ODOWN, OSETTADJ, &
                                 PPABS, PZZ, PDXDY, ZTIMEC, &
                                 PT, PRV, PRC, PRI, PU, PV, PW, &
                                 IEDUMMY, ZTTENE(:, :, 1), ZRVTENE(:, :, 1), ZRCTENE(:, :, 1), ZRITENE(:, :, 1), &
                                 ZPRLTENE(:, 1), ZPRSTENE(:, 1), &
                                 ICLTOPE(:, 1), ICLBASE(:, 1), ZPRLFLXE(:, :, 1), ZPRSFLXE(:, :, 1), &
                                 ZUMFE(:, :, 1), ZDMFE(:, :, 1), ZEDUMMY, &
                                 OCHTRANS, KCH1, PCH1, ZCH1TENE(:, :, :, 1))
        endif
!
        if(KENS > 1) then
!
            call INI_CONVPAR
!
!* second member (positive vertical velocity perturb for Trigger)
!
            call DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                 PDTCONV, KICE, OREFRESH_ALL, ODOWN, OSETTADJ, &
                                 PPABS, PZZ, PDXDY, ZTIMEC, &
                                 PT, PRV, PRC, PRI, PU, PV, PW * 1.5 + 1.e-4, &
                                 IEDUMMY, ZTTENE(:, :, 2), ZRVTENE(:, :, 2), ZRCTENE(:, :, 2), ZRITENE(:, :, 2), &
                                 ZPRLTENE(:, 2), ZPRSTENE(:, 2), &
                                 ICLTOPE(:, 2), ICLBASE(:, 2), ZPRLFLXE(:, :, 2), ZPRSFLXE(:, :, 2), &
                                 ZUMFE(:, :, 2), ZDMFE(:, :, 2), ZEDUMMY, &
                                 OCHTRANS, KCH1, PCH1, ZCH1TENE(:, :, :, 2))
        endif
!
        if(KENS > 2) then
!
!* third member (negative vertical velocity perturb for Trigger)
!
            call DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                 PDTCONV, KICE, OREFRESH_ALL, ODOWN, OSETTADJ, &
                                 PPABS, PZZ, PDXDY, ZTIMEC, &
                                 PT, PRV, PRC, PRI, PU, PV, PW*.5 - 1.e-4, &
                                 IEDUMMY, ZTTENE(:, :, 3), ZRVTENE(:, :, 3), ZRCTENE(:, :, 3), ZRITENE(:, :, 3), &
                                 ZPRLTENE(:, 3), ZPRSTENE(:, 3), &
                                 ICLTOPE(:, 3), ICLBASE(:, 3), ZPRLFLXE(:, :, 3), ZPRSFLXE(:, :, 3), &
                                 ZUMFE(:, :, 3), ZDMFE(:, :, 3), ZEDUMMY, &
                                 OCHTRANS, KCH1, PCH1, ZCH1TENE(:, :, :, 3))
        endif
!
!print*, "PRV end",MAXVAL(PTTEN),MAXVAL(PRVTEN),MAXVAL(PRCTEN),MAXVAL(PRITEN)
    endif
    if(.not. ODEEP) then
        KCOUNT(:) = 0
        PTTEN(:, :) = 0.
        PRVTEN(:, :) = 0.
        PRCTEN(:, :) = 0.
        PRITEN(:, :) = 0.
        PUMF(:, :) = 0.
        PDMF(:, :) = 0.
        ! KCLTOP(:)  =1
        ! KCLBAS(:)  =1
        PCH1TEN(:, :, :) = 0.
        ZPRLTEN(:) = 0.
        PPRSTEN(:) = 0.
        PPRLFLX(:, :) = 0.
        PPRSFLX(:, :) = 0.
        PCAPE(:) = 0.
    endif

!
!*       4.b  Call shallow convection routine
!             -------------------------------
!
    if(OSHAL) then
!
        if(.not. ODEEP) call INI_CONVPAR
        call INI_CONVPAR_SHAL
!
        call SHALLOW_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                PDTCONV, KICE, OSETTADJ, PTADJS, &
                                PPABS, PZZ, SeBi_TKE(:, KBDIA + 1), &
                                PT, PRV, PRC, PRI, PW, &
                                ZTTENS, ZRVTENS, ZRCTENS, ZRITENS, &
                                ICLTOPS, ICLBASS, ZUMFS, &
                                OCHTRANS, KCH1, PCH1, ZCH1TENS)
    endif
    if(.not. OSHAL) then
        ZTTENS(:, :) = 0.
        ZRVTENS(:, :) = 0.
        ZRCTENS(:, :) = 0.
        ZRITENS(:, :) = 0.
        ZUMFS(:, :) = 0
        ! ICLTOPS(:)  =1
        ! ICLBASS(:)  =1
        ZCH1TENS(:, :, :) = 0.
    endif
!
!*       5.  Add  - if activated - ensemble average values for deep
!            and then shallow convective tendencies
!            ---------------------------------------------------------
!
    ZSUM = 1.
    if(KENS > 0) then
        if(KENS == 1) ZWEIGHT(:) = .5
        if(KENS > 1) ZWEIGHT(:) = 1.
        do JN = 1, KENS
            PTTEN(:, :) = PTTEN(:, :) + ZWEIGHT(JN) * ZTTENE(:, :, JN)
            PRVTEN(:, :) = PRVTEN(:, :) + ZWEIGHT(JN) * ZRVTENE(:, :, JN)
            PRCTEN(:, :) = PRCTEN(:, :) + ZWEIGHT(JN) * ZRCTENE(:, :, JN)
            PRITEN(:, :) = PRITEN(:, :) + ZWEIGHT(JN) * ZRITENE(:, :, JN)
            PPRLFLX(:, :) = PPRLFLX(:, :) + ZWEIGHT(JN) * ZPRLFLXE(:, :, JN)
            PPRSFLX(:, :) = PPRSFLX(:, :) + ZWEIGHT(JN) * ZPRSFLXE(:, :, JN)
            PUMF(:, :) = PUMF(:, :) + ZWEIGHT(JN) * ZUMFE(:, :, JN)
            PDMF(:, :) = PDMF(:, :) + ZWEIGHT(JN) * ZDMFE(:, :, JN)
            ZPRLTEN(:) = ZPRLTEN(:) + ZWEIGHT(JN) * ZPRLTENE(:, JN)
            PPRSTEN(:) = PPRSTEN(:) + ZWEIGHT(JN) * ZPRSTENE(:, JN)
            KCLTOP(:) = MAX(KCLTOP(:), ICLTOPE(:, JN))
            KCLBAS(:) = MAX(KCLBAS(:), ICLBASE(:, JN))
            if(OCHTRANS)  &
              & PCH1TEN(:, :, :) = PCH1TEN(:, :, :) + ZWEIGHT(JN) * ZCH1TENE(:, :, :, JN)
        enddo
!
        ZSUM = 1./(1.+SUM(ZWEIGHT(:)))
    endif
!
    PTTEN(:, :) = PTTEN(:, :) * ZSUM + ZTTENS(:, :)
    PRVTEN(:, :) = PRVTEN(:, :) * ZSUM + ZRVTENS(:, :)
    PRCTEN(:, :) = PRCTEN(:, :) * ZSUM + ZRCTENS(:, :)
    PRITEN(:, :) = PRITEN(:, :) * ZSUM + ZRITENS(:, :)
    PPRLFLX(:, :) = PPRLFLX(:, :) * ZSUM
    PPRSFLX(:, :) = PPRSFLX(:, :) * ZSUM
    PUMF(:, :) = PUMF(:, :) * ZSUM + ZUMFS(:, :)
    PDMF(:, :) = PDMF(:, :) * ZSUM
    PPRTEN(:) = (ZPRLTEN(:) + PPRSTEN(:)) * ZSUM
    PPRSTEN(:) = PPRSTEN(:) * ZSUM
    KCLTOP(:) = MAX(KCLTOP(:), ICLTOPS(:))
    KCLBAS(:) = MAX(KCLBAS(:), ICLBASS(:))
    if(OCHTRANS) then
        PCH1TEN(:, :, :) = PCH1TEN(:, :, :) * ZSUM + ZCH1TENS(:, :, :)
    endif
!
!*       6.  Deallocate local arrays
!
    DEALLOCATE(ICLBASS); DEALLOCATE(ICLTOPS)
    DEALLOCATE(ZUMFS)
    DEALLOCATE(ZCH1TENS)
    DEALLOCATE(ZRCTENS); DEALLOCATE(ZRITENS)
    DEALLOCATE(ZTTENS); DEALLOCATE(ZRVTENS)

    if(KENS > 0) then
        DEALLOCATE(ZTTENE); DEALLOCATE(ZRVTENE)
        DEALLOCATE(ZRCTENE); DEALLOCATE(ZRITENE)
        DEALLOCATE(ZUMFE); DEALLOCATE(ZDMFE)
        DEALLOCATE(ZCH1TENE)
        DEALLOCATE(ZPRLFLXE); DEALLOCATE(ZPRSFLXE)
        DEALLOCATE(ZPRLTENE); DEALLOCATE(ZPRSTENE)
        DEALLOCATE(ZEDUMMY); DEALLOCATE(IEDUMMY)
        DEALLOCATE(ZWEIGHT)
! XF BUG BUG 26/09/2016
        DEALLOCATE(ICLTOPE)
        DEALLOCATE(ICLBASE)
! XF BUG BUG
    endif
!
!
ENDsubroutine CONVECTION

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 modd 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     ###############
MODULE MODD_CST
!     ###############
!
!!****  *MODD_CST* - declaration of Physic constants
!!
!!    PURPOSE
!!    -------
!       The purpose of this declarative module is to declare  the
!     Physics constants.
!
!!
!!**  IMPLICIT ARGUMENTS
!!    ------------------
!!      None
!!
!!    REFERENCE
!!    ---------
!!      Book2 of documentation of Meso-NH (MODD_CST)
!!
!!    AUTHOR
!!    ------
!!      V. Ducrocq   *Meteo France*
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    16/05/94
!!      J. Stein    02/01/95  add xrholw
!!      J.-P. Pinty 13/12/95  add XALPI,XBETAI,XGAMI
!!      J. Stein    25/07/97  add XTH00
!!      V. Masson   05/10/98  add XRHOLI
!!      C. Mari     31/10/00  add NDAYSEC
!!      V. Masson   01/03/03  add conductivity of ice
!-------------------------------------------------------------------------------
!
!*       0.   DECLARATIONS
!             ------------
!
    implicit none
    REAL, PARAMETER :: XPI = 2.*ASIN(1.)               ! Pi
!
    REAL, PARAMETER :: XDAY = 86400.
    REAL, PARAMETER :: XSIYEA = 365.25 * XDAY * 2.*XPI / 6.283076
    REAL, PARAMETER :: XSIDAY = XDAY / (1.+XDAY / XSIYEA) ! day duration, sideral year duration,
    ! sideral day duration
!
    REAL, PARAMETER :: XKARMAN = 0.4             ! von karman constant
    REAL, PARAMETER :: XLIGHTSPEED = 299792458.      ! light speed
    REAL, PARAMETER :: XPLANCK = 6.6260755E-34   ! Planck constant
    REAL, PARAMETER :: XBOLTZ = 1.380658E-23    ! Boltzman constant
    REAL, PARAMETER :: XAVOGADRO = 6.0221367E+23   ! Avogadro number
!
    REAL, PARAMETER :: XRADIUS = 6371229.
    REAL, PARAMETER :: XOMEGA = 2.*XPI / XSIDAY     ! Earth radius, earth rotation
    REAL, PARAMETER :: XG = 9.80665                 ! Gravity constant
!
    REAL, PARAMETER :: XP00 = 1.E5               ! Reference pressure
!
    REAL, PARAMETER :: XSTEFAN = (2.*XPI**5 / 15.) * ((XBOLTZ / XPLANCK) * XBOLTZ) * (XBOLTZ / (XLIGHTSPEED * XPLANCK))**2
    REAL, PARAMETER :: XI0 = 1370.       ! Stefan-Boltzman constant, solar constant
!
    REAL, PARAMETER :: XMD = 28.9644E-3
    REAL, PARAMETER :: XMV = 18.0153E-3                 ! Molar mass of dry air and molar mass of vapor
    REAL, PARAMETER :: XRD = XAVOGADRO * XBOLTZ / XMD
    REAL, PARAMETER :: XRV = XAVOGADRO * XBOLTZ / XMV   ! Gaz constant for dry air, gaz constant for vapor
    REAL, PARAMETER :: XCPD = 7.*XRD / 2.
    REAL, PARAMETER :: XCPV = 4.*XRV                    ! Cpd (dry air), Cpv (vapor)
    REAL, PARAMETER :: XRHOLW = 1000.                      ! Volumic mass of liquid water
    REAL, PARAMETER :: XCL = 4.218E+3
    REAL, PARAMETER :: XCI = 2.106E+3                   ! Cl (liquid), Ci (ice)
    REAL, PARAMETER :: XTT = 273.16                     ! Triple point temperature
    REAL, PARAMETER :: XLVTT = 2.5008E+6                  ! Vaporization heat constant
    REAL, PARAMETER :: XLSTT = 2.8345E+6                  ! Sublimation heat constant
    REAL, PARAMETER :: XLMTT = XLSTT - XLVTT              ! Melting heat constant
    REAL, PARAMETER :: XESTT = 611.14                     ! Saturation vapor pressure  at triple point
    ! temperature
    REAL, PARAMETER :: XGAMW = (XCL - XCPV) / XRV ! Constants for saturation vapor pressure function
    REAL, PARAMETER :: XBETAW = (XLVTT / XRV) + (XGAMW * XTT)
    REAL, PARAMETER :: XALPW = LOG(XESTT) + (XBETAW / XTT) + (XGAMW * LOG(XTT))
!
    REAL, PARAMETER :: XGAMI = (XCI - XCPV) / XRV ! Constants for saturation vapor pressure function over solid ice
    REAL, PARAMETER :: XBETAI = (XLSTT / XRV) + (XGAMI * XTT)
    REAL, PARAMETER :: XALPI = LOG(XESTT) + (XBETAI / XTT) + (XGAMI * LOG(XTT))
!
    REAL, PARAMETER :: XCONDI = 2.22      ! thermal conductivity of ice (W m-1 K-1)
    REAL, PARAMETER :: XTH00 = 300.      ! reference value  for the potential
    ! temperature
    REAL, PARAMETER :: XRHOLI = 900.            ! Volumic mass of liquid water
!
    INTEGER, PARAMETER :: NDAYSEC = 24 * 3600     ! Number of seconds in a day
!
!
!   Some machine precision value depending of real4/8 use
!
    REAL, PARAMETER     :: XMNH_TINY = 1.0e-30                   ! minimum real on this machine
    REAL, PARAMETER     :: XMNH_TINY_12 = SQRT(XMNH_TINY)   ! sqrt(minimum real on this machine)
    REAL, PARAMETER     :: XMNH_EPSILON = EPSILON(XMNH_EPSILON)   ! minimum space with 1.0
    REAL, PARAMETER     :: XMNH_HUGE = HUGE(XMNH_HUGE)      ! minimum real on this machine

    REAL, PARAMETER     :: XEPS_DT = 1.0e-5          ! default value for DT test
    REAL, PARAMETER     :: XRES_FLAT_CART = 1.0e-12         ! default     flat&cart residual tolerance
    REAL, PARAMETER     :: XRES_OTHER = 1.0e-9          ! default not flat&cart residual tolerance
    REAL, PARAMETER     :: XRES_PREP = 1.0e-8          ! default     prep      residual tolerance

!
ENDMODULE MODD_CST

!! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
!! **************************************************************************************************
!! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/
!
!
!
!
!!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!!MNH_LIC for details. version 1.
!!-----------------------------------------------------------------
!!--------------- special set of characters for RCS information
!!-----------------------------------------------------------------
!! $Source$ $Revision$
!! MASDEV4_7 init 2006/05/18 13:07:25
!!-----------------------------------------------------------------
!!     ###################
!      MODULE MODI_INI_CST
!!     ###################
!!
!INTERFACE
!!
!subroutine INI_CST
!END subroutine INI_CST
!!
!END INTERFACE
!!
!END MODULE MODI_INI_CST
!!
!!
!!
!!     ##################
!      subroutine INI_CST
!!     ##################
!!
!!!****  *INI_CST * - routine to initialize the module MODD_CST
!!!
!!!    PURPOSE
!!!    -------
!!       The purpose of this routine is to initialize  the physical constants
!!     stored in  module MODD_CST.
!!
!!
!!!**  METHOD
!!!    ------
!!!      The physical constants are set to their numerical values
!!!
!!!
!!!    EXTERNAL
!!!    --------
!!!      FMLOOK : to retrieve logical unit number associated to a file
!!!
!!!    IMPLICIT ARGUMENTS
!!!    ------------------
!!!      Module MODD_CST     : contains physical constants
!!!
!!!    REFERENCE
!!!    ---------
!!!      Book2 of the documentation (module MODD_CST, routine INI_CST)
!!!
!!!
!!!    AUTHOR
!!!    ------
!!!          V. Ducrocq       * Meteo France *
!!!
!!!    MODIFICATIONS
!!!    -------------
!!!      Original    18/05/94
!!!      J. Stein    02/01/95  add the volumic mass of liquid water
!!!      J.-P. Pinty 13/12/95  add the water vapor pressure over solid ice
!!!      J. Stein    29/06/97  add XTH00
!!!      V. Masson   05/10/98  add XRHOLI
!!!      C. Mari     31/10/00  add NDAYSEC
!!!      V. Masson   01/03/03  add XCONDI
!!!      J. Escobar  28/03/2014 for pb with emissivity/aerosol reset XMNH_TINY=1.0e-80 in real8 case
!!!
!!-------------------------------------------------------------------------------
!!
!!*       0.    DECLARATIONS
!!              ------------
!!
!USE MODD_CST
!!
!implicit none
!!
!!-------------------------------------------------------------------------------
!!
!!*         1.     FUNDAMENTAL CONSTANTS
!!                ---------------------
!!
!!XPI         = 2.*ASIN(1.)
!!XKARMAN     = 0.4
!!XLIGHTSPEED = 299792458.
!!XPLANCK     = 6.6260755E-34
!!XBOLTZ      = 1.380658E-23
!!XAVOGADRO   = 6.0221367E+23
!!
!!-------------------------------------------------------------------------------
!!
!!*       2.     ASTRONOMICAL CONSTANTS
!!                ----------------------
!!
!!XDAY   = 86400.
!!XSIYEA = 365.25*XDAY*2.*XPI/ 6.283076
!!XSIDAY = XDAY/(1.+XDAY/XSIYEA)
!!XOMEGA = 2.*XPI/XSIDAY
!!NDAYSEC = 24*3600 ! Number of seconds in a day
!!
!!-------------------------------------------------------------------------------!
!!
!!
!!*       3.     TERRESTRIAL GEOIDE CONSTANTS
!!                ----------------------------
!!
!!XRADIUS = 6371229.
!!XG      = 9.80665
!!
!!-------------------------------------------------------------------------------
!!
!!*         4.     REFERENCE PRESSURE
!!                -------------------
!!
!!XP00 = 1.E5
!!XTH00 = 300.
!!-------------------------------------------------------------------------------
!!
!!*         5.     RADIATION CONSTANTS
!!                -------------------
!!
!!JUAN OVERFLOW XSTEFAN = 2.* XPI**5 * XBOLTZ**4 / (15.* XLIGHTSPEED**2 * XPLANCK**3)
!!XSTEFAN = ( 2.* XPI**5 / 15. ) * ( (XBOLTZ / XPLANCK) * XBOLTZ ) * (XBOLTZ/(XLIGHTSPEED*XPLANCK))**2
!!XI0     = 1370.
!!
!!-------------------------------------------------------------------------------
!!
!!*         6.     THERMODYNAMIC CONSTANTS
!!                -----------------------
!!
!!XMD    = 28.9644E-3
!!XMV    = 18.0153E-3
!!XRD    = XAVOGADRO * XBOLTZ / XMD
!!XRV    = XAVOGADRO * XBOLTZ / XMV
!!XCPD   = 7.* XRD /2.
!!XCPV   = 4.* XRV
!!XRHOLW = 1000.
!!XRHOLI = 900.
!!XCONDI = 2.22
!!XCL    = 4.218E+3
!!XCI    = 2.106E+3
!!XTT    = 273.16
!!XLVTT  = 2.5008E+6
!!XLSTT  = 2.8345E+6
!!XLMTT  = XLSTT - XLVTT
!!XESTT  = 611.14
!!XGAMW  = (XCL - XCPV) / XRV
!!XBETAW = (XLVTT/XRV) + (XGAMW * XTT)
!!XALPW  = LOG(XESTT) + (XBETAW /XTT) + (XGAMW *LOG(XTT))
!!XGAMI  = (XCI - XCPV) / XRV
!!XBETAI = (XLSTT/XRV) + (XGAMI * XTT)
!!XALPI  = LOG(XESTT) + (XBETAI /XTT) + (XGAMI *LOG(XTT))
!
!!
!!   Some machine precision value depending of real4/8 use
!!
!
!
!!XMNH_EPSILON = EPSILON (XMNH_EPSILON )
!!XMNH_HUGE    = HUGE    (XMNH_HUGE )
!
!!SeBi #ifdef MNH_MPI_DOUBLE_PRECISION
!!XMNH_TINY      = 1.0e-80
!!XEPS_DT        = 1.0e-5
!!XRES_FLAT_CART = 1.0e-12
!!XRES_OTHER     = 1.0e-9
!!XRES_PREP      = 1.0e-8
!!SeBi #else
!!XMNH_TINY      = TINY    (XMNH_TINY    )
!!XEPS_DT        = 1.0e-4
!!XRES_FLAT_CART = 1.0e-12
!!XRES_OTHER     = 1.0e-7
!!XRES_PREP      = 1.0e-4
!!SeBi #endif
!!XMNH_TINY_12 = SQRT    (XMNH_TINY    )
!
!
!
!!
!!-------------------------------------------------------------------------------
!!
!END subroutine INI_CST

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_SATMIXRATIO
!     #################
!
    INTERFACE
!
        subroutine CONVECT_SATMIXRATIO(KLON, &
                                       PPRES, PT, PEW, PLV, PLS, PCPH)
!
            INTEGER, INTENT(IN) :: KLON    ! horizontal loop index
            REAL, DIMENSION(KLON), INTENT(IN) :: PPRES   ! pressure
            REAL, DIMENSION(KLON), INTENT(IN) :: PT      ! temperature
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PEW     ! vapor saturation mixing ratio
            REAL, DIMENSION(KLON), INTENT(OUT):: PLV     ! latent heat L_v
            REAL, DIMENSION(KLON), INTENT(OUT):: PLS     ! latent heat L_s
            REAL, DIMENSION(KLON), INTENT(OUT):: PCPH    ! specific heat C_ph
!
        ENDsubroutine CONVECT_SATMIXRATIO
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_SATMIXRATIO
!     ######spl
subroutine CONVECT_SATMIXRATIO(KLON, &
                               PPRES, PT, PEW, PLV, PLS, PCPH)
!     ################################################################
!
!!**** Compute vapor saturation mixing ratio over liquid water
!!
!!
!!    PDRPOSE
!!    -------
!!     The purpose of this routine is to determine saturation mixing ratio
!!     and to return values for L_v L_s and C_ph
!!
!!
!!**  METHOD
!!    ------
!!
!!
!!    EXTERNAL
!!    --------
!!     None
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XALPW, XBETAW, XGAMW ! constants for water saturation pressure
!!          XRD, XRV             ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV           ! specific heat for dry air and water vapor
!!          XCL, XCI             ! specific heat for liquid water and ice
!!          XTT                  ! triple point temperature
!!          XLVTT, XLSTT         ! vaporization, sublimation heat constant
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_SATMIXRATIO)
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!------------------------- ------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON    ! horizontal loop index
    REAL, DIMENSION(KLON), INTENT(IN) :: PPRES   ! pressure
    REAL, DIMENSION(KLON), INTENT(IN) :: PT      ! temperature
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PEW     ! vapor saturation mixing ratio
    REAL, DIMENSION(KLON), INTENT(OUT):: PLV     ! latent heat L_v
    REAL, DIMENSION(KLON), INTENT(OUT):: PLS     ! latent heat L_s
    REAL, DIMENSION(KLON), INTENT(OUT):: PCPH    ! specific heat C_ph
!
!*       0.2   Declarations of local variables :
!
    REAL, DIMENSION(KLON)              :: ZT      ! temperature
    real :: ZEPS           ! R_d / R_v
!
!
!-------------------------------------------------------------------------------
!
    ZEPS = XRD / XRV
!
    ZT(:) = MIN(400., MAX(PT(:), 10.)) ! overflow bound
    PEW(:) = EXP(XALPW - XBETAW / ZT(:) - XGAMW * ALOG(ZT(:)))
    PEW(:) = ZEPS * PEW(:) / (PPRES(:) - PEW(:))
!
    PLV(:) = XLVTT + (XCPV - XCL) * (ZT(:) - XTT) ! compute L_v
    PLS(:) = XLSTT + (XCPV - XCI) * (ZT(:) - XTT) ! compute L_i
!
    PCPH(:) = XCPD + XCPV * PEW(:)                     ! compute C_ph
!
ENDsubroutine CONVECT_SATMIXRATIO

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 modd 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     ######spl
MODULE MODD_CONVPAREXT
!     ######################
!
    implicit none
!
    INTEGER, SAVE :: JCVEXB ! start vertical computations at
    ! 1 + JCVEXB = 1 + ( KBDIA - 1 )
    INTEGER, SAVE :: JCVEXT ! limit vertical computations to
    ! KLEV - JCVEXT = KLEV - ( KTDIA - 1 )
!$OMP threadprivate(JCVEXB,JCVEXT)
ENDMODULE MODD_CONVPAREXT

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 modd 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     ######spl

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_TRIGGER_FUNCT
!     #################
!
    INTERFACE
!
        subroutine CONVECT_TRIGGER_FUNCT(KLON, KLEV, &
                                         PPRES, PTH, PTHV, PTHES, &
                                         PRV, PW, PZ, PDXDY, &
                                         PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, &
                                         PTHVELCL, KLCL, KDPL, KPBL, OTRIG, &
                                         PCAPE)
!
            INTEGER, INTENT(IN)                   :: KLON      ! horizontal loop index
            INTEGER, INTENT(IN)                   :: KLEV      ! vertical loop index
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY     ! grid area
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH, PTHV ! theta, theta_v
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES     ! envir. satur. theta_e
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV       ! vapor mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES     ! pressure
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ        ! height of grid point (m)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PW        ! vertical velocity
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL    ! theta at LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL     ! temp. at LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL    ! vapor mixing ratio at  LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PWLCL     ! parcel velocity at  LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL     ! height at LCL (m)
            REAL, DIMENSION(KLON), INTENT(OUT):: PTHVELCL  ! environm. theta_v at LCL (K)
            LOGICAL, DIMENSION(KLON), INTENT(OUT):: OTRIG     ! logical mask for convection
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KLCL    ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KDPL    ! contains vert. index of DPL
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KPBL    ! contains index of source layer top
            REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE     ! CAPE (J/kg) for diagnostics
!
        ENDsubroutine CONVECT_TRIGGER_FUNCT
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_TRIGGER_FUNCT
!     #########################################################################
subroutine CONVECT_TRIGGER_FUNCT(KLON, KLEV, &
                                 PPRES, PTH, PTHV, PTHES, &
                                 PRV, PW, PZ, PDXDY, &
                                 PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, &
                                 PTHVELCL, KLCL, KDPL, KPBL, OTRIG, &
                                 PCAPE)
!     #########################################################################
!
!!**** Determine convective columns as well as the cloudy values of theta,
!!     and qv at the lifting condensation level (LCL)
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine convective columns
!!
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!      What we look for is the undermost unstable level at each grid point.
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_SATMIXRATIO
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! Reference pressure
!!          XRD, XRV           ! Gaz  constants for dry air and water vapor
!!          XCPD               ! Cpd (dry air)
!!          XTT                ! triple point temperature
!!          XBETAW, XGAMW      ! constants for vapor saturation pressure
!!
!!      Module MODD_CONVPAR
!!          XA25               ! reference grid area
!!          XZLCL              ! maximum height difference between
!!                             ! the surface and the DPL
!!          XZPBL              ! minimum mixed layer depth to sustain convection
!!          XWTRIG             ! constant in vertical velocity trigger
!!          XCDEPTH            ! minimum necessary cloud depth
!!          XNHGAM             ! coefficient for buoyancy term in w eq.
!!                             ! accounting for nh-pressure
!!
!!      Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book2 of documentation ( routine TRIGGER_FUNCT)
!!      Fritsch and Chappell (1980), J. Atm. Sci., Vol. 37, 1722-1761.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  20/03/97  Select first departure level
!!                            that produces a cloud thicker than XCDEPTH
!!   Last modified  12/12/97  add small perturbation
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
    USE MODD_CONVPAREXT
    USE MODI_CONVECT_SATMIXRATIO
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                   :: KLON      ! horizontal loop index
    INTEGER, INTENT(IN)                   :: KLEV      ! vertical loop index
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY     ! grid area
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH, PTHV ! theta, theta_v
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES     ! envir. satur. theta_e
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV       ! vapor mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES     ! pressure
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ        ! height of grid point (m)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PW        ! vertical velocity
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL    ! theta at LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL     ! temp. at LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL    ! vapor mixing ratio at  LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PWLCL     ! parcel velocity at  LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL     ! height at LCL (m)
    REAL, DIMENSION(KLON), INTENT(OUT):: PTHVELCL  ! environm. theta_v at LCL (K)
    LOGICAL, DIMENSION(KLON), INTENT(OUT):: OTRIG     ! logical mask for convection
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KLCL    ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KDPL    ! contains vert. index of DPL
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KPBL    ! contains index of source layer top
    REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE     ! CAPE (J/kg) for diagnostics
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: JKK, JK, JKP, JKM, JKDL, JL, JKT, JT! vertical loop index
    INTEGER :: JI                                  ! horizontal loop index
    INTEGER :: IIE, IKB, IKE                       ! horizontal + vertical loop bounds
    real :: ZEPS, ZEPSA                         ! R_d / R_v, R_v / R_d
    real :: ZCPORD, ZRDOCP                      ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON) :: ZTHLCL, ZTLCL, ZRVLCL, & ! locals for PTHLCL,PTLCL
                             ZWLCL, ZZLCL, ZTHVELCL  ! PRVLCL, ....
    INTEGER, DIMENSION(KLON) :: IDPL, IPBL, ILCL      ! locals for KDPL, ...
    REAL, DIMENSION(KLON) :: ZPLCL    ! pressure at LCL
    REAL, DIMENSION(KLON) :: ZZDPL    ! height of DPL
    REAL, DIMENSION(KLON) :: ZTHVLCL  ! theta_v at LCL = mixed layer value
    REAL, DIMENSION(KLON) :: ZTMIX    ! mixed layer temperature
    REAL, DIMENSION(KLON) :: ZEVMIX   ! mixed layer water vapor pressure
    REAL, DIMENSION(KLON) :: ZDPTHMIX, ZPRESMIX ! mixed layer depth and pressure
    REAL, DIMENSION(KLON) :: ZCAPE    ! convective available energy (m^2/s^2/g)
    REAL, DIMENSION(KLON) :: ZTHEUL   ! updraft equiv. pot. temperature (K)
    REAL, DIMENSION(KLON) :: ZLV, ZCPH! specific heats of vaporisation, dry air
    REAL, DIMENSION(KLON) :: ZDP      ! pressure between LCL and model layer
    REAL, DIMENSION(KLON) :: ZTOP     ! estimated cloud top (m)
    REAL, DIMENSION(KLON, KLEV):: ZCAP ! CAPE at every level for diagnostics
    REAL, DIMENSION(KLON) :: ZWORK1, ZWORK2, ZWORK3    ! work arrays
    LOGICAL, DIMENSION(KLON) :: GTRIG, GTRIG2          ! local arrays for OTRIG
    LOGICAL, DIMENSION(KLON) :: GWORK1                 ! work array
!
!
!-------------------------------------------------------------------------------
!
!*       0.3    Compute array bounds
!               --------------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Initialize local variables
!               --------------------------
!
    ZEPS = XRD / XRV
    ZEPSA = XRV / XRD
    ZCPORD = XCPD / XRD
    ZRDOCP = XRD / XCPD
    OTRIG(:) = .false.
    IDPL(:) = KDPL(:)
    IPBL(:) = KPBL(:)
    ILCL(:) = KLCL(:)
    PWLCL(:) = 0.
    ZWLCL(:) = 0.
    PTHLCL(:) = 1.
    PTHVELCL(:) = 1.
    PTLCL(:) = 1.
    PRVLCL(:) = 0.
    PWLCL(:) = 0.
    PZLCL(:) = PZ(:, IKB)
    ZZDPL(:) = PZ(:, IKB)
    GTRIG2(:) = .true.
    ZCAP(:, :) = 0.
!
!
!
!       1.     Determine highest necessary loop test layer
!              -------------------------------------------
!
    JT = IKE - 2
    do JK = IKB + 1, IKE - 2
        if(PZ(1, JK) - PZ(1, IKB) < 12.E3) JT = JK
    enddo
!
!
!*       2.     Enter loop for convection test
!               ------------------------------
!
    JKP = MINVAL(IDPL(:)) + 1
    JKT = JT
    do JKK = JKP, JKT
!
        GWORK1(:) = ZZDPL(:) - PZ(:, IKB) < XZLCL
        ! we exit the trigger test when the center of the mixed layer is more
        ! than 3500 m  above soil level.
        WHERE(GWORK1(:))
            ZDPTHMIX(:) = 0.
            ZPRESMIX(:) = 0.
            ZTHLCL(:) = 0.
            ZRVLCL(:) = 0.
            ZZDPL(:) = PZ(:, JKK)
            IDPL(:) = JKK
        ENDWHERE
!
!
!*       3.     Construct a mixed layer of at least 60 hPa (XZPBL)
!               ------------------------------------------
!
        do JK = JKK, IKE - 1
            JKM = JK + 1
            do JI = 1, IIE
                if(GWORK1(JI) .and. ZDPTHMIX(JI) < XZPBL) then
                    IPBL(JI) = JK
                    ZWORK1(JI) = PPRES(JI, JK) - PPRES(JI, JKM)
                    ZDPTHMIX(JI) = ZDPTHMIX(JI) + ZWORK1(JI)
                    ZPRESMIX(JI) = ZPRESMIX(JI) + PPRES(JI, JK) * ZWORK1(JI)
                    ZTHLCL(JI) = ZTHLCL(JI) + PTH(JI, JK) * ZWORK1(JI)
                    ZRVLCL(JI) = ZRVLCL(JI) + PRV(JI, JK) * ZWORK1(JI)
                endif
            enddo
            if(MINVAL(ZDPTHMIX(:)) >= XZPBL) EXIT
        enddo
!
!
        WHERE(GWORK1(:))
!
            ZPRESMIX(:) = ZPRESMIX(:) / ZDPTHMIX(:)
            ZTHLCL(:) = ZTHLCL(:) / ZDPTHMIX(:) + .3   ! add small perturbation
            ZRVLCL(:) = ZRVLCL(:) / ZDPTHMIX(:) + 1.e-4
            ZTHVLCL(:) = ZTHLCL(:) * (1.+ZEPSA * ZRVLCL(:)) &
                         / (1.+ZRVLCL(:))
!
!*       4.1    Use an empirical direct solution ( Bolton formula )
!               to determine temperature and pressure at LCL.
!               Nota: the adiabatic saturation temperature is not
!                     equal to the dewpoint temperature
!               ----------------------------------------------------
!
!
            ZTMIX(:) = ZTHLCL(:) * (ZPRESMIX(:) / XP00)**ZRDOCP
            ZEVMIX(:) = ZRVLCL(:) * ZPRESMIX(:) / (ZRVLCL(:) + ZEPS)
            ZEVMIX(:) = MAX(1.E-8, ZEVMIX(:))
            ZWORK1(:) = LOG(ZEVMIX(:) / 613.3)
            ! dewpoint temperature
            ZWORK1(:) = (4780.8 - 32.19 * ZWORK1(:)) / (17.502 - ZWORK1(:))
            ! adiabatic saturation temperature
            ZTLCL(:) = ZWORK1(:) - (.212 + 1.571E-3 * (ZWORK1(:) - XTT) &
                                    - 4.36E-4 * (ZTMIX(:) - XTT)) * (ZTMIX(:) - ZWORK1(:))
            ZTLCL(:) = MIN(ZTLCL(:), ZTMIX(:))
            ZPLCL(:) = XP00 * (ZTLCL(:) / ZTHLCL(:))**ZCPORD
!
        ENDWHERE
!
!
!*       4.2    Correct ZTLCL in order to be completely consistent
!               with MNH saturation formula
!               ---------------------------------------------
!
        call CONVECT_SATMIXRATIO(KLON, ZPLCL, ZTLCL, ZWORK1, ZLV, ZWORK2, ZCPH)
        WHERE(GWORK1(:))
            ZWORK2(:) = ZWORK1(:) / ZTLCL(:) * (XBETAW / ZTLCL(:) - XGAMW) ! dr_sat/dT
            ZWORK2(:) = (ZWORK1(:) - ZRVLCL(:)) / &
                        (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
            ZTLCL(:) = ZTLCL(:) - ZLV(:) / ZCPH(:) * ZWORK2(:)
!
        ENDWHERE
!
!
!*       4.3    If ZRVLCL = PRVMIX is oversaturated set humidity
!               and temperature to saturation values.
!               ---------------------------------------------
!
        call CONVECT_SATMIXRATIO(KLON, ZPRESMIX, ZTMIX, ZWORK1, ZLV, ZWORK2, ZCPH)
        WHERE(GWORK1(:) .and. ZRVLCL(:) > ZWORK1(:))
            ZWORK2(:) = ZWORK1(:) / ZTMIX(:) * (XBETAW / ZTMIX(:) - XGAMW) ! dr_sat/dT
            ZWORK2(:) = (ZWORK1(:) - ZRVLCL(:)) / &
                        (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
            ZTLCL(:) = ZTMIX(:) - ZLV(:) / ZCPH(:) * ZWORK2(:)
            ZRVLCL(:) = ZRVLCL(:) - ZWORK2(:)
            ZPLCL(:) = ZPRESMIX(:)
            ZTHLCL(:) = ZTLCL(:) * (XP00 / ZPLCL(:))**ZRDOCP
            ZTHVLCL(:) = ZTHLCL(:) * (1.+ZEPSA * ZRVLCL(:)) &
                         / (1.+ZRVLCL(:))
        ENDWHERE
!
!
!*        5.1   Determine  vertical loop index at the LCL and DPL
!               --------------------------------------------------
!
        do JK = JKK, IKE - 1
            do JI = 1, IIE
                if(ZPLCL(JI) <= PPRES(JI, JK) .and. GWORK1(JI)) ILCL(JI) = JK + 1
            enddo
        enddo
!
!
!*        5.2   Estimate height and environm. theta_v at LCL
!               --------------------------------------------------
!
        do JI = 1, IIE
            JK = ILCL(JI)
            JKM = JK - 1
            ZDP(JI) = LOG(ZPLCL(JI) / PPRES(JI, JKM)) / &
                      LOG(PPRES(JI, JK) / PPRES(JI, JKM))
            ZWORK1(JI) = PTHV(JI, JKM) + (PTHV(JI, JK) - PTHV(JI, JKM)) * ZDP(JI)
            ! we compute the precise value of the LCL
            ! The precise height is between the levels ILCL and ILCL-1.
            ZWORK2(JI) = PZ(JI, JKM) + (PZ(JI, JK) - PZ(JI, JKM)) * ZDP(JI)
        enddo
        WHERE(GWORK1(:))
            ZTHVELCL(:) = ZWORK1(:)
            ZZLCL(:) = ZWORK2(:)
        ENDWHERE
!
!
!*       6.     Check to see if cloud is bouyant
!               --------------------------------
!
!*      6.1    Compute grid scale vertical velocity perturbation term ZWORK1
!               -------------------------------------------------------------
!
        !  normalize w grid scale to a 25 km refer. grid
        do JI = 1, IIE
            JK = ILCL(JI)
            JKM = JK - 1
            JKDL = IDPL(JI)
            !ZWORK1(JI) =  ( PW(JI,JKM)  + ( PW(JI,JK) - PW(JI,JKM) ) * ZDP(JI) )  &
            ZWORK1(JI) = (PW(JI, JK) + PW(JI, JKDL) * ZZLCL(JI) / PZ(JI, JKDL))*.5 &
                         * SQRT(PDXDY(JI) / XA25)
!                         - 0.02 * ZZLCL(JI) / XZLCL ! avoid spurious convection
        enddo
        ! compute sign of normalized grid scale w
        ZWORK2(:) = SIGN(1., ZWORK1(:))
        ZWORK1(:) = XWTRIG * ZWORK2(:) * ABS(ZWORK1(:))**0.333 &
                    * (XP00 / ZPLCL(:))**ZRDOCP
!
!*       6.2    Compute parcel vertical velocity at LCL
!               ---------------------------------------
!
        do JI = 1, IIE
            JKDL = IDPL(JI)
            ZWORK3(JI) = XG * ZWORK1(JI) * (ZZLCL(JI) - PZ(JI, JKDL)) &
                         / (PTHV(JI, JKDL) + ZTHVELCL(JI))
        enddo
        WHERE(GWORK1(:))
            ZWLCL(:) = 1.+.5 * ZWORK2(:) * SQRT(ABS(ZWORK3(:)))
            GTRIG(:) = ZTHVLCL(:) - ZTHVELCL(:) + ZWORK1(:) > 0. .and. &
                       ZWLCL(:) > 0.
        ENDWHERE
!
!
!*       6.3    Look for parcel that produces sufficient cloud depth.
!               The cloud top is estimated as the level where the CAPE
!               is smaller  than a given value (based on vertical velocity eq.)
!               --------------------------------------------------------------
!
        ZTHEUL(:) = ZTLCL(:) * (ZTHLCL(:) / ZTLCL(:)) &
                    **(1.-0.28 * ZRVLCL(:)) &
                    * EXP((3374.6525 / ZTLCL(:) - 2.5403) * &
                          ZRVLCL(:) * (1.+0.81 * ZRVLCL(:)))
!
        ZCAPE(:) = 0.
        ZTOP(:) = 0.
        ZWORK3(:) = 0.
        JKM = MINVAL(ILCL(:))
        do JL = JKM, JT
            JK = JL + 1
            do JI = 1, IIE
                ZWORK1(JI) = (2.*ZTHEUL(JI) / &
                              (PTHES(JI, JK) + PTHES(JI, JL)) - 1.) * (PZ(JI, JK) - PZ(JI, JL))
                if(JL < ILCL(JI)) ZWORK1(JI) = 0.
                !  if ( JL <= ILCL(JI) ) ZWORK1(JI) = 0.
                ZCAPE(JI) = ZCAPE(JI) + ZWORK1(JI)
                ZCAP(JI, JKK) = ZCAP(JI, JKK) + XG * MAX(0., ZWORK1(JI)) ! actual CAPE
                ZWORK2(JI) = XNHGAM * XG * ZCAPE(JI) + 1.05 * ZWLCL(JI) * ZWLCL(JI)
                ! the factor 1.05 takes entrainment into account
                ZWORK2(JI) = SIGN(1., ZWORK2(JI))
                ZWORK3(JI) = ZWORK3(JI) + MIN(0., ZWORK2(JI))
                ZWORK3(JI) = MAX(-1., ZWORK3(JI))
                ! Nota, the factors ZWORK2 and ZWORK3 are only used to avoid
                ! if and goto statements, the difficulty is to extract only
                ! the level where the criterium is first fullfilled
                ZTOP(JI) = PZ(JI, JL)*.5 * (1.+ZWORK2(JI)) * (1.+ZWORK3(JI)) + &
                           ZTOP(JI)*.5 * (1.-ZWORK2(JI))
            enddo
        enddo
!
!
        WHERE(ZTOP(:) - ZZLCL(:) >= XCDEPTH .and. GTRIG(:) .and. GTRIG2(:))
            GTRIG2(:) = .false.
            OTRIG(:) = GTRIG(:)     ! we  select the first departure level
            PTHLCL(:) = ZTHLCL(:)    ! that gives sufficient cloud depth
            PRVLCL(:) = ZRVLCL(:)
            PTLCL(:) = ZTLCL(:)
            PWLCL(:) = ZWLCL(:)
            PZLCL(:) = ZZLCL(:)
            PTHVELCL(:) = ZTHVELCL(:)
            KDPL(:) = IDPL(:)
            KPBL(:) = IPBL(:)
            KLCL(:) = ILCL(:)
        ENDWHERE
!
    enddo
!
    do JI = 1, IIE
        PCAPE(JI) = MAXVAL(ZCAP(JI, :)) ! maximum CAPE for diagnostics
    enddo
!
!
ENDsubroutine CONVECT_TRIGGER_FUNCT

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #############################################################################
!     #################
MODULE MODI_CONVECT_CONDENS
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CONDENS(KLON, &
                                   KICE, PPRES, PTHL, PRW, PRCO, PRIO, PZ, OWORK1, &
                                   PT, PEW, PRC, PRI, PLV, PLS, PCPH)
!
            INTEGER, INTENT(IN)                :: KLON    ! horizontal loop index
            INTEGER, INTENT(IN)                :: KICE    ! flag for ice ( 1 = yes,
            !                0 = no ice )
            REAL, DIMENSION(KLON), INTENT(IN) :: PPRES  ! pressure
            REAL, DIMENSION(KLON), INTENT(IN) :: PTHL   ! enthalpy (J/kg)
            REAL, DIMENSION(KLON), INTENT(IN) :: PRW    ! total water mixing ratio
            REAL, DIMENSION(KLON), INTENT(IN) :: PRCO   ! cloud water estimate (kg/kg)
            REAL, DIMENSION(KLON), INTENT(IN) :: PRIO   ! cloud ice   estimate (kg/kg)
            REAL, DIMENSION(KLON), INTENT(IN) :: PZ     ! level height (m)
            LOGICAL, DIMENSION(KLON), INTENT(IN) :: OWORK1 ! logical mask
!
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PT     ! temperature
            REAL, DIMENSION(KLON), INTENT(OUT):: PRC    ! cloud water mixing ratio(kg/kg)
            REAL, DIMENSION(KLON), INTENT(OUT):: PRI    ! cloud ice mixing ratio  (kg/kg)
            REAL, DIMENSION(KLON), INTENT(OUT):: PLV    ! latent heat L_v
            REAL, DIMENSION(KLON), INTENT(OUT):: PLS    ! latent heat L_s
            REAL, DIMENSION(KLON), INTENT(OUT):: PCPH   ! specific heat C_ph
            REAL, DIMENSION(KLON), INTENT(OUT):: PEW    ! water saturation mixing ratio
!
        ENDsubroutine CONVECT_CONDENS
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CONDENS
subroutine CONVECT_CONDENS(KLON, &
                           KICE, PPRES, PTHL, PRW, PRCO, PRIO, PZ, OWORK1, &
                           PT, PEW, PRC, PRI, PLV, PLS, PCPH)
!     #############################################################################
!
!!**** Compute temperature cloud and ice water content from enthalpy and r_w
!!
!!
!!    PURPOSE
!!    -------
!!     The purpose of this routine is to determine cloud condensate
!!     and to return values for L_v, L_s and C_ph
!!
!!
!!**  METHOD
!!    ------
!!     Condensate is extracted iteratively
!!
!!
!!    EXTERNAL
!!    --------
!!     None
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!      Module MODD_CST
!!          XG                   ! gravity constant
!!          XALPW, XBETAW, XGAMW ! constants for water saturation pressure
!!          XALPI, XBETAI, XGAMI ! constants for ice saturation pressure
!!          XP00                 ! reference pressure
!!          XRD, XRV             ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV           ! specific heat for dry air and water vapor
!!          XCL, XCI             ! specific heat for liquid water and ice
!!          XTT                  ! triple point temperature
!!          XLVTT, XLSTT         ! vaporization, sublimation heat constant
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CONVPAR
!!          XTFRZ1               ! begin of freezing interval
!!          XTFRZ2               ! end of freezing interval
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_CONDENS)
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                :: KLON    ! horizontal loop index
    INTEGER, INTENT(IN)                :: KICE    ! flag for ice ( 1 = yes,
    !                0 = no ice )
    REAL, DIMENSION(KLON), INTENT(IN) :: PPRES  ! pressure
    REAL, DIMENSION(KLON), INTENT(IN) :: PTHL   ! enthalpy (J/kg)
    REAL, DIMENSION(KLON), INTENT(IN) :: PRW    ! total water mixing ratio
    REAL, DIMENSION(KLON), INTENT(IN) :: PRCO   ! cloud water estimate (kg/kg)
    REAL, DIMENSION(KLON), INTENT(IN) :: PRIO   ! cloud ice   estimate (kg/kg)
    REAL, DIMENSION(KLON), INTENT(IN) :: PZ     ! level height (m)
    LOGICAL, DIMENSION(KLON), INTENT(IN) :: OWORK1 ! logical mask
!
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PT     ! temperature
    REAL, DIMENSION(KLON), INTENT(OUT):: PRC    ! cloud water mixing ratio(kg/kg)
    REAL, DIMENSION(KLON), INTENT(OUT):: PRI    ! cloud ice mixing ratio  (kg/kg)
    REAL, DIMENSION(KLON), INTENT(OUT):: PLV    ! latent heat L_v
    REAL, DIMENSION(KLON), INTENT(OUT):: PLS    ! latent heat L_s
    REAL, DIMENSION(KLON), INTENT(OUT):: PCPH   ! specific heat C_ph
    REAL, DIMENSION(KLON), INTENT(OUT):: PEW    ! water saturation mixing ratio
!
!*       0.2   Declarations of local variables KLON
!
    INTEGER :: JITER          ! iteration index
    real :: ZEPS           ! R_d / R_v
!
    REAL, DIMENSION(KLON)    :: ZEI           ! ice saturation mixing ratio
    REAL, DIMENSION(KLON)    :: ZWORK1, ZWORK2, ZWORK3, ZT ! work arrays
!
!
!-------------------------------------------------------------------------------
!
!*       1.     Initialize temperature and Exner function
!               -----------------------------------------
!
    ZEPS = XRD / XRV
!
!
    ! Make a first temperature estimate, based e.g. on values of
    !  r_c and r_i at lower level
!
      !! Note that the definition of ZCPH is not the same as used in
      !! routine CONVECT_SATMIXRATIO
    PCPH(:) = XCPD + XCPV * PRW(:)
    ZWORK1(:) = (1.+PRW(:)) * XG * PZ(:)
    PT(:) = (PTHL(:) + PRCO(:) * XLVTT + PRIO(:) * XLSTT - ZWORK1(:)) &
            / PCPH(:)
    PT(:) = MAX(180., MIN(330., PT(:))) ! set overflow bounds in
    ! case that PTHL=0
!
!
!*       2.     Enter the iteration loop
!               ------------------------
!
    do JITER = 1, 6
        PEW(:) = EXP(XALPW - XBETAW / PT(:) - XGAMW * ALOG(PT(:)))
        ZEI(:) = EXP(XALPI - XBETAI / PT(:) - XGAMI * ALOG(PT(:)))
        PEW(:) = ZEPS * PEW(:) / (PPRES(:) - PEW(:))
        ZEI(:) = ZEPS * ZEI(:) / (PPRES(:) - ZEI(:))
!
        PLV(:) = XLVTT + (XCPV - XCL) * (PT(:) - XTT) ! compute L_v
        PLS(:) = XLSTT + (XCPV - XCI) * (PT(:) - XTT) ! compute L_i
!
        ZWORK2(:) = (XTFRZ1 - PT(:)) / (XTFRZ1 - XTFRZ2) ! freezing interval
        ZWORK2(:) = MAX(0., MIN(1., ZWORK2(:))) * REAL(KICE)
        ZWORK3(:) = (1.-ZWORK2(:)) * PEW(:) + ZWORK2(:) * ZEI(:)
        PRC(:) = MAX(0.,(1.-ZWORK2(:)) * (PRW(:) - ZWORK3(:)))
        PRI(:) = MAX(0., ZWORK2(:) * (PRW(:) - ZWORK3(:)))
        ZT(:) = (PTHL(:) + PRC(:) * PLV(:) + PRI(:) * PLS(:) - ZWORK1(:)) &
                / PCPH(:)
        PT(:) = PT(:) + (ZT(:) - PT(:)) * 0.4  ! force convergence
        PT(:) = MAX(175., MIN(330., PT(:)))
    enddo
!
!
ENDsubroutine CONVECT_CONDENS

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_MIXING_FUNCT
!     #################
!
    INTERFACE
!
        subroutine CONVECT_MIXING_FUNCT(KLON, &
                                        PMIXC, KMF, PER, PDR)
!
            INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
            INTEGER, INTENT(IN) :: KMF    ! switch for dist. function
            REAL, DIMENSION(KLON), INTENT(IN) :: PMIXC  ! critical mixed fraction
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PER    ! normalized entrainment rate
            REAL, DIMENSION(KLON), INTENT(OUT):: PDR    ! normalized detrainment rate
!
        ENDsubroutine CONVECT_MIXING_FUNCT
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_MIXING_FUNCT
!     ######spl
subroutine CONVECT_MIXING_FUNCT(KLON, &
                                PMIXC, KMF, PER, PDR)
!     #######################################################
!
!!**** Determine the area under the distribution function
!!     KMF = 1 : gaussian  KMF = 2 : triangular distribution function
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the entrainment and
!!      detrainment rate by evaluating the are under the distribution
!!      function. The integration interval is limited by the critical
!!      mixed fraction PMIXC
!!
!!
!!
!!**  METHOD
!!    ------
!!      Use handbook of mathemat. functions by Abramowitz and Stegun, 1968
!!
!!
!!
!!    EXTERNAL
!!    --------
!!      None
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      None
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Book2 of documentation ( routine MIXING_FUNCT)
!!      Abramovitz and Stegun (1968), handbook of math. functions
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
    INTEGER, INTENT(IN) :: KMF    ! switch for dist. function
    REAL, DIMENSION(KLON), INTENT(IN) :: PMIXC  ! critical mixed fraction
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PER    ! normalized entrainment rate
    REAL, DIMENSION(KLON), INTENT(OUT):: PDR    ! normalized detrainment rate
!
!*       0.2   Declarations of local variables :
!
    real :: ZSIGMA = 0.166666667                   ! standard deviation
    real :: ZFE = 4.931813949                   ! integral normalization
    real :: ZSQRTP = 2.506628, ZP = 0.33267      ! constants
    real :: ZA1 = 0.4361836, ZA2 = -0.1201676    ! constants
    real :: ZA3 = 0.9372980, ZT1 = 0.500498     ! constants
    real :: ZE45 = 0.01111                       ! constant
!
    REAL, DIMENSION(KLON) :: ZX, ZY, ZW1, ZW2         ! work variables
    real :: ZW11
!
!
!-------------------------------------------------------------------------------
!
!       1.     Use gaussian function for KMF=1
!              -------------------------------
!
    if(KMF == 1) then
        ! ZX(:)  = ( PMIXC(:) - 0.5 ) / ZSIGMA
        ZX(:) = 6.*PMIXC(:) - 3.
        ZW1(:) = 1./(1.+ZP * ABS(ZX(:)))
        ZY(:) = EXP(-0.5 * ZX(:) * ZX(:))
        ZW2(:) = ZA1 * ZW1(:) + ZA2 * ZW1(:) * ZW1(:) + &
                 ZA3 * ZW1(:) * ZW1(:) * ZW1(:)
        ZW11 = ZA1 * ZT1 + ZA2 * ZT1 * ZT1 + ZA3 * ZT1 * ZT1 * ZT1
    endif
!
    WHERE(KMF == 1 .and. ZX(:) >= 0.)
        PER(:) = ZSIGMA * (0.5 * (ZSQRTP - ZE45 * ZW11 &
                                  - ZY(:) * ZW2(:)) + ZSIGMA * (ZE45 - ZY(:))) &
                 - 0.5 * ZE45 * PMIXC(:) * PMIXC(:)
        PDR(:) = ZSIGMA * (0.5 * (ZY(:) * ZW2(:) - ZE45 * ZW11) &
                           + ZSIGMA * (ZE45 - ZY(:))) &
                 - ZE45 * (0.5 + 0.5 * PMIXC(:) * PMIXC(:) - PMIXC(:))
    ENDWHERE
    WHERE(KMF == 1 .and. ZX(:) < 0.)
        PER(:) = ZSIGMA * (0.5 * (ZY(:) * ZW2(:) - ZE45 * ZW11) &
                           + ZSIGMA * (ZE45 - ZY(:))) &
                 - 0.5 * ZE45 * PMIXC(:) * PMIXC(:)
        PDR(:) = ZSIGMA * (0.5 * (ZSQRTP - ZE45 * ZW11 - ZY(:) &
                                  * ZW2(:)) + ZSIGMA * (ZE45 - ZY(:))) &
                 - ZE45 * (0.5 + 0.5 * PMIXC(:) * PMIXC(:) - PMIXC(:))
    ENDWHERE
!
    PER(:) = PER(:) * ZFE
    PDR(:) = PDR(:) * ZFE
!
!
!       2.     Use triangular function KMF=2
!              -------------------------------
!
!     not yet released
!
!
ENDsubroutine CONVECT_MIXING_FUNCT

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_UPDRAFT
!     #################
!
    INTERFACE
!
        subroutine CONVECT_UPDRAFT(KLON, KLEV, &
                                   KICE, PPRES, PDPRES, PZ, PTHL, PTHV, PTHES, PRW, &
                                   PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, PTHVELCL, &
                                   PMFLCL, OTRIG, KLCL, KDPL, KPBL, &
                                   PUMF, PUER, PUDR, PUTHL, PUTHV, PURW, &
                                   PURC, PURI, PURR, PURS, PUPR, &
                                   PUTPR, PCAPE, KCTL, KETL, PUTT)
!
            INTEGER, INTENT(IN)                    :: KLON  ! horizontal dimension
            INTEGER, INTENT(IN)                    :: KLEV  ! vertical dimension
            INTEGER, INTENT(IN)                    :: KICE  ! flag for ice ( 1 = yes,
            !                0 = no ice )
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL  ! grid scale enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHV  ! grid scale theta_v
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
            ! mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (P)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
            ! bottom and top of layer (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTHLCL ! theta at LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PTLCL  ! temp. at LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PRVLCL ! vapor mixing ratio at  LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PWLCL  ! parcel velocity at LCL (m/s)
            REAL, DIMENSION(KLON), INTENT(IN) :: PMFLCL ! cloud  base unit mass flux
            ! (kg/s)
            REAL, DIMENSION(KLON), INTENT(IN) :: PZLCL  ! height at LCL (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTHVELCL  ! environm. theta_v at LCL (K)
            LOGICAL, DIMENSION(KLON), INTENT(INOUT):: OTRIG! logical mask for convection
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! contains vert. index of DPL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   !  " vert. index of source layertop
!
!
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KCTL   ! contains vert. index of CTL
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KETL   ! contains vert. index of        &
            !equilibrium (zero buoyancy) level
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHL ! updraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHV ! updraft theta_v (K)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTT  ! updraft temperature(K)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURW  ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURC  ! updraft cloud water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURI  ! updraft cloud ice   (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURR  ! liquid precipit. (kg/kg)
            ! produced in  model layer
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)::PURS ! solid precipit. (kg/kg)
            ! produced in  model layer
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)::PUPR ! updraft precipitation in
            ! flux units (kg water / s)
            REAL, DIMENSION(KLON), INTENT(OUT):: PUTPR  ! total updraft precipitation
            ! in flux units (kg water / s)
            REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE  ! available potent. energy
!
        ENDsubroutine CONVECT_UPDRAFT
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_UPDRAFT
!     ##########################################################################
subroutine CONVECT_UPDRAFT(KLON, KLEV, &
                           KICE, PPRES, PDPRES, PZ, PTHL, PTHV, PTHES, PRW, &
                           PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, PTHVELCL, &
                           PMFLCL, OTRIG, KLCL, KDPL, KPBL, &
                           PUMF, PUER, PUDR, PUTHL, PUTHV, PURW, &
                           PURC, PURI, PURR, PURS, PUPR, &
                           PUTPR, PCAPE, KCTL, KETL, PUTT)
!     ##########################################################################
!
!!**** Compute updraft properties from DPL to CTL.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine updraft properties
!!      ( mass flux, thermodynamics, precipitation )
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_MIXING_FUNCT
!!     Routine CONVECT_CONDENS
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! reference pressure
!!          XRD, XRV           ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV, XCL    ! Cp of dry air, water vapor and liquid water
!!          XTT                ! triple point temperature
!!          XLVTT              ! vaporisation heat at XTT
!!
!!
!!      Module MODD_CONVPAR
!!          XA25               ! reference grid area
!!          XCRAD              ! cloud radius
!!          XCDEPTH            ! minimum necessary cloud depth
!!          XENTR              ! entrainment constant
!!          XRCONV             ! constant in precipitation conversion
!!          XNHGAM             ! coefficient for buoyancy term in w eq.
!!                             ! accounting for nh-pressure
!!          XTFRZ1             ! begin of freezing interval
!!          XTFRZ2             ! begin of freezing interval
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_UPDRAFT)
!!      Kain and Fritsch, 1990, J. Atmos. Sci., Vol.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  10/12/97
!!   V.Masson, C.Lac, Sept. 2010 : Correction of a loop for reproducibility
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
    USE MODD_CONVPAREXT
!
    USE MODI_CONVECT_CONDENS
    USE MODI_CONVECT_MIXING_FUNCT
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                    :: KLON  ! horizontal dimension
    INTEGER, INTENT(IN)                    :: KLEV  ! vertical dimension
    INTEGER, INTENT(IN)                    :: KICE  ! flag for ice ( 1 = yes,
    !                0 = no ice )
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL  ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHV  ! grid scale theta_v
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
    ! mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (P)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTHLCL ! theta at LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PTLCL  ! temp. at LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PRVLCL ! vapor mixing ratio at  LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PWLCL  ! parcel velocity at LCL (m/s)
    REAL, DIMENSION(KLON), INTENT(IN) :: PMFLCL ! cloud  base unit mass flux
    ! (kg/s)
    REAL, DIMENSION(KLON), INTENT(IN) :: PZLCL  ! height at LCL (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTHVELCL  ! environm. theta_v at LCL (K)
    LOGICAL, DIMENSION(KLON), INTENT(INOUT):: OTRIG! logical mask for convection
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! contains vert. index of DPL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   !  " vert. index of source layertop
!
!
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KCTL   ! contains vert. index of CTL
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KETL   ! contains vert. index of        &
    !equilibrium (zero buoyancy) level
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHL ! updraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHV ! updraft theta_v (K)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTT  ! updraft temperature(K)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURW  ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURC  ! updraft cloud water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURI  ! updraft cloud ice   (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURR  ! liquid precipit. (kg/kg)
    ! produced in  model layer
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)::PURS ! solid precipit. (kg/kg)
    ! produced in  model layer
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)::PUPR ! updraft precipitation in
    ! flux units (kg water / s)
    REAL, DIMENSION(KLON), INTENT(OUT):: PUTPR  ! total updraft precipitation
    ! in flux units (kg water / s)
    REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE  ! available potent. energy
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE  ! horizontal and vertical loop bounds
    INTEGER :: JI             ! horizontal loop index
    INTEGER :: JK, JKP, JKM, JK1, JK2, JKMIN   ! vertical loop index
    real :: ZEPSA          ! R_v / R_d, C_pv / C_pd
    real :: ZRDOCP         ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON)    :: ZUT             ! updraft temperature (K)
    REAL, DIMENSION(KLON)    :: ZUW1, ZUW2      ! square of updraft vert.
    ! velocity at levels k and k+1
    REAL, DIMENSION(KLON)    :: ZE1, ZE2, ZD1, ZD2 ! fractional entrainm./detrain
    ! rates at levels k and k+1
    REAL, DIMENSION(KLON)    :: ZMIXF           ! critical mixed fraction
    REAL, DIMENSION(KLON)    :: ZCPH            ! specific heat C_ph
    REAL, DIMENSION(KLON)    :: ZLV, ZLS        ! latent heat of vaporis., sublim.
    REAL, DIMENSION(KLON)    :: ZURV            ! updraft water vapor at level k+1
    REAL, DIMENSION(KLON)    :: ZPI             ! Pi=(P0/P)**(Rd/Cpd)
    REAL, DIMENSION(KLON)    :: ZTHEUL          ! theta_e for undilute ascent
    REAL, DIMENSION(KLON)    :: ZWORK1, ZWORK2, ZWORK3, ZWORK4, ZWORK5, &
                                ZWORK6          ! work arrays
    INTEGER, DIMENSION(KLON) :: IWORK           ! wok array
    LOGICAL, DIMENSION(KLON) :: GWORK1, GWORK2, GWORK4
    ! work arrays
    LOGICAL, DIMENSION(KLON, KLEV) :: GWORK6     ! work array
!
!
!-------------------------------------------------------------------------------
!
!        0.3   Set loop bounds
!              ---------------
!
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
    IIE = KLON
!
!
!*       1.     Initialize updraft properties and local variables
!               -------------------------------------------------
!
    ZEPSA = XRV / XRD
    ZRDOCP = XRD / XCPD
!
    PUMF(:, :) = 0.
    PUER(:, :) = 0.
    PUDR(:, :) = 0.
    PUTHL(:, :) = 0.
    PUTHV(:, :) = 0.
    PUTT(:, :) = 0.
    PURW(:, :) = 0.
    PURC(:, :) = 0.
    PURI(:, :) = 0.
    PUPR(:, :) = 0.
    PURR(:, :) = 0.
    PURS(:, :) = 0.
    PUTPR(:) = 0.
    ZUW1(:) = PWLCL(:) * PWLCL(:)
    ZUW2(:) = 0.
    ZE1(:) = 1.
    ZD1(:) = 0.
    PCAPE(:) = 0.
    KCTL(:) = IKB
    KETL(:) = KLCL(:)
    GWORK2(:) = .true.
    ZPI(:) = 1.
    ZWORK3(:) = 0.
    ZWORK4(:) = 0.
    ZWORK5(:) = 0.
    ZWORK6(:) = 0.
    GWORK1(:) = .false.
    GWORK4(:) = .false.
!
!
!*       1.1    Compute undilute updraft theta_e for CAPE computations
!               Bolton (1980) formula.
!               Define accurate enthalpy for updraft
!               -----------------------------------------------------
!
    ZTHEUL(:) = PTLCL(:) * (PTHLCL(:) / PTLCL(:))**(1.-0.28 * PRVLCL(:)) &
                * EXP((3374.6525 / PTLCL(:) - 2.5403) * &
                      PRVLCL(:) * (1.+0.81 * PRVLCL(:)))
!
!
    ZWORK1(:) = (XCPD + PRVLCL(:) * XCPV) * PTLCL(:) &
                + (1.+PRVLCL(:)) * XG * PZLCL(:)
!
!
!*       2.     Set updraft properties between DPL and LCL
!               ------------------------------------------
!
    JKP = MAXVAL(KLCL(:))
    JKM = MINVAL(KDPL(:))
    do JK = JKM, JKP
        do JI = 1, IIE
        if(JK >= KDPL(JI) .and. JK < KLCL(JI)) then
            PUMF(JI, JK) = PMFLCL(JI)
            PUTHL(JI, JK) = ZWORK1(JI)
            PUTHV(JI, JK) = PTHLCL(JI) * (1.+ZEPSA * PRVLCL(JI)) / &
                            (1.+PRVLCL(JI))
            PURW(JI, JK) = PRVLCL(JI)
        endif
        enddo
    enddo
!
!
!*       3.     Enter loop for updraft computations
!               ------------------------------------
!
! Correction for reproduciblity
!JKMIN = MINVAL( KLCL(:) ) - 1
    JKMIN = MINVAL(KLCL(:)) - 2
    do JK = MAX(IKB + 1, JKMIN), IKE - 1
        ZWORK6(:) = 1.
        JKP = JK + 1
!
        GWORK4(:) = JK >= KLCL(:) - 1
        GWORK1(:) = GWORK4(:) .and. GWORK2(:) ! this mask is used to confine
        ! updraft computations between the LCL and the CTL
!
        WHERE(JK == KLCL(:) - 1) ZWORK6(:) = 0. ! factor that is used in buoyancy
        ! computation at first level above LCL
!
!
!*       4.     Estimate condensate, L_v L_i, Cph and theta_v at level k+1
!               ----------------------------------------------------------
!
        ZWORK1(:) = PURC(:, JK) + PURR(:, JK)
        ZWORK2(:) = PURI(:, JK) + PURS(:, JK)
        call CONVECT_CONDENS(KLON, KICE, PPRES(:, JKP), PUTHL(:, JK), PURW(:, JK), &
                             ZWORK1, ZWORK2, PZ(:, JKP), GWORK1, ZUT, ZURV, &
                             PURC(:, JKP), PURI(:, JKP), ZLV, ZLS, ZCPH)
!
!
        ZPI(:) = (XP00 / PPRES(:, JKP))**ZRDOCP
        WHERE(GWORK1(:))
!
            PUTHV(:, JKP) = ZPI(:) * ZUT(:) * (1.+ZEPSA * ZURV(:)) &
                            / (1.+PURW(:, JK))
            PUTT(:, JKP) = ZUT(:)
!
!
!*       5.     Compute square of vertical velocity using entrainment
!               at level k
!               -----------------------------------------------------
!
            ZWORK3(:) = PZ(:, JKP) - PZ(:, JK) * ZWORK6(:) - &
                        (1.-ZWORK6(:)) * PZLCL(:)          ! level thickness
            ZWORK4(:) = PTHV(:, JK) * ZWORK6(:) + &
                        (1.-ZWORK6(:)) * PTHVELCL(:)
            ZWORK5(:) = 2.*ZUW1(:) * PUER(:, JK) / MAX(.1, PUMF(:, JK))
            ZUW2(:) = ZUW1(:) + ZWORK3(:) * XNHGAM * XG * &
                      ((PUTHV(:, JK) + PUTHV(:, JKP)) / &
                       (ZWORK4(:) + PTHV(:, JKP)) - 1.) & ! buoyancy term
                      - ZWORK5(:)                                  ! entrainment term
!
!
!*       6.     Update total precipitation: dr_r=(r_c+r_i)*exp(-rate*dz)
!               --------------------------------------------------------
!
!                    compute level mean vertical velocity
            ZWORK2(:) = 0.5 * &
                        (SQRT(MAX(1.E-2, ZUW2(:))) + &
                         SQRT(MAX(1.E-2, ZUW1(:))))
            PURR(:, JKP) = 0.5 * (PURC(:, JK) + PURC(:, JKP) + PURI(:, JK) + PURI(:, JKP)) &
                           * (1.-EXP(-XRCONV * ZWORK3(:) / ZWORK2(:)))
            PUPR(:, JKP) = PURR(:, JKP) * PUMF(:, JK) ! precipitation rate ( kg water / s)
            PUTPR(:) = PUTPR(:) + PUPR(:, JKP)   ! total precipitation rate
            ZWORK2(:) = PURR(:, JKP) / MAX(1.E-8, PURC(:, JKP) + PURI(:, JKP))
            PURR(:, JKP) = ZWORK2(:) * PURC(:, JKP)          ! liquid precipitation
            PURS(:, JKP) = ZWORK2(:) * PURI(:, JKP)          ! solid precipitation
!
!
!*       7.     Update r_c, r_i, enthalpy, r_w  for precipitation
!               -------------------------------------------------------
!
            PURW(:, JKP) = PURW(:, JK) - PURR(:, JKP) - PURS(:, JKP)
            PURC(:, JKP) = PURC(:, JKP) - PURR(:, JKP)
            PURI(:, JKP) = PURI(:, JKP) - PURS(:, JKP)
            PUTHL(:, JKP) = (XCPD + PURW(:, JKP) * XCPV) * ZUT(:) &
                            + (1.+PURW(:, JKP)) * XG * PZ(:, JKP) &
                            - ZLV(:) * PURC(:, JKP) - ZLS(:) * PURI(:, JKP)
!
            ZUW1(:) = ZUW2(:)
!
        ENDWHERE
!
!
!*       8.     Compute entrainment and detrainment using conservative
!               variables adjusted for precipitation ( not for entrainment)
!               -----------------------------------------------------------
!
!*       8.1    Compute critical mixed fraction by estimating unknown
!               T^mix r_c^mix and r_i^mix from enthalpy^mix and r_w^mix
!               We determine the zero crossing of the linear curve
!               evaluating the derivative using ZMIXF=0.1.
!               -----------------------------------------------------
!
        ZMIXF(:) = 0.1   ! starting value for critical mixed fraction
        ZWORK1(:) = ZMIXF(:) * PTHL(:, JKP) &
                    + (1.-ZMIXF(:)) * PUTHL(:, JKP) ! mixed enthalpy
        ZWORK2(:) = ZMIXF(:) * PRW(:, JKP) &
                    + (1.-ZMIXF(:)) * PURW(:, JKP)  ! mixed r_w
!
        call CONVECT_CONDENS(KLON, KICE, PPRES(:, JKP), ZWORK1, ZWORK2, &
                             PURC(:, JKP), PURI(:, JKP), PZ(:, JKP), GWORK1, ZUT, &
                             ZWORK3, ZWORK4, ZWORK5, ZLV, ZLS, ZCPH)
!        put in enthalpy and r_w and get T r_c, r_i (ZUT, ZWORK4-5)
!
        ! compute theta_v of mixture
        ZWORK3(:) = ZUT(:) * ZPI(:) * (1.+ZEPSA * ( &
                                       ZWORK2(:) - ZWORK4(:) - ZWORK5(:))) / (1.+ZWORK2(:))
        ! compute final value of critical mixed fraction using theta_v
        ! of mixture, grid-scale and updraft
        ZMIXF(:) = MAX(0., PUTHV(:, JKP) - PTHV(:, JKP)) * ZMIXF(:) / &
                   (PUTHV(:, JKP) - ZWORK3(:) + 1.E-10)
        ZMIXF(:) = MAX(0., MIN(1., ZMIXF(:)))
!
!
!*       8.2     Compute final midlevel values for entr. and detrainment
!                after call of distribution function
!                -------------------------------------------------------
!
!
        call CONVECT_MIXING_FUNCT(KLON, ZMIXF, 1, ZE2, ZD2)
!       Note: routine MIXING_FUNCT returns fractional entrainm/detrainm. rates
!
! ZWORK1(:) = XENTR * PMFLCL(:) * PDPRES(:,JKP) / XCRAD ! rate of env. inflow
!*MOD
        zwork1(:) = xentr * xg / xcrad * pumf(:, jk) * (pz(:, jkp) - pz(:, jk))
! ZWORK1(:) = XENTR * pumf(:,jk) * PDPRES(:,JKP) / XCRAD ! rate of env. inflow
!*MOD
        ZWORK2(:) = 0.
        WHERE(GWORK1(:)) ZWORK2(:) = 1.
        ZE2(:) = .5; ZD2(:) = .6 ! set entrainment=detrainment for better
        ! mass flux profiles in deep continental convection
        WHERE(PUTHV(:, JKP) > PTHV(:, JKP))
            PUER(:, JKP) = 0.5 * ZWORK1(:) * (ZE1(:) + ZE2(:)) * ZWORK2(:)
            PUDR(:, JKP) = 0.5 * ZWORK1(:) * (ZD1(:) + ZD2(:)) * ZWORK2(:)
        elseWHERE
            PUER(:, JKP) = 0.
            PUDR(:, JKP) = ZWORK1(:) * ZWORK2(:)
        ENDWHERE
!
!*       8.3     Determine equilibrium temperature level
!                --------------------------------------
!
        WHERE(PUTHV(:, JKP) > PTHV(:, JKP) .and. JK > KLCL(:) + 1 &
              .and. GWORK1(:))
            KETL(:) = JKP            ! equilibrium temperature level
        ENDWHERE
!
!*       8.4     If the calculated detrained mass flux is greater than
!                the total updraft mass flux, or vertical velocity is
!                negative, all cloud mass detrains at previous model level,
!                exit updraft calculations - CTL is attained
!                -------------------------------------------------------
!
        WHERE(GWORK1(:)) &
            GWORK2(:) = PUMF(:, JK) - PUDR(:, JKP) > 10. .and. ZUW2(:) > 0.
        WHERE(GWORK2(:)) KCTL(:) = JKP   ! cloud top level
!!!! Correction Bug C.Lac 30/10/08 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        KCTL(:) = MIN(KCTL(:), IKE - 1)
        GWORK1(:) = GWORK2(:) .and. GWORK4(:)
!
        if(COUNT(GWORK2(:)) == 0) EXIT
!
!
!*       9.   Compute CAPE for undilute ascent using theta_e and
!             theta_es instead of theta_v. This estimation produces
!             a significantly larger value for CAPE than the actual one.
!             ----------------------------------------------------------
!
        WHERE(GWORK1(:))
!
            ZWORK3(:) = PZ(:, JKP) - PZ(:, JK) * ZWORK6(:) - &
                        (1.-ZWORK6(:)) * PZLCL(:)              ! level thickness
            ZWORK2(:) = PTHES(:, JK) + (1.-ZWORK6(:)) * &
                        (PTHES(:, JKP) - PTHES(:, JK)) / (PZ(:, JKP) - PZ(:, JK)) * &
                        (PZLCL(:) - PZ(:, JK)) ! linear interpolation for theta_es at LCL
            ! ( this is only done for model level just above LCL
!
            ZWORK1(:) = (2.*ZTHEUL(:)) / (ZWORK2(:) + PTHES(:, JKP)) - 1.
            PCAPE(:) = PCAPE(:) + XG * ZWORK3(:) * MAX(0., ZWORK1(:))
!
!
!*       10.   Compute final values of updraft mass flux, enthalpy, r_w
!              at level k+1
!              --------------------------------------------------------
!
            PUMF(:, JKP) = PUMF(:, JK) - PUDR(:, JKP) + PUER(:, JKP)
            PUMF(:, JKP) = MAX(PUMF(:, JKP), 0.1)
            PUTHL(:, JKP) = (PUMF(:, JK) * PUTHL(:, JK) + &
                             PUER(:, JKP) * PTHL(:, JK) - PUDR(:, JKP) * PUTHL(:, JK)) &
                            / PUMF(:, JKP) + PUTHL(:, JKP) - PUTHL(:, JK)
            PURW(:, JKP) = (PUMF(:, JK) * PURW(:, JK) + &
                            PUER(:, JKP) * PRW(:, JK) - PUDR(:, JKP) * PURW(:, JK)) &
                           / PUMF(:, JKP) - PURR(:, JKP) - PURS(:, JKP)
!
!
            ZE1(:) = ZE2(:) ! update fractional entrainment/detrainment
            ZD1(:) = ZD2(:)
!
        ENDWHERE
!
    enddo
!
!*       12.1    Set OTRIG to False if cloud thickness < XCDEPTH
!                or CAPE < 1
!                ------------------------------------------------
!
    do JI = 1, IIE
        JK = KCTL(JI)
        OTRIG(JI) = PZ(JI, JK) - PZLCL(JI) >= XCDEPTH &
                    .and. PCAPE(JI) > 1.
    enddo
    WHERE(.not. OTRIG(:))
        KCTL(:) = IKB
    ENDWHERE
    KETL(:) = MAX(KETL(:), KLCL(:) + 2)
    KETL(:) = MIN(KETL(:), KCTL(:))
!
!
!*       12.2    If the ETL and CTL are the same detrain updraft mass
!                flux at this level
!                -------------------------------------------------------
!
    ZWORK1(:) = 0.
    WHERE(KETL(:) == KCTL(:)) ZWORK1(:) = 1.
!
    do JI = 1, IIE
        JK = KETL(JI)
        PUDR(JI, JK) = PUDR(JI, JK) + &
                       (PUMF(JI, JK) - PUER(JI, JK)) * ZWORK1(JI)
        PUER(JI, JK) = PUER(JI, JK) * (1.-ZWORK1(JI))
        PUMF(JI, JK) = PUMF(JI, JK) * (1.-ZWORK1(JI))
        JKP = KCTL(JI) + 1
        PUER(JI, JKP) = 0. ! entrainm/detr rates have been already computed
        PUDR(JI, JKP) = 0. ! at level KCTL+1, set them to zero
        PURW(JI, JKP) = 0.
        PURC(JI, JKP) = 0.
        PURI(JI, JKP) = 0.
        PUTHL(JI, JKP) = 0.
        PURI(JI, JKP + 1) = 0.
        PURC(JI, JKP + 1) = 0.
    enddo
!
!*       12.3    Adjust mass flux profiles, detrainment rates, and
!                precipitation fallout rates to reflect linear decrease
!                in mass flux between the ETL and CTL
!                -------------------------------------------------------
!
    ZWORK1(:) = 0.
    JK1 = MINVAL(KETL(:))
    JK2 = MAXVAL(KCTL(:))
    do JK = JK1, JK2
        do JI = 1, IIE
        if(JK > KETL(JI) .and. JK <= KCTL(JI)) then
            ZWORK1(JI) = ZWORK1(JI) + PDPRES(JI, JK)
        endif
        enddo
    enddo
!
    do JI = 1, IIE
        JK = KETL(JI)
        ZWORK1(JI) = PUMF(JI, JK) / MAX(1., ZWORK1(JI))
    enddo
!
    do JK = JK1 + 1, JK2
        JKP = JK - 1
        do JI = 1, IIE
        if(JK > KETL(JI) .and. JK <= KCTL(JI)) then
            ! PUTPR(JI)    = PUTPR(JI) - ( PURR(JI,JK) + PURS(JI,JK) ) * PUMF(JI,JKP)
            PUTPR(JI) = PUTPR(JI) - PUPR(JI, JK)
            PUDR(JI, JK) = PDPRES(JI, JK) * ZWORK1(JI)
            PUMF(JI, JK) = PUMF(JI, JKP) - PUDR(JI, JK)
            PUPR(JI, JK) = PUMF(JI, JKP) * (PURR(JI, JK) + PURS(JI, JK))
            PUTPR(JI) = PUTPR(JI) + PUPR(JI, JK)
        endif
        enddo
    enddo
!
!         12.4   Set mass flux and entrainment in the source layer.
!                Linear increase throughout the source layer.
!                -------------------------------------------------------
!
!IWORK(:) = MIN( KPBL(:), KLCL(:) - 1 )
    IWORK(:) = KPBL(:)
    do JI = 1, IIE
        JK = KDPL(JI)
        JKP = IWORK(JI)
!          mixed layer depth
        ZWORK2(JI) = PPRES(JI, JK) - PPRES(JI, JKP) + PDPRES(JI, JK)
    enddo
!
    JKP = MAXVAL(IWORK(:))
    do JK = JKM, JKP
        do JI = 1, IIE
        if(JK >= KDPL(JI) .and. JK <= IWORK(JI)) then
            PUER(JI, JK) = PUER(JI, JK) + PMFLCL(JI) * PDPRES(JI, JK) / (ZWORK2(JI) + 0.1)
            PUMF(JI, JK) = PUMF(JI, JK - 1) + PUER(JI, JK)
        endif
        enddo
    enddo
!
!
!*       13.   If cloud thickness is smaller than  3 km, no
!              convection is allowed
!              Nota: For technical reasons, we stop the convection
!                    computations in this case and do not go back to
!                    TRIGGER_FUNCT to look for the next unstable LCL
!                    which could produce a thicker cloud.
!              ---------------------------------------------------
!
    GWORK6(:, :) = SPREAD(OTRIG(:), DIM=2, NCOPIES=KLEV)
    WHERE(.not. OTRIG(:)) PUTPR(:) = 0.
    WHERE(.not. GWORK6(:, :))
        PUMF(:, :) = 0.
        PUDR(:, :) = 0.
        PUER(:, :) = 0.
        PUTHL(:, :) = PTHL(:, :)
        PURW(:, :) = PRW(:, :)
        PUPR(:, :) = 0.
        PURC(:, :) = 0.
        PURI(:, :) = 0.
        PURR(:, :) = 0.
        PURS(:, :) = 0.
    ENDWHERE
!
ENDsubroutine CONVECT_UPDRAFT

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_TSTEP_PREF
!     #################
!
    INTERFACE
!
        subroutine CONVECT_TSTEP_PREF(KLON, KLEV, &
                                      PU, PV, PPRES, PZ, PDXDY, KLCL, KCTL, &
                                      PTIMEA, PPREF)
!
            INTEGER, INTENT(IN)                    :: KLON   ! horizontal dimension
            INTEGER, INTENT(IN)                    :: KLEV   ! vertical dimension
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PU     ! grid scale horiz. wind u
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PV     ! grid scale horiz. wind v
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! lifting condensation level index
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! cloud top level index
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PTIMEA ! advective time period
            REAL, DIMENSION(KLON), INTENT(OUT):: PPREF  ! precipitation efficiency
!
        ENDsubroutine CONVECT_TSTEP_PREF
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_TSTEP_PREF
!     ######################################################################
subroutine CONVECT_TSTEP_PREF(KLON, KLEV, &
                              PU, PV, PPRES, PZ, PDXDY, KLCL, KCTL, &
                              PTIMEA, PPREF)
!     ######################################################################
!
!!**** Routine to compute convective advection time step and precipitation
!!     efficiency
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the convective
!!      advection time step PTIMEC as a function of the mean ambient
!!      wind as well as the precipitation efficiency as a function
!!      of wind shear and cloud base height.
!!
!!
!!**  METHOD
!!    ------
!!
!!
!!    EXTERNAL
!!    --------
!!     None
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation
!!      Fritsch and Chappell, 1980, J. Atmos. Sci.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAREXT
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                    :: KLON   ! horizontal dimension
    INTEGER, INTENT(IN)                    :: KLEV   ! vertical dimension
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PU     ! grid scale horiz. wind u
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PV     ! grid scale horiz. wind v
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! lifting condensation level index
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! cloud top level index
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PTIMEA ! advective time period
    REAL, DIMENSION(KLON), INTENT(OUT):: PPREF  ! precipitation efficiency
!
!
!*       0.2   Declarations of local variables KLON
!
    INTEGER :: IIE, IKB, IKE                      ! horizontal + vertical loop bounds
    INTEGER :: JI                                 ! horizontal loop index
    INTEGER :: JK, JKLC, JKP5, JKCT               ! vertical loop index
!
    INTEGER, DIMENSION(KLON)  :: IP500       ! index of 500 hPa levels
    REAL, DIMENSION(KLON)     :: ZCBH        ! cloud base height
    REAL, DIMENSION(KLON)     :: ZWORK1, ZWORK2, ZWORK3  ! work arrays
!
!
!-------------------------------------------------------------------------------
!
!        0.3   Set loop bounds
!              ---------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Determine vertical index for 500 hPa levels
!               ------------------------------------------
!
!
    IP500(:) = IKB
    do JK = IKB, IKE
        WHERE(PPRES(:, JK) >= 500.E2) IP500(:) = JK
    enddo
!
!
!*       2.     Compute convective time step
!               ----------------------------
!
    ! compute wind speed at LCL, 500 hPa, CTL

    do JI = 1, IIE
        JKLC = KLCL(JI)
        JKP5 = IP500(JI)
        JKCT = KCTL(JI)
        ZWORK1(JI) = SQRT(PU(JI, JKLC) * PU(JI, JKLC) + &
                          PV(JI, JKLC) * PV(JI, JKLC))
        ZWORK2(JI) = SQRT(PU(JI, JKP5) * PU(JI, JKP5) + &
                          PV(JI, JKP5) * PV(JI, JKP5))
        ZWORK3(JI) = SQRT(PU(JI, JKCT) * PU(JI, JKCT) + &
                          PV(JI, JKCT) * PV(JI, JKCT))
    enddo
!
    ZWORK2(:) = MAX(0.1, 0.5 * (ZWORK1(:) + ZWORK2(:)))
!
    PTIMEA(:) = SQRT(PDXDY(:)) / ZWORK2(:)
!
!
!*       3.     Compute precipitation efficiency
!               -----------------------------------
!
!*       3.1    Precipitation efficiency as a function of wind shear
!               ----------------------------------------------------
!
    ZWORK2(:) = SIGN(1., ZWORK3(:) - ZWORK1(:))
    do JI = 1, IIE
        JKLC = KLCL(JI)
        JKCT = KCTL(JI)
        ZWORK1(JI) = (PU(JI, JKCT) - PU(JI, JKLC)) * &
                     (PU(JI, JKCT) - PU(JI, JKLC)) + &
                     (PV(JI, JKCT) - PV(JI, JKLC)) * &
                     (PV(JI, JKCT) - PV(JI, JKLC))
        ZWORK1(JI) = 1.E3 * ZWORK2(JI) * SQRT(ZWORK1(JI)) / &
                     MAX(1.E-2, PZ(JI, JKCT) - PZ(JI, JKLC))
    enddo
!
    PPREF(:) = 1.591 + ZWORK1(:) * (-.639 + ZWORK1(:) * ( &
                                    9.53E-2 - ZWORK1(:) * 4.96E-3))
    PPREF(:) = MAX(.4, MIN(PPREF(:), .9))
!
!*       3.2    Precipitation efficiency as a function of cloud base height
!               ----------------------------------------------------------
!
    do JI = 1, IIE
        JKLC = KLCL(JI)
        ZCBH(JI) = MAX(3.,(PZ(JI, JKLC) - PZ(JI, IKB)) * 3.281E-3)
    enddo
    ZWORK1(:) = .9673 + ZCBH(:) * (-.7003 + ZCBH(:) * (.1622 + &
                                                       ZCBH(:) * (-1.2570E-2 + ZCBH(:) * (4.2772E-4 - &
                                                                                          ZCBH(:) * 5.44E-6))))
    ZWORK1(:) = MAX(.4, MIN(.9, 1./(1.+ZWORK1(:))))
!
!*       3.3    Mean precipitation efficiency is used to compute rainfall
!               ----------------------------------------------------------
!
    PPREF(:) = 0.5 * (PPREF(:) + ZWORK1(:))
!
!
ENDsubroutine CONVECT_TSTEP_PREF

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_DOWNDRAFT
!     #################
!
    INTERFACE
!
        subroutine CONVECT_DOWNDRAFT(KLON, KLEV, &
                                     KICE, PPRES, PDPRES, PZ, PTH, PTHES, &
                                     PRW, PRC, PRI, &
                                     PPREF, KLCL, KCTL, KETL, &
                                     PUTHL, PURW, PURC, PURI, &
                                     PDMF, PDER, PDDR, PDTHL, PDRW, &
                                     PMIXF, PDTEVR, KLFS, KDBL, KML, &
                                     PDTEVRF)
!
            INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
            INTEGER, INTENT(IN) :: KICE  ! flag for ice ( 1 = yes,
            !                0 = no ice )
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH   ! grid scale theta
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
            ! mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC   ! grid scale r_c (cloud water)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI   ! grid scale r_i (cloud ice)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
            ! bottom and top of layer (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! level height (m)
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL  ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL  ! contains vert. index of CTL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KETL  ! contains vert. index of
            ! equilibrium (zero buoyancy) level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KML   ! " vert. index of melting level
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PUTHL ! updraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURW  ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURC  ! updraft r_c (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURI  ! updraft r_i (kg/kg)
            REAL, DIMENSION(KLON), INTENT(IN) :: PPREF ! precipitation efficiency
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDMF   ! downdraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDER   ! downdraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDDR   ! downdraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDTHL  ! downdraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDRW   ! downdraft total water (kg/kg)
            REAL, DIMENSION(KLON), INTENT(OUT):: PMIXF  ! mixed fraction at LFS
            REAL, DIMENSION(KLON), INTENT(OUT):: PDTEVR ! total downdraft evaporation
            ! rate at LFS (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDTEVRF! downdraft evaporation rate
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KLFS    ! contains vert. index of LFS
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KDBL    ! contains vert. index of DBL
!
        ENDsubroutine CONVECT_DOWNDRAFT
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_DOWNDRAFT
!    ##########################################################################
subroutine CONVECT_DOWNDRAFT(KLON, KLEV, &
                             KICE, PPRES, PDPRES, PZ, PTH, PTHES, &
                             PRW, PRC, PRI, &
                             PPREF, KLCL, KCTL, KETL, &
                             PUTHL, PURW, PURC, PURI, &
                             PDMF, PDER, PDDR, PDTHL, PDRW, &
                             PMIXF, PDTEVR, KLFS, KDBL, KML, &
                             PDTEVRF)
!    ##########################################################################
!
!!**** Compute downdraft properties from LFS to DBL.
!!
!!
!!    PDRPOSE
!!    -------
!!      The purpose of this routine is to determine downdraft properties
!!      ( mass flux, thermodynamics )
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from top.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_SATMIXRATIO
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XPI                ! Pi
!!          XP00               ! reference pressure
!!          XRD, XRV           ! gaz  constants for dry air and water vapor
!!          XCPD               ! Cpd (dry air)
!!          XCPV, XCL, XCI     ! Cp of water vapor, liquid water and ice
!!          XTT                ! triple point temperature
!!          XLVTT, XLSTT       ! vaporisation/sublimation heat at XTT
!!
!!      Module MODD_CONVPAR
!!          XCRAD              ! cloud radius
!!          XZPBL              ! thickness of downdraft detrainment layer
!!          XENTR              ! entrainment constant in pressure coordinates
!!          XRHDBC             ! relative humidity in downdraft below cloud
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_DOWNDRAFT)
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!!   C.Lac          27/09/10 modification loop index for reproducibility
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
    USE MODD_CONVPAREXT
!
    USE MODI_CONVECT_SATMIXRATIO
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
    INTEGER, INTENT(IN) :: KICE  ! flag for ice ( 1 = yes,
    !                0 = no ice )
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH   ! grid scale theta
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
    ! mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC   ! grid scale r_c (cloud water)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI   ! grid scale r_i (cloud ice)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! level height (m)
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL  ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL  ! contains vert. index of CTL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KETL  ! contains vert. index of
    ! equilibrium (zero buoyancy) level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KML   ! " vert. index of melting level
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PUTHL ! updraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURW  ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURC  ! updraft r_c (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURI  ! updraft r_i (kg/kg)
    REAL, DIMENSION(KLON), INTENT(IN) :: PPREF ! precipitation efficiency
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDMF   ! downdraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDER   ! downdraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDDR   ! downdraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDTHL  ! downdraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDRW   ! downdraft total water (kg/kg)
    REAL, DIMENSION(KLON), INTENT(OUT):: PMIXF  ! mixed fraction at LFS
    REAL, DIMENSION(KLON), INTENT(OUT):: PDTEVR ! total downdraft evaporation
    ! rate at LFS (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PDTEVRF! downdraft evaporation rate
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KLFS    ! contains vert. index of LFS
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KDBL    ! contains vert. index of DBL
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE     ! horizontal + vertical loop bounds
    INTEGER :: JK, JKP, JKM, JKT ! vertical loop index
    INTEGER :: JI, JL            ! horizontal loop index
    INTEGER :: JITER          ! iteration loop index
    real :: ZRDOCP         ! R_d / C_pd
    real :: ZEPS           ! R_d / R_v
!
    INTEGER, DIMENSION(KLON) :: IDDT      ! top level of detrainm. layer
    REAL, DIMENSION(KLON)    :: ZTHE      ! environm. theta_e (K)
    REAL, DIMENSION(KLON)    :: ZDT, ZDTP ! downdraft temperature (K)
    REAL, DIMENSION(KLON)    :: ZCPH      ! specific heat C_ph
    REAL, DIMENSION(KLON)    :: ZLV, ZLS  ! latent heat of vaporis., sublim.
    REAL, DIMENSION(KLON)    :: ZDDT      ! thickness (hPa) of detrainm. layer
    REAL, DIMENSION(KLON)    :: ZPI       ! Pi=(P0/P)**(Rd/Cpd)
    REAL, DIMENSION(KLON)    :: ZWORK1, ZWORK2, ZWORK3, ZWORK4 ! work arrays
    LOGICAL, DIMENSION(KLON) :: GWORK1                         ! work array
!
!
!-------------------------------------------------------------------------------
!
!        0.3    Set loop bounds
!               ---------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Initialize downdraft properties
!               -------------------------------
!
    ZRDOCP = XRD / XCPD
    ZEPS = XRD / XRV
    PDMF(:, :) = 0.
    PDER(:, :) = 0.
    PDDR(:, :) = 0.
    PDRW(:, :) = 0.
    PDTHL(:, :) = 0.
    PDTEVR(:) = 0.
    PMIXF(:) = 0.
    ZTHE(:) = 0.
    ZDDT(:) = PDPRES(:, IKB + 2)
    KDBL(:) = IKB + 1
    KLFS(:) = IKB + 1
    IDDT(:) = KDBL(:) + 1
!
!
!*       2.     Determine the LFS by looking for minimum of environmental
!               saturated theta_e
!               ----------------------------------------------------------
!
    ZWORK1(:) = 900.   ! starting value for search of minimum envir. theta_e
    do JK = MINVAL(KLCL(:)) + 2, MAXVAL(KETL(:))
        do JI = 1, IIE
            GWORK1(JI) = JK >= KLCL(JI) + 2 .and. JK < KETL(JI)
            if(GWORK1(JI) .and. ZWORK1(JI) > PTHES(JI, JK)) then
                KLFS(JI) = JK
                ZWORK1(JI) = MIN(ZWORK1(JI), PTHES(JI, JK))
            endif
        enddo
    enddo
!
!
!*       3.     Determine the mixed fraction using environmental and updraft
!               values of theta_e at LFS
!               ---------------------------------------------------------
!
    do JI = 1, IIE
        JK = KLFS(JI)
        ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
        ! compute updraft theta_e
        ZWORK3(JI) = PURW(JI, JK) - PURC(JI, JK) - PURI(JI, JK)
        ZDT(JI) = PTH(JI, JK) / ZPI(JI)
        ZLV(JI) = XLVTT + (XCPV - XCL) * (ZDT(JI) - XTT)
        ZLS(JI) = XLSTT + (XCPV - XCI) * (ZDT(JI) - XTT)
        ZCPH(JI) = XCPD + XCPV * PURW(JI, JK)
        ZDT(JI) = (PUTHL(JI, JK) - (1.+PURW(JI, JK)) * XG * PZ(JI, JK) &
                   + ZLV(JI) * PURC(JI, JK) + ZLS(JI) * PURI(JI, JK)) / ZCPH(JI)
        ZWORK1(JI) = ZDT(JI) * ZPI(JI)**(1.-0.28 * ZWORK3(JI)) &
                     * EXP((3374.6525 / ZDT(JI) - 2.5403) &
                           * ZWORK3(JI) * (1.+0.81 * ZWORK3(JI)))
        ! compute environmental theta_e
        ZDT(JI) = PTH(JI, JK) / ZPI(JI)
        ZLV(JI) = XLVTT + (XCPV - XCL) * (ZDT(JI) - XTT)
        ZLS(JI) = XLSTT + (XCPV - XCI) * (ZDT(JI) - XTT)
        ZWORK3(JI) = PRW(JI, JK) - PRC(JI, JK) - PRI(JI, JK)
        ZCPH(JI) = XCPD + XCPV * PRW(JI, JK)
        ZWORK2(JI) = ZDT(JI) * ZPI(JI)**(1.-0.28 * ZWORK3(JI)) &
                     * EXP((3374.6525 / ZDT(JI) - 2.5403) &
                           * ZWORK3(JI) * (1.+0.81 * ZWORK3(JI)))
        ! compute mixed fraction
        PMIXF(JI) = MAX(0.,(ZWORK1(JI) - PTHES(JI, JK))) &
                    / (ZWORK1(JI) - ZWORK2(JI) + 1.E-10)
        PMIXF(JI) = MAX(0., MIN(1., PMIXF(JI)))
        ZWORK4(JI) = PPRES(JI, JK)
    enddo
!
!
!*       4.     Estimate the effect of melting on the downdraft
!               ---------------------------------------------
!
    ZWORK1(:) = 0.
    ! use total solid precipitation
!do JK = IKB + 1, IKE
!    ZWORK1(:) = ZWORK1(:) + PURS(:,JK) ! total snow/hail content
!end do
!
    do JI = 1, IIE
        JK = KLCL(JI)
        JKP = KCTL(JI)
        ZWORK1(JI) = 0.5 * (PURW(JI, JK) - PURW(JI, JKP))
    enddo
!
    ! temperature perturbation due to melting at LFS
    ZWORK3(:) = 0.
    WHERE(KML(:) > IKB + 2)
        ZWORK3(:) = ZWORK1(:) * (ZLS(:) - ZLV(:)) / ZCPH(:)
        ZDT(:) = ZDT(:) - ZWORK3(:) * REAL(KICE)
    ENDWHERE
!
!
!*       5.     Initialize humidity at LFS as a saturated mixture of
!               updraft and environmental air
!               -----------------------------------------------------
!
    do JI = 1, IIE
        JK = KLFS(JI)
        PDRW(JI, JK) = PMIXF(JI) * PRW(JI, JK) + (1.-PMIXF(JI)) * PURW(JI, JK)
        ZWORK2(JI) = PDRW(JI, JK) - (1.-PMIXF(JI)) &
                     * (PURC(JI, JK) + PURI(JI, JK))
    enddo
!
!
!*       6.1    Determine the DBL by looking for level where the envir.
!               theta_es at the LFS corrected by melting effects  becomes
!               larger than envir. value
!               ---------------------------------------------------------
!
    ! compute satur. mixing ratio for melting corrected temperature
    call CONVECT_SATMIXRATIO(KLON, ZWORK4, ZDT, ZWORK3, ZLV, ZLS, ZCPH)
!
    ! compute envir. saturated theta_e for melting corrected temperature
    ZWORK1(:) = MIN(ZWORK2(:), ZWORK3(:))
    ZWORK3(:) = ZWORK3(:) * ZWORK4(:) / (ZWORK3(:) + ZEPS) ! sat. pressure
    ZWORK3(:) = ALOG(ZWORK3(:) / 613.3)
    ! dewp point temperature
    ZWORK3(:) = (4780.8 - 32.19 * ZWORK3(:)) / (17.502 - ZWORK3(:))
    ! adiabatic saturation temperature
    ZWORK3(:) = ZWORK3(:) - (.212 + 1.571E-3 * (ZWORK3(:) - XTT) &
                             - 4.36E-4 * (ZDT(:) - XTT)) * (ZDT(:) - ZWORK3(:))
    ZWORK4(:) = SIGN(0.5, ZWORK2(:) - ZWORK3(:))
    ZDT(:) = ZDT(:) * (.5 + ZWORK4(:)) + (.5 - ZWORK4(:)) * ZWORK3(:)
    ZWORK2(:) = ZDT(:) * ZPI(:)**(1.-0.28 * ZWORK2(:)) &
                * EXP((3374.6525 / ZDT(:) - 2.5403) &
                      * ZWORK1(:) * (1.+0.81 * ZWORK1(:)))
!
    GWORK1(:) = .true.
    JKM = MAXVAL(KLFS(:))
    do JK = JKM - 1, IKB + 1, -1
        do JI = 1, IIE
            if(JK < KLFS(JI) .and. ZWORK2(JI) > PTHES(JI, JK) .and. GWORK1(JI)) then
                KDBL(JI) = JK
                GWORK1(JI) = .false.
            endif
        enddo
    enddo
!
!
!*       7.     Define mass flux and entr/detr. rates at LFS
!               -------------------------------------------
!
    do JI = 1, IIE
        JK = KLFS(JI)
        ZWORK1(JI) = PPRES(JI, JK) / &
                     (XRD * ZDT(JI) * (1.+ZEPS * ZWORK1(JI))) ! density
        PDMF(JI, JK) = -(1.-PPREF(JI)) * ZWORK1(JI) * XPI * XCRAD * XCRAD
        PDTHL(JI, JK) = ZWORK2(JI)   ! theta_l is here actually theta_e
        ZWORK2(JI) = PDMF(JI, JK)
        PDDR(JI, JK) = 0.
        PDER(JI, JK) = -PMIXF(JI) * PDMF(JI, JK)
    enddo
!
!
!         7.1   Downdraft detrainment is assumed to occur in a layer
!               of 60 hPa, determine top level IDDT of this layer
!               ---------------------------------------------------------
!
    ZWORK1(:) = 0.
    do JK = IKB + 2, JKM
        ZWORK1(:) = ZWORK1(:) + PDPRES(:, JK)
        !WHERE ( JK > KDBL(:) .and. ZWORK1(:) <= XZPBL )
        WHERE(JK > KDBL(:) .and. JK <= KLCL(:))
            ZDDT(:) = ZWORK1(:)
            IDDT(:) = JK
        ENDWHERE
    enddo
!
!
!*       8.     Enter loop for downdraft computations. Make a first guess
!               of initial downdraft mass flux.
!               In the downdraft computations we use theta_es instead of
!               enthalpy as it allows to better take into account evaporation
!               effects. As the downdraft detrainment rate is zero apart
!               from the detrainment layer, we just compute enthalpy
!               downdraft from theta_es in this layer.
!               ----------------------------------------------------------
!
!
!
    do JK = JKM - 1, IKB + 1, -1
        JKP = JK + 1
        do JI = 1, IIE
            if(JK < KLFS(JI) .and. JK >= IDDT(JI)) then
                PDER(JI, JK) = -ZWORK2(JI) * XENTR * PDPRES(JI, JKP) / XCRAD
                ! DER and DPRES are positive
                PDMF(JI, JK) = PDMF(JI, JKP) - PDER(JI, JK)
                ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
                ZDT(JI) = PTH(JI, JK) / ZPI(JI)
                ZWORK1(JI) = PRW(JI, JK) - PRC(JI, JK) - PRI(JI, JK)
                ZTHE(JI) = ZDT(JI) * ZPI(JI)**(1.-0.28 * ZWORK1(JI)) &
                           * EXP((3374.6525 / ZDT(JI) - 2.5403) &
                                 * ZWORK1(JI) * (1.+0.81 * ZWORK1(JI)))
                ! PDTHL is here theta_es, later on in this routine this table is
                ! reskipped to enthalpy
                PDTHL(JI, JK) = (PDTHL(JI, JKP) * PDMF(JI, JKP) - ZTHE(JI) * PDER(JI, JK) &
                                 ) / (PDMF(JI, JK) - 1.E-7)
                PDRW(JI, JK) = (PDRW(JI, JKP) * PDMF(JI, JKP) - PRW(JI, JK) * PDER(JI, JK) &
                                ) / (PDMF(JI, JK) - 1.E-7)
            endif
            if(JK < IDDT(JI) .and. JK >= KDBL(JI)) then
                JL = IDDT(JI)
                PDDR(JI, JK) = -PDMF(JI, JL) * PDPRES(JI, JKP) / ZDDT(JI)
                PDMF(JI, JK) = PDMF(JI, JKP) + PDDR(JI, JK)
                PDTHL(JI, JK) = PDTHL(JI, JKP)
                PDRW(JI, JK) = PDRW(JI, JKP)
            endif
        enddo
    enddo
!
!
!*       9.     Calculate total downdraft evaporation
!               rate for given mass flux (between DBL and IDDT)
!               -----------------------------------------------
!
    PDTEVRF(:, :) = 0.
! Reproducibility
!JKT = MAXVAL( IDDT(:) )
!do JK = IKB + 1, JKT
    do JK = IKB + 1, IKE
!
        ZPI(:) = (XP00 / PPRES(:, JK))**ZRDOCP
        ZDT(:) = PTH(:, JK) / ZPI(:)
!
!*       9.1    Determine wet bulb temperature at DBL from theta_e.
!               The iteration algoritm is similar to that used in
!               routine CONVECT_CONDENS
!               --------------------------------------------------
!
        do JITER = 1, 4
            call CONVECT_SATMIXRATIO(KLON, PPRES(:, JK), ZDT, ZWORK1, ZLV, ZLS, ZCPH)
            ZDTP(:) = PDTHL(:, JK) / (ZPI(:)**(1.-0.28 * ZWORK1(:)) &
                                      * EXP((3374.6525 / ZDT(:) - 2.5403) &
                                            * ZWORK1(:) * (1.+0.81 * ZWORK1(:))))
            ZDT(:) = 0.4 * ZDTP(:) + 0.6 * ZDT(:) ! force convergence
        enddo
!
!
!*       9.2    Sum total downdraft evaporation rate. No evaporation
!               if actual humidity is larger than specified one.
!               -----------------------------------------------------
!
        ZWORK2(:) = ZWORK1(:) / ZDT(:) * (XBETAW / ZDT(:) - XGAMW) ! dr_sat/dT
        ZWORK2(:) = ZLV(:) / ZCPH(:) * ZWORK1(:) * (1.-XRHDBC) / &
                    (1.+ZLV(:) / ZCPH(:) * ZWORK2(:)) ! temperature perturb                                                           ! due to evaporation
        ZDT(:) = ZDT(:) + ZWORK2(:)
!
        call CONVECT_SATMIXRATIO(KLON, PPRES(:, JK), ZDT, ZWORK3, ZLV, ZLS, ZCPH)
!
        ZWORK3(:) = ZWORK3(:) * XRHDBC
        ZWORK1(:) = MAX(0., ZWORK3(:) - PDRW(:, JK))
        PDTEVR(:) = PDTEVR(:) + ZWORK1(:) * PDDR(:, JK)
        PDTEVRF(:, JK) = PDTEVRF(:, JK) + ZWORK1(:) * PDDR(:, JK)
        ! compute enthalpie and humidity in the detrainment layer
        PDRW(:, JK) = MAX(PDRW(:, JK), ZWORK3(:))
        PDTHL(:, JK) = ((XCPD + PDRW(:, JK) * XCPV) * ZDT(:) &
                        + (1.+PDRW(:, JK)) * XG * PZ(:, JK))
!
    enddo
!
!
!*      12.     If downdraft does not evaporate any water for specified
!               relative humidity, no downdraft is allowed
!               ---------------------------------------------------------
!
    ZWORK2(:) = 1.
    WHERE(PDTEVR(:) < 1. .OR. KLFS(:) == IKB + 1) ZWORK2(:) = 0.
    do JK = IKB, JKM
        KDBL(:) = KDBL(:) * INT(ZWORK2(:)) + (1 - INT(ZWORK2(:))) * IKB
        KLFS(:) = KLFS(:) * INT(ZWORK2(:)) + (1 - INT(ZWORK2(:))) * IKB
        PDMF(:, JK) = PDMF(:, JK) * ZWORK2(:)
        PDER(:, JK) = PDER(:, JK) * ZWORK2(:)
        PDDR(:, JK) = PDDR(:, JK) * ZWORK2(:)
        ZWORK1(:) = REAL(KLFS(:) - JK)         ! use this to reset thl_d
        ZWORK1(:) = MAX(0., MIN(1., ZWORK1(:))) ! and rv_d to zero above LFS
        PDTHL(:, JK) = PDTHL(:, JK) * ZWORK2(:) * ZWORK1(:)
        PDRW(:, JK) = PDRW(:, JK) * ZWORK2(:) * ZWORK1(:)
        PDTEVR(:) = PDTEVR(:) * ZWORK2(:)
        PDTEVRF(:, JK) = PDTEVRF(:, JK) * ZWORK2(:)
    enddo
!
ENDsubroutine CONVECT_DOWNDRAFT

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_PRECIP_ADJUST
!     #################
!
    INTERFACE
!
        subroutine CONVECT_PRECIP_ADJUST(KLON, KLEV, &
                                         PPRES, PUMF, PUER, PUDR, &
                                         PUPR, PUTPR, PURW, &
                                         PDMF, PDER, PDDR, PDTHL, PDRW, &
                                         PPREF, PTPR, PMIXF, PDTEVR, &
                                         KLFS, KDBL, KLCL, KCTL, KETL, &
                                         PDTEVRF)

!
            INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURW  ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON), INTENT(IN) :: PUTPR ! updraft  total precipit. (kg/s
            REAL, DIMENSION(KLON), INTENT(IN) :: PPREF ! precipitation efficiency
            REAL, DIMENSION(KLON), INTENT(IN) :: PMIXF ! critical mixed fraction at LCL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL  ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL  ! contains vert. index of CTL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KETL  ! contains vert. index of equilibrium
            ! (zero buoyancy) level
            INTEGER, DIMENSION(KLON), INTENT(INOUT) :: KLFS ! contains vert. index of LFS
            INTEGER, DIMENSION(KLON), INTENT(INOUT) :: KDBL ! contains vert. index of DBL
!
            REAL, DIMENSION(KLON), INTENT(INOUT) :: PDTEVR ! total downdraft evaporation
            ! rate at LFS
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDTEVRF! downdraft evaporation rate
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF   ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER   ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR   ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUPR   ! updraft  precipit. (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDMF   ! downdraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDER   ! downdraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDDR   ! downdraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDTHL  ! downdraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDRW   ! downdraft total water (kg/kg)
!
            REAL, DIMENSION(KLON), INTENT(OUT)   :: PTPR    ! total precipitation (kg/s)
            ! = downdraft precipitation
!
        ENDsubroutine CONVECT_PRECIP_ADJUST
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_PRECIP_ADJUST
!     ######################################################################
subroutine CONVECT_PRECIP_ADJUST(KLON, KLEV, &
                                 PPRES, PUMF, PUER, PUDR, &
                                 PUPR, PUTPR, PURW, &
                                 PDMF, PDER, PDDR, PDTHL, PDRW, &
                                 PPREF, PTPR, PMIXF, PDTEVR, &
                                 KLFS, KDBL, KLCL, KCTL, KETL, &
                                 PDTEVRF)
!     ######################################################################
!
!!**** Adjust up- and downdraft mass fluxes to be consistent with the
!!     mass transport at the LFS given by the precipitation efficiency
!!     relation.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to adjust up- and downdraft mass
!!      fluxes below the LFS to be consistent with the precipitation
!!      efficiency relation
!!
!!
!!
!!**  METHOD
!!    ------
!!
!!
!!    EXTERNAL
!!    --------
!!     None
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!     Module MODD_CONVPAR
!!        XUSRDPTH             ! pressure depth to compute updraft humidity
!!                             ! supply rate for downdraft
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_PRECIP_ADJUST)
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAREXT
    USE MODD_CONVPAR
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PURW  ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON), INTENT(IN) :: PUTPR ! updraft  total precipit. (kg/s
    REAL, DIMENSION(KLON), INTENT(IN) :: PPREF ! precipitation efficiency
    REAL, DIMENSION(KLON), INTENT(IN) :: PMIXF ! critical mixed fraction at LCL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL  ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL  ! contains vert. index of CTL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KETL  ! contains vert. index of equilibrium
    ! (zero buoyancy) level
    INTEGER, DIMENSION(KLON), INTENT(INOUT) :: KLFS ! contains vert. index of LFS
    INTEGER, DIMENSION(KLON), INTENT(INOUT) :: KDBL ! contains vert. index of DBL
!
    REAL, DIMENSION(KLON), INTENT(INOUT) :: PDTEVR ! total downdraft evaporation
    ! rate at LFS
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDTEVRF! downdraft evaporation rate
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF   ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER   ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR   ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUPR   ! updraft  precipit. (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDMF   ! downdraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDER   ! downdraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDDR   ! downdraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDTHL  ! downdraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDRW   ! downdraft total water (kg/kg)
!
    REAL, DIMENSION(KLON), INTENT(OUT)   :: PTPR    ! total precipitation (kg/s)
    ! = downdraft precipitation
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE        ! horizontal + vertical loop bounds
    INTEGER :: JK, JKT1, JKT2, JKT3 ! vertical loop index
    INTEGER :: JI                   ! horizontal loop index
!
    INTEGER, DIMENSION(KLON) :: IPRL
    REAL, DIMENSION(KLON)    :: ZWORK1, ZWORK2, ZWORK3, &
                                ZWORK4, ZWORK5, ZWORK6 ! work arrays
!
!
!-------------------------------------------------------------------------------
!
!        0.3   Set loop bounds
!              ---------------
!
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
    IIE = KLON
    JKT1 = MAXVAL(KLFS(:))
    JKT2 = MAXVAL(KCTL(:))
    JKT3 = MINVAL(KLCL(:))
!
!
!        1.    Set some output variables for columns where no downdraft
!              exists. Exit if there is no downdraft at all.
!              --------------------------------------------------------
!
    IPRL(:) = IKB
    PTPR(:) = 0.
!
    WHERE(PDTEVR(:) == 0.)
        PTPR(:) = PUTPR(:)  ! no downdraft evaporation => no downdraft, all
        ! precipitation occurs in updraft
    ENDWHERE
    if(COUNT(PDTEVR(:) > 0.) == 0) then  ! exit routine if no downdraft exists
        RETURN
    endif
!
!*       2.     The total mass transported from the updraft to the down-
!               draft at the LFS must be consistent with the three water
!               budget terms :
!               ---------------------------------------------------------
!
!*       2.1    Downdraft evaporation rate at the DBL. The evaporation
!               rate in downdraft must be consistent with precipitation
!               efficiency relation.
!               --------------------------------------------------------
!
!
    do JI = 1, IIE
        JK = KLFS(JI)
        ZWORK1(JI) = PDTEVR(JI) / MIN(-1.E-1, PDMF(JI, JK))
        ZWORK6(JI) = PDMF(JI, JK)
    enddo
!
!*       2.2    Some preliminar computations for downdraft = total
!               precipitation rate. The precipitation is evaluated in
!               a layer thickness DP=XUSRDPTH=165 hPa above the LCL.
!               The difference between updraft precipitation and downdraft
!               precipitation (updraft supply rate) is used to drive the
!               downdraft through evaporational cooling.
!               --------------------------------------------------------
!
    do JI = 1, IIE
        JK = KLCL(JI)
        ZWORK5(JI) = PPRES(JI, JK)
    enddo
!
    PTPR(:) = 0.
    do JK = JKT3, JKT2
        WHERE(JK >= KLCL(:) .and. PPRES(:, JK) >= ZWORK5(:) - XUSRDPTH)
            PTPR(:) = PTPR(:) + PUPR(:, JK)
            IPRL(:) = JK
        ENDWHERE
    enddo
    IPRL(:) = MIN(KETL(:), IPRL(:))
!
    do JI = 1, IIE
        JK = IPRL(JI)
        PTPR(JI) = PUMF(JI, JK + 1) * PURW(JI, JK + 1) + PTPR(JI)
    enddo
!
    PTPR(:) = PPREF(:) * MIN(PUTPR(:), PTPR(:))
    ZWORK4(:) = PUTPR(:) - PTPR(:)
!
!
!*       2.3    Total amount of precipitation that falls out of the up-
!               draft between the LCL and the LFS.
!               Condensate transfer from up to downdraft at LFS
!               ---------------------------------------------------------
!
    ZWORK5(:) = 0.
    do JK = JKT3, JKT1
        WHERE(JK >= KLCL(:) .and. JK <= KLFS(:))
            ZWORK5(:) = ZWORK5(:) + PUPR(:, JK)
        ENDWHERE
    enddo
!
    do JI = 1, IIE
        JK = KLFS(JI)
        ZWORK2(JI) = (1.-PPREF(JI)) * ZWORK5(JI) * &
                     (1.-PMIXF(JI)) / MAX(1.E-1, PUMF(JI, JK))
    enddo
!
!
!*       2.4    Increase the first guess downdraft mass flux to satisfy
!               precipitation efficiency relation.
!               If downdraft does not evaporate any water at the DBL for
!               the specified relative humidity, or if the corrected mass
!               flux at the LFS is positive no downdraft is allowed
!               ---------------------------------------------------------
!
!
!ZWORK1(:) = ZWORK4(:) / ( ZWORK1(:) + ZWORK2(:) + 1.E-8 )
    ZWORK1(:) = -ZWORK4(:) / (-ZWORK1(:) + ZWORK2(:) + 1.E-8)
    ZWORK2(:) = ZWORK1(:) / MIN(-1.E-1, ZWORK6(:)) ! ratio of budget consistent to actual DMF
!
    ZWORK3(:) = 1.
    ZWORK6(:) = 1.
    WHERE(ZWORK1(:) > 0. .OR. PDTEVR(:) < 1.)
        KDBL(:) = IKB
        KLFS(:) = IKB
        PDTEVR(:) = 0.
        ZWORK2(:) = 0.
        ZWORK3(:) = 0.
        ZWORK6(:) = 0.
    ENDWHERE
!
    do JK = IKB, JKT1
        PDMF(:, JK) = PDMF(:, JK) * ZWORK2(:)
        PDER(:, JK) = PDER(:, JK) * ZWORK2(:)
        PDDR(:, JK) = PDDR(:, JK) * ZWORK2(:)
        PDTEVRF(:, JK) = PDTEVRF(:, JK) * ZWORK2(:)
        PDRW(:, JK) = PDRW(:, JK) * ZWORK3(:)
        PDTHL(:, JK) = PDTHL(:, JK) * ZWORK3(:)
    enddo
    ZWORK4(:) = ZWORK2(:)
!
!
!*       3.     Increase updraft mass flux, mass detrainment rate, and water
!               substance detrainment rates to be consistent with the transfer
!               of the estimated mass from the up- to the downdraft at the LFS
!               --------------------------------------------------------------
!
    do JI = 1, IIE
        JK = KLFS(JI)
        ZWORK2(JI) = (1.-ZWORK6(JI)) + ZWORK6(JI) * &
                     (PUMF(JI, JK) - (1.-PMIXF(JI)) * ZWORK1(JI)) / &
                     MAX(1.E-1, PUMF(JI, JK))
    enddo
!
!
    JKT1 = MAXVAL(KLFS(:))  ! value of KLFS might have been reset to IKB above
    do JK = IKB, JKT1
        do JI = 1, IIE
            if(JK <= KLFS(JI)) then
                PUMF(JI, JK) = PUMF(JI, JK) * ZWORK2(JI)
                PUER(JI, JK) = PUER(JI, JK) * ZWORK2(JI)
                PUDR(JI, JK) = PUDR(JI, JK) * ZWORK2(JI)
                PUPR(JI, JK) = PUPR(JI, JK) * ZWORK2(JI)
            endif
        enddo
    enddo
!
!
!*       4.     Increase total = downdraft precipitation and evaporation rate
!               -------------------------------------------------------------
!
    WHERE(PDTEVR(:) > 0.)
        PTPR(:) = PTPR(:) + PPREF(:) * ZWORK5(:) * (ZWORK2(:) - 1.)
        PDTEVR(:) = PUTPR(:) - PTPR(:)
        PDTEVRF(:, IKB + 1) = PDTEVR(:)
    elseWHERE
        PTPR(:) = PUTPR(:)
    ENDWHERE
!
!
ENDsubroutine CONVECT_PRECIP_ADJUST

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_CLOSURE_THRVLCL
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CLOSURE_THRVLCL(KLON, KLEV, &
                                           PPRES, PTH, PRV, PZ, OWORK1, &
                                           PTHLCL, PRVLCL, PZLCL, PTLCL, PTELCL, &
                                           KLCL, KDPL, KPBL)
!
            INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH   ! theta
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV   ! vapor mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of grid point (m)
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL  ! contains vert. index of DPL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL  ! " vert. index of source layer top
            LOGICAL, DIMENSION(KLON), INTENT(IN) :: OWORK1! logical mask
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL ! theta at LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL ! vapor mixing ratio at  LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL  ! height at LCL (m)
            REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL  ! temperature at LCL (m)
            REAL, DIMENSION(KLON), INTENT(OUT):: PTELCL ! environm. temp. at LCL (K)
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KLCL   ! contains vert. index of LCL
!
        ENDsubroutine CONVECT_CLOSURE_THRVLCL
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CLOSURE_THRVLCL
!     #########################################################################
subroutine CONVECT_CLOSURE_THRVLCL(KLON, KLEV, &
                                   PPRES, PTH, PRV, PZ, OWORK1, &
                                   PTHLCL, PRVLCL, PZLCL, PTLCL, PTELCL, &
                                   KLCL, KDPL, KPBL)
!     #########################################################################
!
!!**** Determine thermodynamic properties at new LCL
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the thermodynamic
!!      properties at the new lifting condensation level LCL
!!
!!
!!
!!**  METHOD
!!    ------
!!    see CONVECT_TRIGGER_FUNCT
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_SATMIXRATIO
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! Reference pressure
!!          XRD, XRV           ! Gaz  constants for dry air and water vapor
!!          XCPD               ! Cpd (dry air)
!!          XTT                ! triple point temperature
!!          XBETAW, XGAMW      ! constants for vapor saturation pressure
!!
!!      Module MODD_CONVPAR
!!          XA25               ! reference grid area
!!          XZLCL              ! lowest allowed pressure difference between
!!                             ! surface and LCL
!!          XZPBL              ! minimum mixed layer depth to sustain convection
!!          XWTRIG             ! constant in vertical velocity trigger
!!
!!      Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book2 of documentation ( routine TRIGGER_FUNCT)
!!      Fritsch and Chappell (1980), J. Atm. Sci., Vol. 37, 1722-1761.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
    USE MODD_CONVPAREXT
    USE MODI_CONVECT_SATMIXRATIO
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN) :: KLON  ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV  ! vertical dimension
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH   ! theta
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV   ! vapor mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of grid point (m)
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL  ! contains vert. index of DPL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL  ! " vert. index of source layer top
    LOGICAL, DIMENSION(KLON), INTENT(IN) :: OWORK1! logical mask
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL ! theta at LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL ! vapor mixing ratio at  LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL  ! height at LCL (m)
    REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL  ! temperature at LCL (m)
    REAL, DIMENSION(KLON), INTENT(OUT):: PTELCL ! environm. temp. at LCL (K)
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KLCL   ! contains vert. index of LCL
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: JK, JKM, JKMIN, JKMAX      ! vertical loop index
    INTEGER :: JI                         ! horizontal loop index
    INTEGER :: IIE, IKB, IKE              ! horizontal + vertical loop bounds
    real :: ZEPS           ! R_d / R_v
    real :: ZCPORD, ZRDOCP ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON) :: ZPLCL    ! pressure at LCL
    REAL, DIMENSION(KLON) :: ZTMIX    ! mixed layer temperature
    REAL, DIMENSION(KLON) :: ZEVMIX   ! mixed layer water vapor pressure
    REAL, DIMENSION(KLON) :: ZDPTHMIX, ZPRESMIX ! mixed layer depth and pressure
    REAL, DIMENSION(KLON) :: ZLV, ZCPH! specific heats of vaporisation, dry air
    REAL, DIMENSION(KLON) :: ZDP      ! pressure between LCL and model layer
    REAL, DIMENSION(KLON) :: ZWORK1, ZWORK2     ! work arrays
!
!
!-------------------------------------------------------------------------------
!
!*       0.3    Compute array bounds
!               --------------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Initialize local variables
!               --------------------------
!
    ZEPS = XRD / XRV
    ZCPORD = XCPD / XRD
    ZRDOCP = XRD / XCPD
!
    ZDPTHMIX(:) = 0.
    ZPRESMIX(:) = 0.
    PTHLCL(:) = 300.
    PTLCL(:) = 300.
    PTELCL(:) = 300.
    PRVLCL(:) = 0.
    PZLCL(:) = PZ(:, IKB)
    ZTMIX(:) = 230.
    ZPLCL(:) = 1.E4
    KLCL(:) = IKB + 1
!
!
!*       2.     Construct a mixed layer as in TRIGGER_FUNCT
!               -------------------------------------------
!
    JKMAX = MAXVAL(KPBL(:))
    JKMIN = MINVAL(KDPL(:))
    do JK = IKB + 1, JKMAX
        JKM = JK + 1
        do JI = 1, IIE
        if(JK >= KDPL(JI) .and. JK <= KPBL(JI)) then
!
            ZWORK1(JI) = PPRES(JI, JK) - PPRES(JI, JKM)
            ZDPTHMIX(JI) = ZDPTHMIX(JI) + ZWORK1(JI)
            ZPRESMIX(JI) = ZPRESMIX(JI) + PPRES(JI, JK) * ZWORK1(JI)
            PTHLCL(JI) = PTHLCL(JI) + PTH(JI, JK) * ZWORK1(JI)
            PRVLCL(JI) = PRVLCL(JI) + PRV(JI, JK) * ZWORK1(JI)
!
        endif
        enddo
    enddo
!
!
    WHERE(OWORK1(:))
!
        ZPRESMIX(:) = ZPRESMIX(:) / ZDPTHMIX(:)
        PTHLCL(:) = PTHLCL(:) / ZDPTHMIX(:)
        PRVLCL(:) = PRVLCL(:) / ZDPTHMIX(:)
!
!*       3.1    Use an empirical direct solution ( Bolton formula )
!               to determine temperature and pressure at LCL.
!               Nota: the adiabatic saturation temperature is not
!                     equal to the dewpoint temperature
!               --------------------------------------------------
!
!
        ZTMIX(:) = PTHLCL(:) * (ZPRESMIX(:) / XP00)**ZRDOCP
        ZEVMIX(:) = PRVLCL(:) * ZPRESMIX(:) / (PRVLCL(:) + ZEPS)
        ZEVMIX(:) = MAX(1.E-8, ZEVMIX(:))
        ZWORK1(:) = ALOG(ZEVMIX(:) / 613.3)
        ! dewpoint temperature
        ZWORK1(:) = (4780.8 - 32.19 * ZWORK1(:)) / (17.502 - ZWORK1(:))
        ! adiabatic saturation temperature
        PTLCL(:) = ZWORK1(:) - (.212 + 1.571E-3 * (ZWORK1(:) - XTT) &
                                - 4.36E-4 * (ZTMIX(:) - XTT)) * (ZTMIX(:) - ZWORK1(:))
        PTLCL(:) = MIN(PTLCL(:), ZTMIX(:))
        ZPLCL(:) = XP00 * (PTLCL(:) / PTHLCL(:))**ZCPORD
!
    ENDWHERE
!
    ZPLCL(:) = MIN(2.E5, MAX(10., ZPLCL(:))) ! bound to avoid overflow
!
!
!*       3.2    Correct PTLCL in order to be completely consistent
!               with MNH saturation formula
!               --------------------------------------------------
!
    call CONVECT_SATMIXRATIO(KLON, ZPLCL, PTLCL, ZWORK1, ZLV, ZWORK2, ZCPH)
    WHERE(OWORK1(:))
        ZWORK2(:) = ZWORK1(:) / PTLCL(:) * (XBETAW / PTLCL(:) - XGAMW) ! dr_sat/dT
        ZWORK2(:) = (ZWORK1(:) - PRVLCL(:)) / &
                    (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
        PTLCL(:) = PTLCL(:) - ZLV(:) / ZCPH(:) * ZWORK2(:)
!
    ENDWHERE
!
!
!*       3.3    If PRVLCL is oversaturated set humidity and temperature
!               to saturation values.
!               -------------------------------------------------------
!
    call CONVECT_SATMIXRATIO(KLON, ZPRESMIX, ZTMIX, ZWORK1, ZLV, ZWORK2, ZCPH)
    WHERE(OWORK1(:) .and. PRVLCL(:) > ZWORK1(:))
        ZWORK2(:) = ZWORK1(:) / ZTMIX(:) * (XBETAW / ZTMIX(:) - XGAMW) ! dr_sat/dT
        ZWORK2(:) = (ZWORK1(:) - PRVLCL(:)) / &
                    (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
        PTLCL(:) = ZTMIX(:) + ZLV(:) / ZCPH(:) * ZWORK2(:)
        PRVLCL(:) = PRVLCL(:) - ZWORK2(:)
        ZPLCL(:) = ZPRESMIX(:)
        PTHLCL(:) = PTLCL(:) * (XP00 / ZPLCL(:))**ZRDOCP
    ENDWHERE
!
!
!*        4.1   Determine  vertical loop index at the LCL
!               -----------------------------------------
!
    do JK = JKMIN, IKE - 1
        do JI = 1, IIE
        if(ZPLCL(JI) <= PPRES(JI, JK) .and. OWORK1(JI)) then
            KLCL(JI) = JK + 1
            PZLCL(JI) = PZ(JI, JK + 1)
        endif
        enddo
    enddo
!
!
!*        4.2   Estimate height and environmental temperature at LCL
!               ----------------------------------------------------
!
    do JI = 1, IIE
        JK = KLCL(JI)
        JKM = JK - 1
        ZDP(JI) = ALOG(ZPLCL(JI) / PPRES(JI, JKM)) / &
                  ALOG(PPRES(JI, JK) / PPRES(JI, JKM))
        ZWORK1(JI) = PTH(JI, JK) * (PPRES(JI, JK) / XP00)**ZRDOCP
        ZWORK2(JI) = PTH(JI, JKM) * (PPRES(JI, JKM) / XP00)**ZRDOCP
        ZWORK1(JI) = ZWORK2(JI) + (ZWORK1(JI) - ZWORK2(JI)) * ZDP(JI)
        ! we compute the precise value of the LCL
        ! The precise height is between the levels KLCL and KLCL-1.
        ZWORK2(JI) = PZ(JI, JKM) + (PZ(JI, JK) - PZ(JI, JKM)) * ZDP(JI)
    enddo
    WHERE(OWORK1(:))
        PTELCL(:) = ZWORK1(:)
        PZLCL(:) = ZWORK2(:)
    ENDWHERE
!
!
!
ENDsubroutine CONVECT_CLOSURE_THRVLCL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_CLOSURE_ADJUST
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CLOSURE_ADJUST(KLON, KLEV, PADJ, &
                                          PUMF, PZUMF, PUER, PZUER, PUDR, PZUDR, &
                                          PDMF, PZDMF, PDER, PZDER, PDDR, PZDDR, &
                                          PPRMELT, PZPRMELT, PDTEVR, PZDTEVR, &
                                          PTPR, PZTPR, &
                                          PPRLFLX, PZPRLFL, PPRSFLX, PZPRSFL)
!
            INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
            REAL, DIMENSION(KLON), INTENT(IN) :: PADJ     ! mass adjustment factor
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUMF ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUER ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUDR ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDMF  ! downdraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDMF ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDER  ! downdraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDER ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDDR  ! downdraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDDR ! initial value of  "
            REAL, DIMENSION(KLON), INTENT(INOUT):: PTPR     ! total precipitation (kg/s)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PZTPR    ! initial value of "
            REAL, DIMENSION(KLON), INTENT(INOUT):: PDTEVR   ! donwndraft evapor. (kg/s)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PZDTEVR  ! initial value of "
            REAL, DIMENSION(KLON), INTENT(INOUT):: PPRMELT  ! melting of precipitation
            REAL, DIMENSION(KLON), INTENT(INOUT):: PZPRMELT ! initial value of "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PPRLFLX! liquid precip flux
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PZPRLFL! initial value "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PPRSFLX! solid  precip flux
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PZPRSFL! initial value "
!
        ENDsubroutine CONVECT_CLOSURE_ADJUST
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CLOSURE_ADJUST
!    ###########################################################################
subroutine CONVECT_CLOSURE_ADJUST(KLON, KLEV, PADJ, &
                                  PUMF, PZUMF, PUER, PZUER, PUDR, PZUDR, &
                                  PDMF, PZDMF, PDER, PZDER, PDDR, PZDDR, &
                                  PPRMELT, PZPRMELT, PDTEVR, PZDTEVR, &
                                  PTPR, PZTPR, &
                                  PPRLFLX, PZPRLFL, PPRSFLX, PZPRSFL)
!    ###########################################################################
!
!!**** Uses closure adjustment factor to adjust mass flux and to modify
!!     precipitation efficiency  when necessary. The computations are
!!     similar to routine CONVECT_PRECIP_ADJUST.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to adjust the mass flux using the
!!      factor PADJ computed in CONVECT_CLOSURE
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!    EXTERNAL
!!    --------
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    None
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!    None
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_CLOSURE_ADJUST)
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  04/10/97
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAREXT
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
    REAL, DIMENSION(KLON), INTENT(IN) :: PADJ     ! mass adjustment factor
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUMF ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUER ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUDR ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDMF  ! downdraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDMF ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDER  ! downdraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDER ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PDDR  ! downdraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZDDR ! initial value of  "
    REAL, DIMENSION(KLON), INTENT(INOUT):: PTPR     ! total precipitation (kg/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PZTPR    ! initial value of "
    REAL, DIMENSION(KLON), INTENT(INOUT):: PDTEVR   ! donwndraft evapor. (kg/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PZDTEVR  ! initial value of "
    REAL, DIMENSION(KLON), INTENT(INOUT):: PPRMELT  ! melting of precipitation
    REAL, DIMENSION(KLON), INTENT(INOUT):: PZPRMELT ! initial value of "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PPRLFLX! liquid precip flux
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PZPRLFL! initial value "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PPRSFLX! solid  precip flux
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT)  :: PZPRSFL! initial value "
!
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IKB, IKE                 !  vert. loop bounds
    INTEGER :: JK                       ! vertical loop index
!
!
!-------------------------------------------------------------------------------
!
!*       0.3   Compute loop bounds
!              -------------------
!
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Adjust mass flux by the factor PADJ to converge to
!               specified degree of stabilization
!               ----------------------------------------------------
!
    PPRMELT(:) = PZPRMELT(:) * PADJ(:)
    PDTEVR(:) = PZDTEVR(:) * PADJ(:)
    PTPR(:) = PZTPR(:) * PADJ(:)
!
    do JK = IKB + 1, IKE
        PUMF(:, JK) = PZUMF(:, JK) * PADJ(:)
        PUER(:, JK) = PZUER(:, JK) * PADJ(:)
        PUDR(:, JK) = PZUDR(:, JK) * PADJ(:)
        PDMF(:, JK) = PZDMF(:, JK) * PADJ(:)
        PDER(:, JK) = PZDER(:, JK) * PADJ(:)
        PDDR(:, JK) = PZDDR(:, JK) * PADJ(:)
        PPRLFLX(:, JK) = PZPRLFL(:, JK) * PADJ(:)
        PPRSFLX(:, JK) = PZPRSFL(:, JK) * PADJ(:)
    enddo
!
ENDsubroutine CONVECT_CLOSURE_ADJUST

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_CLOSURE
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CLOSURE(KLON, KLEV, &
                                   PPRES, PDPRES, PZ, PDXDY, PLMASS, &
                                   PTHL, PTH, PRW, PRC, PRI, OTRIG1, &
                                   PTHC, PRWC, PRCC, PRIC, PWSUB, &
                                   KLCL, KDPL, KPBL, KLFS, KCTL, KML, &
                                   PUMF, PUER, PUDR, PUTHL, PURW, &
                                   PURC, PURI, PUPR, &
                                   PDMF, PDER, PDDR, PDTHL, PDRW, &
                                   PTPR, PSPR, PDTEVR, &
                                   PCAPE, PTIMEC, &
                                   KFTSTEPS, &
                                   PDTEVRF, PPRLFLX, PPRSFLX)
!
            INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV   ! vertical dimension
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLFS   ! index for level of free sink
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! index lifting condens. level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! index for cloud top level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! index for departure level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   ! index for top of source layer
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KML    ! index for melting level
            REAL, DIMENSION(KLON), INTENT(INOUT) :: PTIMEC ! convection time step
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL   ! grid scale enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH    ! grid scale theta
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW    ! grid scale total water
            ! mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC    ! grid scale r_c
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI    ! grid scale r_i
            LOGICAL, DIMENSION(KLON), INTENT(IN) :: OTRIG1 ! logical to keep trace of
            ! convective arrays modified in UPDRAFT
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (P)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES ! pressure difference between
            ! bottom and top of layer (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PLMASS ! mass of model layer (kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN)  :: PCAPE  ! available potent. energy
            INTEGER, INTENT(OUT)   :: KFTSTEPS! maximum of fract time steps
            ! only used for chemical tracers
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUPR  ! updraft precipitation in
            ! flux units (kg water / s)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PUTHL  ! updraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURW   ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURC   ! updraft cloud water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURI   ! updraft cloud ice   (kg/kg)
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDMF  ! downdraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDER  ! downdraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDDR  ! downdraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)   :: PDTHL ! downdraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)   :: PDRW  ! downdraft total water (kg/kg)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PTPR  ! total surf precipitation (kg/s)
            REAL, DIMENSION(KLON), INTENT(OUT)  :: PSPR  ! solid surf precipitation (kg/s)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PDTEVR! donwndraft evapor. (kg/s)
!
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PTHC  ! conv. adj. grid scale theta
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRWC  ! conv. adj. grid scale r_w
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRCC  ! conv. adj. grid scale r_c
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRIC  ! conv. adj. grid scale r_i
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PWSUB ! envir. compensating subsidence(Pa/s)
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDTEVRF! downdraft evaporation rate
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PPRLFLX! liquid precip flux
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PPRSFLX! solid  precip flux
!
        ENDsubroutine CONVECT_CLOSURE
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CLOSURE
!    #########################################################################
subroutine CONVECT_CLOSURE(KLON, KLEV, &
                           PPRES, PDPRES, PZ, PDXDY, PLMASS, &
                           PTHL, PTH, PRW, PRC, PRI, OTRIG1, &
                           PTHC, PRWC, PRCC, PRIC, PWSUB, &
                           KLCL, KDPL, KPBL, KLFS, KCTL, KML, &
                           PUMF, PUER, PUDR, PUTHL, PURW, &
                           PURC, PURI, PUPR, &
                           PDMF, PDER, PDDR, PDTHL, PDRW, &
                           PTPR, PSPR, PDTEVR, &
                           PCAPE, PTIMEC, &
                           KFTSTEPS, &
                           PDTEVRF, PPRLFLX, PPRSFLX)
!    #########################################################################
!
!!**** Uses modified Fritsch-Chappell closure
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the final adjusted
!!     (over a time step PTIMEC) environmental values of THETA_l, R_w, R_c, R_i
!!      The final convective tendencies can then be evaluated in the main
!!      routine DEEP_CONVECT by (PTHC-PTH)/PTIMEC
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!
!!    EXTERNAL
!!    --------
!!
!!    CONVECT_CLOSURE_THRVLCL
!!    CONVECT_CLOSURE_ADJUST
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! reference pressure
!!          XRD, XRV           ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV         ! specific heat for dry air and water vapor
!!          XCL, XCI           ! specific heat for liquid water and ice
!!          XTT                ! triple point temperature
!!          XLVTT, XLSTT       ! vaporization, sublimation heat constant
!!
!!      Module MODD_CONVPAR
!!          XA25               ! reference grid area
!!          XSTABT             ! stability factor in time integration
!!          XSTABC             ! stability factor in CAPE adjustment
!!          XMELDPTH           ! allow melting over specific pressure depth
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_CLOSURE)
!!      Fritsch and Chappell, 1980, J. Atmos. Sci.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Peter Bechtold 04/10/97 change for enthalpie, r_c + r_i tendencies
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR
    USE MODD_CONVPAREXT
!
    USE MODI_CONVECT_SATMIXRATIO
    USE MODI_CONVECT_CLOSURE_THRVLCL
    USE MODI_CONVECT_CLOSURE_ADJUST
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV   ! vertical dimension
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLFS   ! index for level of free sink
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! index lifting condens. level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! index for cloud top level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! index for departure level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   ! index for top of source layer
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KML    ! index for melting level
    REAL, DIMENSION(KLON), INTENT(INOUT) :: PTIMEC ! convection time step
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL   ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH    ! grid scale theta
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW    ! grid scale total water
    ! mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC    ! grid scale r_c
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI    ! grid scale r_i
    LOGICAL, DIMENSION(KLON), INTENT(IN) :: OTRIG1 ! logical to keep trace of
    ! convective arrays modified in UPDRAFT
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (P)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES ! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PLMASS ! mass of model layer (kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN)  :: PCAPE  ! available potent. energy
    INTEGER, INTENT(OUT)   :: KFTSTEPS! maximum of fract time steps
    ! only used for chemical tracers
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUPR  ! updraft precipitation in
    ! flux units (kg water / s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PUTHL  ! updraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURW   ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURC   ! updraft cloud water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURI   ! updraft cloud ice   (kg/kg)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDMF  ! downdraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDER  ! downdraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDDR  ! downdraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)   :: PDTHL ! downdraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)   :: PDRW  ! downdraft total water (kg/kg)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PTPR  ! total surf precipitation (kg/s)
    REAL, DIMENSION(KLON), INTENT(OUT)  :: PSPR  ! solid surf precipitation (kg/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PDTEVR! donwndraft evapor. (kg/s)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PTHC  ! conv. adj. grid scale theta
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRWC  ! conv. adj. grid scale r_w
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRCC  ! conv. adj. grid scale r_c
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRIC  ! conv. adj. grid scale r_i
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PWSUB ! envir. compensating subsidence(Pa/s)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDTEVRF! downdraft evaporation rate
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PPRLFLX! liquid precip flux
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PPRSFLX! solid  precip flux
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE  ! horizontal + vertical loop bounds
    INTEGER :: IKS            ! vertical dimension
    INTEGER :: JK, JKP, JKMAX ! vertical loop index
    INTEGER :: JI             ! horizontal loop index
    INTEGER :: JITER          ! iteration loop index
    INTEGER :: JSTEP          ! fractional time loop index
    real :: ZCPORD, ZRDOCP ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON, KLEV) :: ZTHLC       ! convectively adjusted
    ! grid scale enthalpy
    REAL, DIMENSION(KLON, KLEV) :: ZOMG        ! conv. environm. subsidence (Pa/s)
    REAL, DIMENSION(KLON, KLEV) :: ZUMF        ! non-adjusted updraft mass flux
    REAL, DIMENSION(KLON, KLEV) :: ZUER        !   "     updraft  entrainm. rate
    REAL, DIMENSION(KLON, KLEV) :: ZUDR        !   "     updraft  detrainm. rate
    REAL, DIMENSION(KLON, KLEV) :: ZDMF        !   "   downdraft mass flux
    REAL, DIMENSION(KLON, KLEV) :: ZDER        !   "   downdraft  entrainm. rate
    REAL, DIMENSION(KLON, KLEV) :: ZDDR        !   "   downdraft  detrainm. rate
    REAL, DIMENSION(KLON)     :: ZTPR         !   "   total precipitation
    REAL, DIMENSION(KLON)     :: ZDTEVR       !   "   total downdraft evapor.
    REAL, DIMENSION(KLON, KLEV):: ZPRLFLX      !   "   liquid precip flux
    REAL, DIMENSION(KLON, KLEV):: ZPRSFLX      !   "   solid  precip flux
    REAL, DIMENSION(KLON)     :: ZPRMELT      ! melting of precipitation
    REAL, DIMENSION(KLON)     :: ZPRMELTO     ! non-adjusted  "
    REAL, DIMENSION(KLON)     :: ZADJ         ! mass adjustment factor
    REAL, DIMENSION(KLON)     :: ZADJMAX      ! limit value for ZADJ
    REAL, DIMENSION(KLON)     :: ZCAPE        ! new CAPE after adjustment
    REAL, DIMENSION(KLON)     :: ZTIMEC       ! fractional convective time step
    REAL, DIMENSION(KLON, KLEV):: ZTIMC        ! 2D work array for ZTIMEC
!
    REAL, DIMENSION(KLON)     :: ZTHLCL       ! new  theta at LCL
    REAL, DIMENSION(KLON)     :: ZRVLCL       ! new  r_v at LCL
    REAL, DIMENSION(KLON)     :: ZZLCL        ! height of LCL
    REAL, DIMENSION(KLON)     :: ZTLCL        ! temperature at LCL
    REAL, DIMENSION(KLON)     :: ZTELCL       ! envir. temper. at LCL
    REAL, DIMENSION(KLON)     :: ZTHEUL       ! theta_e for undilute ascent
    REAL, DIMENSION(KLON)     :: ZTHES1, ZTHES2! saturation environm. theta_e
    REAL, DIMENSION(KLON, KLEV) :: ZTHMFIN, ZTHMFOUT, ZRWMFIN, ZRWMFOUT
    REAL, DIMENSION(KLON, KLEV) :: ZRCMFIN, ZRCMFOUT, ZRIMFIN, ZRIMFOUT
    ! work arrays for environm. compensat. mass flux
    REAL, DIMENSION(KLON)     :: ZPI          ! (P/P00)**R_d/C_pd
    REAL, DIMENSION(KLON)     :: ZLV          ! latent heat of vaporisation
    REAL, DIMENSION(KLON)     :: ZLS          ! latent heat of sublimation
    REAL, DIMENSION(KLON)     :: ZLM          ! latent heat of melting
    REAL, DIMENSION(KLON)     :: ZCPH         ! specific heat C_ph
    REAL, DIMENSION(KLON)     :: ZMELDPTH     ! actual depth of melting layer
    INTEGER, DIMENSION(KLON)  :: ITSTEP       ! fractional convective time step
    INTEGER, DIMENSION(KLON)  :: ICOUNT       ! timestep counter
    INTEGER, DIMENSION(KLON)  :: ILCL         ! index lifting condens. level
    INTEGER, DIMENSION(KLON)  :: IWORK1       ! work array
    REAL, DIMENSION(KLON)     :: ZWORK1, ZWORK2, ZWORK3, ZWORK4, ZWORK5
    REAL, DIMENSION(KLON, KLEV):: ZWORK6
    LOGICAL, DIMENSION(KLON)  :: GWORK1, GWORK3! work arrays
    LOGICAL, DIMENSION(KLON, KLEV) :: GWORK4    ! work array
!
!
!-------------------------------------------------------------------------------
!
!*       0.2    Initialize  local variables
!               ----------------------------
!
!
    PSPR(:) = 0.
    ZTIMC(:, :) = 0.
    ZTHES2(:) = 0.
    ZWORK1(:) = 0.
    ZWORK2(:) = 0.
    ZWORK3(:) = 0.
    ZWORK4(:) = 0.
    ZWORK5(:) = 0.
    GWORK1(:) = .false.
    GWORK3(:) = .false.
    GWORK4(:, :) = .false.
    ILCL(:) = KLCL(:)
!
    ZCPORD = XCPD / XRD
    ZRDOCP = XRD / XCPD
!
    ZADJ(:) = 1.
    ZWORK5(:) = 1.
    WHERE(.not. OTRIG1(:)) ZWORK5(:) = 0.
!
!
!*       0.3   Compute loop bounds
!              -------------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKS = KLEV
    IKE = KLEV - JCVEXT
    JKMAX = MAXVAL(KCTL(:))
!
!
!*       2.     Save initial mass flux values to be used in adjustment procedure
!               ---------------------------------------------------------------
!
    ZUMF(:, :) = PUMF(:, :)
    ZUER(:, :) = PUER(:, :)
    ZUDR(:, :) = PUDR(:, :)
    ZDMF(:, :) = PDMF(:, :)
    ZDER(:, :) = PDER(:, :)
    ZDDR(:, :) = PDDR(:, :)
    ZTPR(:) = PTPR(:)
    ZDTEVR(:) = PDTEVR(:)
    ZOMG(:, :) = 0.
    PWSUB(:, :) = 0.
    ZPRMELT(:) = 0.
    PPRLFLX(:, :) = 0.
    ZPRLFLX(:, :) = 0.
    PPRSFLX(:, :) = 0.
    ZPRSFLX(:, :) = 0.
!
!
!*       2.1    Some preliminar computations for melting of precipitation
!               used later in section 9 and computation of precip fluxes
!               Precipitation fluxes are updated for melting and evaporation
!               ---------------------------------------------------------
!
!
    ZWORK1(:) = 0.
    ZMELDPTH(:) = 0.
    ZWORK6(:, :) = 0.
    do JK = JKMAX + 1, IKB + 1, -1
        ! Nota: PUPR is total precipitation flux, but the solid, liquid
        !       precipitation is stored in units kg/kg; therefore we compute here
        !       the solid fraction of the total precipitation flux.
        do JI = 1, IIE
            ZWORK2(JI) = PUPR(JI, JK) / (PURC(JI, JK) + PURI(JI, JK) + 1.E-8)
            ZPRMELT(JI) = ZPRMELT(JI) + PURI(JI, JK) * ZWORK2(JI)
            ZWORK1(JI) = ZWORK1(JI) + PURC(JI, JK) * ZWORK2(JI) - PDTEVRF(JI, JK)
            ZPRLFLX(JI, JK) = MAX(0., ZWORK1(JI))
            ZPRMELT(JI) = ZPRMELT(JI) + MIN(0., ZWORK1(JI))
            ZPRSFLX(JI, JK) = ZPRMELT(JI)
            if(KML(JI) >= JK .and. ZMELDPTH(JI) <= XMELDPTH) then
                ZPI(JI) = (PPRES(JI, JK) / XP00)**ZRDOCP
                ZWORK3(JI) = PTH(JI, JK) * ZPI(JI)            ! temperature estimate
                ZLM(JI) = XLSTT + (XCPV - XCI) * (ZWORK3(JI) - XTT) - &
                          (XLVTT + (XCPV - XCL) * (ZWORK3(JI) - XTT)) ! L_s - L_v
                ZCPH(JI) = XCPD + XCPV * PRW(JI, JK)
                ZMELDPTH(JI) = ZMELDPTH(JI) + PDPRES(JI, JK)
                ZWORK6(JI, JK) = ZLM(JI) * PTIMEC(JI) / PLMASS(JI, JK) * PDPRES(JI, JK)
                ZOMG(JI, JK) = 1. ! at this place only used as work variable
            endif
        enddo
!
    enddo
!
    ZWORK2(:) = 0.
    do JK = JKMAX, IKB + 1, -1
        ZWORK1(:) = ZPRMELT(:) * PDPRES(:, JK) / MAX(XMELDPTH, ZMELDPTH(:))
        ZWORK2(:) = ZWORK2(:) + ZWORK1(:) * ZOMG(:, JK)
        ZPRLFLX(:, JK) = ZPRLFLX(:, JK) + ZWORK2(:)
        ZPRSFLX(:, JK) = ZPRSFLX(:, JK) - ZWORK2(:)
    enddo
    WHERE(ZPRSFLX(:, :) < 1.) ZPRSFLX(:, :) = 0.
    ZPRMELTO(:) = ZPRMELT(:)
!
!
!*       3.     Compute limits on the closure adjustment factor so that the
!               inflow in convective drafts from a given layer can't be larger
!               than the mass contained in this layer initially.
!               ---------------------------------------------------------------
!
    ZADJMAX(:) = 1000.
    IWORK1(:) = MAX(ILCL(:), KLFS(:))
    JKP = MINVAL(KDPL(:))
    do JK = JKP, IKE
        do JI = 1, IIE
            if(JK > KDPL(JI) .and. JK <= IWORK1(JI)) then
                ZWORK1(JI) = PLMASS(JI, JK) / &
                             ((PUER(JI, JK) + PDER(JI, JK) + 1.E-5) * PTIMEC(JI))
                ZADJMAX(JI) = MIN(ZADJMAX(JI), ZWORK1(JI))
            endif
        enddo
    enddo
!
!
    GWORK1(:) = OTRIG1(:)  ! logical array to limit adjustment to not definitively
    ! adjusted columns
!
    do JK = IKB, IKE
        ZTHLC(:, JK) = PTHL(:, JK) ! initialize adjusted envir. values
        PRWC(:, JK) = PRW(:, JK)
        PRCC(:, JK) = PRC(:, JK)
        PRIC(:, JK) = PRI(:, JK)
        PTHC(:, JK) = PTH(:, JK)
    enddo
!
!
!
    do JITER = 1, 6  ! Enter adjustment loop to assure that all CAPE is
        ! removed within the advective time interval TIMEC
!
        ZTIMEC(:) = PTIMEC(:)
        GWORK4(:, :) = SPREAD(GWORK1(:), DIM=2, NCOPIES=IKS)
        WHERE(GWORK4(:, :)) PWSUB(:, :) = 0.
        ZOMG(:, :) = 0.
!
        do JK = IKB + 1, JKMAX
            JKP = MAX(IKB + 1, JK - 1)
            WHERE(GWORK1(:) .and. JK <= KCTL(:))
!
!
!*       4.     Determine vertical velocity at top and bottom of each layer
!               to satisfy mass continuity.
!               ---------------------------------------------------------------
                ! we compute here Domega/Dp = - g rho Dw/Dz = 1/Dt
!
                ZWORK1(:) = -(PUER(:, JKP) + PDER(:, JKP) - &
                              PUDR(:, JKP) - PDDR(:, JKP)) / PLMASS(:, JKP)
!
                PWSUB(:, JK) = PWSUB(:, JKP) - PDPRES(:, JK - 1) * ZWORK1(:)
                ! we use PDPRES(JK-1) and not JKP in order to have zero subsidence
                ! at the first layer
!
!
!*       5.     Compute fractional time step. For stability or
!               mass conservation reasons one must split full time step PTIMEC)
!               ---------------------------------------------------------------
!
                ZWORK1(:) = XSTABT * PDPRES(:, JKP) / (ABS(PWSUB(:, JK)) + 1.E-10)
                ! the factor XSTABT is used for stability reasons
                ZTIMEC(:) = MIN(ZTIMEC(:), ZWORK1(:))
!
                ! transform vertical velocity in mass flux units
                ZOMG(:, JK) = PWSUB(:, JK) * PDXDY(:) / XG
            ENDWHERE
        enddo
!
!
        WHERE(GWORK4(:, :))
            ZTHLC(:, :) = PTHL(:, :) ! reinitialize adjusted envir. values
            PRWC(:, :) = PRW(:, :)  ! when iteration criterium not attained
            PRCC(:, :) = PRC(:, :)
            PRIC(:, :) = PRI(:, :)
            PTHC(:, :) = PTH(:, :)
        ENDWHERE
!
!
!        6. Check for mass conservation, i.e. ZWORK1 > 1.E-2
!           If mass is not conserved, the convective tendencies
!           automatically become zero.
!           ----------------------------------------------------
!
        do JI = 1, IIE
            JK = KCTL(JI)
            ZWORK1(JI) = PUDR(JI, JK) * PDPRES(JI, JK) / (PLMASS(JI, JK) + .1) &
                         - PWSUB(JI, JK)
        enddo
        WHERE(GWORK1(:) .and. ABS(ZWORK1(:)) - .01 > 0.)
            GWORK1(:) = .false.
            PTIMEC(:) = 1.E-1
            ZTPR(:) = 0.
            ZWORK5(:) = 0.
        ENDWHERE
        do JK = IKB, IKE
            PWSUB(:, JK) = PWSUB(:, JK) * ZWORK5(:)
            ZPRLFLX(:, JK) = ZPRLFLX(:, JK) * ZWORK5(:)
            ZPRSFLX(:, JK) = ZPRSFLX(:, JK) * ZWORK5(:)
        enddo
        GWORK4(:, 1:IKB) = .false.
        GWORK4(:, IKE:IKS) = .false.
!
        ITSTEP(:) = INT(PTIMEC(:) / ZTIMEC(:)) + 1
        ZTIMEC(:) = PTIMEC(:) / REAL(ITSTEP(:)) ! adjust  fractional time step
        ! to be an integer multiple of PTIMEC
        ZTIMC(:, :) = SPREAD(ZTIMEC(:), DIM=2, NCOPIES=IKS)
        ICOUNT(:) = 0
!
!
!
        KFTSTEPS = MAXVAL(ITSTEP(:))
        do JSTEP = 1, KFTSTEPS ! Enter the fractional time step loop here
!
            ICOUNT(:) = ICOUNT(:) + 1
!
            GWORK3(:) = ITSTEP(:) >= ICOUNT(:) .and. GWORK1(:)
!
!
!*       7.     Assign enthalpy and r_w values at the top and bottom of each
!               layer based on the sign of w
!               ------------------------------------------------------------
!
            ZTHMFIN(:, :) = 0.
            ZRWMFIN(:, :) = 0.
            ZRCMFIN(:, :) = 0.
            ZRIMFIN(:, :) = 0.
            ZTHMFOUT(:, :) = 0.
            ZRWMFOUT(:, :) = 0.
            ZRCMFOUT(:, :) = 0.
            ZRIMFOUT(:, :) = 0.
!
            do JK = IKB + 1, JKMAX
                do JI = 1, IIE
                    GWORK4(JI, JK) = GWORK3(JI) .and. JK <= KCTL(JI)
                enddo
                JKP = MAX(IKB + 1, JK - 1)
                do JI = 1, IIE
                if(GWORK3(JI)) then
!
                    ZWORK1(JI) = SIGN(1., ZOMG(JI, JK))
                    ZWORK2(JI) = 0.5 * (1.+ZWORK1(JI))
                    ZWORK1(JI) = 0.5 * (1.-ZWORK1(JI))
                    ZTHMFIN(JI, JK) = -ZOMG(JI, JK) * ZTHLC(JI, JKP) * ZWORK1(JI)
                    ZTHMFOUT(JI, JK) = ZOMG(JI, JK) * ZTHLC(JI, JK) * ZWORK2(JI)
                    ZRWMFIN(JI, JK) = -ZOMG(JI, JK) * PRWC(JI, JKP) * ZWORK1(JI)
                    ZRWMFOUT(JI, JK) = ZOMG(JI, JK) * PRWC(JI, JK) * ZWORK2(JI)
                    ZRCMFIN(JI, JK) = -ZOMG(JI, JK) * PRCC(JI, JKP) * ZWORK1(JI)
                    ZRCMFOUT(JI, JK) = ZOMG(JI, JK) * PRCC(JI, JK) * ZWORK2(JI)
                    ZRIMFIN(JI, JK) = -ZOMG(JI, JK) * PRIC(JI, JKP) * ZWORK1(JI)
                    ZRIMFOUT(JI, JK) = ZOMG(JI, JK) * PRIC(JI, JK) * ZWORK2(JI)
                endif
                enddo
                do JI = 1, IIE
                if(GWORK3(JI)) then
                    ZTHMFIN(JI, JKP) = ZTHMFIN(JI, JKP) + ZTHMFOUT(JI, JK) * ZWORK2(JI)
                    ZTHMFOUT(JI, JKP) = ZTHMFOUT(JI, JKP) + ZTHMFIN(JI, JK) * ZWORK1(JI)
                    ZRWMFIN(JI, JKP) = ZRWMFIN(JI, JKP) + ZRWMFOUT(JI, JK) * ZWORK2(JI)
                    ZRWMFOUT(JI, JKP) = ZRWMFOUT(JI, JKP) + ZRWMFIN(JI, JK) * ZWORK1(JI)
                    ZRCMFIN(JI, JKP) = ZRCMFIN(JI, JKP) + ZRCMFOUT(JI, JK) * ZWORK2(JI)
                    ZRCMFOUT(JI, JKP) = ZRCMFOUT(JI, JKP) + ZRCMFIN(JI, JK) * ZWORK1(JI)
                    ZRIMFIN(JI, JKP) = ZRIMFIN(JI, JKP) + ZRIMFOUT(JI, JK) * ZWORK2(JI)
                    ZRIMFOUT(JI, JKP) = ZRIMFOUT(JI, JKP) + ZRIMFIN(JI, JK) * ZWORK1(JI)
!
                endif
                enddo
            enddo
!
            WHERE(GWORK4(:, :))
!
!******************************************************************************
!
!*       8.     Update the environmental values of enthalpy and r_w at each level
!               NOTA: These are the MAIN EQUATIONS of the scheme
!               -----------------------------------------------------------------
!
!
                ZTHLC(:, :) = ZTHLC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                              ZTHMFIN(:, :) + PUDR(:, :) * PUTHL(:, :) + &
                              PDDR(:, :) * PDTHL(:, :) - ZTHMFOUT(:, :) - &
                              (PUER(:, :) + PDER(:, :)) * PTHL(:, :))
                PRWC(:, :) = PRWC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRWMFIN(:, :) + PUDR(:, :) * PURW(:, :) + &
                             PDDR(:, :) * PDRW(:, :) - ZRWMFOUT(:, :) - &
                             (PUER(:, :) + PDER(:, :)) * PRW(:, :))
                PRCC(:, :) = PRCC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRCMFIN(:, :) + PUDR(:, :) * PURC(:, :) - ZRCMFOUT(:, :) - &
                             (PUER(:, :) + PDER(:, :)) * PRC(:, :))
                PRIC(:, :) = PRIC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRIMFIN(:, :) + PUDR(:, :) * PURI(:, :) - ZRIMFOUT(:, :) - &
                             (PUER(:, :) + PDER(:, :)) * PRI(:, :))
!
!
!******************************************************************************
!
            ENDWHERE
!
        enddo ! Exit the fractional time step loop
!
!
!*           9.    Allow frozen precipitation to melt over a 200 mb deep layer
!                  -----------------------------------------------------------
!
        do JK = JKMAX, IKB + 1, -1
            ZTHLC(:, JK) = ZTHLC(:, JK) - &
                           ZPRMELT(:) * ZWORK6(:, JK) / MAX(XMELDPTH, ZMELDPTH(:))
        enddo
!
!
!*          10.    Compute final linearized value of theta envir.
!                  ----------------------------------------------
!
        do JK = IKB + 1, JKMAX
            do JI = 1, IIE
            if(GWORK1(JI) .and. JK <= KCTL(JI)) then
                ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
                ZCPH(JI) = XCPD + PRWC(JI, JK) * XCPV
                ZWORK2(JI) = PTH(JI, JK) / ZPI(JI)  ! first temperature estimate
                ZLV(JI) = XLVTT + (XCPV - XCL) * (ZWORK2(JI) - XTT)
                ZLS(JI) = XLVTT + (XCPV - XCI) * (ZWORK2(JI) - XTT)
                ! final linearized temperature
                ZWORK2(JI) = (ZTHLC(JI, JK) + ZLV(JI) * PRCC(JI, JK) + ZLS(JI) * PRIC(JI, JK) &
                              - (1.+PRWC(JI, JK)) * XG * PZ(JI, JK)) / ZCPH(JI)
                ZWORK2(JI) = MAX(180., MIN(340., ZWORK2(JI)))
                PTHC(JI, JK) = ZWORK2(JI) * ZPI(JI) ! final adjusted envir. theta
            endif
            enddo
        enddo
!
!
!*         11.     Compute new cloud ( properties at new LCL )
!                     NOTA: The computations are very close to
!                           that in routine TRIGGER_FUNCT
!                  ---------------------------------------------
!
        call CONVECT_CLOSURE_THRVLCL(KLON, KLEV, &
                                     PPRES, PTHC, PRWC, PZ, GWORK1, &
                                     ZTHLCL, ZRVLCL, ZZLCL, ZTLCL, ZTELCL, &
                                     ILCL, KDPL, KPBL)
!
!
        ZTLCL(:) = MAX(230., MIN(335., ZTLCL(:)))  ! set some overflow bounds
        ZTELCL(:) = MAX(230., MIN(335., ZTELCL(:)))
        ZTHLCL(:) = MAX(230., MIN(345., ZTHLCL(:)))
        ZRVLCL(:) = MAX(0., MIN(1., ZRVLCL(:)))
!
!
!*         12.    Compute adjusted CAPE
!                 ---------------------
!
        ZCAPE(:) = 0.
        ZPI(:) = ZTHLCL(:) / ZTLCL(:)
        ZPI(:) = MAX(0.95, MIN(1.5, ZPI(:)))
        ZWORK1(:) = XP00 / ZPI(:)**ZCPORD ! pressure at LCL
!
        call CONVECT_SATMIXRATIO(KLON, ZWORK1, ZTELCL, ZWORK3, ZLV, ZLS, ZCPH)
        ZWORK3(:) = MIN(.1, MAX(0., ZWORK3(:)))
!
        ! compute theta_e updraft undilute
        ZTHEUL(:) = ZTLCL(:) * ZPI(:)**(1.-0.28 * ZRVLCL(:)) &
                    * EXP((3374.6525 / ZTLCL(:) - 2.5403) &
                          * ZRVLCL(:) * (1.+0.81 * ZRVLCL(:)))
!
        ! compute theta_e saturated environment at LCL
        ZTHES1(:) = ZTELCL(:) * ZPI(:)**(1.-0.28 * ZWORK3(:)) &
                    * EXP((3374.6525 / ZTELCL(:) - 2.5403) &
                          * ZWORK3(:) * (1.+0.81 * ZWORK3(:)))
!
        do JK = MINVAL(ILCL(:)), JKMAX
            JKP = JK - 1
            do JI = 1, IIE
                ZWORK4(JI) = 1.
                if(JK == ILCL(JI)) ZWORK4(JI) = 0.
!
                ! compute theta_e saturated environment and adjusted values
                ! of theta
!
                GWORK3(JI) = JK >= ILCL(JI) .and. JK <= KCTL(JI) .and. GWORK1(JI)
!
                ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
                ZWORK2(JI) = PTHC(JI, JK) / ZPI(JI)
            enddo
!
            call CONVECT_SATMIXRATIO(KLON, PPRES(:, JK), ZWORK2, ZWORK3, ZLV, ZLS, ZCPH)
!
!
            do JI = 1, IIE
                if(GWORK3(JI)) then
                    ZTHES2(JI) = ZWORK2(JI) * ZPI(JI)**(1.-0.28 * ZWORK3(JI)) &
                                 * EXP((3374.6525 / ZWORK2(JI) - 2.5403) &
                                       * ZWORK3(JI) * (1.+0.81 * ZWORK3(JI)))
!
                    ZWORK3(JI) = PZ(JI, JK) - PZ(JI, JKP) * ZWORK4(JI) - &
                                 (1.-ZWORK4(JI)) * ZZLCL(JI)    ! level thickness
                    ZWORK1(JI) = (2.*ZTHEUL(JI)) / (ZTHES1(JI) + ZTHES2(JI)) - 1.
                    ZCAPE(JI) = ZCAPE(JI) + XG * ZWORK3(JI) * MAX(0., ZWORK1(JI))
                    ZTHES1(JI) = ZTHES2(JI)
                endif
            enddo
        enddo
!
!
!*         13.     Determine mass adjustment factor knowing how much
!                  CAPE has been removed.
!                  -------------------------------------------------
!
        WHERE(GWORK1(:))
            ZWORK1(:) = MAX(PCAPE(:) - ZCAPE(:), 0.1 * PCAPE(:))
            ZWORK2(:) = ZCAPE(:) / (PCAPE(:) + 1.E-8)
!
            GWORK1(:) = ZWORK2(:) > 0.1 .OR. ZCAPE(:) == 0. ! mask for adjustment
        ENDWHERE
!
        WHERE(ZCAPE(:) == 0. .and. GWORK1(:)) ZADJ(:) = ZADJ(:) * 0.5
        WHERE(ZCAPE(:) /= 0. .and. GWORK1(:)) &
            ZADJ(:) = ZADJ(:) * XSTABC * PCAPE(:) / (ZWORK1(:) + 1.E-8)
        ZADJ(:) = MIN(ZADJ(:), ZADJMAX(:))
!
!
!*         13.     Adjust mass flux by the factor ZADJ to converge to
!                  specified degree of stabilization
!                 ----------------------------------------------------
!
        call CONVECT_CLOSURE_ADJUST(KLON, KLEV, ZADJ, &
                                    PUMF, ZUMF, PUER, ZUER, PUDR, ZUDR, &
                                    PDMF, ZDMF, PDER, ZDER, PDDR, ZDDR, &
                                    ZPRMELT, ZPRMELTO, PDTEVR, ZDTEVR, &
                                    PTPR, ZTPR, &
                                    PPRLFLX, ZPRLFLX, PPRSFLX, ZPRSFLX)
!
!
        if(COUNT(GWORK1(:)) == 0) EXIT ! exit big adjustment iteration loop
        ! when all columns have reached
        ! desired degree of stabilization.
!
    enddo ! end of big adjustment iteration loop
!
!
    ! skip adj. total water array  to water vapor
    do JK = IKB, IKE
        PRWC(:, JK) = MAX(0., PRWC(:, JK) - PRCC(:, JK) - PRIC(:, JK))
    enddo
!
    ! compute surface solid (ice) precipitation
    PSPR(:) = ZPRMELT(:) * (1.-ZMELDPTH(:) / XMELDPTH)
    PSPR(:) = MAX(0., PSPR(:))
!
!
ENDsubroutine CONVECT_CLOSURE

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!     ######################
MODULE MODI_DEEP_CONVECTION
!     ######################
!
    INTERFACE
!
        subroutine DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                   PDTCONV, KICE, OREFRESH, ODOWN, OSETTADJ, &
                                   PPABST, PZZ, PDXDY, PTIMEC, &
                                   PTT, PRVT, PRCT, PRIT, PUT, PVT, PWT, &
                                   KCOUNT, PTTEN, PRVTEN, PRCTEN, PRITEN, &
                                   PPRLTEN, PPRSTEN, &
                                   KCLTOP, KCLBAS, PPRLFLX, PPRSFLX, &
                                   PUMF, PDMF, PCAPE, &
                                   OCH1CONV, KCH1, PCH1, PCH1TEN)

            INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
            INTEGER, INTENT(IN) :: KIDIA    ! value of the first point in x
            INTEGER, INTENT(IN) :: KFDIA    ! value of the last point in x
            INTEGER, INTENT(IN) :: KBDIA    ! vertical  computations start at
!                                                  ! KBDIA that is at least 1
            INTEGER, INTENT(IN) :: KTDIA    ! vertical computations can be
            ! limited to KLEV + 1 - KTDIA
            ! default=1
            REAL, INTENT(IN) :: PDTCONV  ! Interval of time between two
            ! calls of the deep convection
            ! scheme
            INTEGER, INTENT(IN) :: KICE     ! flag for ice ( 1 = yes,
            !                0 = no ice )
            LOGICAL, INTENT(IN) :: OREFRESH ! refresh or not tendencies
            ! at every call
            LOGICAL, INTENT(IN) :: ODOWN    ! take or not convective
            ! downdrafts into account
            LOGICAL, INTENT(IN) :: OSETTADJ ! logical to set convective
            ! adjustment time by user
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTT      ! grid scale temperature at t
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRVT     ! grid scale water vapor "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRCT     ! grid scale r_c  "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRIT     ! grid scale r_i "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PUT      ! grid scale horiz. wind u "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PVT      ! grid scale horiz. wind v "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PWT      ! grid scale vertical
            ! velocity (m/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPABST   ! grid scale pressure at t
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZZ      ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY    ! horizontal grid area (m-a2)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTIMEC   ! value of convective adjustment
            ! time if OSETTADJ=.true.
!
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCOUNT ! convective counter (recompute
            ! tendency or keep it)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PTTEN  ! convective temperature
            ! tendency (K/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRVTEN ! convective r_v tendency (1/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRCTEN ! convective r_c tendency (1/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRITEN ! convective r_i tendency (1/s)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PPRLTEN! liquid surf. precipitation
            ! tendency (m/s)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PPRSTEN! solid surf. precipitation
            ! tendency (m/s)
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLTOP ! cloud top level
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLBAS ! cloud base level
            ! they are given a value of
            ! 0 if no convection
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PPRLFLX! liquid precip flux (m/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PPRSFLX! solid  precip flux (m/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF   ! updraft mass flux (kg/s m2)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDMF   ! downdraft mass flux (kg/s m2)
            REAL, DIMENSION(KLON), INTENT(INOUT):: PCAPE  ! maximum CAPE (J/kg)
!
            LOGICAL, INTENT(IN) :: OCH1CONV ! include tracer transport
            INTEGER, INTENT(IN) :: KCH1     ! number of species
            REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(IN) :: PCH1! grid scale chemical species
            REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(INOUT):: PCH1TEN! species conv. tendency (1/s)
            LOGICAL                       :: OUSECHEM      ! flag for chemistry
            LOGICAL                       :: OCH_CONV_SCAV !  & scavenging
            LOGICAL                       :: OCH_CONV_LINOX ! & LiNOx
            LOGICAL                       :: ODUST         ! flag for dust
            LOGICAL                       :: OSALT         ! flag for sea salt
            REAL, DIMENSION(KLON, KLEV)    :: PRHODREF      ! grid scale density
            REAL, DIMENSION(KLON)         :: PIC_RATE ! IC lightning frequency
            REAL, DIMENSION(KLON)         :: PCG_RATE ! CG lightning frequency

!
        ENDsubroutine DEEP_CONVECTION
!
    ENDINTERFACE
!
ENDMODULE MODI_DEEP_CONVECTION
!
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/09/21 10:55:01
!-----------------------------------------------------------------
!   ############################################################################
subroutine DEEP_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                           PDTCONV, KICE, OREFRESH, ODOWN, OSETTADJ, &
                           PPABST, PZZ, PDXDY, PTIMEC, &
                           PTT, PRVT, PRCT, PRIT, PUT, PVT, PWT, &
                           KCOUNT, PTTEN, PRVTEN, PRCTEN, PRITEN, &
                           PPRLTEN, PPRSTEN, &
                           KCLTOP, KCLBAS, PPRLFLX, PPRSFLX, &
                           PUMF, PDMF, PCAPE, &
                           OCH1CONV, KCH1, PCH1, PCH1TEN)
!   ############################################################################
!
!!**** Monitor routine to compute all convective tendencies by calls
!!     of several subroutines.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the convective
!!      tendencies. The routine first prepares all necessary grid-scale
!!      variables. The final convective tendencies are then computed by
!!      calls of different subroutines.
!!
!!
!!**  METHOD
!!    ------
!!      We start by selecting convective columns in the model domain through
!!      the call of routine TRIGGER_FUNCT. Then, we allocate memory for the
!!      convection updraft and downdraft variables and gather the grid scale
!!      variables in convective arrays.
!!      The updraft and downdraft computations are done level by level starting
!!      at the  bottom and top of the domain, respectively.
!!      All computations are done on MNH thermodynamic levels. The depth
!!      of the current model layer k is defined by DP(k)=P(k-1)-P(k)
!!
!!
!!
!!    EXTERNAL
!!    --------
!!    CONVECT_TRIGGER_FUNCT
!!    CONVECT_SATMIXRATIO
!!    CONVECT_UPDRAFT
!!        CONVECT_CONDENS
!!        CONVECT_MIXING_FUNCT
!!    CONVECT_TSTEP_PREF
!!    CONVECT_DOWNDRAFT
!!    CONVECT_PRECIP_ADJUST
!!    CONVECT_CLOSURE
!!        CONVECT_CLOSURE_THRVLCL
!!        CONVECT_CLOSURE_ADJUST
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                   ! gravity constant
!!          XPI                  ! number Pi
!!          XP00                 ! reference pressure
!!          XRD, XRV             ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV           ! specific heat for dry air and water vapor
!!          XRHOLW               ! density of liquid water
!!          XALPW, XBETAW, XGAMW ! constants for water saturation pressure
!!          XTT                  ! triple point temperature
!!          XLVTT, XLSTT         ! vaporization, sublimation heat constant
!!          XCL, XCI             ! specific heat for liquid water and ice
!!
!!      Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT       ! extra levels on the vertical boundaries
!!
!!      Module MODD_CONVPAR
!!          XA25                 ! reference grid area
!!          XCRAD                ! cloud radius
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Bechtold, 1997 : Meso-NH scientific  documentation (31 pp)
!!      Bechtold et al., 2001, Quart. J. Roy. Met. Soc.
!!      Kain and Fritsch, 1990, J. Atmos. Sci., Vol. 47, 2784-2801.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol. 24, 165-170.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Peter Bechtold 04/10/97 replace theta_il by enthalpy
!!         "        10/12/98 changes for ARPEGE
!!         "        12/12/00 add conservation correction
!!      C. Mari     13/02/01 add scavenging of chemical species in updraft
!!     P. Jabouille 02/07/01 case of lagragian variables
!!     P. Tulet     02/03/05 update for dust
!!     C.Lac        27/09/10 modification loop index for reproducibility
!!    Juan 24/09/2012: for BUG Pgi rewrite PACK function on mode_pack_pgi
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAREXT
    USE MODD_CONVPAR
!USE MODD_NSV,       ONLY : NSV_LGBEG,NSV_LGEND, &
!                           NSV_CHEMBEG,NSV_CHEMEND, &
!                           NSV_LNOXBEG
!USE MODD_CH_M9_n,   ONLY : CNAMES
!
!USE MODI_CH_CONVECT_LINOX
    USE MODI_CONVECT_TRIGGER_FUNCT
    USE MODI_CONVECT_UPDRAFT
    USE MODI_CONVECT_TSTEP_PREF
    USE MODI_CONVECT_DOWNDRAFT
    USE MODI_CONVECT_PRECIP_ADJUST
    USE MODI_CONVECT_CLOSURE
!USE MODI_CH_CONVECT_SCAVENGING
!USE MODI_CONVECT_CHEM_TRANSPORT
!
!SeBi #ifdef MNH_PGI
!SeBi USE MODE_PACK_PGI
!SeBi #endif
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
    INTEGER, INTENT(IN) :: KIDIA    ! value of the first point in x
    INTEGER, INTENT(IN) :: KFDIA    ! value of the last point in x
    INTEGER, INTENT(IN) :: KBDIA    ! vertical  computations start at
!                                                  ! KBDIA that is at least 1
    INTEGER, INTENT(IN) :: KTDIA    ! vertical computations can be
    ! limited to KLEV + 1 - KTDIA
    ! default=1
    REAL, INTENT(IN) :: PDTCONV  ! Interval of time between two
    ! calls of the deep convection
    ! scheme
    INTEGER, INTENT(IN) :: KICE     ! flag for ice ( 1 = yes,
    !                0 = no ice )
    LOGICAL, INTENT(IN) :: OREFRESH ! refresh or not tendencies
    ! at every call
    LOGICAL, INTENT(IN) :: ODOWN    ! take or not convective
    ! downdrafts into account
    LOGICAL, INTENT(IN) :: OSETTADJ ! logical to set convective
    ! adjustment time by user
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTT      ! grid scale temperature at t
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRVT     ! grid scale water vapor "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRCT     ! grid scale r_c  "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRIT     ! grid scale r_i "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PUT      ! grid scale horiz. wind u "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PVT      ! grid scale horiz. wind v "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PWT      ! grid scale vertical
    ! velocity (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPABST   ! grid scale pressure at t
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZZ      ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY    ! horizontal grid area (m-a2)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTIMEC   ! value of convective adjustment
    ! time if OSETTADJ=.true.
!
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCOUNT ! convective counter (recompute
    ! tendency or keep it)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PTTEN  ! convective temperature
    ! tendency (K/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRVTEN ! convective r_v tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRCTEN ! convective r_c tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRITEN ! convective r_i tendency (1/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PPRLTEN! liquid surf. precipitation
    ! tendency (m/s)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PPRSTEN! solid surf. precipitation
    ! tendency (m/s)
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLTOP ! cloud top level
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLBAS ! cloud base level
    ! they are given a value of
    ! 0 if no convection
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PPRLFLX! liquid precip flux (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PPRSFLX! solid  precip flux (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF   ! updraft mass flux (kg/s m2)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PDMF   ! downdraft mass flux (kg/s m2)
    REAL, DIMENSION(KLON), INTENT(INOUT):: PCAPE  ! maximum CAPE (J/kg)
!
    LOGICAL, INTENT(IN) :: OCH1CONV ! include tracer transport
    INTEGER, INTENT(IN) :: KCH1     ! number of species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(IN) :: PCH1! grid scale chemical species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(INOUT):: PCH1TEN! species conv. tendency (1/s)
    LOGICAL                    :: OUSECHEM          ! flag for chemistry
    LOGICAL                    :: OCH_CONV_SCAV !  & scavenging
    LOGICAL                    :: OCH_CONV_LINOX ! & LiNOx
    LOGICAL                    :: ODUST          ! flag for dust
    LOGICAL                    :: OSALT          ! flag for sea salt
    REAL, DIMENSION(KLON, KLEV) :: PRHODREF           ! grid scale density
    REAL, DIMENSION(KLON)      :: PIC_RATE ! IC lightning frequency
    REAL, DIMENSION(KLON)      :: PCG_RATE ! CG lightning frequency
!
!
!*       0.2   Declarations of local fixed memory variables :
!
    INTEGER  :: ITEST, ICONV, ICONV1    ! number of convective columns
    INTEGER  :: IIB, IIE                ! horizontal loop bounds
    INTEGER  :: IKB, IKE                ! vertical loop bounds
    INTEGER  :: IKS                     ! vertical dimension
    INTEGER  :: JI, JL, JJ              ! horizontal loop index
    INTEGER  :: JN                      ! number of tracers
    INTEGER  :: JK, JKP, JKM            ! vertical loop index
    INTEGER  :: IFTSTEPS                ! only used for chemical tracers
    real  :: ZEPS, ZEPSA             ! R_d / R_v, R_v / R_d
    real  :: ZRDOCP                  ! R_d/C_p
!
    LOGICAL, DIMENSION(KLON, KLEV)     :: GTRIG3 ! 3D logical mask for convection
    LOGICAL, DIMENSION(KLON)           :: GTRIG  ! 2D logical mask for trigger test
    REAL, DIMENSION(KLON, KLEV)         :: ZTHT, ZSTHV, ZSTHES  ! grid scale theta,
    ! theta_v, theta_es
    REAL, DIMENSION(KLON)              :: ZTIME  ! convective time period
    REAL, DIMENSION(KLON)              :: ZWORK2, ZWORK2B ! work array
    real                :: ZW1    ! work variable
!
!
!*       0.2   Declarations of local allocatable  variables :
!
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IDPL    ! index for parcel departure level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IPBL    ! index for source layer top
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ILCL    ! index for lifting condensation level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IETL    ! index for zero buoyancy level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ICTL    ! index for cloud top level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ILFS    ! index for level of free sink
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IDBL    ! index for downdraft base level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IML     ! melting level
!
    INTEGER, DIMENSION(:), ALLOCATABLE :: ISDPL   ! index for parcel departure level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ISPBL   ! index for source layer top
    INTEGER, DIMENSION(:), ALLOCATABLE :: ISLCL   ! index for lifting condensation level
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTHLCL ! updraft theta at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTLCL  ! updraft temp. at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSRVLCL ! updraft rv at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSWLCL  ! updraft w at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSZLCL  ! LCL height
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTHVELCL! envir. theta_v at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSDXDY  ! grid area (m^2)
!
! grid scale variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZZ      ! height of model layer (m)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZPRES   ! grid scale pressure
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDPRES  ! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZU      ! grid scale horiz. u component on theta grid
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZV      ! grid scale horiz. v component on theta grid
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZW      ! grid scale vertical velocity on theta grid
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTT     ! temperature
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTH     ! grid scale theta
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHV    ! grid scale theta_v
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHL    ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHES, ZTHEST ! grid scale saturated theta_e
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRW     ! grid scale total water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRV     ! grid scale water vapor (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRC     ! grid scale cloud water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRI     ! grid scale cloud ice (kg/kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZDXDY   ! grid area (m^2)
!
! updraft variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUMF    ! updraft mass flux (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUER    ! updraft entrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUDR    ! updraft detrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUPR    ! updraft precipitation in
    ! flux units (kg water / s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUTHL   ! updraft enthalpy (J/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUTHV   ! updraft theta_v (K)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUTT    ! updraft temperature (K)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURW    ! updraft total water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURC    ! updraft cloud water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURI    ! updraft cloud ice   (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURR    ! liquid precipit. (kg/kg)
    ! produced in  model layer
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURS    ! solid precipit. (kg/kg)
    ! produced in  model layer
    REAL, DIMENSION(:), ALLOCATABLE  :: ZUTPR   ! total updraft precipitation (kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZMFLCL  ! cloud base unit mass flux(kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZCAPE   ! available potent. energy
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTHLCL  ! updraft theta at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTLCL   ! updraft temp. at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZRVLCL  ! updraft rv at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZWLCL   ! updraft w at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZZLCL   ! LCL height
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTHVELCL! envir. theta_v at LCL
!
! downdraft variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDMF    ! downdraft mass flux (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDER    ! downdraft entrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDDR    ! downdraft detrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDTHL   ! downdraft enthalpy (J/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDRW    ! downdraft total water (kg/kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZMIXF   ! mixed fraction at LFS
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTPR    ! total surf precipitation (kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZSPR    ! solid surf precipitation (kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZDTEVR  ! donwndraft evapor. (kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZPREF   ! precipitation efficiency
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDTEVRF ! donwndraft evapor. (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZPRLFLX ! liquid precip flux
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZPRSFLX ! solid precip flux
!
! closure variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZLMASS  ! mass of model layer (kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTIMEA  ! advective time period
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTIMEC, ZTIMED! time during which convection is
    ! active at grid point (as ZTIME)
!
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHC    ! conv. adj. grid scale theta
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRVC    ! conv. adj. grid scale r_w
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRCC    ! conv. adj. grid scale r_c
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRIC    ! conv. adj. grid scale r_i
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZWSUB   ! envir. compensating subsidence (Pa/s)
!
    LOGICAL, DIMENSION(:), ALLOCATABLE  :: GTRIG1  ! logical mask for convection
    LOGICAL, DIMENSION(:), ALLOCATABLE  :: GWORK   ! logical work array
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IINDEX, IJINDEX, IJSINDEX, IJPINDEX!hor.index
    REAL, DIMENSION(:), ALLOCATABLE  :: ZCPH    ! specific heat C_ph
    REAL, DIMENSION(:), ALLOCATABLE  :: ZLV, ZLS! latent heat of vaporis., sublim.
    real                :: ZES     ! saturation vapor mixng ratio
!
! Chemical Tracers:
    REAL, DIMENSION(:, :, :), ALLOCATABLE:: ZCH1    ! grid scale chemical specy (kg/kg)
    REAL, DIMENSION(:, :, :), ALLOCATABLE:: ZCH1C   ! conv. adjust. chemical specy 1
    REAL, DIMENSION(:, :), ALLOCATABLE:: ZWORK3  ! work array
    LOGICAL, DIMENSION(:, :, :), ALLOCATABLE::GTRIG4 ! logical mask
    integer                :: JN_NO   ! index of NO compound in PCH1
    REAL, DIMENSION(:, :), ALLOCATABLE   :: ZWORK4, ZWORK4C
    ! LiNOx conc. and tendency
    REAL, DIMENSION(:, :), ALLOCATABLE   :: ZZZ, ZRHODREF
    REAL, DIMENSION(:), ALLOCATABLE     :: ZIC_RATE, ZCG_RATE
!
!-------------------------------------------------------------------------------
!
!* SeBi     set some logical to false and real to nul because unused here
!           -------------------------------------------------------------
    OUSECHEM = .false.
    OCH_CONV_SCAV = .false.
    OCH_CONV_LINOX = .false.
    ODUST = .false.
    OSALT = .false.
    PRHODREF = 0.0
    PIC_RATE = 0.0
    PCG_RATE = 0.0

!
!*       0.3    Compute loop bounds
!               -------------------
!
    IIB = KIDIA
    IIE = KFDIA
    JCVEXB = MAX(0, KBDIA - 1)
    IKB = 1 + JCVEXB
    IKS = KLEV
    JCVEXT = MAX(0, KTDIA - 1)
    IKE = IKS - JCVEXT
!
!
!*       0.5    Update convective counter ( where KCOUNT > 0
!               convection is still active ).
!               ---------------------------------------------
!
    KCOUNT(IIB:IIE) = KCOUNT(IIB:IIE) - 1
!
    if(OREFRESH) then
        KCOUNT(:) = 1
        KCOUNT(IIB:IIE) = 0 ! refresh or not at every call
    endif
!
    GTRIG(:) = KCOUNT(:) <= 0
    ITEST = COUNT(GTRIG(:))
    if(ITEST == 0) then   ! if convection is already active at every grid point
        RETURN
    endif
    ! exit DEEP_CONVECTION
!
!
!*       0.7    Reset convective tendencies to zero if convective
!               counter becomes negative
!               -------------------------------------------------
!
    do JJ = 1, KLEV; do JI = 1, KLON
            GTRIG3(JI, JJ) = GTRIG(JI)
        ENDdo; ENDdo
    WHERE(GTRIG3(:, :))
        PTTEN(:, :) = 0.
        PRVTEN(:, :) = 0.
        PRCTEN(:, :) = 0.
        PRITEN(:, :) = 0.
        PPRLFLX(:, :) = 0.
        PPRSFLX(:, :) = 0.
! PUTEN(:,:)  = 0.
! PVTEN(:,:)  = 0.
        PUMF(:, :) = 0.
        PDMF(:, :) = 0.
    ENDWHERE
    WHERE(GTRIG(:))
        PPRLTEN(:) = 0.
        PPRSTEN(:) = 0.
        KCLTOP(:) = 0
        KCLBAS(:) = 0
        PCAPE(:) = 0.
    ENDWHERE
    ALLOCATE(GTRIG4(KLON, KLEV, KCH1))
    do JK = 1, KCH1; do JJ = 1, KLEV; do JI = 1, KLON
!GTRIG4(:,:,:) = SPREAD( GTRIG3(:,:), DIM=3, NCOPIES=KCH1 )
            GTRIG4(JI, JJ, JK) = GTRIG3(JI, JJ)
        ENDdo; ENDdo; enddo
    WHERE(GTRIG4(:, :, :)) PCH1TEN(:, :, :) = 0.
    DEALLOCATE(GTRIG4)
!
!-------------------------------------------------------------------------------
!
!*       1.     Initialize  local variables
!               ----------------------------
!
    ZEPS = XRD / XRV
    ZEPSA = XRV / XRD
    ZRDOCP = XRD / XCPD
!
!
!*       1.1    Set up grid scale theta, theta_v, theta_es
!               ------------------------------------------
!
    ZTHT(:, :) = 300.
    ZSTHV(:, :) = 300.
    ZSTHES(:, :) = 400.
    do JK = IKB, IKE
    do JI = IIB, IIE
        if(PPABST(JI, JK) > 40.E2) then
            ZTHT(JI, JK) = PTT(JI, JK) * (XP00 / PPABST(JI, JK))**ZRDOCP
            ZSTHV(JI, JK) = ZTHT(JI, JK) * (1.+ZEPSA * PRVT(JI, JK)) / &
                            (1.+PRVT(JI, JK) + PRCT(JI, JK) + PRIT(JI, JK))
!
            ! use conservative Bolton (1980) formula for theta_e
            ! it is used to compute CAPE for undilute parcel ascent
            ! For economical reasons we do not use routine CONVECT_SATMIXRATIO here
!
            ZES = EXP(XALPW - XBETAW / PTT(JI, JK) - XGAMW * LOG(PTT(JI, JK)))
            ZES = ZEPS * ZES / (PPABST(JI, JK) - ZES)
            ZSTHES(JI, JK) = PTT(JI, JK) * (ZTHT(JI, JK) / PTT(JI, JK))** &
                             (1.-0.28 * ZES) * EXP((3374.6525 / PTT(JI, JK) - 2.5403) &
                                                   * ZES * (1.+0.81 * ZES))
        endif
    enddo
    enddo
!
!-------------------------------------------------------------------------------
!
!*       2.     Test for convective columns and determine properties at the LCL
!               --------------------------------------------------------------
!
!*       2.1    Allocate arrays depending on number of model columns that need
!               to be tested for convection (i.e. where no convection is present
!               at the moment.
!               --------------------------------------------------------------
!
    ALLOCATE(ZPRES(ITEST, IKS))
    ALLOCATE(ZZ(ITEST, IKS))
    ALLOCATE(ZW(ITEST, IKS))
    ALLOCATE(ZTH(ITEST, IKS))
    ALLOCATE(ZTHV(ITEST, IKS))
    ALLOCATE(ZTHEST(ITEST, IKS))
    ALLOCATE(ZRV(ITEST, IKS))
    ALLOCATE(ZSTHLCL(ITEST))
    ALLOCATE(ZSTLCL(ITEST))
    ALLOCATE(ZSRVLCL(ITEST))
    ALLOCATE(ZSWLCL(ITEST))
    ALLOCATE(ZSZLCL(ITEST))
    ALLOCATE(ZSTHVELCL(ITEST))
    ALLOCATE(ISDPL(ITEST))
    ALLOCATE(ISPBL(ITEST))
    ALLOCATE(ISLCL(ITEST))
    ALLOCATE(ZSDXDY(ITEST))
    ALLOCATE(GTRIG1(ITEST))
    ALLOCATE(ZCAPE(ITEST))
    ALLOCATE(IINDEX(KLON))
    ALLOCATE(IJSINDEX(ITEST))
    do JI = 1, KLON
        IINDEX(JI) = JI
    enddo
    IJSINDEX(:) = PACK(IINDEX(:), MASK=GTRIG(:))
!
    ZPRES = 0.
    ZZ = 0.
    ZTH = 0.
    ZTHV = 0.
    ZTHEST = 0.
    ZRV = 0.
    ZW = 0.
!
    do JK = IKB, IKE
    do JI = 1, ITEST
        JL = IJSINDEX(JI)
        ZPRES(JI, JK) = PPABST(JL, JK)
        ZZ(JI, JK) = PZZ(JL, JK)
        ZTH(JI, JK) = ZTHT(JL, JK)
        ZTHV(JI, JK) = ZSTHV(JL, JK)
        ZTHEST(JI, JK) = ZSTHES(JL, JK)
        ZRV(JI, JK) = MAX(0., PRVT(JL, JK))
        ZW(JI, JK) = PWT(JL, JK)
    enddo
    enddo
    do JI = 1, ITEST
        JL = IJSINDEX(JI)
        ZSDXDY(JI) = PDXDY(JL)
    enddo
!
!*       2.2    Compute environm. enthalpy and total water = r_v + r_i + r_c
!               and envir. saturation theta_e
!               ------------------------------------------------------------
!
!
!*       2.3    Test for convective columns and determine properties at the LCL
!               --------------------------------------------------------------
!
    ISLCL(:) = MAX(IKB, 2)   ! initialize DPL PBL and LCL
    ISDPL(:) = IKB
    ISPBL(:) = IKB
!
!
    call CONVECT_TRIGGER_FUNCT(ITEST, KLEV, &
                               ZPRES, ZTH, ZTHV, ZTHEST, &
                               ZRV, ZW, ZZ, ZSDXDY, &
                               ZSTHLCL, ZSTLCL, ZSRVLCL, ZSWLCL, ZSZLCL, &
                               ZSTHVELCL, ISLCL, ISDPL, ISPBL, GTRIG1, &
                               ZCAPE)
!
    do JI = 1, ITEST
        JL = IJSINDEX(JI)
        PCAPE(JL) = ZCAPE(JI)
    enddo
!
    DEALLOCATE(ZPRES)
    DEALLOCATE(ZZ)
    DEALLOCATE(ZTH)
    DEALLOCATE(ZTHV)
    DEALLOCATE(ZTHEST)
    DEALLOCATE(ZRV)
    DEALLOCATE(ZW)
    DEALLOCATE(ZCAPE)
!
!-------------------------------------------------------------------------------
!
!*       3.     After the call of TRIGGER_FUNCT we allocate all the dynamic
!               arrays used in the convection scheme using the mask GTRIG, i.e.
!               we do calculus only in convective columns. This corresponds to
!               a GATHER operation.
!               --------------------------------------------------------------
!
    ICONV = COUNT(GTRIG1(:))
    if(ICONV == 0) then
        DEALLOCATE(ZSTHLCL)
        DEALLOCATE(ZSTLCL)
        DEALLOCATE(ZSRVLCL)
        DEALLOCATE(ZSWLCL)
        DEALLOCATE(ZSZLCL)
        DEALLOCATE(ZSTHVELCL)
        DEALLOCATE(ZSDXDY)
        DEALLOCATE(ISLCL)
        DEALLOCATE(ISDPL)
        DEALLOCATE(ISPBL)
        DEALLOCATE(GTRIG1)
        DEALLOCATE(IINDEX)
        DEALLOCATE(IJSINDEX)
        RETURN   ! no convective column has been found, exit DEEP_CONVECTION
    endif
!
    ! vertical index variables
!
    ALLOCATE(IDPL(ICONV))
    ALLOCATE(IPBL(ICONV))
    ALLOCATE(ILCL(ICONV))
    ALLOCATE(ICTL(ICONV))
    ALLOCATE(IETL(ICONV))
!
    ! grid scale variables
!
    ALLOCATE(ZZ(ICONV, IKS)); ZZ = 0.0
    ALLOCATE(ZPRES(ICONV, IKS)); ZPRES = 0.0
    ALLOCATE(ZDPRES(ICONV, IKS+1)); ZDPRES = 0.0
    ALLOCATE(ZU(ICONV, IKS)); ZU = 0.0
    ALLOCATE(ZV(ICONV, IKS)); ZV = 0.0
    ALLOCATE(ZTT(ICONV, IKS)); ZTT = 0.0
    ALLOCATE(ZTH(ICONV, IKS)); ZTH = 0.0
    ALLOCATE(ZTHV(ICONV, IKS)); ZTHV = 0.0
    ALLOCATE(ZTHL(ICONV, IKS)); ZTHL = 0.0
    ALLOCATE(ZTHES(ICONV, IKS)); ZTHES = 0.0
    ALLOCATE(ZRV(ICONV, IKS)); ZRV = 0.0
    ALLOCATE(ZRC(ICONV, IKS)); ZRC = 0.0
    ALLOCATE(ZRI(ICONV, IKS)); ZRI = 0.0
    ALLOCATE(ZRW(ICONV, IKS)); ZRW = 0.0
    ALLOCATE(ZDXDY(ICONV)); ZDXDY = 0.0
!
    ! updraft variables
!
    ALLOCATE(ZUMF(ICONV, IKS))
    ALLOCATE(ZUER(ICONV, IKS))
    ALLOCATE(ZUDR(ICONV, IKS))
    ALLOCATE(ZUPR(ICONV, IKS))
    ALLOCATE(ZUTHL(ICONV, IKS))
    ALLOCATE(ZUTHV(ICONV, IKS))
    ALLOCATE(ZUTT(ICONV, IKS))
    ALLOCATE(ZURW(ICONV, IKS))
    ALLOCATE(ZURC(ICONV, IKS))
    ALLOCATE(ZURI(ICONV, IKS))
    ALLOCATE(ZURR(ICONV, IKS))
    ALLOCATE(ZURS(ICONV, IKS))
    ALLOCATE(ZUTPR(ICONV))
    ALLOCATE(ZTHLCL(ICONV))
    ALLOCATE(ZTLCL(ICONV))
    ALLOCATE(ZRVLCL(ICONV))
    ALLOCATE(ZWLCL(ICONV))
    ALLOCATE(ZMFLCL(ICONV))
    ALLOCATE(ZZLCL(ICONV))
    ALLOCATE(ZTHVELCL(ICONV))
    ALLOCATE(ZCAPE(ICONV))
!
! work variables
!
    ALLOCATE(IJINDEX(ICONV))
    ALLOCATE(IJPINDEX(ICONV))
    ALLOCATE(ZCPH(ICONV))
    ALLOCATE(ZLV(ICONV))
    ALLOCATE(ZLS(ICONV))
!
!
!*           3.1    Gather grid scale and updraft base variables in
!                   arrays using mask GTRIG
!                   ---------------------------------------------------
!
    GTRIG(:) = UNPACK(GTRIG1(:), MASK=GTRIG, FIELD=.false.)
    IJINDEX(:) = PACK(IINDEX(:), MASK=GTRIG(:))
!
    do JK = IKB, IKE
    do JI = 1, ICONV
        JL = IJINDEX(JI)
        ZZ(JI, JK) = PZZ(JL, JK)
        ZPRES(JI, JK) = PPABST(JL, JK)
        ZTT(JI, JK) = PTT(JL, JK)
        ZTH(JI, JK) = ZTHT(JL, JK)
        ZTHES(JI, JK) = ZSTHES(JL, JK)
        ZRV(JI, JK) = MAX(0., PRVT(JL, JK))
        ZRC(JI, JK) = MAX(0., PRCT(JL, JK))
        ZRI(JI, JK) = MAX(0., PRIT(JL, JK))
        ZTHV(JI, JK) = ZSTHV(JL, JK)
        ZU(JI, JK) = PUT(JL, JK)
        ZV(JI, JK) = PVT(JL, JK)
    enddo
    enddo
    if(OSETTADJ) then
        ALLOCATE(ZTIMED(ICONV))
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            ZTIMED(JI) = PTIMEC(JL)
        enddo
    endif
!
    do JI = 1, ITEST
        IJSINDEX(JI) = JI
    enddo
    IJPINDEX(:) = PACK(IJSINDEX(:), MASK=GTRIG1(:))
    do JI = 1, ICONV
        JL = IJPINDEX(JI)
        IDPL(JI) = ISDPL(JL)
        IPBL(JI) = ISPBL(JL)
        ILCL(JI) = ISLCL(JL)
        ZTHLCL(JI) = ZSTHLCL(JL)
        ZTLCL(JI) = ZSTLCL(JL)
        ZRVLCL(JI) = ZSRVLCL(JL)
        ZWLCL(JI) = ZSWLCL(JL)
        ZZLCL(JI) = ZSZLCL(JL)
        ZTHVELCL(JI) = ZSTHVELCL(JL)
        ZDXDY(JI) = ZSDXDY(JL)
    enddo
    ALLOCATE(GWORK(ICONV))
    GWORK(:) = PACK(GTRIG1(:), MASK=GTRIG1(:))
    DEALLOCATE(GTRIG1)
    ALLOCATE(GTRIG1(ICONV))
    GTRIG1(:) = GWORK(:)
!
    DEALLOCATE(GWORK)
    DEALLOCATE(IJPINDEX)
    DEALLOCATE(ISDPL)
    DEALLOCATE(ISPBL)
    DEALLOCATE(ISLCL)
    DEALLOCATE(ZSTHLCL)
    DEALLOCATE(ZSTLCL)
    DEALLOCATE(ZSRVLCL)
    DEALLOCATE(ZSWLCL)
    DEALLOCATE(ZSZLCL)
    DEALLOCATE(ZSTHVELCL)
    DEALLOCATE(ZSDXDY)
!
!
!*           3.2    Compute pressure difference
!            ---------------------------------------------------
!
    ZDPRES(:, IKB) = 0.
    do JK = IKB + 1, IKE
        ZDPRES(:, JK) = ZPRES(:, JK - 1) - ZPRES(:, JK)
    enddo
!
!*           3.3   Compute environm. enthalpy and total water = r_v + r_i + r_c
!                  ----------------------------------------------------------
!
    do JK = IKB, IKE, 1
        ZRW(:, JK) = ZRV(:, JK) + ZRC(:, JK) + ZRI(:, JK)
        ZCPH(:) = XCPD + XCPV * ZRW(:, JK)
        ZLV(:) = XLVTT + (XCPV - XCL) * (ZTT(:, JK) - XTT) ! compute L_v
        ZLS(:) = XLSTT + (XCPV - XCI) * (ZTT(:, JK) - XTT) ! compute L_i
        ZTHL(:, JK) = ZCPH(:) * ZTT(:, JK) + (1.+ZRW(:, JK)) * XG * ZZ(:, JK) &
                      - ZLV(:) * ZRC(:, JK) - ZLS(:) * ZRI(:, JK)
    enddo
!
!-------------------------------------------------------------------------------
!
!*           4.     Compute updraft properties
!                   ----------------------------
!
!*           4.1    Set mass flux at LCL ( here a unit mass flux with w = 1 m/s )
!                   -------------------------------------------------------------
!
    do JI = 1, ICONV
        JK = ILCL(JI) - 1
        ZMFLCL(JI) = ZPRES(JI, JK) / (XRD * ZTT(JI, JK) * &
                                      (1.+ZEPS * ZRVLCL(JI))) * XPI * XCRAD * XCRAD &
                     * MAX(1., ZDXDY(JI) / XA25)
    enddo
!
    DEALLOCATE(ZCPH)
    DEALLOCATE(ZLV)
    DEALLOCATE(ZLS)
!
!
    call CONVECT_UPDRAFT(ICONV, KLEV, &
                         KICE, ZPRES, ZDPRES, ZZ, ZTHL, ZTHV, ZTHES, ZRW, &
                         ZTHLCL, ZTLCL, ZRVLCL, ZWLCL, ZZLCL, ZTHVELCL, &
                         ZMFLCL, GTRIG1, ILCL, IDPL, IPBL, &
                         ZUMF, ZUER, ZUDR, ZUTHL, ZUTHV, ZURW, &
                         ZURC, ZURI, ZURR, ZURS, ZUPR, &
                         ZUTPR, ZCAPE, ICTL, IETL, ZUTT)
!
!
!
!*           4.2    In routine UPDRAFT GTRIG1 has been set to false when cloud
!                   thickness is smaller than 3 km
!                   -----------------------------------------------------------
!
!
    ICONV1 = COUNT(GTRIG1)
!
    if(ICONV1 > 0) then
!
!*       4.3    Allocate memory for downdraft variables
!               ---------------------------------------
!
! downdraft variables
!
        ALLOCATE(ILFS(ICONV))
        ALLOCATE(IDBL(ICONV))
        ALLOCATE(IML(ICONV))
        ALLOCATE(ZDMF(ICONV, IKS))
        ALLOCATE(ZDER(ICONV, IKS))
        ALLOCATE(ZDDR(ICONV, IKS))
        ALLOCATE(ZDTHL(ICONV, IKS))
        ALLOCATE(ZDRW(ICONV, IKS))
        ALLOCATE(ZLMASS(ICONV, IKS)); ZLMASS = 0.0
        do JK = IKB, IKE
            ZLMASS(:, JK) = ZDXDY(:) * ZDPRES(:, JK) / XG  ! mass of model layer
        enddo
        ZLMASS(:, IKB) = ZLMASS(:, IKB + 1)
        ALLOCATE(ZMIXF(ICONV))
        ALLOCATE(ZTPR(ICONV))
        ALLOCATE(ZSPR(ICONV))
        ALLOCATE(ZDTEVR(ICONV))
        ALLOCATE(ZPREF(ICONV))
        ALLOCATE(ZDTEVRF(ICONV, IKS))
        ALLOCATE(ZPRLFLX(ICONV, IKS))
        ALLOCATE(ZPRSFLX(ICONV, IKS))
!
! closure variables
!
        ALLOCATE(ZTIMEA(ICONV))
        ALLOCATE(ZTIMEC(ICONV))
        ALLOCATE(ZTHC(ICONV, IKS))
        ALLOCATE(ZRVC(ICONV, IKS))
        ALLOCATE(ZRCC(ICONV, IKS))
        ALLOCATE(ZRIC(ICONV, IKS))
        ALLOCATE(ZWSUB(ICONV, IKS))
!
!-------------------------------------------------------------------------------
!
!*           5.     Compute downdraft properties
!                   ----------------------------
!
!*           5.1    Compute advective time period and precipitation
!                   efficiency as a function of mean ambient wind (shear)
!                   --------------------------------------------------------
!
        call CONVECT_TSTEP_PREF(ICONV, KLEV, &
                                ZU, ZV, ZPRES, ZZ, ZDXDY, ILCL, ICTL, &
                                ZTIMEA, ZPREF)
!
        ! exclude convective downdrafts if desired
        if(.not. ODOWN) ZPREF(:) = 1.
!
! Compute the period during which convection is active
        ZTIMEC(:) = MAX(1800., MIN(3600., ZTIMEA(:)))
        ZTIMEC(:) = REAL(INT(ZTIMEC(:) / PDTCONV)) * PDTCONV
        ZTIMEC(:) = MAX(PDTCONV, ZTIMEC(:)) ! necessary if PDTCONV > 1800
        if(OSETTADJ) then
            ZTIMEC(:) = MAX(PDTCONV, ZTIMED(:))
        endif
!
!
!*           5.2    Compute melting level
!                   ----------------------
!
        IML(:) = IKB
        do JK = IKE, IKB, -1
            WHERE(ZTT(:, JK) <= XTT) IML(:) = JK
        enddo
!
        call CONVECT_DOWNDRAFT(ICONV, KLEV, &
                               KICE, ZPRES, ZDPRES, ZZ, ZTH, ZTHES, &
                               ZRW, ZRC, ZRI, &
                               ZPREF, ILCL, ICTL, IETL, &
                               ZUTHL, ZURW, ZURC, ZURI, &
                               ZDMF, ZDER, ZDDR, ZDTHL, ZDRW, &
                               ZMIXF, ZDTEVR, ILFS, IDBL, IML, &
                               ZDTEVRF)
!
!-------------------------------------------------------------------------------
!
!*           6.     Adjust up and downdraft mass flux to be consistent
!                   with precipitation efficiency relation.
!                   ---------------------------------------------------
!
        call CONVECT_PRECIP_ADJUST(ICONV, KLEV, &
                                   ZPRES, ZUMF, ZUER, ZUDR, ZUPR, ZUTPR, ZURW, &
                                   ZDMF, ZDER, ZDDR, ZDTHL, ZDRW, &
                                   ZPREF, ZTPR, ZMIXF, ZDTEVR, &
                                   ILFS, IDBL, ILCL, ICTL, IETL, &
                                   ZDTEVRF)
!
!-------------------------------------------------------------------------------
!
!*           7.     Determine adjusted environmental values assuming
!                   that all available buoyant energy must be removed
!                   within an advective time step ZTIMEC.
!                   ---------------------------------------------------
!
        call CONVECT_CLOSURE(ICONV, KLEV, &
                             ZPRES, ZDPRES, ZZ, ZDXDY, ZLMASS, &
                             ZTHL, ZTH, ZRW, ZRC, ZRI, GTRIG1, &
                             ZTHC, ZRVC, ZRCC, ZRIC, ZWSUB, &
                             ILCL, IDPL, IPBL, ILFS, ICTL, IML, &
                             ZUMF, ZUER, ZUDR, ZUTHL, ZURW, &
                             ZURC, ZURI, ZUPR, &
                             ZDMF, ZDER, ZDDR, ZDTHL, ZDRW, &
                             ZTPR, ZSPR, ZDTEVR, &
                             ZCAPE, ZTIMEC, &
                             IFTSTEPS, &
                             ZDTEVRF, ZPRLFLX, ZPRSFLX)
!
!-------------------------------------------------------------------------------
!
!*           8.     Determine the final grid-scale (environmental) convective
!                   tendencies and set convective counter
!                   --------------------------------------------------------
!
!
!*           8.1    Grid scale tendencies
!                   ---------------------
!
!     in order to save memory, the tendencies are temporarily stored
!     in the tables for the adjusted grid-scale values
!
        do JK = IKB, IKE
            ZTHC(:, JK) = (ZTHC(:, JK) - ZTH(:, JK)) / ZTIMEC(:) &
                          * (ZPRES(:, JK) / XP00)**ZRDOCP ! change theta in temperature
            ZRVC(:, JK) = (ZRVC(:, JK) - ZRW(:, JK) + ZRC(:, JK) + ZRI(:, JK)) / ZTIMEC(:)
            ZRCC(:, JK) = (ZRCC(:, JK) - ZRC(:, JK)) / ZTIMEC(:)
            ZRIC(:, JK) = (ZRIC(:, JK) - ZRI(:, JK)) / ZTIMEC(:)
!
            ZPRLFLX(:, JK) = ZPRLFLX(:, JK) / (XRHOLW * ZDXDY(:))
            ZPRSFLX(:, JK) = ZPRSFLX(:, JK) / (XRHOLW * ZDXDY(:))
!
        enddo
!
        ZPRLFLX(:, IKB) = ZPRLFLX(:, IKB + 1)
        ZPRSFLX(:, IKB) = ZPRSFLX(:, IKB + 1)
!
!
!*           8.2    Apply conservation correction
!                   -----------------------------
!
        ! Compute vertical integrals
!
! Reproducibility
! JKM = MAXVAL( ICTL(:) )
        JKM = IKE - 1
        ZWORK2(:) = 0.
        ZWORK2B(:) = 0.
        do JK = IKB + 1, JKM
            JKP = JK + 1
            do JI = 1, ICONV
                ZW1 = .5 * (ZPRES(JI, JK - 1) - ZPRES(JI, JKP)) / XG
                ZWORK2(JI) = ZWORK2(JI) + (ZRVC(JI, JK) + ZRCC(JI, JK) + ZRIC(JI, JK)) * ZW1 ! moisture
                ZWORK2B(JI) = ZWORK2B(JI) + ((XCPD + XCPV * ZRW(JI, JK)) * ZTHC(JI, JK) - &
                                             (XLVTT + (XCPV - XCL) * (ZTT(JI, JK) - XTT)) * ZRCC(JI, JK) - &
                                             (XLSTT + (XCPV - XCL) * (ZTT(JI, JK) - XTT)) * ZRIC(JI, JK)) * &
                              ZW1                                       ! enthalpy
            enddo
        enddo
!
        ! Budget error (compare integral to surface precip.)
!
        do JI = 1, ICONV
            if(ZTPR(JI) > 0.) then
                ZW1 = XG / (ZPRES(JI, IKB) - ZPRES(JI, JKP) - .5 * ( &
                            ZDPRES(JI, IKB + 1) - ZDPRES(JI, JKP + 1)))
                ZWORK2(JI) = (ZTPR(JI) / ZDXDY(JI) + ZWORK2(JI)) * ZW1
                ZWORK2B(JI) = (ZTPR(JI) / ZDXDY(JI) * &
                               (XLVTT + (XCPV - XCL) * (ZTT(JI, IKB) - XTT)) - ZWORK2B(JI)) &
                              * ZW1
            endif
        enddo
!
        ! Apply uniform correction
!
        do JK = JKM, IKB + 1, -1
        do JI = 1, ICONV
            if(ZTPR(JI) > 0. .and. JK <= ICTL(JI)) then
                ! ZW1 = ABS(ZRVC(JI,JK)) +  ABS(ZRCC(JI,JK)) +  ABS(ZRIC(JI,JK)) + 1.E-12
                ! ZRVC(JI,JK) = ZRVC(JI,JK) - ABS(ZRVC(JI,JK))/ZW1*ZWORK2(JI)           ! moisture
                ZRVC(JI, JK) = ZRVC(JI, JK) - ZWORK2(JI)                                ! moisture
                ! ZRCC(JI,JK) = ZRCC(JI,JK) - ABS(ZRCC(JI,JK))/ZW1*ZWORK2(JI)
                ! ZRIC(JI,JK) = ZRIC(JI,JK) - ABS(ZRIC(JI,JK))/ZW1*ZWORK2(JI)
                ZTHC(JI, JK) = ZTHC(JI, JK) + ZWORK2B(JI) / (XCPD + XCPV * ZRW(JI, JK))! energy
            endif
        enddo
        enddo
!
!
!     execute a "scatter"= pack command to store the tendencies in
!     the final 2D tables
!
        do JK = IKB, IKE
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            PTTEN(JL, JK) = ZTHC(JI, JK)
            PRVTEN(JL, JK) = ZRVC(JI, JK)
            PRCTEN(JL, JK) = ZRCC(JI, JK)
            PRITEN(JL, JK) = ZRIC(JI, JK)
!
            PPRLFLX(JL, JK) = ZPRLFLX(JI, JK)
            PPRSFLX(JL, JK) = ZPRSFLX(JI, JK)
        enddo
        enddo
!
!
!*           8.3    Convective rainfall tendency
!                   ----------------------------
!
        ! liquid and solid surface rainfall tendency in m/s
        ZTPR(:) = ZTPR(:) / (XRHOLW * ZDXDY(:)) ! total surf precip
        ZSPR(:) = ZSPR(:) / (XRHOLW * ZDXDY(:)) ! solid surf precip
        ZTPR(:) = ZTPR(:) - ZSPR(:) ! compute liquid part
!
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            PPRLTEN(JL) = ZTPR(JI)
            PPRSTEN(JL) = ZSPR(JI)
        enddo
!
!
!                   Cloud base and top levels
!                   -------------------------
!
        ILCL(:) = MIN(ILCL(:), ICTL(:))
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            KCLTOP(JL) = ICTL(JI)
            KCLBAS(JL) = ILCL(JI)
        enddo
!
!
!*           8.4    Set convective counter
!                   ----------------------
!
        ! compute convective counter for just activated convective
        ! grid points
        ! If the advective time period is less than specified
        ! minimum for convective period, allow feedback to occur only
        ! during advective time
!
        ZTIME(:) = 1.
        ZWORK2(:) = 0.
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            ZTIME(JL) = ZTIMEC(JI)
            ZWORK2(JL) = ZTIMEA(JI)
            ZWORK2(JL) = MIN(ZWORK2(JL), ZTIME(JL))
            ZWORK2(JL) = MAX(ZWORK2(JL), PDTCONV)
            if(GTRIG(JL)) KCOUNT(JL) = INT(ZWORK2(JL) / PDTCONV)
            if(GTRIG(JL) .and. PPRLTEN(JL) < 1.E-14) KCOUNT(JL) = 0
        enddo
!
!
!*           8.7    Compute convective tendencies for Tracers
!                   ------------------------------------------
!
!  if ( OCH1CONV ) then
!!
!    ALLOCATE( ZCH1(ICONV,IKS,KCH1) )    ; ZCH1 = 0.0
!    ALLOCATE( ZCH1C(ICONV,IKS,KCH1) )   ; ZCH1C = 0.0
!    ALLOCATE( ZWORK3(ICONV,KCH1) )
!!
!    ALLOCATE( ZRHODREF(ICONV,IKS) )
!    ZRHODREF=0.
!    if ( OCH_CONV_LINOX ) then
!      ALLOCATE( ZZZ(ICONV,IKS) )
!      ALLOCATE( ZIC_RATE(ICONV) )
!      ALLOCATE( ZCG_RATE(ICONV) )
!      ALLOCATE( ZWORK4(ICONV,IKS) )
!      ALLOCATE( ZWORK4C(ICONV,IKS) )
!      ZZZ=0.
!      ZIC_RATE=0.
!      ZCG_RATE=0.
!      ZWORK4=0.
!      ZWORK4C=0.
!    end if
!!
!    do JI = 1, ICONV
!     do JK = IKB, IKE
!      JL = IJINDEX(JI)
!      ZCH1(JI,JK,:) = PCH1(JL,JK,:)
!      ZRHODREF(JI,JK)=PRHODREF(JL,JK)
!     end do
!     ZRHODREF(JI,1) = PRHODREF(JL,IKB)
!     ZRHODREF(JI,IKS) = PRHODREF(JL,IKE)
!    end do
!    ZCH1(:,1,:) = ZCH1(:,IKB,:)
!    ZCH1(:,IKS,:) = ZCH1(:,IKE,:)
!!
!    JN_NO = 0
!    if ( OCH_CONV_LINOX ) then
!      do JK = IKB, IKE
!      do JI = 1, ICONV
!        JL = IJINDEX(JI)
!        ZZZ(JI,JK)=PZZ(JL,JK)
!        ZIC_RATE(JI)=PIC_RATE(JL)
!        ZCG_RATE(JI)=PCG_RATE(JL)
!      end do
!      end do
!      if (OUSECHEM) then
!        do JN = NSV_CHEMBEG,NSV_CHEMEND
!          if (CNAMES(JN-NSV_CHEMBEG+1)=='NO') JN_NO = JN
!        end do
!      else
!        JN_NO = NSV_LNOXBEG
!      end if
!      ZWORK4(:,:) = ZCH1(:,:,JN_NO)
!      call CH_CONVECT_LINOX( ICONV, KLEV, ZWORK4, ZWORK4C,        &
!                             IDPL, IPBL, ILCL, ICTL, ILFS, IDBL,  &
!                             ZUMF, ZUER, ZUDR, ZDMF, ZDER, ZDDR,  &
!                             ZTIMEC, ZDXDY, ZMIXF, ZLMASS, ZWSUB, &
!                             IFTSTEPS, ZUTT, ZRHODREF,            &
!                             OUSECHEM, ZZZ, ZIC_RATE, ZCG_RATE    )
!      do JI = 1, ICONV
!        JL = IJINDEX(JI)
!        PIC_RATE(JL)=ZIC_RATE(JI)
!        PCG_RATE(JL)=ZCG_RATE(JI)
!      end do
!    end if
!!
!    if ((OUSECHEM .and. OCH_CONV_SCAV).OR.(ODUST .and.  OCH_CONV_SCAV).OR.&
!        (OSALT .and.  OCH_CONV_SCAV)  ) then
!!
!      call CH_CONVECT_SCAVENGING( ICONV, KLEV, KCH1, ZCH1, ZCH1C,      &
!                                  IDPL, IPBL, ILCL, ICTL, ILFS, IDBL,  &
!                                  ZUMF, ZUER, ZUDR, ZDMF, ZDER, ZDDR,  &
!                                  ZTIMEC, ZDXDY, ZMIXF, ZLMASS, ZWSUB, &
!                                  IFTSTEPS,                            &
!                                  ZURC, ZURR, ZURI, ZURS, ZUTT, ZPRES, &
!                                  ZRHODREF, PPABST, ZTHT               )
!!
!      if (OCH_CONV_LINOX) then
!        ZCH1C(:,:,JN_NO) = ZWORK4C(:,:)
!      end if
!!    no conservation correction for scavenging
!     do JI = 1, ICONV
!        JL = IJINDEX(JI)
!        if ( ZTPR(JI) > 0. ) then
!          do JK = IKB, IKE
!            PCH1TEN(JL,JK,:) = (ZCH1C(JI,JK,:)- ZCH1(JI,JK,:)) /ZTIMEC(JI)
!          end do
!        else
!          do JK = IKB, IKE
!            PCH1TEN(JL,JK,:) = 0.
!          end do
!        end if
!      end do
!
!!
!    else
!!
!      call CONVECT_CHEM_TRANSPORT( ICONV, KLEV, KCH1, ZCH1, ZCH1C,      &
!                                   IDPL, IPBL, ILCL, ICTL, ILFS, IDBL,  &
!                                   ZUMF, ZUER, ZUDR, ZDMF, ZDER, ZDDR,  &
!                                   ZTIMEC, ZDXDY, ZMIXF, ZLMASS, ZWSUB, &
!                                   IFTSTEPS )
!!
!      if (OCH_CONV_LINOX) then
!        ZCH1C(:,:,JN_NO) = ZWORK4C(:,:)
!      end if
!!
!!*           8.8    Apply conservation correction
!!                   -----------------------------
!!
!          ! Compute vertical integrals
!!
!! Reproducibility
!!     JKM = MAXVAL( ICTL(:) )
!      JKM = IKE - 1
!      do JN = 1, KCH1
!        if((JN < NSV_LGBEG .OR. JN>NSV_LGEND-1) .and. JN .NE. JN_NO ) then
!          ! no correction for Lagrangian and LiNOx variables
!          ZWORK3(:,JN) = 0.
!          ZWORK2(:)    = 0.
!          do JK = IKB+1, JKM
!            JKP = JK + 1
!            do JI = 1, ICONV
!              ZW1 = .5 * (ZPRES(JI,JK-1) - ZPRES(JI,JKP))
!              ZWORK3(JI,JN) = ZWORK3(JI,JN) + (ZCH1C(JI,JK,JN)-ZCH1(JI,JK,JN)) * ZW1
!              ZWORK2(JI)    = ZWORK2(JI)    + ABS(ZCH1C(JI,JK,JN)) * ZW1
!            end do
!          end do
!!
!             ! Apply concentration weighted correction
!!
!          do JK = JKM, IKB+1, -1
!          do JI = 1, ICONV
!            if ( ZTPR(JI) > 0. .and. JK <= ICTL(JI) ) then
!              ZCH1C(JI,JK,JN) = ZCH1C(JI,JK,JN) -       &
!              ZWORK3(JI,JN)*ABS(ZCH1C(JI,JK,JN))/MAX(1.E-30,ZWORK2(JI))
!                ! ZCH1C(JI,JK,JN) = MAX( ZCH1C(JI,JK,JN), -ZCH1(JI,JK,JN)/ZTIMEC(JI) )
!            end if
!          end do
!          end do
!        end if
!!
!        do JI = 1, ICONV
!          JL = IJINDEX(JI)
!          if ( ZTPR(JI) > 0. ) then
!            do JK = IKB, IKE
!              PCH1TEN(JL,JK,JN) = (ZCH1C(JI,JK,JN)-ZCH1(JI,JK,JN) ) /ZTIMEC(JI)
!            end do
!          else
!            do JK = IKB, IKE
!              PCH1TEN(JL,JK,JN) = 0.
!            end do
!          end if
!        end do
!      end do
!    end if
!  end if
!
!-------------------------------------------------------------------------------
!
!*           9.     Write up- and downdraft mass fluxes
!                   ------------------------------------
!
        do JK = IKB, IKE
            ZUMF(:, JK) = ZUMF(:, JK) / ZDXDY(:) ! Mass flux per unit area
            ZDMF(:, JK) = ZDMF(:, JK) / ZDXDY(:)
        enddo
        ZWORK2(:) = 1.
        WHERE(PPRLTEN(:) < 1.E-14) ZWORK2(:) = 0.
        do JK = IKB, IKE
        do JI = 1, ICONV
            JL = IJINDEX(JI)
            PUMF(JL, JK) = ZUMF(JI, JK) * ZWORK2(JL)
            PDMF(JL, JK) = ZDMF(JI, JK) * ZWORK2(JL)
        enddo
        enddo
!
!-------------------------------------------------------------------------------
!
!*           10.    Deallocate all local arrays
!                   ---------------------------
!
! downdraft variables
!
        DEALLOCATE(ZDMF)
        DEALLOCATE(ZDER)
        DEALLOCATE(ZDDR)
        DEALLOCATE(ZDTHL)
        DEALLOCATE(ZDRW)
        DEALLOCATE(ZLMASS)
        DEALLOCATE(ZMIXF)
        DEALLOCATE(ZTPR)
        DEALLOCATE(ZSPR)
        DEALLOCATE(ZDTEVR)
        DEALLOCATE(ZPREF)
        DEALLOCATE(IML)
        DEALLOCATE(ILFS)
        DEALLOCATE(IDBL)
        DEALLOCATE(ZDTEVRF)
        DEALLOCATE(ZPRLFLX)
        DEALLOCATE(ZPRSFLX)
!
!   closure variables
!
        DEALLOCATE(ZTIMEA)
        DEALLOCATE(ZTIMEC)
        DEALLOCATE(ZTHC)
        DEALLOCATE(ZRVC)
        DEALLOCATE(ZRCC)
        DEALLOCATE(ZRIC)
        DEALLOCATE(ZWSUB)
!
        if(OCH1CONV) then
            DEALLOCATE(ZCH1)
            DEALLOCATE(ZCH1C)
            DEALLOCATE(ZWORK3)
            DEALLOCATE(ZRHODREF)
            if(OCH_CONV_LINOX) then
                DEALLOCATE(ZZZ)
                DEALLOCATE(ZIC_RATE)
                DEALLOCATE(ZCG_RATE)
                DEALLOCATE(ZWORK4)
                DEALLOCATE(ZWORK4C)
            endif
        endif
!
    endif
!
!    vertical index
!
    DEALLOCATE(IDPL)
    DEALLOCATE(IPBL)
    DEALLOCATE(ILCL)
    DEALLOCATE(ICTL)
    DEALLOCATE(IETL)
!
! grid scale variables
!
    DEALLOCATE(ZZ)
    DEALLOCATE(ZPRES)
    DEALLOCATE(ZDPRES)
    DEALLOCATE(ZU)
    DEALLOCATE(ZV)
    DEALLOCATE(ZTT)
    DEALLOCATE(ZTH)
    DEALLOCATE(ZTHV)
    DEALLOCATE(ZTHL)
    DEALLOCATE(ZTHES)
    DEALLOCATE(ZRW)
    DEALLOCATE(ZRV)
    DEALLOCATE(ZRC)
    DEALLOCATE(ZRI)
    DEALLOCATE(ZDXDY)
!
! updraft variables
!
    DEALLOCATE(ZUMF)
    DEALLOCATE(ZUER)
    DEALLOCATE(ZUDR)
    DEALLOCATE(ZUTHL)
    DEALLOCATE(ZUTHV)
    DEALLOCATE(ZUTT)
    DEALLOCATE(ZURW)
    DEALLOCATE(ZURC)
    DEALLOCATE(ZURI)
    DEALLOCATE(ZURR)
    DEALLOCATE(ZURS)
    DEALLOCATE(ZUPR)
    DEALLOCATE(ZUTPR)
    DEALLOCATE(ZTHLCL)
    DEALLOCATE(ZTLCL)
    DEALLOCATE(ZRVLCL)
    DEALLOCATE(ZWLCL)
    DEALLOCATE(ZZLCL)
    DEALLOCATE(ZTHVELCL)
    DEALLOCATE(ZMFLCL)
    DEALLOCATE(ZCAPE)
    if(OSETTADJ) DEALLOCATE(ZTIMED)
!
! work arrays
!
    DEALLOCATE(IINDEX)
    DEALLOCATE(IJINDEX)
    DEALLOCATE(IJSINDEX)
    DEALLOCATE(GTRIG1)
!
!
ENDsubroutine DEEP_CONVECTION

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 modd 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     ######spl

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_CLOSURE_ADJUST_SHAL
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CLOSURE_ADJUST_SHAL(KLON, KLEV, PADJ, &
                                               PUMF, PZUMF, PUER, PZUER, PUDR, PZUDR)
!
            INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
            REAL, DIMENSION(KLON), INTENT(IN) :: PADJ     ! mass adjustment factor
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUMF ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUER ! initial value of  "
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUDR ! initial value of  "
!
        ENDsubroutine CONVECT_CLOSURE_ADJUST_SHAL
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CLOSURE_ADJUST_SHAL
!    ################################################################################
subroutine CONVECT_CLOSURE_ADJUST_SHAL(KLON, KLEV, PADJ, &
                                       PUMF, PZUMF, PUER, PZUER, PUDR, PZUDR)
!    ################################################################################
!
!!**** Uses closure adjustment factor to adjust mass flux and to modify
!!     precipitation efficiency  when necessary. The computations are
!!     similar to routine CONVECT_PRECIP_ADJUST.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to adjust the mass flux using the
!!      factor PADJ computed in CONVECT_CLOSURE
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!    EXTERNAL
!!    --------
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    None
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!
!!    None
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_CLOSURE_ADJUST)
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Last modified  15/11/96
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CONVPAREXT
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
    REAL, DIMENSION(KLON), INTENT(IN) :: PADJ     ! mass adjustment factor
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUMF ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUER ! initial value of  "
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT) :: PZUDR ! initial value of  "
!
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IKB, IKE                 ! vert. loop bounds
    INTEGER :: JK                       ! vertical loop index
!
!
!-------------------------------------------------------------------------------
!
!*       0.3   Compute loop bounds
!              -------------------
!
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Adjust mass flux by the factor PADJ to converge to
!               specified degree of stabilization
!               ----------------------------------------------------
!
    do JK = IKB + 1, IKE
        PUMF(:, JK) = PZUMF(:, JK) * PADJ(:)
        PUER(:, JK) = PZUER(:, JK) * PADJ(:)
        PUDR(:, JK) = PZUDR(:, JK) * PADJ(:)
    enddo
!
ENDsubroutine CONVECT_CLOSURE_ADJUST_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_CLOSURE_SHAL
!     #################
!
    INTERFACE
!
        subroutine CONVECT_CLOSURE_SHAL(KLON, KLEV, &
                                        PPRES, PDPRES, PZ, PDXDY, PLMASS, &
                                        PTHL, PTH, PRW, PRC, PRI, OTRIG1, &
                                        PTHC, PRWC, PRCC, PRIC, PWSUB, &
                                        KLCL, KDPL, KPBL, KCTL, &
                                        PUMF, PUER, PUDR, PUTHL, PURW, &
                                        PURC, PURI, PCAPE, PTIMEC, KFTSTEPS)

!
            INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV   ! vertical dimension
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! index lifting condens. level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! index for cloud top level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! index for departure level
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   ! index for top of source layer
            REAL, DIMENSION(KLON), INTENT(INOUT) :: PTIMEC ! convection time step
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL   ! grid scale enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH    ! grid scale theta
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW    ! grid scale total water
            ! mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC    ! grid scale r_c
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI    ! grid scale r_i
            LOGICAL, DIMENSION(KLON), INTENT(IN) :: OTRIG1 ! logical to keep trace of
            ! convective arrays modified in UPDRAFT
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (P)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES ! pressure difference between
            ! bottom and top of layer (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PLMASS ! mass of model layer (kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN)  :: PCAPE  ! available potent. energy
            INTEGER, INTENT(OUT)   :: KFTSTEPS! maximum of fract time steps
            ! only used for chemical tracers
!
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PUTHL  ! updraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURW   ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURC   ! updraft cloud water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURI   ! updraft cloud ice   (kg/kg)
!
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PTHC  ! conv. adj. grid scale theta
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRWC  ! conv. adj. grid scale r_w
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRCC  ! conv. adj. grid scale r_c
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRIC  ! conv. adj. grid scale r_i
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PWSUB ! envir. compensating subsidence(Pa/s)
!
        ENDsubroutine CONVECT_CLOSURE_SHAL
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_CLOSURE_SHAL
!    ##############################################################################
subroutine CONVECT_CLOSURE_SHAL(KLON, KLEV, &
                                PPRES, PDPRES, PZ, PDXDY, PLMASS, &
                                PTHL, PTH, PRW, PRC, PRI, OTRIG1, &
                                PTHC, PRWC, PRCC, PRIC, PWSUB, &
                                KLCL, KDPL, KPBL, KCTL, &
                                PUMF, PUER, PUDR, PUTHL, PURW, &
                                PURC, PURI, PCAPE, PTIMEC, KFTSTEPS)
!    ##############################################################################
!
!!**** Uses modified Fritsch-Chappell closure
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the final adjusted
!!     (over a time step PTIMEC) environmental values of THETA_l, R_w, R_c, R_i
!!      The final convective tendencies can then be evaluated in the main
!!      routine DEEP_CONVECT by (PTHC-PTH)/PTIMEC
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!
!!    EXTERNAL
!!    --------
!!
!!    CONVECT_CLOSURE_THRVLCL
!!    CONVECT_CLOSURE_ADJUST_SHAL
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! reference pressure
!!          XRD, XRV           ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV         ! specific heat for dry air and water vapor
!!          XCL, XCI           ! specific heat for liquid water and ice
!!          XTT                ! triple point temperature
!!          XLVTT, XLSTT       ! vaporization, sublimation heat constant
!!
!!      Module MODD_CONVPAR_SHAL
!!          XA25               ! reference grid area
!!          XSTABT             ! stability factor in time integration
!!          XSTABC             ! stability factor in CAPE adjustment
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_CLOSURE)
!!      Fritsch and Chappell, 1980, J. Atmos. Sci.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Peter Bechtold 15/11/96 change for enthalpie, r_c + r_i tendencies
!!      Tony Dore   14/10/96 Initialise local variables
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR_SHAL
    USE MODD_CONVPAREXT
!
    USE MODI_CONVECT_CLOSURE_THRVLCL
    USE MODI_CONVECT_SATMIXRATIO
    USE MODI_CONVECT_CLOSURE_ADJUST_SHAL
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN) :: KLON   ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV   ! vertical dimension
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! index lifting condens. level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KCTL   ! index for cloud top level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! index for departure level
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   ! index for top of source layer
    REAL, DIMENSION(KLON), INTENT(INOUT) :: PTIMEC ! convection time step
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY  ! grid area (m^2)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL   ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH    ! grid scale theta
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW    ! grid scale total water
    ! mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRC    ! grid scale r_c
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRI    ! grid scale r_i
    LOGICAL, DIMENSION(KLON), INTENT(IN) :: OTRIG1 ! logical to keep trace of
    ! convective arrays modified in UPDRAFT
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES  ! pressure (P)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES ! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PLMASS ! mass of model layer (kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ     ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN)  :: PCAPE  ! available potent. energy
    INTEGER, INTENT(OUT)   :: KFTSTEPS! maximum of fract time steps
    ! only used for chemical tracers
!
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PUTHL  ! updraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURW   ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURC   ! updraft cloud water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN)  :: PURI   ! updraft cloud ice   (kg/kg)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PTHC  ! conv. adj. grid scale theta
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRWC  ! conv. adj. grid scale r_w
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRCC  ! conv. adj. grid scale r_c
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PRIC  ! conv. adj. grid scale r_i
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT)  :: PWSUB ! envir. compensating subsidence(Pa/s)
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE  ! horizontal + vertical loop bounds
    INTEGER :: IKS            ! vertical dimension
    INTEGER :: JK, JKP, JKMAX ! vertical loop index
    INTEGER :: JI             ! horizontal loop index
    INTEGER :: JITER          ! iteration loop index
    INTEGER :: JSTEP          ! fractional time loop index
    real :: ZCPORD, ZRDOCP ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON, KLEV) :: ZTHLC       ! convectively adjusted
    ! grid scale enthalpy
    REAL, DIMENSION(KLON, KLEV) :: ZOMG        ! conv. environm. subsidence (Pa/s)
    REAL, DIMENSION(KLON, KLEV) :: ZUMF        ! non-adjusted updraft mass flux
    REAL, DIMENSION(KLON, KLEV) :: ZUER        !   "     updraft  entrainm. rate
    REAL, DIMENSION(KLON, KLEV) :: ZUDR        !   "     updraft  detrainm. rate
    REAL, DIMENSION(KLON)     :: ZADJ         ! mass adjustment factor
    REAL, DIMENSION(KLON)     :: ZADJMAX      ! limit value for ZADJ
    REAL, DIMENSION(KLON)     :: ZCAPE        ! new CAPE after adjustment
    REAL, DIMENSION(KLON)     :: ZTIMEC       ! fractional convective time step
    REAL, DIMENSION(KLON, KLEV):: ZTIMC        ! 2D work array for ZTIMEC
!
    REAL, DIMENSION(KLON)     :: ZTHLCL       ! new  theta at LCL
    REAL, DIMENSION(KLON)     :: ZRVLCL       ! new  r_v at LCL
    REAL, DIMENSION(KLON)     :: ZZLCL        ! height of LCL
    REAL, DIMENSION(KLON)     :: ZTLCL        ! temperature at LCL
    REAL, DIMENSION(KLON)     :: ZTELCL       ! envir. temper. at LCL
    REAL, DIMENSION(KLON)     :: ZTHEUL       ! theta_e for undilute ascent
    REAL, DIMENSION(KLON)     :: ZTHES1, ZTHES2! saturation environm. theta_e
    REAL, DIMENSION(KLON, KLEV) :: ZTHMFIN, ZTHMFOUT, ZRWMFIN, ZRWMFOUT
    REAL, DIMENSION(KLON, KLEV) :: ZRCMFIN, ZRCMFOUT, ZRIMFIN, ZRIMFOUT
    ! work arrays for environm. compensat. mass flux
    REAL, DIMENSION(KLON)     :: ZPI          ! (P/P00)**R_d/C_pd
    REAL, DIMENSION(KLON)     :: ZLV          ! latent heat of vaporisation
    REAL, DIMENSION(KLON)     :: ZLS          ! latent heat of sublimation
    REAL, DIMENSION(KLON)     :: ZCPH         ! specific heat C_ph
    INTEGER, DIMENSION(KLON)  :: ITSTEP       ! fractional convective time step
    INTEGER, DIMENSION(KLON)  :: ICOUNT       ! timestep counter
    INTEGER, DIMENSION(KLON)  :: ILCL         ! index lifting condens. level
    INTEGER, DIMENSION(KLON)  :: IWORK1       ! work array
    REAL, DIMENSION(KLON)     :: ZWORK1, ZWORK2, ZWORK3, ZWORK4, ZWORK5
    LOGICAL, DIMENSION(KLON)  :: GWORK1, GWORK3! work arrays
    LOGICAL, DIMENSION(KLON, KLEV) :: GWORK4    ! work array
!
!
!-------------------------------------------------------------------------------
!
!*       0.2    Initialize  local variables
!               ----------------------------
!
!
    ZTIMC(:, :) = 0.
    ZTHES2(:) = 0.
    ZWORK1(:) = 0.
    ZWORK2(:) = 0.
    ZWORK3(:) = 0.
    ZWORK4(:) = 0.
    ZWORK5(:) = 0.
    GWORK1(:) = .false.
    GWORK3(:) = .false.
    GWORK4(:, :) = .false.
    ILCL(:) = KLCL(:)
!
    ZCPORD = XCPD / XRD
    ZRDOCP = XRD / XCPD
!
    ZADJ(:) = 1.
    ZWORK5(:) = 1.
    WHERE(.not. OTRIG1(:)) ZWORK5(:) = 0.
!
!
!*       0.3   Compute loop bounds
!              -------------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKS = KLEV
    IKE = KLEV - JCVEXT
    JKMAX = MAXVAL(KCTL(:))
!
!
!*       2.     Save initial mass flux values to be used in adjustment procedure
!               ---------------------------------------------------------------
!
    ZUMF(:, :) = PUMF(:, :)
    ZUER(:, :) = PUER(:, :)
    ZUDR(:, :) = PUDR(:, :)
    ZOMG(:, :) = 0.
    PWSUB(:, :) = 0.
!
!
!*       3.     Compute limits on the closure adjustment factor so that the
!               inflow in convective drafts from a given layer can't be larger
!               than the mass contained in this layer initially.
!               ---------------------------------------------------------------
!
    ZADJMAX(:) = 1000.
    IWORK1(:) = ILCL(:)
    JKP = MINVAL(KDPL(:))
    do JK = JKP, IKE
        do JI = 1, IIE
            if(JK > KDPL(JI) .and. JK <= IWORK1(JI)) then
                ZWORK1(JI) = PLMASS(JI, JK) / ((PUER(JI, JK) + 1.E-5) * PTIMEC(JI))
                ZADJMAX(JI) = MIN(ZADJMAX(JI), ZWORK1(JI))
            endif
        enddo
    enddo
!
!
    GWORK1(:) = OTRIG1(:)  ! logical array to limit adjustment to not definitively
    ! adjusted columns
!
    do JK = IKB, IKE
        ZTHLC(:, JK) = PTHL(:, JK) ! initialize adjusted envir. values
        PRWC(:, JK) = PRW(:, JK)
        PRCC(:, JK) = PRC(:, JK)
        PRIC(:, JK) = PRI(:, JK)
        PTHC(:, JK) = PTH(:, JK)
    enddo
!
!
!
    do JITER = 1, 4  ! Enter adjustment loop to assure that all CAPE is
        ! removed within the advective time interval TIMEC
!
        ZTIMEC(:) = PTIMEC(:)
        GWORK4(:, :) = SPREAD(GWORK1(:), DIM=2, NCOPIES=IKS)
        WHERE(GWORK4(:, :)) PWSUB(:, :) = 0.
        ZOMG(:, :) = 0.
!
        do JK = IKB + 1, JKMAX
            JKP = MAX(IKB + 1, JK - 1)
            WHERE(GWORK1(:) .and. JK <= KCTL(:))
!
!
!*       4.     Determine vertical velocity at top and bottom of each layer
!               to satisfy mass continuity.
!               ---------------------------------------------------------------
                ! we compute here Domega/Dp = - g rho Dw/Dz = 1/Dt
!
                ZWORK1(:) = -(PUER(:, JKP) - PUDR(:, JKP)) / PLMASS(:, JKP)
!
                PWSUB(:, JK) = PWSUB(:, JKP) - PDPRES(:, JK - 1) * ZWORK1(:)
                ! we use PDPRES(JK-1) and not JKP in order to have zero subsidence
                ! at the first layer
!
!
!*       5.     Compute fractional time step. For stability or
!               mass conservation reasons one must split full time step PTIMEC)
!               ---------------------------------------------------------------
!
                ZWORK1(:) = XSTABT * PDPRES(:, JKP) / (ABS(PWSUB(:, JK)) + 1.E-10)
                ! the factor XSTABT is used for stability reasons
                ZTIMEC(:) = MIN(ZTIMEC(:), ZWORK1(:))
!
                ! transform vertical velocity in mass flux units
                ZOMG(:, JK) = PWSUB(:, JK) * PDXDY(:) / XG
            ENDWHERE
        enddo
!
!
        WHERE(GWORK4(:, :))
            ZTHLC(:, :) = PTHL(:, :) ! reinitialize adjusted envir. values
            PRWC(:, :) = PRW(:, :)  ! when iteration criterium not attained
            PRCC(:, :) = PRC(:, :)
            PRIC(:, :) = PRI(:, :)
            PTHC(:, :) = PTH(:, :)
        ENDWHERE
!
!
!        6. Check for mass conservation, i.e. ZWORK1 > 1.E-2
!           If mass is not conserved, the convective tendencies
!           automatically become zero.
!           ----------------------------------------------------
!
        do JI = 1, IIE
            JK = KCTL(JI)
            ZWORK1(JI) = PUDR(JI, JK) * PDPRES(JI, JK) / (PLMASS(JI, JK) + .1) &
                         - PWSUB(JI, JK)
        enddo
        WHERE(GWORK1(:) .and. ABS(ZWORK1(:)) - .01 > 0.)
            GWORK1(:) = .false.
            PTIMEC(:) = 1.E-1
            ZWORK5(:) = 0.
        ENDWHERE
        do JK = IKB, IKE
            PWSUB(:, JK) = PWSUB(:, JK) * ZWORK5(:)
        enddo
        GWORK4(:, 1:IKB) = .false.
        GWORK4(:, IKE:IKS) = .false.
!
        ITSTEP(:) = INT(PTIMEC(:) / ZTIMEC(:)) + 1
        ZTIMEC(:) = PTIMEC(:) / REAL(ITSTEP(:)) ! adjust  fractional time step
        ! to be an integer multiple of PTIMEC
        ZTIMC(:, :) = SPREAD(ZTIMEC(:), DIM=2, NCOPIES=IKS)
        ICOUNT(:) = 0
!
!
!
        KFTSTEPS = MAXVAL(ITSTEP(:))
        do JSTEP = 1, KFTSTEPS ! Enter the fractional time step loop here
!
            ICOUNT(:) = ICOUNT(:) + 1
!
            GWORK3(:) = ITSTEP(:) >= ICOUNT(:) .and. GWORK1(:)
!
!
!*       7.     Assign enthalpy and r_w values at the top and bottom of each
!               layer based on the sign of w
!               ------------------------------------------------------------
!
            ZTHMFIN(:, :) = 0.
            ZRWMFIN(:, :) = 0.
            ZRCMFIN(:, :) = 0.
            ZRIMFIN(:, :) = 0.
            ZTHMFOUT(:, :) = 0.
            ZRWMFOUT(:, :) = 0.
            ZRCMFOUT(:, :) = 0.
            ZRIMFOUT(:, :) = 0.
!
            do JK = IKB + 1, JKMAX
                do JI = 1, IIE
                    GWORK4(JI, JK) = GWORK3(JI) .and. JK <= KCTL(JI)
                enddo
                JKP = MAX(IKB + 1, JK - 1)
                do JI = 1, IIE
                if(GWORK3(JI)) then
!
                    ZWORK1(JI) = SIGN(1., ZOMG(JI, JK))
                    ZWORK2(JI) = 0.5 * (1.+ZWORK1(JI))
                    ZWORK1(JI) = 0.5 * (1.-ZWORK1(JI))
                    ZTHMFIN(JI, JK) = -ZOMG(JI, JK) * ZTHLC(JI, JKP) * ZWORK1(JI)
                    ZTHMFOUT(JI, JK) = ZOMG(JI, JK) * ZTHLC(JI, JK) * ZWORK2(JI)
                    ZRWMFIN(JI, JK) = -ZOMG(JI, JK) * PRWC(JI, JKP) * ZWORK1(JI)
                    ZRWMFOUT(JI, JK) = ZOMG(JI, JK) * PRWC(JI, JK) * ZWORK2(JI)
                    ZRCMFIN(JI, JK) = -ZOMG(JI, JK) * PRCC(JI, JKP) * ZWORK1(JI)
                    ZRCMFOUT(JI, JK) = ZOMG(JI, JK) * PRCC(JI, JK) * ZWORK2(JI)
                    ZRIMFIN(JI, JK) = -ZOMG(JI, JK) * PRIC(JI, JKP) * ZWORK1(JI)
                    ZRIMFOUT(JI, JK) = ZOMG(JI, JK) * PRIC(JI, JK) * ZWORK2(JI)
                endif
                enddo
                do JI = 1, IIE
                if(GWORK3(JI)) then
                    ZTHMFIN(JI, JKP) = ZTHMFIN(JI, JKP) + ZTHMFOUT(JI, JK) * ZWORK2(JI)
                    ZTHMFOUT(JI, JKP) = ZTHMFOUT(JI, JKP) + ZTHMFIN(JI, JK) * ZWORK1(JI)
                    ZRWMFIN(JI, JKP) = ZRWMFIN(JI, JKP) + ZRWMFOUT(JI, JK) * ZWORK2(JI)
                    ZRWMFOUT(JI, JKP) = ZRWMFOUT(JI, JKP) + ZRWMFIN(JI, JK) * ZWORK1(JI)
                    ZRCMFIN(JI, JKP) = ZRCMFIN(JI, JKP) + ZRCMFOUT(JI, JK) * ZWORK2(JI)
                    ZRCMFOUT(JI, JKP) = ZRCMFOUT(JI, JKP) + ZRCMFIN(JI, JK) * ZWORK1(JI)
                    ZRIMFIN(JI, JKP) = ZRIMFIN(JI, JKP) + ZRIMFOUT(JI, JK) * ZWORK2(JI)
                    ZRIMFOUT(JI, JKP) = ZRIMFOUT(JI, JKP) + ZRIMFIN(JI, JK) * ZWORK1(JI)
!
                endif
                enddo
            enddo
!
            WHERE(GWORK4(:, :))
!
!******************************************************************************
!
!*       8.     Update the environmental values of enthalpy and r_w at each level
!               NOTA: These are the MAIN EQUATIONS of the scheme
!               -----------------------------------------------------------------
!
!
                ZTHLC(:, :) = ZTHLC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                              ZTHMFIN(:, :) + PUDR(:, :) * PUTHL(:, :) &
                              - ZTHMFOUT(:, :) - PUER(:, :) * PTHL(:, :))
                PRWC(:, :) = PRWC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRWMFIN(:, :) + PUDR(:, :) * PURW(:, :) &
                             - ZRWMFOUT(:, :) - PUER(:, :) * PRW(:, :))
                PRCC(:, :) = PRCC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRCMFIN(:, :) + PUDR(:, :) * PURC(:, :) - ZRCMFOUT(:, :) - &
                             PUER(:, :) * PRC(:, :))
                PRIC(:, :) = PRIC(:, :) + ZTIMC(:, :) / PLMASS(:, :) * ( &
                             ZRIMFIN(:, :) + PUDR(:, :) * PURI(:, :) - ZRIMFOUT(:, :) - &
                             PUER(:, :) * PRI(:, :))
!
!
!******************************************************************************
!
            ENDWHERE
!
        enddo ! Exit the fractional time step loop
!
!
!*          10.    Compute final linearized value of theta envir.
!                  ----------------------------------------------
!
        do JK = IKB + 1, JKMAX
            do JI = 1, IIE
            if(GWORK1(JI) .and. JK <= KCTL(JI)) then
                ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
                ZCPH(JI) = XCPD + PRWC(JI, JK) * XCPV
                ZWORK2(JI) = PTH(JI, JK) / ZPI(JI)  ! first temperature estimate
                ZLV(JI) = XLVTT + (XCPV - XCL) * (ZWORK2(JI) - XTT)
                ZLS(JI) = XLVTT + (XCPV - XCI) * (ZWORK2(JI) - XTT)
                ! final linearized temperature
                ZWORK2(JI) = (ZTHLC(JI, JK) + ZLV(JI) * PRCC(JI, JK) + ZLS(JI) * PRIC(JI, JK) &
                              - (1.+PRWC(JI, JK)) * XG * PZ(JI, JK)) / ZCPH(JI)
                ZWORK2(JI) = MAX(180., MIN(340., ZWORK2(JI)))
                PTHC(JI, JK) = ZWORK2(JI) * ZPI(JI) ! final adjusted envir. theta
            endif
            enddo
        enddo
!
!
!*         11.     Compute new cloud ( properties at new LCL )
!                     NOTA: The computations are very close to
!                           that in routine TRIGGER_FUNCT
!                  ---------------------------------------------
!
        call CONVECT_CLOSURE_THRVLCL(KLON, KLEV, &
                                     PPRES, PTHC, PRWC, PZ, GWORK1, &
                                     ZTHLCL, ZRVLCL, ZZLCL, ZTLCL, ZTELCL, &
                                     ILCL, KDPL, KPBL)
!
!
        ZTLCL(:) = MAX(230., MIN(335., ZTLCL(:)))  ! set some overflow bounds
        ZTELCL(:) = MAX(230., MIN(335., ZTELCL(:)))
        ZTHLCL(:) = MAX(230., MIN(345., ZTHLCL(:)))
        ZRVLCL(:) = MAX(0., MIN(1., ZRVLCL(:)))
!
!
!*         12.    Compute adjusted CAPE
!                 ---------------------
!
        ZCAPE(:) = 0.
        ZPI(:) = ZTHLCL(:) / ZTLCL(:)
        ZPI(:) = MAX(0.95, MIN(1.5, ZPI(:)))
        ZWORK1(:) = XP00 / ZPI(:)**ZCPORD ! pressure at LCL
!
        call CONVECT_SATMIXRATIO(KLON, ZWORK1, ZTELCL, ZWORK3, ZLV, ZLS, ZCPH)
        ZWORK3(:) = MIN(.1, MAX(0., ZWORK3(:)))
!
        ! compute theta_e updraft undilute
        ZTHEUL(:) = ZTLCL(:) * ZPI(:)**(1.-0.28 * ZRVLCL(:)) &
                    * EXP((3374.6525 / ZTLCL(:) - 2.5403) &
                          * ZRVLCL(:) * (1.+0.81 * ZRVLCL(:)))
!
        ! compute theta_e saturated environment at LCL
        ZTHES1(:) = ZTELCL(:) * ZPI(:)**(1.-0.28 * ZWORK3(:)) &
                    * EXP((3374.6525 / ZTELCL(:) - 2.5403) &
                          * ZWORK3(:) * (1.+0.81 * ZWORK3(:)))
!
        do JK = MINVAL(ILCL(:)), JKMAX
            JKP = JK - 1
            do JI = 1, IIE
                ZWORK4(JI) = 1.
                if(JK == ILCL(JI)) ZWORK4(JI) = 0.
!
                ! compute theta_e saturated environment and adjusted values
                ! of theta
!
                GWORK3(JI) = JK >= ILCL(JI) .and. JK <= KCTL(JI) .and. GWORK1(JI)
!
                ZPI(JI) = (XP00 / PPRES(JI, JK))**ZRDOCP
                ZWORK2(JI) = PTHC(JI, JK) / ZPI(JI)
            enddo
!
            call CONVECT_SATMIXRATIO(KLON, PPRES(:, JK), ZWORK2, ZWORK3, ZLV, ZLS, ZCPH)
!
!
            do JI = 1, IIE
                if(GWORK3(JI)) then
                    ZTHES2(JI) = ZWORK2(JI) * ZPI(JI)**(1.-0.28 * ZWORK3(JI)) &
                                 * EXP((3374.6525 / ZWORK2(JI) - 2.5403) &
                                       * ZWORK3(JI) * (1.+0.81 * ZWORK3(JI)))
!
                    ZWORK3(JI) = PZ(JI, JK) - PZ(JI, JKP) * ZWORK4(JI) - &
                                 (1.-ZWORK4(JI)) * ZZLCL(JI)    ! level thickness
                    ZWORK1(JI) = (2.*ZTHEUL(JI)) / (ZTHES1(JI) + ZTHES2(JI)) - 1.
                    ZCAPE(JI) = ZCAPE(JI) + XG * ZWORK3(JI) * MAX(0., ZWORK1(JI))
                    ZTHES1(JI) = ZTHES2(JI)
                endif
            enddo
        enddo
!
!
!*         13.     Determine mass adjustment factor knowing how much
!                  CAPE has been removed.
!                  -------------------------------------------------
!
        WHERE(GWORK1(:))
            ZWORK1(:) = MAX(PCAPE(:) - ZCAPE(:), 0.2 * PCAPE(:))
            ZWORK2(:) = ZCAPE(:) / (PCAPE(:) + 1.E-8)
!
            GWORK1(:) = ZWORK2(:) > 0.2 .OR. ZCAPE(:) == 0. ! mask for adjustment
        ENDWHERE
!
        WHERE(ZCAPE(:) == 0. .and. GWORK1(:)) ZADJ(:) = ZADJ(:) * 0.5
        WHERE(ZCAPE(:) /= 0. .and. GWORK1(:)) &
            ZADJ(:) = ZADJ(:) * XSTABC * PCAPE(:) / (ZWORK1(:) + 1.E-8)
        ZADJ(:) = MIN(ZADJ(:), ZADJMAX(:))
!
!
!*         13.     Adjust mass flux by the factor ZADJ to converge to
!                  specified degree of stabilization
!                 ----------------------------------------------------
!
        call CONVECT_CLOSURE_ADJUST_SHAL(KLON, KLEV, ZADJ, &
                                         PUMF, ZUMF, PUER, ZUER, PUDR, ZUDR)
!
!
        if(COUNT(GWORK1(:)) == 0) EXIT ! exit big adjustment iteration loop
        ! when all columns have reached
        ! desired degree of stabilization.
!
    enddo ! end of big adjustment iteration loop
!
!
    ! skip adj. total water array  to water vapor
    do JK = IKB, IKE
        PRWC(:, JK) = MAX(0., PRWC(:, JK) - PRCC(:, JK) - PRIC(:, JK))
    enddo
!
!
ENDsubroutine CONVECT_CLOSURE_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_TRIGGER_SHAL
!     #################
!
    INTERFACE
!
        subroutine CONVECT_TRIGGER_SHAL(KLON, KLEV, &
                                        PPRES, PTH, PTHV, PTHES, &
                                        PRV, PW, PZ, PDXDY, PTKECLS, &
                                        PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, &
                                        PTHVELCL, KLCL, KDPL, KPBL, OTRIG)
!
            INTEGER, INTENT(IN)                   :: KLON      ! horizontal loop index
            INTEGER, INTENT(IN)                   :: KLEV      ! vertical loop index
            REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY     ! grid area
            REAL, DIMENSION(KLON), INTENT(IN) :: PTKECLS   ! TKE CLS
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH, PTHV ! theta, theta_v
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES     ! envir. satur. theta_e
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV       ! vapor mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES     ! pressure
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ        ! height of grid point (m)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PW        ! vertical velocity
!
            REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL    ! theta at LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL     ! temp. at LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL    ! vapor mixing ratio at  LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PWLCL     ! parcel velocity at  LCL
            REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL     ! height at LCL (m)
            REAL, DIMENSION(KLON), INTENT(OUT):: PTHVELCL  ! environm. theta_v at LCL (K)
            LOGICAL, DIMENSION(KLON), INTENT(OUT):: OTRIG     ! logical mask for convection
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KLCL    ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KDPL    ! contains vert. index of DPL
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KPBL    ! contains index of source layer top
!
        ENDsubroutine CONVECT_TRIGGER_SHAL
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_TRIGGER_SHAL
!     ########################################################################
subroutine CONVECT_TRIGGER_SHAL(KLON, KLEV, &
                                PPRES, PTH, PTHV, PTHES, &
                                PRV, PW, PZ, PDXDY, PTKECLS, &
                                PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, &
                                PTHVELCL, KLCL, KDPL, KPBL, OTRIG)
!     ########################################################################
!
!!**** Determine convective columns as well as the cloudy values of theta,
!!     and qv at the lifting condensation level (LCL)
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine convective columns
!!
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!      What we look for is the undermost unstable level at each grid point.
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_SATMIXRATIO
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! Reference pressure
!!          XRD, XRV           ! Gaz  constants for dry air and water vapor
!!          XCPD               ! Cpd (dry air)
!!          XTT                ! triple point temperature
!!          XBETAW, XGAMW      ! constants for vapor saturation pressure
!!
!!      Module MODD_CONVPAR
!!          XA25               ! reference grid area
!!          XZLCL              ! maximum height difference between
!!                             ! the surface and the DPL
!!          XZPBL              ! minimum mixed layer depth to sustain convection
!!          XCDEPTH            ! minimum necessary cloud depth
!!          XCDEPTH_D          ! maximum allowed cloud depth
!!          XDTPERT            ! add small Temp peturbation
!!          XNHGAM             ! coefficient for buoyancy term in w eq.
!!                             ! accounting for nh-pressure
!!          XAW, XBW, XATPERT, XBTPERT
!!
!!      Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book2 of documentation ( routine TRIGGER_FUNCT)
!!      Fritsch and Chappell (1980), J. Atm. Sci., Vol. 37, 1722-1761.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  20/03/97  Select first departure level
!!                            that produces a cloud thicker than XCDEPTH
!!   F. Bouyssel    05/11/08  Modifications for reproductibility
!!   E. Bazile      05/05/09  Modifications for using really W and the tempe.
!!                            perturbation function of the TKE.
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR_SHAL
    USE MODD_CONVPAREXT
    USE MODI_CONVECT_SATMIXRATIO
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                   :: KLON      ! horizontal loop index
    INTEGER, INTENT(IN)                   :: KLEV      ! vertical loop index
    REAL, DIMENSION(KLON), INTENT(IN) :: PDXDY     ! grid area
    REAL, DIMENSION(KLON), INTENT(IN) :: PTKECLS   ! TKE CLS
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTH, PTHV ! theta, theta_v
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES     ! envir. satur. theta_e
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRV       ! vapor mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES     ! pressure
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ        ! height of grid point (m)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PW        ! vertical velocity
!
    REAL, DIMENSION(KLON), INTENT(OUT):: PTHLCL    ! theta at LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PTLCL     ! temp. at LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PRVLCL    ! vapor mixing ratio at  LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PWLCL     ! parcel velocity at  LCL
    REAL, DIMENSION(KLON), INTENT(OUT):: PZLCL     ! height at LCL (m)
    REAL, DIMENSION(KLON), INTENT(OUT):: PTHVELCL  ! environm. theta_v at LCL (K)
    LOGICAL, DIMENSION(KLON), INTENT(OUT):: OTRIG     ! logical mask for convection
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KLCL    ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KDPL    ! contains vert. index of DPL
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KPBL    ! contains index of source layer top
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: JKK, JK, JKP, JKM, JKDL, JL, JKT, JT! vertical loop index
    INTEGER :: JI                                  ! horizontal loop index
    INTEGER :: IIE, IKB, IKE                       ! horizontal + vertical loop bounds
    real :: ZEPS, ZEPSA                         ! R_d / R_v, R_v / R_d
    real :: ZCPORD, ZRDOCP                      ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON) :: ZTHLCL, ZTLCL, ZRVLCL, & ! locals for PTHLCL,PTLCL
                             ZWLCL, ZZLCL, ZTHVELCL  ! PRVLCL, ....
    INTEGER, DIMENSION(KLON) :: IDPL, IPBL, ILCL      ! locals for KDPL, ...
    REAL, DIMENSION(KLON) :: ZPLCL    ! pressure at LCL
    REAL, DIMENSION(KLON) :: ZZDPL    ! height of DPL
    REAL, DIMENSION(KLON) :: ZTHVLCL  ! theta_v at LCL = mixed layer value
    REAL, DIMENSION(KLON) :: ZTMIX    ! mixed layer temperature
    REAL, DIMENSION(KLON) :: ZEVMIX   ! mixed layer water vapor pressure
    REAL, DIMENSION(KLON) :: ZDPTHMIX, ZPRESMIX ! mixed layer depth and pressure
    REAL, DIMENSION(KLON) :: ZCAPE    ! convective available energy (m^2/s^2/g)
    REAL, DIMENSION(KLON) :: ZCAP     ! pseudo fro CAPE
    REAL, DIMENSION(KLON) :: ZTHEUL   ! updraft equiv. pot. temperature (K)
    REAL, DIMENSION(KLON) :: ZLV, ZCPH! specific heats of vaporisation, dry air
    REAL, DIMENSION(KLON) :: ZDP      ! pressure between LCL and model layer
    REAL, DIMENSION(KLON) :: ZTOP, ZTOPP     ! estimated cloud top (m)
    REAL, DIMENSION(KLON) :: ZWORK1, ZWORK2, ZWORK3    ! work arrays
    LOGICAL, DIMENSION(KLON) :: GTRIG, GTRIG2          ! local arrays for OTRIG
    LOGICAL, DIMENSION(KLON) :: GWORK1                 ! work array
!
!
!-------------------------------------------------------------------------------
!
!*       0.3    Compute array bounds
!               --------------------
!
    IIE = KLON
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
!
!
!*       1.     Initialize local variables
!               --------------------------
!
    ZEPS = XRD / XRV
    ZEPSA = XRV / XRD
    ZCPORD = XCPD / XRD
    ZRDOCP = XRD / XCPD
    OTRIG(:) = .false.
    IDPL(:) = KDPL(:)
    IPBL(:) = KPBL(:)
    ILCL(:) = KLCL(:)
    PWLCL(:) = 0.
    ZWLCL(:) = 0.
    PTHLCL(:) = 1.
    PTHVELCL(:) = 1.
    PTLCL(:) = 1.
    PRVLCL(:) = 0.
    PWLCL(:) = 0.
    PZLCL(:) = PZ(:, IKB)
    ZZDPL(:) = PZ(:, IKB)
    GTRIG2(:) = .true.
!
!
!
!       1.     Determine highest necessary loop test layer
!              -------------------------------------------
!
    JT = IKE - 2
! FBy
!do JK = IKB + 1, IKE - 2
!  if ( PZ(1,JK) - PZ(1,IKB) < 5.E3 ) JT = JK
!end do
!
!
!*       2.     Enter loop for convection test
!               ------------------------------
!
    JKP = MINVAL(IDPL(:)) + 1
    JKT = JT
    JKT = JKP ! do not allow for looping anymore, test only for surface mixed layer
    do JKK = JKP, JKT
!
        GWORK1(:) = ZZDPL(:) - PZ(:, IKB) < XZLCL
        ! we exit the trigger test when the center of the mixed layer is more
        ! than 1500 m  above soil level.
        WHERE(GWORK1(:))
            ZDPTHMIX(:) = 0.
            ZPRESMIX(:) = 0.
            ZTHLCL(:) = 0.
            ZRVLCL(:) = 0.
            ZZDPL(:) = PZ(:, JKK)
            IDPL(:) = JKK
        ENDWHERE
!
!
!*       3.     Construct a mixed layer of at least 50 hPa (XZPBL)
!               ------------------------------------------
!
        do JK = JKK, IKE - 1
            JKM = JK + 1
            do JI = 1, IIE
                if(GWORK1(JI) .and. ZDPTHMIX(JI) < XZPBL) then
                    IPBL(JI) = JK
                    ZWORK1(JI) = PPRES(JI, JK) - PPRES(JI, JKM)
                    ZDPTHMIX(JI) = ZDPTHMIX(JI) + ZWORK1(JI)
                    ZPRESMIX(JI) = ZPRESMIX(JI) + PPRES(JI, JK) * ZWORK1(JI)
                    ZTHLCL(JI) = ZTHLCL(JI) + PTH(JI, JK) * ZWORK1(JI)
                    ZRVLCL(JI) = ZRVLCL(JI) + PRV(JI, JK) * ZWORK1(JI)
                endif
            enddo
            if(MINVAL(ZDPTHMIX(:)) >= XZPBL) EXIT
        enddo
!
!
        WHERE(GWORK1(:))
!
            ZPRESMIX(:) = ZPRESMIX(:) / ZDPTHMIX(:)
            ZTHLCL(:) = ZTHLCL(:) / ZDPTHMIX(:) + &
          & (XATPERT * MAX(3., PTKECLS(:)) / XCPD + XBTPERT) * XDTPERT ! add small Temp Perturb.
            ZRVLCL(:) = ZRVLCL(:) / ZDPTHMIX(:)
            ZTHVLCL(:) = ZTHLCL(:) * (1.+ZEPSA * ZRVLCL(:)) &
                         / (1.+ZRVLCL(:))
!
!*       4.1    Use an empirical direct solution ( Bolton formula )
!               to determine temperature and pressure at LCL.
!               Nota: the adiabatic saturation temperature is not
!                     equal to the dewpoint temperature
!               ----------------------------------------------------
!
!
            ZTMIX(:) = ZTHLCL(:) * (ZPRESMIX(:) / XP00)**ZRDOCP
            ZEVMIX(:) = ZRVLCL(:) * ZPRESMIX(:) / (ZRVLCL(:) + ZEPS)
            ZEVMIX(:) = MAX(1.E-8, ZEVMIX(:))
            ZWORK1(:) = LOG(ZEVMIX(:) / 613.3)
            ! dewpoint temperature
            ZWORK1(:) = (4780.8 - 32.19 * ZWORK1(:)) / (17.502 - ZWORK1(:))
            ! adiabatic saturation temperature
            ZTLCL(:) = ZWORK1(:) - (.212 + 1.571E-3 * (ZWORK1(:) - XTT) &
                                    - 4.36E-4 * (ZTMIX(:) - XTT)) * (ZTMIX(:) - ZWORK1(:))
            ZTLCL(:) = MIN(ZTLCL(:), ZTMIX(:))
            ZPLCL(:) = XP00 * (ZTLCL(:) / ZTHLCL(:))**ZCPORD
!
        ENDWHERE
!
!
!*       4.2    Correct ZTLCL in order to be completely consistent
!               with MNH saturation formula
!               ---------------------------------------------
!
        call CONVECT_SATMIXRATIO(KLON, ZPLCL, ZTLCL, ZWORK1, ZLV, ZWORK2, ZCPH)
        WHERE(GWORK1(:))
            ZWORK2(:) = ZWORK1(:) / ZTLCL(:) * (XBETAW / ZTLCL(:) - XGAMW) ! dr_sat/dT
            ZWORK2(:) = (ZWORK1(:) - ZRVLCL(:)) / &
                        (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
            ZTLCL(:) = ZTLCL(:) - ZLV(:) / ZCPH(:) * ZWORK2(:)
!
        ENDWHERE
!
!
!*       4.3    If ZRVLCL = PRVMIX is oversaturated set humidity
!               and temperature to saturation values.
!               ---------------------------------------------
!
        call CONVECT_SATMIXRATIO(KLON, ZPRESMIX, ZTMIX, ZWORK1, ZLV, ZWORK2, ZCPH)
        WHERE(GWORK1(:) .and. ZRVLCL(:) > ZWORK1(:))
            ZWORK2(:) = ZWORK1(:) / ZTMIX(:) * (XBETAW / ZTMIX(:) - XGAMW) ! dr_sat/dT
            ZWORK2(:) = (ZWORK1(:) - ZRVLCL(:)) / &
                        (1.+ZLV(:) / ZCPH(:) * ZWORK2(:))
            ZTLCL(:) = ZTMIX(:) - ZLV(:) / ZCPH(:) * ZWORK2(:)
            ZRVLCL(:) = ZRVLCL(:) - ZWORK2(:)
            ZPLCL(:) = ZPRESMIX(:)
            ZTHLCL(:) = ZTLCL(:) * (XP00 / ZPLCL(:))**ZRDOCP
            ZTHVLCL(:) = ZTHLCL(:) * (1.+ZEPSA * ZRVLCL(:)) &
                         / (1.+ZRVLCL(:))
        ENDWHERE
!
!
!*        5.1   Determine  vertical loop index at the LCL and DPL
!               --------------------------------------------------
!
        do JK = JKK, IKE - 1
            do JI = 1, IIE
                if(ZPLCL(JI) <= PPRES(JI, JK) .and. GWORK1(JI)) ILCL(JI) = JK + 1
            enddo
        enddo
!
!
!*        5.2   Estimate height and environm. theta_v at LCL
!               --------------------------------------------------
!
        do JI = 1, IIE
            JK = ILCL(JI)
            JKM = JK - 1
            ZDP(JI) = LOG(ZPLCL(JI) / PPRES(JI, JKM)) / &
                      LOG(PPRES(JI, JK) / PPRES(JI, JKM))
            ZWORK1(JI) = PTHV(JI, JKM) + (PTHV(JI, JK) - PTHV(JI, JKM)) * ZDP(JI)
            ! we compute the precise value of the LCL
            ! The precise height is between the levels ILCL and ILCL-1.
            ZWORK2(JI) = PZ(JI, JKM) + (PZ(JI, JK) - PZ(JI, JKM)) * ZDP(JI)
        enddo
        WHERE(GWORK1(:))
            ZTHVELCL(:) = ZWORK1(:)
            ZZLCL(:) = ZWORK2(:)
        ENDWHERE
!
!
!*       6.     Check to see if cloud is bouyant
!               --------------------------------
!
!*      6.1    Compute grid scale vertical velocity perturbation term ZWORK1
!               -------------------------------------------------------------
!
!            !  normalize w grid scale to a 25 km refer. grid
!    do JI = 1, IIE
!       JK  = ILCL(JI)
!       JKM = JK - 1
!       ZWORK1(JI) =  ( PW(JI,JKM)  + ( PW(JI,JK) - PW(JI,JKM) ) * ZDP(JI) )  &
!                          * SQRT( PDXDY(JI) / XA25 )
!                         - 0.02 * ZZLCL(JI) / XZLCL ! avoid spurious convection
!    end do
!            ! compute sign of normalized grid scale w
!       ZWORK2(:) = SIGN( 1., ZWORK1(:) )
!       ZWORK1(:) = XWTRIG * ZWORK2(:) * ABS( ZWORK1(:) ) ** 0.333       &
!                          * ( XP00 / ZPLCL(:) ) ** ZRDOCP
!
!*       6.2    Compute parcel vertical velocity at LCL
!               ---------------------------------------
!
!    do JI = 1, IIE
!       JKDL = IDPL(JI)
!       ZWORK3(JI) = XG * ZWORK1(JI) * ( ZZLCL(JI) - PZ(JI,JKDL) )       &
!                      / ( PTHV(JI,JKDL) + ZTHVELCL(JI) )
!    end do
!    WHERE( GWORK1(:) )
!      ZWLCL(:)  = 1. + .5 * ZWORK2(:) * SQRT( ABS( ZWORK3(:) ) )
!      GTRIG(:)  = ZTHVLCL(:) - ZTHVELCL(:) + ZWORK1(:) > 0. .and.       &
!                  ZWLCL(:) > 0.
!    END WHERE
        ZWLCL(:) = XAW * MAX(0., PW(:, IKB)) + XBW
!
!
!*       6.3    Look for parcel that produces sufficient cloud depth.
!               The cloud top is estimated as the level where the CAPE
!               is smaller  than a given value (based on vertical velocity eq.)
!               --------------------------------------------------------------
!
        ZTHEUL(:) = ZTLCL(:) * (ZTHLCL(:) / ZTLCL(:)) &
                    **(1.-0.28 * ZRVLCL(:)) &
                    * EXP((3374.6525 / ZTLCL(:) - 2.5403) * &
                          ZRVLCL(:) * (1.+0.81 * ZRVLCL(:)))
!
        ZCAPE(:) = 0.
        ZCAP(:) = 0.
        ZTOP(:) = 0.
        ZTOPP(:) = 0.
        ZWORK3(:) = 0.
        JKM = MINVAL(ILCL(:))
        do JL = JKM, JT
            JK = JL + 1
            do JI = 1, IIE
                ZWORK1(JI) = (2.*ZTHEUL(JI) / &
                              (PTHES(JI, JK) + PTHES(JI, JL)) - 1.) * (PZ(JI, JK) - PZ(JI, JL))
                if(JL < ILCL(JI)) ZWORK1(JI) = 0.
                ZCAPE(JI) = ZCAPE(JI) + XG * MAX(1., ZWORK1(JI))
                ZCAP(JI) = ZCAP(JI) + ZWORK1(JI)
                ZWORK2(JI) = XNHGAM * XG * ZCAP(JI) + 1.05 * ZWLCL(JI) * ZWLCL(JI)
                ! the factor 1.05 takes entrainment into account
                ZWORK2(JI) = SIGN(1., ZWORK2(JI))
                ZWORK3(JI) = ZWORK3(JI) + MIN(0., ZWORK2(JI))
                ZWORK3(JI) = MAX(-1., ZWORK3(JI))
                ! Nota, the factors ZWORK2 and ZWORK3 are only used to avoid
                ! if and goto statements, the difficulty is to extract only
                ! the level where the criterium is first fullfilled
                ZTOPP(JI) = ZTOP(JI)
                ZTOP(JI) = PZ(JI, JL)*.5 * (1.+ZWORK2(JI)) * (1.+ZWORK3(JI)) + &
                           ZTOP(JI)*.5 * (1.-ZWORK2(JI))
                ZTOP(JI) = MAX(ZTOP(JI), ZTOPP(JI))
                ZTOPP(JI) = ZTOP(JI)
            enddo
        enddo
!
!
        ZWORK2(:) = ZTOP(:) - ZZLCL(:)
        ! WHERE( ZWORK2(:)  .GE. XCDEPTH  .and. ZWORK2(:) < XCDEPTH_D .and. GTRIG2(:) &
        WHERE(ZWORK2(:) >= XCDEPTH .and. GTRIG2(:) &
              .and. ZCAPE(:) > 10.)
            GTRIG2(:) = .false.
            OTRIG(:) = .true.
            ! OTRIG(:)    = GTRIG(:)     ! we  select the first departure level
            PTHLCL(:) = ZTHLCL(:)    ! that gives sufficient cloud depth
            PRVLCL(:) = ZRVLCL(:)
            PTLCL(:) = ZTLCL(:)
            PWLCL(:) = ZWLCL(:)
            PZLCL(:) = ZZLCL(:)
            PTHVELCL(:) = ZTHVELCL(:)
            KDPL(:) = IDPL(:)
            KPBL(:) = IPBL(:)
            KLCL(:) = ILCL(:)
        ENDWHERE
!
    enddo
!
!
ENDsubroutine CONVECT_TRIGGER_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     #################
MODULE MODI_CONVECT_UPDRAFT_SHAL
!     #################
!
    INTERFACE
!
        subroutine CONVECT_UPDRAFT_SHAL(KLON, KLEV, &
                                        KICE, PPRES, PDPRES, PZ, PTHL, PTHV, PTHES, PRW, &
                                        PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, PTHVELCL, &
                                        PMFLCL, OTRIG, KLCL, KDPL, KPBL, &
                                        PUMF, PUER, PUDR, PUTHL, PUTHV, PURW, &
                                        PURC, PURI, PCAPE, KCTL, KETL)
!
            INTEGER, INTENT(IN)                    :: KLON  ! horizontal dimension
            INTEGER, INTENT(IN)                    :: KLEV  ! vertical dimension
            INTEGER, INTENT(IN)                    :: KICE  ! flag for ice ( 1 = yes,
            !                0 = no ice )
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL  ! grid scale enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHV  ! grid scale theta_v
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
            ! mixing ratio
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (P)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
            ! bottom and top of layer (Pa)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTHLCL ! theta at LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PTLCL  ! temp. at LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PRVLCL ! vapor mixing ratio at  LCL
            REAL, DIMENSION(KLON), INTENT(IN) :: PWLCL  ! parcel velocity at LCL (m/s)
            REAL, DIMENSION(KLON), INTENT(IN) :: PMFLCL ! cloud  base unit mass flux
            ! (kg/s)
            REAL, DIMENSION(KLON), INTENT(IN) :: PZLCL  ! height at LCL (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTHVELCL  ! environm. theta_v at LCL (K)
            LOGICAL, DIMENSION(KLON), INTENT(INOUT):: OTRIG! logical mask for convection
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! contains vert. index of LCL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! contains vert. index of DPL
            INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   !  " vert. index of source layertop
!
!
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KCTL   ! contains vert. index of CTL
            INTEGER, DIMENSION(KLON), INTENT(OUT):: KETL   ! contains vert. index of        &
            !equilibrium (zero buoyancy) level
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUMF  ! updraft mass flux (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUER  ! updraft entrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUDR  ! updraft detrainment (kg/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHL ! updraft enthalpy (J/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHV ! updraft theta_v (K)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURW  ! updraft total water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURC  ! updraft cloud water (kg/kg)
            REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURI  ! updraft cloud ice   (kg/kg)
            REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE  ! available potent. energy
!
        ENDsubroutine CONVECT_UPDRAFT_SHAL
!
    ENDINTERFACE
!
ENDMODULE MODI_CONVECT_UPDRAFT_SHAL
!    ###############################################################################
subroutine CONVECT_UPDRAFT_SHAL(KLON, KLEV, &
                                KICE, PPRES, PDPRES, PZ, PTHL, PTHV, PTHES, PRW, &
                                PTHLCL, PTLCL, PRVLCL, PWLCL, PZLCL, PTHVELCL, &
                                PMFLCL, OTRIG, KLCL, KDPL, KPBL, &
                                PUMF, PUER, PUDR, PUTHL, PUTHV, PURW, &
                                PURC, PURI, PCAPE, KCTL, KETL)
!    ###############################################################################
!
!!**** Compute updraft properties from DPL to CTL.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine updraft properties
!!      ( mass flux, thermodynamics, precipitation )
!!
!!
!!**  METHOD
!!    ------
!!      Computations are done at every model level starting from bottom.
!!      The use of masks allows to optimise the inner loops (horizontal loops).
!!
!!
!!
!!    EXTERNAL
!!    --------
!!     Routine CONVECT_MIXING_FUNCT
!!     Routine CONVECT_CONDENS
!!
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                 ! gravity constant
!!          XP00               ! reference pressure
!!          XRD, XRV           ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV, XCL    ! Cp of dry air, water vapor and liquid water
!!          XTT                ! triple point temperature
!!          XLVTT              ! vaporisation heat at XTT
!!
!!
!!      Module MODD_CONVPAR_SHAL
!!          XA25               ! reference grid area
!!          XCRAD              ! cloud radius
!!          XCDEPTH            ! minimum necessary cloud depth
!!          XENTR              ! entrainment constant
!!          XNHGAM             ! coefficient for buoyancy term in w eq.
!!                             ! accounting for nh-pressure
!!          XTFRZ1             ! begin of freezing interval
!!          XTFRZ2             ! begin of freezing interval
!!
!!     Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT     ! extra levels on the vertical boundaries
!!
!!    REFERENCE
!!    ---------
!!
!!      Book1,2 of documentation ( routine CONVECT_UPDRAFT)
!!      Kain and Fritsch, 1990, J. Atmos. Sci., Vol.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    07/11/95
!!   Last modified  10/12/97
!!   F. Bouyssel    05/11/08  Modifications for reproductibility
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAR_SHAL
    USE MODD_CONVPAREXT
    USE MODI_CONVECT_CONDENS
    USE MODI_CONVECT_MIXING_FUNCT
!
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
    INTEGER, INTENT(IN)                    :: KLON  ! horizontal dimension
    INTEGER, INTENT(IN)                    :: KLEV  ! vertical dimension
    INTEGER, INTENT(IN)                    :: KICE  ! flag for ice ( 1 = yes,
    !                0 = no ice )
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHL  ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHV  ! grid scale theta_v
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTHES ! grid scale saturated theta_e
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRW   ! grid scale total water
    ! mixing ratio
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPRES ! pressure (P)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PDPRES! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZ    ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTHLCL ! theta at LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PTLCL  ! temp. at LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PRVLCL ! vapor mixing ratio at  LCL
    REAL, DIMENSION(KLON), INTENT(IN) :: PWLCL  ! parcel velocity at LCL (m/s)
    REAL, DIMENSION(KLON), INTENT(IN) :: PMFLCL ! cloud  base unit mass flux
    ! (kg/s)
    REAL, DIMENSION(KLON), INTENT(IN) :: PZLCL  ! height at LCL (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTHVELCL  ! environm. theta_v at LCL (K)
    LOGICAL, DIMENSION(KLON), INTENT(INOUT):: OTRIG! logical mask for convection
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KLCL   ! contains vert. index of LCL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KDPL   ! contains vert. index of DPL
    INTEGER, DIMENSION(KLON), INTENT(IN) :: KPBL   !  " vert. index of source layertop
!
!
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KCTL   ! contains vert. index of CTL
    INTEGER, DIMENSION(KLON), INTENT(OUT):: KETL   ! contains vert. index of        &
    !equilibrium (zero buoyancy) level
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUMF  ! updraft mass flux (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUER  ! updraft entrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUDR  ! updraft detrainment (kg/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHL ! updraft enthalpy (J/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PUTHV ! updraft theta_v (K)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURW  ! updraft total water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURC  ! updraft cloud water (kg/kg)
    REAL, DIMENSION(KLON, KLEV), INTENT(OUT):: PURI  ! updraft cloud ice   (kg/kg)
    REAL, DIMENSION(KLON), INTENT(OUT):: PCAPE  ! available potent. energy
!
!*       0.2   Declarations of local variables :
!
    INTEGER :: IIE, IKB, IKE  ! horizontal and vertical loop bounds
    INTEGER :: JI             ! horizontal loop index
    INTEGER :: JK, JKP, JKM, JK1, JK2   ! vertical loop index
    real :: ZEPSA          ! R_v / R_d, C_pv / C_pd
    real :: ZRDOCP         ! C_pd / R_d, R_d / C_pd
!
    REAL, DIMENSION(KLON)    :: ZUT             ! updraft temperature (K)
    REAL, DIMENSION(KLON)    :: ZUW1, ZUW2      ! square of updraft vert.
    ! velocity at levels k and k+1
    REAL, DIMENSION(KLON)    :: ZE1, ZE2, ZD1, ZD2 ! fractional entrainm./detrain
    ! rates at levels k and k+1
    REAL, DIMENSION(KLON)    :: ZMIXF           ! critical mixed fraction
    REAL, DIMENSION(KLON)    :: ZCPH            ! specific heat C_ph
    REAL, DIMENSION(KLON)    :: ZLV, ZLS        ! latent heat of vaporis., sublim.
    REAL, DIMENSION(KLON)    :: ZURV            ! updraft water vapor at level k+1
    REAL, DIMENSION(KLON)    :: ZPI             ! Pi=(P0/P)**(Rd/Cpd)
    REAL, DIMENSION(KLON)    :: ZTHEUL          ! theta_e for undilute ascent
    REAL, DIMENSION(KLON)    :: ZWORK1, ZWORK2, ZWORK3, ZWORK4, ZWORK5, &
                                ZWORK6          ! work arrays
    INTEGER, DIMENSION(KLON) :: IWORK           ! wok array
    LOGICAL, DIMENSION(KLON) :: GWORK1, GWORK2, GWORK4, GWORK5
    ! work arrays
    LOGICAL, DIMENSION(KLON, KLEV) :: GWORK6     ! work array
!
!
!-------------------------------------------------------------------------------
!
!        0.3   Set loop bounds
!              ---------------
!
    IKB = 1 + JCVEXB
    IKE = KLEV - JCVEXT
    IIE = KLON
!
!
!*       1.     Initialize updraft properties and local variables
!               -------------------------------------------------
!
    ZEPSA = XRV / XRD
    ZRDOCP = XRD / XCPD
!
    PUMF(:, :) = 0.
    PUER(:, :) = 0.
    PUDR(:, :) = 0.
    PUTHL(:, :) = 0.
    PUTHV(:, :) = 0.
    PURW(:, :) = 0.
    PURC(:, :) = 0.
    PURI(:, :) = 0.
    ZUW1(:) = PWLCL(:) * PWLCL(:)
    ZUW2(:) = 0.
    ZE1(:) = 0.
    ZD1(:) = 0.
    PCAPE(:) = 0.
    KCTL(:) = IKB
    KETL(:) = KLCL(:)
    GWORK2(:) = .true.
    ZPI(:) = 1.
    ZWORK3(:) = 0.
    ZWORK4(:) = 0.
    ZWORK5(:) = 0.
    ZWORK6(:) = 0.
    GWORK1(:) = .false.
    GWORK4(:) = .false.
!
!
!*       1.1    Compute undilute updraft theta_e for CAPE computations
!               Bolton (1980) formula.
!               Define accurate enthalpy for updraft
!               -----------------------------------------------------
!
    ZTHEUL(:) = PTLCL(:) * (PTHLCL(:) / PTLCL(:))**(1.-0.28 * PRVLCL(:)) &
                * EXP((3374.6525 / PTLCL(:) - 2.5403) * &
                      PRVLCL(:) * (1.+0.81 * PRVLCL(:)))
!
!
    ZWORK1(:) = (XCPD + PRVLCL(:) * XCPV) * PTLCL(:) &
                + (1.+PRVLCL(:)) * XG * PZLCL(:)
!
!
!*       2.     Set updraft properties between DPL and LCL
!               ------------------------------------------
!
    JKP = MAXVAL(KLCL(:))
    JKM = MINVAL(KDPL(:))
    do JK = JKM, JKP
        do JI = 1, IIE
        if(JK >= KDPL(JI) .and. JK < KLCL(JI)) then
            PUMF(JI, JK) = PMFLCL(JI)
            PUTHL(JI, JK) = ZWORK1(JI)
            PUTHV(JI, JK) = PTHLCL(JI) * (1.+ZEPSA * PRVLCL(JI)) / &
                            (1.+PRVLCL(JI))
            PURW(JI, JK) = PRVLCL(JI)
        endif
        enddo
    enddo
!
!
!*       3.     Enter loop for updraft computations
!               ------------------------------------
!
    do JK = IKB + 1, IKE - 1
        ZWORK6(:) = 1.
        JKP = JK + 1
!
        GWORK4(:) = JK >= KLCL(:) - 1
        GWORK1(:) = GWORK4(:) .and. GWORK2(:) ! this mask is used to confine
        ! updraft computations between the LCL and the CTL
!
        WHERE(JK == KLCL(:) - 1) ZWORK6(:) = 0. ! factor that is used in buoyancy
        ! computation at first level above LCL
!
!
!*       4.     Estimate condensate, L_v L_i, Cph and theta_v at level k+1
!               ----------------------------------------------------------
!
        ZWORK1(:) = PURC(:, JK)
        ZWORK2(:) = PURI(:, JK)
        call CONVECT_CONDENS(KLON, KICE, PPRES(:, JKP), PUTHL(:, JK), PURW(:, JK), &
                             ZWORK1, ZWORK2, PZ(:, JKP), GWORK1, ZUT, ZURV, &
                             PURC(:, JKP), PURI(:, JKP), ZLV, ZLS, ZCPH)
!
!
        ZPI(:) = (XP00 / PPRES(:, JKP))**ZRDOCP
        WHERE(GWORK1(:))
!
            PUTHV(:, JKP) = ZPI(:) * ZUT(:) * (1.+ZEPSA * ZURV(:)) &
                            / (1.+PURW(:, JK))
!
!
!*       5.     Compute square of vertical velocity using entrainment
!               at level k
!               -----------------------------------------------------
!
            ZWORK3(:) = PZ(:, JKP) - PZ(:, JK) * ZWORK6(:) - &
                        (1.-ZWORK6(:)) * PZLCL(:)          ! level thickness
            ZWORK4(:) = PTHV(:, JK) * ZWORK6(:) + &
                        (1.-ZWORK6(:)) * PTHVELCL(:)
            ZWORK5(:) = 2.*ZUW1(:) * PUER(:, JK) / MAX(.1, PUMF(:, JK))
            ZUW2(:) = ZUW1(:) + ZWORK3(:) * XNHGAM * XG * &
                      ((PUTHV(:, JK) + PUTHV(:, JKP)) / &
                       (ZWORK4(:) + PTHV(:, JKP)) - 1.) & ! buoyancy term
                      - ZWORK5(:)                                  ! entrainment term
!
!
!*       6.     Update total precipitation: dr_r=(r_c+r_i)*exp(-rate*dz)
!               --------------------------------------------------------
!
!                    compute level mean vertical velocity
            ZWORK2(:) = 0.5 * &
                        (SQRT(MAX(1.E-2, ZUW2(:))) + &
                         SQRT(MAX(1.E-2, ZUW1(:))))
!
!
!*       7.     Update r_c, r_i, enthalpy, r_w  for precipitation
!               -------------------------------------------------------
!
            PURW(:, JKP) = PURW(:, JK)
            PURC(:, JKP) = PURC(:, JKP)
            PURI(:, JKP) = PURI(:, JKP)
            PUTHL(:, JKP) = PUTHL(:, JK)
!
            ZUW1(:) = ZUW2(:)
!
        ENDWHERE
!
!
!*       8.     Compute entrainment and detrainment using conservative
!               variables adjusted for precipitation ( not for entrainment)
!               -----------------------------------------------------------
!
!*       8.1    Compute critical mixed fraction by estimating unknown
!               T^mix r_c^mix and r_i^mix from enthalpy^mix and r_w^mix
!               We determine the zero crossing of the linear curve
!               evaluating the derivative using ZMIXF=0.1.
!               -----------------------------------------------------
!
        ZMIXF(:) = 0.1   ! starting value for critical mixed fraction
        ZWORK1(:) = ZMIXF(:) * PTHL(:, JKP) &
                    + (1.-ZMIXF(:)) * PUTHL(:, JKP) ! mixed enthalpy
        ZWORK2(:) = ZMIXF(:) * PRW(:, JKP) &
                    + (1.-ZMIXF(:)) * PURW(:, JKP)  ! mixed r_w
!
        call CONVECT_CONDENS(KLON, KICE, PPRES(:, JKP), ZWORK1, ZWORK2, &
                             PURC(:, JKP), PURI(:, JKP), PZ(:, JKP), GWORK1, ZUT, &
                             ZWORK3, ZWORK4, ZWORK5, ZLV, ZLS, ZCPH)
!        put in enthalpy and r_w and get T r_c, r_i (ZUT, ZWORK4-5)
!
        ! compute theta_v of mixture
        ZWORK3(:) = ZUT(:) * ZPI(:) * (1.+ZEPSA * ( &
                                       ZWORK2(:) - ZWORK4(:) - ZWORK5(:))) / (1.+ZWORK2(:))
        ! compute final value of critical mixed fraction using theta_v
        ! of mixture, grid-scale and updraft
        ZMIXF(:) = MAX(0., PUTHV(:, JKP) - PTHV(:, JKP)) * ZMIXF(:) / &
                   (PUTHV(:, JKP) - ZWORK3(:) + 1.E-10)
        ZMIXF(:) = MAX(0., MIN(1., ZMIXF(:)))
!
!
!*       8.2     Compute final midlevel values for entr. and detrainment
!                after call of distribution function
!                -------------------------------------------------------
!
!
        call CONVECT_MIXING_FUNCT(KLON, ZMIXF, 1, ZE2, ZD2)
!       Note: routine MIXING_FUNCT returns fractional entrainm/detrainm. rates
!
        ZE2 = MIN(ZD2, MAX(.3, ZE2))
!
! ZWORK1(:) = XENTR * PMFLCL(:) * PDPRES(:,JKP) / XCRAD ! rate of env. inflow
!*MOD
        zwork1(:) = xentr * xg / xcrad * pumf(:, jk) * (pz(:, jkp) - pz(:, jk))
! ZWORK1(:) = XENTR * pumf(:,jk) * PDPRES(:,JKP) / XCRAD ! rate of env. inflow
!*MOD
        ZWORK2(:) = 0.
        WHERE(GWORK1(:)) ZWORK2(:) = 1.
        WHERE(PUTHV(:, JKP) > PTHV(:, JKP))
            PUER(:, JKP) = 0.5 * ZWORK1(:) * (ZE1(:) + ZE2(:)) * ZWORK2(:)
            PUDR(:, JKP) = 0.5 * ZWORK1(:) * (ZD1(:) + ZD2(:)) * ZWORK2(:)
        elseWHERE
            PUER(:, JKP) = 0.
            PUDR(:, JKP) = ZWORK1(:) * ZWORK2(:)
        ENDWHERE
!
!*       8.3     Determine equilibrium temperature level
!                --------------------------------------
!
        WHERE(PUTHV(:, JKP) > PTHV(:, JKP) .and. JK > KLCL(:) + 1 &
              .and. GWORK1(:))
            KETL(:) = JKP            ! equilibrium temperature level
        ENDWHERE
!
!*       8.4     If the calculated detrained mass flux is greater than
!                the total updraft mass flux, or vertical velocity is
!                negative, all cloud mass detrains at previous model level,
!                exit updraft calculations - CTL is attained
!                -------------------------------------------------------
!
        WHERE(GWORK1(:)) &
            GWORK2(:) = PUMF(:, JK) - PUDR(:, JKP) > 10. .and. ZUW2(:) > 0.
        WHERE(GWORK2(:)) KCTL(:) = JKP   ! cloud top level
        GWORK1(:) = GWORK2(:) .and. GWORK4(:)
!
        if(COUNT(GWORK2(:)) == 0) EXIT
!
!
!*       9.   Compute CAPE for undilute ascent using theta_e and
!             theta_es instead of theta_v. This estimation produces
!             a significantly larger value for CAPE than the actual one.
!             ----------------------------------------------------------
!
        WHERE(GWORK1(:))
!
            ZWORK3(:) = PZ(:, JKP) - PZ(:, JK) * ZWORK6(:) - &
                        (1.-ZWORK6(:)) * PZLCL(:)              ! level thickness
            ZWORK2(:) = PTHES(:, JK) + (1.-ZWORK6(:)) * &
                        (PTHES(:, JKP) - PTHES(:, JK)) / (PZ(:, JKP) - PZ(:, JK)) * &
                        (PZLCL(:) - PZ(:, JK)) ! linear interpolation for theta_es at LCL
            ! ( this is only done for model level just above LCL
!
            ZWORK1(:) = (2.*ZTHEUL(:)) / (ZWORK2(:) + PTHES(:, JKP)) - 1.
            PCAPE(:) = PCAPE(:) + XG * ZWORK3(:) * MAX(0., ZWORK1(:))
!
!
!*       10.   Compute final values of updraft mass flux, enthalpy, r_w
!              at level k+1
!              --------------------------------------------------------
!
            PUMF(:, JKP) = PUMF(:, JK) - PUDR(:, JKP) + PUER(:, JKP)
            PUMF(:, JKP) = MAX(PUMF(:, JKP), 0.1)
            PUTHL(:, JKP) = (PUMF(:, JK) * PUTHL(:, JK) + &
                             PUER(:, JKP) * PTHL(:, JK) - PUDR(:, JKP) * PUTHL(:, JK)) &
                            / PUMF(:, JKP)
            PURW(:, JKP) = (PUMF(:, JK) * PURW(:, JK) + &
                            PUER(:, JKP) * PRW(:, JK) - PUDR(:, JKP) * PURW(:, JK)) &
                           / PUMF(:, JKP)
!
!
            ZE1(:) = ZE2(:) ! update fractional entrainment/detrainment
            ZD1(:) = ZD2(:)
!
        ENDWHERE
!
    enddo
!
!*       12.1    Set OTRIG to False if cloud thickness < 0.5km
!                or > 3km (deep convection) or CAPE < 1
!                ------------------------------------------------
!
    do JI = 1, IIE
        JK = KCTL(JI)
        ZWORK1(JI) = PZ(JI, JK) - PZLCL(JI)
        OTRIG(JI) = ZWORK1(JI) >= XCDEPTH .and. ZWORK1(JI) < XCDEPTH_D &
                    .and. PCAPE(JI) > 1.
    enddo
    WHERE(.not. OTRIG(:))
        KCTL(:) = IKB
    ENDWHERE
    KETL(:) = MAX(KETL(:), KLCL(:) + 2)
    KETL(:) = MIN(KETL(:), KCTL(:))
!
!
!*       12.2    If the ETL and CTL are the same detrain updraft mass
!                flux at this level
!                -------------------------------------------------------
!
    ZWORK1(:) = 0.
    WHERE(KETL(:) == KCTL(:)) ZWORK1(:) = 1.
!
    do JI = 1, IIE
        JK = KETL(JI)
        PUDR(JI, JK) = PUDR(JI, JK) + &
                       (PUMF(JI, JK) - PUER(JI, JK)) * ZWORK1(JI)
        PUER(JI, JK) = PUER(JI, JK) * (1.-ZWORK1(JI))
        PUMF(JI, JK) = PUMF(JI, JK) * (1.-ZWORK1(JI))
        JKP = KCTL(JI) + 1
        PUER(JI, JKP) = 0. ! entrainm/detr rates have been already computed
        PUDR(JI, JKP) = 0. ! at level KCTL+1, set them to zero
        PURW(JI, JKP) = 0.
        PURC(JI, JKP) = 0.
        PURI(JI, JKP) = 0.
        PUTHL(JI, JKP) = 0.
        PURC(JI, JKP + 1) = 0.
        PURI(JI, JKP + 1) = 0.
    enddo
!
!*       12.3    Adjust mass flux profiles, detrainment rates, and
!                precipitation fallout rates to reflect linear decrease
!                in mass flux between the ETL and CTL
!                -------------------------------------------------------
!
    ZWORK1(:) = 0.
    JK1 = MINVAL(KETL(:))
    JK2 = MAXVAL(KCTL(:))

    do JK = JK1, JK2
        do JI = 1, IIE
        if(JK > KETL(JI) .and. JK <= KCTL(JI)) then
            ZWORK1(JI) = ZWORK1(JI) + PDPRES(JI, JK)
        endif
        enddo
    enddo
!
    do JI = 1, IIE
        JK = KETL(JI)
        ZWORK1(JI) = PUMF(JI, JK) / MAX(1., ZWORK1(JI))
    enddo
!
    do JK = JK1 + 1, JK2
        JKP = JK - 1
        do JI = 1, IIE
        if(JK > KETL(JI) .and. JK <= KCTL(JI)) then
            PUDR(JI, JK) = PDPRES(JI, JK) * ZWORK1(JI)
            PUMF(JI, JK) = PUMF(JI, JKP) - PUDR(JI, JK)
        endif
        enddo
    enddo
!
!         12.4   Set mass flux and entrainment in the source layer.
!                Linear increase throughout the source layer.
!                -------------------------------------------------------
!
!IWORK(:) = MIN( KPBL(:), KLCL(:) - 1 )
    IWORK(:) = KPBL(:)
    do JI = 1, IIE
        JK = KDPL(JI)
        JKP = IWORK(JI)
!          mixed layer depth
        ZWORK2(JI) = PPRES(JI, JK) - PPRES(JI, JKP) + PDPRES(JI, JK)
    enddo
!
    JKP = MAXVAL(IWORK(:))
    do JK = JKM, JKP
        do JI = 1, IIE
        if(JK >= KDPL(JI) .and. JK <= IWORK(JI)) then
            PUER(JI, JK) = PUER(JI, JK) + PMFLCL(JI) * PDPRES(JI, JK) / (ZWORK2(JI) + 0.1)
            PUMF(JI, JK) = PUMF(JI, JK - 1) + PUER(JI, JK)
        endif
        enddo
    enddo
!
!
!*       13.   If cloud thickness is smaller than  .5 km or > 3 km
!              no shallow convection is allowed
!              Nota: For technical reasons, we stop the convection
!                    computations in this case and do not go back to
!                    TRIGGER_FUNCT to look for the next unstable LCL
!                    which could produce a thicker cloud.
!              ---------------------------------------------------
!
    GWORK6(:, :) = SPREAD(OTRIG(:), DIM=2, NCOPIES=KLEV)
    WHERE(.not. GWORK6(:, :))
        PUMF(:, :) = 0.
        PUDR(:, :) = 0.
        PUER(:, :) = 0.
        PUTHL(:, :) = PTHL(:, :)
        PURW(:, :) = PRW(:, :)
        PURC(:, :) = 0.
        PURI(:, :) = 0.
    ENDWHERE
!
ENDsubroutine CONVECT_UPDRAFT_SHAL

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

!MNH_LIC Copyright 1994-2014 CNRS, Meteo-France and Universite Paul Sabatier
!MNH_LIC This is part of the Meso-NH software governed by the CeCILL-C licence
!MNH_LIC version 1. See LICENSE, CeCILL-C_V1-en.txt and CeCILL-C_V1-fr.txt
!MNH_LIC for details. version 1.
!-----------------------------------------------------------------
!--------------- special set of characters for RCS information
!-----------------------------------------------------------------
! $Source$ $Revision$
! MASDEV4_7 conv 2006/05/18 13:07:25
!-----------------------------------------------------------------
!     ######################
MODULE MODI_SHALLOW_CONVECTION
!     ######################
!
    INTERFACE
!
        subroutine SHALLOW_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                                      PDTCONV, KICE, OSETTADJ, PTADJS, &
                                      PPABST, PZZ, PTKECLS, &
                                      PTT, PRVT, PRCT, PRIT, PWT, &
                                      PTTEN, PRVTEN, PRCTEN, PRITEN, &
                                      KCLTOP, KCLBAS, PUMF, &
                                      OCH1CONV, KCH1, PCH1, PCH1TEN)
!
            INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
            INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
            INTEGER, INTENT(IN) :: KIDIA    ! value of the first point in x
            INTEGER, INTENT(IN) :: KFDIA    ! value of the last point in x
            INTEGER, INTENT(IN) :: KBDIA    ! vertical  computations start at
!                                                  ! KBDIA that is at least 1
            INTEGER, INTENT(IN) :: KTDIA    ! vertical computations can be
            ! limited to KLEV + 1 - KTDIA
            ! default=1
            REAL, INTENT(IN) :: PDTCONV  ! Interval of time between two
            ! calls of the deep convection
            ! scheme
            INTEGER, INTENT(IN) :: KICE     ! flag for ice ( 1 = yes,
            !                0 = no ice )
            LOGICAL, INTENT(IN) :: OSETTADJ ! logical to set convective
            ! adjustment time by user
            REAL, INTENT(IN) :: PTADJS   ! user defined adjustment time
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTT      ! grid scale temperature at t
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRVT     ! grid scale water vapor "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRCT     ! grid scale r_c  "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRIT     ! grid scale r_i "
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PWT      ! grid scale vertical
            ! velocity (m/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPABST   ! grid scale pressure at t
            REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZZ      ! height of model layer (m)
            REAL, DIMENSION(KLON), INTENT(IN) :: PTKECLS  ! TKE in the CLS  (m2/s2)
!
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PTTEN  ! convective temperature
            ! tendency (K/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRVTEN ! convective r_v tendency (1/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRCTEN ! convective r_c tendency (1/s)
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRITEN ! convective r_i tendency (1/s)
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLTOP ! cloud top level
            INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLBAS ! cloud base level
            ! they are given a value of
            ! 0 if no convection
            REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF   ! updraft mass flux (kg/s m2)
!
            LOGICAL, INTENT(IN) :: OCH1CONV ! include tracer transport
            INTEGER, INTENT(IN) :: KCH1     ! number of species
            REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(IN) :: PCH1! grid scale chemical species
            REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(INOUT):: PCH1TEN! species conv. tendency (1/s)
!
        ENDsubroutine SHALLOW_CONVECTION
!
    ENDINTERFACE
!
ENDMODULE MODI_SHALLOW_CONVECTION
!   ###############################################################################
subroutine SHALLOW_CONVECTION(KLON, KLEV, KIDIA, KFDIA, KBDIA, KTDIA, &
                              PDTCONV, KICE, OSETTADJ, PTADJS, &
                              PPABST, PZZ, PTKECLS, &
                              PTT, PRVT, PRCT, PRIT, PWT, &
                              PTTEN, PRVTEN, PRCTEN, PRITEN, &
                              KCLTOP, KCLBAS, PUMF, &
                              OCH1CONV, KCH1, PCH1, PCH1TEN)
!   ###############################################################################
!
!!**** Monitor routine to compute all convective tendencies by calls
!!     of several subroutines.
!!
!!
!!    PURPOSE
!!    -------
!!      The purpose of this routine is to determine the convective
!!      tendencies. The routine first prepares all necessary grid-scale
!!      variables. The final convective tendencies are then computed by
!!      calls of different subroutines.
!!
!!
!!**  METHOD
!!    ------
!!      We start by selecting convective columns in the model domain through
!!      the call of routine TRIGGER_FUNCT. Then, we allocate memory for the
!!      convection updraft and downdraft variables and gather the grid scale
!!      variables in convective arrays.
!!      The updraft and downdraft computations are done level by level starting
!!      at the  bottom and top of the domain, respectively.
!!      All computations are done on MNH thermodynamic levels. The depth
!!      of the current model layer k is defined by DP(k)=P(k-1)-P(k)
!!
!!
!!
!!    EXTERNAL
!!    --------
!!    CONVECT_TRIGGER_SHAL
!!    CONVECT_SATMIXRATIO
!!    CONVECT_UPDRAFT_SHAL
!!        CONVECT_CONDENS
!!        CONVECT_MIXING_FUNCT
!!    CONVECT_CLOSURE_SHAL
!!        CONVECT_CLOSURE_THRVLCL
!!        CONVECT_CLOSURE_ADJUST_SHAL
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST
!!          XG                   ! gravity constant
!!          XPI                  ! number Pi
!!          XP00                 ! reference pressure
!!          XRD, XRV             ! gaz  constants for dry air and water vapor
!!          XCPD, XCPV           ! specific heat for dry air and water vapor
!!          XRHOLW               ! density of liquid water
!!          XALPW, XBETAW, XGAMW ! constants for water saturation pressure
!!          XTT                  ! triple point temperature
!!          XLVTT, XLSTT         ! vaporization, sublimation heat constant
!!          XCL, XCI             ! specific heat for liquid water and ice
!!
!!      Module MODD_CONVPAREXT
!!          JCVEXB, JCVEXT       ! extra levels on the vertical boundaries
!!
!!      Module MODD_CONVPAR
!!          XA25                 ! reference grid area
!!          XCRAD                ! cloud radius
!!
!!
!!    REFERENCE
!!    ---------
!!
!!      Bechtold, 1997 : Meso-NH scientific  documentation (31 pp)
!!      Fritsch and Chappell, 1980, J. Atmos. Sci., Vol. 37, 1722-1761.
!!      Kain and Fritsch, 1990, J. Atmos. Sci., Vol. 47, 2784-2801.
!!      Kain and Fritsch, 1993, Meteor. Monographs, Vol. 24, 165-170.
!!
!!    AUTHOR
!!    ------
!!      P. BECHTOLD       * Laboratoire d'Aerologie *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    26/03/96
!!   Peter Bechtold 15/11/96 replace theta_il by enthalpy
!!         "        10/12/98 changes for ARPEGE
!!         "        01/01/02 Apply conservation correction
!!   F Bouyssel     05/11/08 Modifications for reproductibility
!!   E. Bazile      20/07/09 Input of TKECLS.
!!     Juan 24/09/2012: for BUG Pgi rewrite PACK function on mode_pack_pgi
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
    USE MODD_CST
    USE MODD_CONVPAREXT
    USE MODD_CONVPAR_SHAL
!USE MODD_NSV,       ONLY : NSV_LGBEG,NSV_LGEND
!
    USE MODI_CONVECT_TRIGGER_SHAL
    USE MODI_CONVECT_UPDRAFT_SHAL
    USE MODI_CONVECT_CLOSURE_SHAL
!USE MODI_CONVECT_CHEM_TRANSPORT
!
!SeBi #ifdef MNH_PGI
!SeBi USE MODE_PACK_PGI
!SeBi #endif
!
    implicit none
!
!*       0.1   Declarations of dummy arguments :
!
!
    INTEGER, INTENT(IN) :: KLON     ! horizontal dimension
    INTEGER, INTENT(IN) :: KLEV     ! vertical dimension
    INTEGER, INTENT(IN) :: KIDIA    ! value of the first point in x
    INTEGER, INTENT(IN) :: KFDIA    ! value of the last point in x
    INTEGER, INTENT(IN) :: KBDIA    ! vertical  computations start at
!                                                  ! KBDIA that is at least 1
    INTEGER, INTENT(IN) :: KTDIA    ! vertical computations can be
    ! limited to KLEV + 1 - KTDIA
    ! default=1
    REAL, INTENT(IN) :: PDTCONV  ! Interval of time between two
    ! calls of the deep convection
    ! scheme
    INTEGER, INTENT(IN) :: KICE     ! flag for ice ( 1 = yes,
    !                0 = no ice )
    LOGICAL, INTENT(IN) :: OSETTADJ ! logical to set convective
    ! adjustment time by user
    REAL, INTENT(IN) :: PTADJS   ! user defined adjustment time
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PTT      ! grid scale temperature at t
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRVT     ! grid scale water vapor "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRCT     ! grid scale r_c  "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PRIT     ! grid scale r_i "
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PWT      ! grid scale vertical
    ! velocity (m/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PPABST   ! grid scale pressure at t
    REAL, DIMENSION(KLON, KLEV), INTENT(IN) :: PZZ      ! height of model layer (m)
    REAL, DIMENSION(KLON), INTENT(IN) :: PTKECLS  ! TKE in the CLS  (m2/s2)
!
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PTTEN  ! convective temperature
    ! tendency (K/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRVTEN ! convective r_v tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRCTEN ! convective r_c tendency (1/s)
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PRITEN ! convective r_i tendency (1/s)
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLTOP ! cloud top level
    INTEGER, DIMENSION(KLON), INTENT(INOUT):: KCLBAS ! cloud base level
    ! they are given a value of
    ! 0 if no convection
    REAL, DIMENSION(KLON, KLEV), INTENT(INOUT):: PUMF   ! updraft mass flux (kg/s m2)
!
    LOGICAL, INTENT(IN) :: OCH1CONV ! include tracer transport
    INTEGER, INTENT(IN) :: KCH1     ! number of species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(IN) :: PCH1! grid scale chemical species
    REAL, DIMENSION(KLON, KLEV, KCH1), INTENT(INOUT):: PCH1TEN! species conv. tendency (1/s)
!
!
!*       0.2   Declarations of local fixed memory variables :
!
    INTEGER  :: ITEST, ICONV            ! number of convective columns
    INTEGER  :: IIB, IIE                ! horizontal loop bounds
    INTEGER  :: IKB, IKE                ! vertical loop bounds
    INTEGER  :: IKS                     ! vertical dimension
    INTEGER  :: JI, JL                  ! horizontal loop index
    INTEGER  :: JN                      ! number of tracers
    INTEGER  :: JK, JKM, JKP            ! vertical loop index
    INTEGER  :: IFTSTEPS                ! only used for chemical tracers
    real  :: ZEPS, ZEPSA             ! R_d / R_v, R_v / R_d
    real  :: ZRDOCP                  ! R_d/C_p
!
    LOGICAL, DIMENSION(KLON, KLEV)     :: GTRIG3 ! 3D logical mask for convection
    LOGICAL, DIMENSION(KLON)           :: GTRIG  ! 2D logical mask for trigger test
    REAL, DIMENSION(KLON, KLEV)         :: ZTHT, ZSTHV, ZSTHES  ! grid scale theta, theta_v
    REAL, DIMENSION(KLON)              :: ZWORK2, ZWORK2B ! work array
    real                :: ZW1     ! work variable
!
!
!*       0.2   Declarations of local allocatable  variables :
!
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IDPL    ! index for parcel departure level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IPBL    ! index for source layer top
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ILCL    ! index for lifting condensation level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IETL    ! index for zero buoyancy level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ICTL    ! index for cloud top level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ILFS    ! index for level of free sink
!
    INTEGER, DIMENSION(:), ALLOCATABLE :: ISDPL   ! index for parcel departure level
    INTEGER, DIMENSION(:), ALLOCATABLE  :: ISPBL   ! index for source layer top
    INTEGER, DIMENSION(:), ALLOCATABLE :: ISLCL   ! index for lifting condensation level
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTHLCL ! updraft theta at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTLCL  ! updraft temp. at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSRVLCL ! updraft rv at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSWLCL  ! updraft w at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSZLCL  ! LCL height
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSTHVELCL! envir. theta_v at LCL
    REAL, DIMENSION(:), ALLOCATABLE    :: ZSDXDY  ! grid area (m^2)
!
! grid scale variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZZ      ! height of model layer (m)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZPRES   ! grid scale pressure
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDPRES  ! pressure difference between
    ! bottom and top of layer (Pa)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZW      ! grid scale vertical velocity on theta grid
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTT     ! temperature
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTH     ! grid scale theta
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHV    ! grid scale theta_v
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHL    ! grid scale enthalpy (J/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHES, ZTHEST ! grid scale saturated theta_e
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRW     ! grid scale total water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRV     ! grid scale water vapor (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRC     ! grid scale cloud water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRI     ! grid scale cloud ice (kg/kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZDXDY   ! grid area (m^2)
!
! updraft variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUMF    ! updraft mass flux (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUER    ! updraft entrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUDR    ! updraft detrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUTHL   ! updraft enthalpy (J/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZUTHV   ! updraft theta_v (K)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURW    ! updraft total water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURC    ! updraft cloud water (kg/kg)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZURI    ! updraft cloud ice   (kg/kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZMFLCL  ! cloud base unit mass flux(kg/s)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZCAPE   ! available potent. energy
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTHLCL  ! updraft theta at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTLCL   ! updraft temp. at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZRVLCL  ! updraft rv at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZWLCL   ! updraft w at LCL
    REAL, DIMENSION(:), ALLOCATABLE  :: ZZLCL   ! LCL height
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTHVELCL! envir. theta_v at LCL
!
! downdraft variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDMF    ! downdraft mass flux (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDER    ! downdraft entrainment (kg/s)
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZDDR    ! downdraft detrainment (kg/s)
!
! closure variables
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZLMASS  ! mass of model layer (kg)
    REAL, DIMENSION(:), ALLOCATABLE  :: ZTIMEC  ! advective time period
!
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZTHC    ! conv. adj. grid scale theta
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRVC    ! conv. adj. grid scale r_w
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRCC    ! conv. adj. grid scale r_c
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZRIC    ! conv. adj. grid scale r_i
    REAL, DIMENSION(:, :), ALLOCATABLE  :: ZWSUB   ! envir. compensating subsidence (Pa/s)
!
    LOGICAL, DIMENSION(:), ALLOCATABLE  :: GTRIG1  ! logical mask for convection
    LOGICAL, DIMENSION(:), ALLOCATABLE  :: GWORK   ! logical work array
    INTEGER, DIMENSION(:), ALLOCATABLE  :: IINDEX, IJINDEX, IJSINDEX, IJPINDEX!hor.index
    REAL, DIMENSION(:), ALLOCATABLE  :: ZCPH    ! specific heat C_ph
    REAL, DIMENSION(:), ALLOCATABLE  :: ZLV, ZLS! latent heat of vaporis., sublim.
    real                :: ZES     ! saturation vapor mixng ratio
!
! Chemical Tracers:
    REAL, DIMENSION(:, :, :), ALLOCATABLE:: ZCH1    ! grid scale chemical specy (kg/kg)
    REAL, DIMENSION(:, :, :), ALLOCATABLE:: ZCH1C   ! conv. adjust. chemical specy 1
    REAL, DIMENSION(:, :), ALLOCATABLE:: ZWORK3  ! conv. adjust. chemical specy 1
    LOGICAL, DIMENSION(:, :, :), ALLOCATABLE::GTRIG4 ! logical mask
!
!-------------------------------------------------------------------------------
!
!
!*       0.3    Compute loop bounds
!               -------------------
!
    IIB = KIDIA
    IIE = KFDIA
    JCVEXB = MAX(0, KBDIA - 1)
    IKB = 1 + JCVEXB
    IKS = KLEV
    JCVEXT = MAX(0, KTDIA - 1)
    IKE = IKS - JCVEXT
!
!
!*       0.5    Update convective counter ( where KCOUNT > 0
!               convection is still active ).
!               ---------------------------------------------
!
    GTRIG(:) = .false.
    GTRIG(IIB:IIE) = .true.
    ITEST = COUNT(GTRIG(:))
    if(ITEST == 0) then
        RETURN
    endif

!
!
!*       0.7    Reset convective tendencies to zero if convective
!               counter becomes negative
!               -------------------------------------------------
!
    GTRIG3(:, :) = SPREAD(GTRIG(:), DIM=2, NCOPIES=IKS)
    WHERE(GTRIG3(:, :))
        PTTEN(:, :) = 0.
        PRVTEN(:, :) = 0.
        PRCTEN(:, :) = 0.
        PRITEN(:, :) = 0.
! PUTEN(:,:)  = 0.
! PVTEN(:,:)  = 0.
        PUMF(:, :) = 0.
    ENDWHERE
    WHERE(GTRIG(:))
        KCLTOP(:) = 0
        KCLBAS(:) = 0
    ENDWHERE
    if(OCH1CONV) then
        ALLOCATE(GTRIG4(KLON, KLEV, KCH1))
        GTRIG4(:, :, :) = SPREAD(GTRIG3(:, :), DIM=3, NCOPIES=KCH1)
        WHERE(GTRIG4(:, :, :)) PCH1TEN(:, :, :) = 0.
        DEALLOCATE(GTRIG4)
    endif
!
!
!*       1.     Initialize  local variables
!               ----------------------------
!
    ZEPS = XRD / XRV
    ZEPSA = XRV / XRD
    ZRDOCP = XRD / XCPD
!
!-------------------------------------------------------------------------------
!
!*       1.1    Set up grid scale theta, theta_v, theta_es
!               ------------------------------------------
!
    ZTHT(:, :) = 300.
    ZSTHV(:, :) = 300.
    ZSTHES(:, :) = 400.
    do JK = IKB, IKE
    do JI = IIB, IIE
        if(PPABST(JI, JK) > 40.E2) then
            ZTHT(JI, JK) = PTT(JI, JK) * (XP00 / PPABST(JI, JK))**ZRDOCP
            ZSTHV(JI, JK) = ZTHT(JI, JK) * (1.+ZEPSA * PRVT(JI, JK)) / &
                            (1.+PRVT(JI, JK) + PRCT(JI, JK) + PRIT(JI, JK))
!
            ! use conservative Bolton (1980) formula for theta_e
            ! it is used to compute CAPE for undilute parcel ascent
            ! For economical reasons we do not use routine CONVECT_SATMIXRATIO here
!
            ZES = EXP(XALPW - XBETAW / PTT(JI, JK) - XGAMW * LOG(PTT(JI, JK)))
            ZES = MIN(1., ZEPS * ZES / (PPABST(JI, JK) - ZES))
            ZSTHES(JI, JK) = PTT(JI, JK) * (ZTHT(JI, JK) / PTT(JI, JK))** &
                             (1.-0.28 * ZES) * EXP((3374.6525 / PTT(JI, JK) - 2.5403) &
                                                   * ZES * (1.+0.81 * ZES))
        endif
    enddo
    enddo
!
!-------------------------------------------------------------------------------
!
!*       2.     Test for convective columns and determine properties at the LCL
!               --------------------------------------------------------------
!
!*       2.1    Allocate arrays depending on number of model columns that need
!               to be tested for convection (i.e. where no convection is present
!               at the moment.
!               --------------------------------------------------------------
!
    ALLOCATE(ZPRES(ITEST, IKS))
    ALLOCATE(ZZ(ITEST, IKS))
    ALLOCATE(ZW(ITEST, IKS))
    ALLOCATE(ZTH(ITEST, IKS))
    ALLOCATE(ZTHV(ITEST, IKS))
    ALLOCATE(ZTHEST(ITEST, IKS))
    ALLOCATE(ZRV(ITEST, IKS))
    ALLOCATE(ZSTHLCL(ITEST))
    ALLOCATE(ZSTLCL(ITEST))
    ALLOCATE(ZSRVLCL(ITEST))
    ALLOCATE(ZSWLCL(ITEST))
    ALLOCATE(ZSZLCL(ITEST))
    ALLOCATE(ZSTHVELCL(ITEST))
    ALLOCATE(ISDPL(ITEST))
    ALLOCATE(ISPBL(ITEST))
    ALLOCATE(ISLCL(ITEST))
    ALLOCATE(ZSDXDY(ITEST))
    ALLOCATE(GTRIG1(ITEST))
    ALLOCATE(IINDEX(KLON))
    ALLOCATE(IJSINDEX(ITEST))
    do JI = 1, KLON
        IINDEX(JI) = JI
    enddo
    IJSINDEX(:) = PACK(IINDEX(:), MASK=GTRIG(:))
!
    do JK = IKB, IKE
    do JI = 1, ITEST
        JL = IJSINDEX(JI)
        ZPRES(JI, JK) = PPABST(JL, JK)
        ZZ(JI, JK) = PZZ(JL, JK)
        ZTH(JI, JK) = ZTHT(JL, JK)
        ZTHV(JI, JK) = ZSTHV(JL, JK)
        ZTHEST(JI, JK) = ZSTHES(JL, JK)
        ZRV(JI, JK) = MAX(0., PRVT(JL, JK))
        ZW(JI, JK) = PWT(JL, JK)
    enddo
    enddo
    do JI = 1, ITEST
        JL = IJSINDEX(JI)
        ZSDXDY(JI) = XA25
    enddo
!
!*       2.2    Compute environm. enthalpy and total water = r_v + r_i + r_c
!               and envir. saturation theta_e
!               ------------------------------------------------------------
!
!
!*       2.3    Test for convective columns and determine properties at the LCL
!               --------------------------------------------------------------
!
    ISLCL(:) = MAX(IKB, 2)   ! initialize DPL PBL and LCL
    ISDPL(:) = IKB
    ISPBL(:) = IKB
!
    call CONVECT_TRIGGER_SHAL(ITEST, KLEV, &
                              ZPRES, ZTH, ZTHV, ZTHEST, &
                              ZRV, ZW, ZZ, ZSDXDY, PTKECLS, &
                              ZSTHLCL, ZSTLCL, ZSRVLCL, ZSWLCL, ZSZLCL, &
                              ZSTHVELCL, ISLCL, ISDPL, ISPBL, GTRIG1)
!
    DEALLOCATE(ZPRES)
    DEALLOCATE(ZZ)
    DEALLOCATE(ZTH)
    DEALLOCATE(ZTHV)
    DEALLOCATE(ZTHEST)
    DEALLOCATE(ZRV)
    DEALLOCATE(ZW)
!
!-------------------------------------------------------------------------------
!
!*       3.     After the call of TRIGGER_FUNCT we allocate all the dynamic
!               arrays used in the convection scheme using the mask GTRIG, i.e.
!               we do calculus only in convective columns. This corresponds to
!               a GATHER operation.
!               --------------------------------------------------------------
!
    ICONV = COUNT(GTRIG1(:))
    if(ICONV == 0) then
        DEALLOCATE(ZSTHLCL)
        DEALLOCATE(ZSTLCL)
        DEALLOCATE(ZSRVLCL)
        DEALLOCATE(ZSWLCL)
        DEALLOCATE(ZSZLCL)
        DEALLOCATE(ZSTHVELCL)
        DEALLOCATE(ZSDXDY)
        DEALLOCATE(ISLCL)
        DEALLOCATE(ISDPL)
        DEALLOCATE(ISPBL)
        DEALLOCATE(GTRIG1)
        DEALLOCATE(IINDEX)
        DEALLOCATE(IJSINDEX)
        RETURN   ! no convective column has been found, exit DEEP_CONVECTION
    endif
!
    ! vertical index variables
!
    ALLOCATE(IDPL(ICONV))
    ALLOCATE(IPBL(ICONV))
    ALLOCATE(ILCL(ICONV))
    ALLOCATE(ICTL(ICONV))
    ALLOCATE(IETL(ICONV))
!
    ! grid scale variables
!
    ALLOCATE(ZZ(ICONV, IKS)); ZZ = 0.0
    ALLOCATE(ZPRES(ICONV, IKS)); ZPRES = 0.0
    ALLOCATE(ZDPRES(ICONV, IKS)); ZDPRES = 0.0
    ALLOCATE(ZTT(ICONV, IKS)); ZTT = 0.0
    ALLOCATE(ZTH(ICONV, IKS)); ZTH = 0.0
    ALLOCATE(ZTHV(ICONV, IKS)); ZTHV = 0.0
    ALLOCATE(ZTHL(ICONV, IKS)); ZTHL = 0.0
    ALLOCATE(ZTHES(ICONV, IKS)); ZTHES = 0.0
    ALLOCATE(ZRV(ICONV, IKS)); ZRV = 0.0
    ALLOCATE(ZRC(ICONV, IKS)); ZRC = 0.0
    ALLOCATE(ZRI(ICONV, IKS)); ZRI = 0.0
    ALLOCATE(ZRW(ICONV, IKS)); ZRW = 0.0
    ALLOCATE(ZDXDY(ICONV)); ZDXDY = 0.0
!
    ! updraft variables
!
    ALLOCATE(ZUMF(ICONV, IKS))
    ALLOCATE(ZUER(ICONV, IKS))
    ALLOCATE(ZUDR(ICONV, IKS))
    ALLOCATE(ZUTHL(ICONV, IKS))
    ALLOCATE(ZUTHV(ICONV, IKS))
    ALLOCATE(ZURW(ICONV, IKS))
    ALLOCATE(ZURC(ICONV, IKS))
    ALLOCATE(ZURI(ICONV, IKS))
    ALLOCATE(ZTHLCL(ICONV))
    ALLOCATE(ZTLCL(ICONV))
    ALLOCATE(ZRVLCL(ICONV))
    ALLOCATE(ZWLCL(ICONV))
    ALLOCATE(ZMFLCL(ICONV))
    ALLOCATE(ZZLCL(ICONV))
    ALLOCATE(ZTHVELCL(ICONV))
    ALLOCATE(ZCAPE(ICONV))
!
    ! work variables
!
    ALLOCATE(IJINDEX(ICONV))
    ALLOCATE(IJPINDEX(ICONV))
    ALLOCATE(ZCPH(ICONV))
    ALLOCATE(ZLV(ICONV))
    ALLOCATE(ZLS(ICONV))
!
!
!*           3.1    Gather grid scale and updraft base variables in
!                   arrays using mask GTRIG
!                   ---------------------------------------------------
!
    GTRIG(:) = UNPACK(GTRIG1(:), MASK=GTRIG, FIELD=.false.)
    IJINDEX(:) = PACK(IINDEX(:), MASK=GTRIG(:))
!
    do JK = IKB, IKE
    do JI = 1, ICONV
        JL = IJINDEX(JI)
        ZZ(JI, JK) = PZZ(JL, JK)
        ZPRES(JI, JK) = PPABST(JL, JK)
        ZTT(JI, JK) = PTT(JL, JK)
        ZTH(JI, JK) = ZTHT(JL, JK)
        ZTHES(JI, JK) = ZSTHES(JL, JK)
        ZRV(JI, JK) = MAX(0., PRVT(JL, JK))
        ZRC(JI, JK) = MAX(0., PRCT(JL, JK))
        ZRI(JI, JK) = MAX(0., PRIT(JL, JK))
        ZTHV(JI, JK) = ZSTHV(JL, JK)
    enddo
    enddo
!
    do JI = 1, ITEST
        IJSINDEX(JI) = JI
    enddo
    IJPINDEX(:) = PACK(IJSINDEX(:), MASK=GTRIG1(:))
    do JI = 1, ICONV
        JL = IJPINDEX(JI)
        IDPL(JI) = ISDPL(JL)
        IPBL(JI) = ISPBL(JL)
        ILCL(JI) = ISLCL(JL)
        ZTHLCL(JI) = ZSTHLCL(JL)
        ZTLCL(JI) = ZSTLCL(JL)
        ZRVLCL(JI) = ZSRVLCL(JL)
        ZWLCL(JI) = ZSWLCL(JL)
        ZZLCL(JI) = ZSZLCL(JL)
        ZTHVELCL(JI) = ZSTHVELCL(JL)
        ZDXDY(JI) = ZSDXDY(JL)
    enddo
    ALLOCATE(GWORK(ICONV))
    GWORK(:) = PACK(GTRIG1(:), MASK=GTRIG1(:))
    DEALLOCATE(GTRIG1)
    ALLOCATE(GTRIG1(ICONV))
    GTRIG1(:) = GWORK(:)
!
    DEALLOCATE(GWORK)
    DEALLOCATE(IJPINDEX)
    DEALLOCATE(ISDPL)
    DEALLOCATE(ISPBL)
    DEALLOCATE(ISLCL)
    DEALLOCATE(ZSTHLCL)
    DEALLOCATE(ZSTLCL)
    DEALLOCATE(ZSRVLCL)
    DEALLOCATE(ZSWLCL)
    DEALLOCATE(ZSZLCL)
    DEALLOCATE(ZSTHVELCL)
    DEALLOCATE(ZSDXDY)
!
!
!*           3.2    Compute pressure difference
!                   ---------------------------------------------------
!
    ZDPRES(:, IKB) = 0.
    do JK = IKB + 1, IKE
        ZDPRES(:, JK) = ZPRES(:, JK - 1) - ZPRES(:, JK)
    enddo
!
!*           3.3   Compute environm. enthalpy and total water = r_v + r_i + r_c
!                  ----------------------------------------------------------
!
    do JK = IKB, IKE, 1
        ZRW(:, JK) = ZRV(:, JK) + ZRC(:, JK) + ZRI(:, JK)
        ZCPH(:) = XCPD + XCPV * ZRW(:, JK)
        ZLV(:) = XLVTT + (XCPV - XCL) * (ZTT(:, JK) - XTT) ! compute L_v
        ZLS(:) = XLSTT + (XCPV - XCI) * (ZTT(:, JK) - XTT) ! compute L_i
        ZTHL(:, JK) = ZCPH(:) * ZTT(:, JK) + (1.+ZRW(:, JK)) * XG * ZZ(:, JK) &
                      - ZLV(:) * ZRC(:, JK) - ZLS(:) * ZRI(:, JK)
    enddo
!
    DEALLOCATE(ZCPH)
    DEALLOCATE(ZLV)
    DEALLOCATE(ZLS)
!
!-------------------------------------------------------------------------------
!
!*           4.     Compute updraft properties
!                   ----------------------------
!
!*           4.1    Set mass flux at LCL ( here a unit mass flux with w = 1 m/s )
!                   -------------------------------------------------------------
!
    ZDXDY(:) = XA25
    ZMFLCL(:) = XA25 * 1.E-3
!
!
!
    call CONVECT_UPDRAFT_SHAL(ICONV, KLEV, &
                              KICE, ZPRES, ZDPRES, ZZ, ZTHL, ZTHV, ZTHES, ZRW, &
                              ZTHLCL, ZTLCL, ZRVLCL, ZWLCL, ZZLCL, ZTHVELCL, &
                              ZMFLCL, GTRIG1, ILCL, IDPL, IPBL, &
                              ZUMF, ZUER, ZUDR, ZUTHL, ZUTHV, ZURW, &
                              ZURC, ZURI, ZCAPE, ICTL, IETL)
!
!
!
!*           4.2    In routine UPDRAFT GTRIG1 has been set to false when cloud
!                   thickness is smaller than 3 km
!                   -----------------------------------------------------------
!
!
!
!*       4.3    Allocate memory for downdraft variables
!               ---------------------------------------
!
! downdraft variables
!
    ALLOCATE(ZDMF(ICONV, IKS))
    ALLOCATE(ZDER(ICONV, IKS))
    ALLOCATE(ZDDR(ICONV, IKS))
    ALLOCATE(ILFS(ICONV))
    ALLOCATE(ZLMASS(ICONV, IKS))
    ZDMF(:, :) = 0.
    ZDER(:, :) = 0.
    ZDDR(:, :) = 0.
    ILFS(:) = IKB
    do JK = IKB, IKE
        ZLMASS(:, JK) = ZDXDY(:) * ZDPRES(:, JK) / XG  ! mass of model layer
    enddo
    ZLMASS(:, IKB) = ZLMASS(:, IKB + 1)
!
! closure variables
!
    ALLOCATE(ZTIMEC(ICONV))
    ALLOCATE(ZTHC(ICONV, IKS))
    ALLOCATE(ZRVC(ICONV, IKS))
    ALLOCATE(ZRCC(ICONV, IKS))
    ALLOCATE(ZRIC(ICONV, IKS))
    ALLOCATE(ZWSUB(ICONV, IKS))
!
!-------------------------------------------------------------------------------
!
!*           5.     Compute downdraft properties
!                   ----------------------------
!
    ZTIMEC(:) = XCTIME_SHAL
    if(OSETTADJ) ZTIMEC(:) = PTADJS
!
!*           7.     Determine adjusted environmental values assuming
!                   that all available buoyant energy must be removed
!                   within an advective time step ZTIMEC.
!                   ---------------------------------------------------
!
    call CONVECT_CLOSURE_SHAL(ICONV, KLEV, &
                              ZPRES, ZDPRES, ZZ, ZDXDY, ZLMASS, &
                              ZTHL, ZTH, ZRW, ZRC, ZRI, GTRIG1, &
                              ZTHC, ZRVC, ZRCC, ZRIC, ZWSUB, &
                              ILCL, IDPL, IPBL, ICTL, &
                              ZUMF, ZUER, ZUDR, ZUTHL, ZURW, &
                              ZURC, ZURI, ZCAPE, ZTIMEC, IFTSTEPS)
!
!-------------------------------------------------------------------------------
!
!*           8.     Determine the final grid-scale (environmental) convective
!                   tendencies and set convective counter
!                   --------------------------------------------------------
!
!
!*           8.1    Grid scale tendencies
!                   ---------------------
!
    ! in order to save memory, the tendencies are temporarily stored
    ! in the tables for the adjusted grid-scale values
!
    do JK = IKB, IKE
        ZTHC(:, JK) = (ZTHC(:, JK) - ZTH(:, JK)) / ZTIMEC(:) &
                      * (ZPRES(:, JK) / XP00)**ZRDOCP ! change theta in temperature
        ZRVC(:, JK) = (ZRVC(:, JK) - ZRW(:, JK) + ZRC(:, JK) + ZRI(:, JK)) &
                      / ZTIMEC(:)

        ZRCC(:, JK) = (ZRCC(:, JK) - ZRC(:, JK)) / ZTIMEC(:)
        ZRIC(:, JK) = (ZRIC(:, JK) - ZRI(:, JK)) / ZTIMEC(:)
!
    enddo
!
!
!*           8.2    Apply conservation correction
!                   -----------------------------
!
    ! adjustment at cloud top to smooth possible discontinuous profiles at PBL inversions
    ! (+ - - tendencies for moisture )
!
!
    if(LLSMOOTH) then
        do JI = 1, ICONV
            JK = ICTL(JI)
            JKM = MAX(2, ICTL(JI) - 1)
            JKP = MAX(2, ICTL(JI) - 2)
            ZRVC(JI, JKM) = ZRVC(JI, JKM) + .5 * ZRVC(JI, JK)
            ZRCC(JI, JKM) = ZRCC(JI, JKM) + .5 * ZRCC(JI, JK)
            ZRIC(JI, JKM) = ZRIC(JI, JKM) + .5 * ZRIC(JI, JK)
            ZTHC(JI, JKM) = ZTHC(JI, JKM) + .5 * ZTHC(JI, JK)
            ZRVC(JI, JKP) = ZRVC(JI, JKP) + .3 * ZRVC(JI, JK)
            ZRCC(JI, JKP) = ZRCC(JI, JKP) + .3 * ZRCC(JI, JK)
            ZRIC(JI, JKP) = ZRIC(JI, JKP) + .3 * ZRIC(JI, JK)
            ZTHC(JI, JKP) = ZTHC(JI, JKP) + .3 * ZTHC(JI, JK)
            ZRVC(JI, JK) = .2 * ZRVC(JI, JK)
            ZRCC(JI, JK) = .2 * ZRCC(JI, JK)
            ZRIC(JI, JK) = .2 * ZRIC(JI, JK)
            ZTHC(JI, JK) = .2 * ZTHC(JI, JK)
        enddo
    endif
!
!
    ! Compute vertical integrals - Fluxes
!
    JKM = MAXVAL(ICTL(:))
    ZWORK2(:) = 0.
    ZWORK2B(:) = 0.
    do JK = IKB + 1, JKM
        JKP = JK + 1
        do JI = 1, ICONV
            if(JK <= ICTL(JI)) then
                ZW1 = ZRVC(JI, JK) + ZRCC(JI, JK) + ZRIC(JI, JK)
                ZWORK2(JI) = ZWORK2(JI) + ZW1* & ! moisture
                             .5 * (ZPRES(JI, JK - 1) - ZPRES(JI, JKP)) / XG
                ZW1 = (XCPD + XCPV * ZRW(JI, JK)) * ZTHC(JI, JK) - &
                      (XLVTT + (XCPV - XCL) * (ZTT(JI, JK) - XTT)) * ZRCC(JI, JK) - &
                      (XLSTT + (XCPV - XCL) * (ZTT(JI, JK) - XTT)) * ZRIC(JI, JK)
                ZWORK2B(JI) = ZWORK2B(JI) + ZW1* & ! energy
                              .5 * (ZPRES(JI, JK - 1) - ZPRES(JI, JKP)) / XG
            endif
        enddo
    enddo
!
    ! Budget error (integral must be zero)
!
    do JI = 1, ICONV
        if(ICTL(JI) > IKB + 1) then
            JKP = ICTL(JI)
            ZW1 = XG / (ZPRES(JI, IKB) - ZPRES(JI, JKP) - &
                        .5 * (ZDPRES(JI, IKB + 1) - ZDPRES(JI, JKP + 1)))
            ZWORK2(JI) = ZWORK2(JI) * ZW1
            ZWORK2B(JI) = ZWORK2B(JI) * ZW1
        endif
    enddo
!
    ! Apply uniform correction
!
    do JK = JKM, IKB + 1, -1
    do JI = 1, ICONV
        if(ICTL(JI) > IKB + 1 .and. JK <= ICTL(JI)) then
            ! ZW1 = ABS(ZRVC(JI,JK)) +  ABS(ZRCC(JI,JK)) +  ABS(ZRIC(JI,JK)) + 1.E-12
            ! ZRVC(JI,JK) = ZRVC(JI,JK) - ABS(ZRVC(JI,JK))/ZW1*ZWORK2(JI)           ! moisture
            ZRVC(JI, JK) = ZRVC(JI, JK) - ZWORK2(JI)                                ! moisture
            ! ZRCC(JI,JK) = ZRCC(JI,JK) - ABS(ZRCC(JI,JK))/ZW1*ZWORK2(JI)
            ! ZRIC(JI,JK) = ZRIC(JI,JK) - ABS(ZRIC(JI,JK))/ZW1*ZWORK2(JI)
            ZTHC(JI, JK) = ZTHC(JI, JK) - ZWORK2B(JI) / XCPD                       ! enthalpy
        endif
    enddo
    enddo
!
    ! execute a "scatter"= pack command to store the tendencies in
    ! the final 2D tables
!
    do JK = IKB, IKE
    do JI = 1, ICONV
        JL = IJINDEX(JI)
        PTTEN(JL, JK) = ZTHC(JI, JK)
        PRVTEN(JL, JK) = ZRVC(JI, JK)
        PRCTEN(JL, JK) = ZRCC(JI, JK)
        PRITEN(JL, JK) = ZRIC(JI, JK)
    enddo
    enddo
!
!
!                   Cloud base and top levels
!                   -------------------------
!
    ILCL(:) = MIN(ILCL(:), ICTL(:))
    do JI = 1, ICONV
        JL = IJINDEX(JI)
        KCLTOP(JL) = ICTL(JI)
        KCLBAS(JL) = ILCL(JI)
    enddo
!
!
!*           8.7    Compute convective tendencies for Tracers
!                   ------------------------------------------
!
!  if ( OCH1CONV ) then
!!
!    ALLOCATE( ZCH1(ICONV,IKS,KCH1) )
!    ALLOCATE( ZCH1C(ICONV,IKS,KCH1) )
!    ALLOCATE( ZWORK3(ICONV,KCH1) )
!!
!    do JK = IKB, IKE
!    do JI = 1, ICONV
!      JL = IJINDEX(JI)
!      ZCH1(JI,JK,:) = PCH1(JL,JK,:)
!    end do
!    end do
!!
!    call CONVECT_CHEM_TRANSPORT( ICONV, KLEV, KCH1, ZCH1, ZCH1C,          &
!                                 IDPL, IPBL, ILCL, ICTL, ILFS, ILFS,      &
!                                 ZUMF, ZUER, ZUDR, ZDMF, ZDER, ZDDR,      &
!                                 ZTIMEC, ZDXDY, ZDMF(:,1), ZLMASS, ZWSUB, &
!                                 IFTSTEPS )
!!
!!
!!*           8.8    Apply conservation correction
!!                   -----------------------------
!!
!          ! Compute vertical integrals
!!
!    JKM = MAXVAL( ICTL(:) )
!    do JN = 1, KCH1
!      if(JN < NSV_LGBEG .OR. JN>NSV_LGEND-1) then ! no correction for xy lagrangian variables
!        ZWORK3(:,JN) = 0.
!        ZWORK2(:)    = 0.
!        do JK = IKB+1, JKM
!          JKP = JK + 1
!          do JI = 1, ICONV
!            ZW1 = .5 * (ZPRES(JI,JK-1) - ZPRES(JI,JKP))
!            ZWORK3(JI,JN) = ZWORK3(JI,JN) + (ZCH1C(JI,JK,JN)-ZCH1(JI,JK,JN)) * ZW1
!            ZWORK2(JI)    = ZWORK2(JI)    + ABS(ZCH1C(JI,JK,JN)) * ZW1
!          end do
!        end do
!!
!             ! Apply concentration weighted correction
!!
!        do JK = JKM, IKB+1, -1
!          do JI = 1, ICONV
!            if ( ICTL(JI) > IKB+1 .and. JK <= ICTL(JI) ) then
!              ZCH1C(JI,JK,JN) = ZCH1C(JI,JK,JN) -   &
!                                ZWORK3(JI,JN)*ABS(ZCH1C(JI,JK,JN))/MAX(1.E-30,ZWORK2(JI))
!            end if
!          end do
!        end do
!      end if
!!
!      do JK = IKB, IKE
!        do JI = 1, ICONV
!          JL = IJINDEX(JI)
!          PCH1TEN(JL,JK,JN) = (ZCH1C(JI,JK,JN)-ZCH1(JI,JK,JN) ) / ZTIMEC(JI)
!        end do
!      end do
!    end do
!  end if
!
!-------------------------------------------------------------------------------
!
!*           9.     Write up- and downdraft mass fluxes
!                   ------------------------------------
!
    do JK = IKB, IKE
        ZUMF(:, JK) = ZUMF(:, JK) / ZDXDY(:) ! Mass flux per unit area
    enddo
    ZWORK2(:) = 1.
    do JK = IKB, IKE
    do JI = 1, ICONV
        JL = IJINDEX(JI)
        if(KCLTOP(JL) <= IKB + 1) ZWORK2(JL) = 0.
        PUMF(JL, JK) = ZUMF(JI, JK) * ZWORK2(JL)
    enddo
    enddo
!
!-------------------------------------------------------------------------------
!
!*           10.    Deallocate all local arrays
!                   ---------------------------
!
! downdraft variables
!
    DEALLOCATE(ZDMF)
    DEALLOCATE(ZDER)
    DEALLOCATE(ZDDR)
    DEALLOCATE(ILFS)
    DEALLOCATE(ZLMASS)
!
!   closure variables
!
    DEALLOCATE(ZTIMEC)
    DEALLOCATE(ZTHC)
    DEALLOCATE(ZRVC)
    DEALLOCATE(ZRCC)
    DEALLOCATE(ZRIC)
    DEALLOCATE(ZWSUB)
!
    if(OCH1CONV) then
        DEALLOCATE(ZCH1)
        DEALLOCATE(ZCH1C)
        DEALLOCATE(ZWORK3)
    endif
!
!    vertical index
!
    DEALLOCATE(IDPL)
    DEALLOCATE(IPBL)
    DEALLOCATE(ILCL)
    DEALLOCATE(ICTL)
    DEALLOCATE(IETL)
!
! grid scale variables
!
    DEALLOCATE(ZZ)
    DEALLOCATE(ZPRES)
    DEALLOCATE(ZDPRES)
    DEALLOCATE(ZTT)
    DEALLOCATE(ZTH)
    DEALLOCATE(ZTHV)
    DEALLOCATE(ZTHL)
    DEALLOCATE(ZTHES)
    DEALLOCATE(ZRW)
    DEALLOCATE(ZRV)
    DEALLOCATE(ZRC)
    DEALLOCATE(ZRI)
    DEALLOCATE(ZDXDY)
!
! updraft variables
!
    DEALLOCATE(ZUMF)
    DEALLOCATE(ZUER)
    DEALLOCATE(ZUDR)
    DEALLOCATE(ZUTHL)
    DEALLOCATE(ZUTHV)
    DEALLOCATE(ZURW)
    DEALLOCATE(ZURC)
    DEALLOCATE(ZURI)
    DEALLOCATE(ZTHLCL)
    DEALLOCATE(ZTLCL)
    DEALLOCATE(ZRVLCL)
    DEALLOCATE(ZWLCL)
    DEALLOCATE(ZZLCL)
    DEALLOCATE(ZTHVELCL)
    DEALLOCATE(ZMFLCL)
    DEALLOCATE(ZCAPE)
!
! work arrays
!
    DEALLOCATE(IINDEX)
    DEALLOCATE(IJINDEX)
    DEALLOCATE(IJSINDEX)
    DEALLOCATE(GTRIG1)
!
!
ENDsubroutine SHALLOW_CONVECTION

! /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\
! **************************************************************************************************
! \/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/

