C   +-------------------------------------------------------------------+
C   |  Subroutine bilSet                                30-03-2022  JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Pre-computes data for bilinear interpolation for each NST point.  |
C   | If the interpolation is repeatedly performed on grids which share |
C   | the same dimensions and positions, the sampling points and other  |
C   | values can be buffered to avoid searching again said sampling     |
C   | points throughout the execution of NESTOR.                        |
C   |                                                                   |
C   | Input : with dimensions provided by NSTdim.inc:                   |
C   | ^^^^^^^ grd_Ix (ni)     : Input grid points position x(i)         |
C   |         grd_Iy (nj)     :   "     "    "       "     y(j)         |
C   |         SPHgrd (T/F)    : If true, spherical coordinates for      |
C   |                           input fields                            |
C   |         grd_Ox (mx, my) : Output grid positions x(i,j)            |
C   |         grd_Oy (mx, my) : Output grid positions y(i,j)            |
C   |                                                                   |
C   | Output: stored via intBil.inc:                                    |
C   | ^^^^^^^ bilNb : Number of useful sampling points for O(i,j)       |
C   |         bilPix (mx, my, 5, 2) : 2 indexes (x and y) for each      |
C   |                                 sampling point (x then y)         |
C   |         bilDat (mx, my, 5, 6) : 6 values for each sampling point  |
C   |                                 of O(i,j) involved in the         |
C   |                                 interpolation (see routine end)   |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE bilSet (grd_Ix, grd_Iy, SPHgrd, grd_Ox, grd_Oy)

      IMPLICIT NONE

      INCLUDE 'NSTdim.inc' ! Dimensions of LSC and NST domains
      INCLUDE 'intBil.inc'

C +---General and local variables
C +   ---------------------------

      ! sm = max samples
      INTEGER sm,LocDim,mmx,mmy,icent1,jcent1,icent2,jcent2,i,j,ii,jj,
     .        p,q,is,i1(4),j1(4),i2
      REAL dist_O,dist_I,AUXlo1,AUXlo2,AUXla1,AUXla2,dx,dy,degrad,x,y,
     .     epsi,AUXlon,MINlon,MAXlon,AUXlat,MINlat,MAXlat

      PARAMETER (sm=5) ! "Sample points maximum" (number of)
      PARAMETER (LocDim=21601) ! Dim. of local 1D arrays
      
      REAL grd_Ix(ni),grd_Iy(nj),grd_Ox(mx,my),grd_Oy(mx,my),
     .     tmp_Ix(0:LocDim+1),tmp_Iy(0:LocDim+1),samOx(sm),
     .     samOy(sm)

      LOGICAL SPHgrd

C +---Data
C +   ----

      DATA epsi   / 1.d-4          /
      DATA degrad / 1.745329252d-2 /

C +---Check dimensions of temporary arrays
C +   ====================================

      IF (ni.gt.LocDim .or. nj.gt.LocDim) THEN
        WRITE(6,*) 'bilSet - fatal error: dimension',LocDim
        WRITE(6,*) 'Please change LocDim   -   STOP'
        STOP
      ENDIF
      
C +---Check if the sampling technique is required
C +   ===========================================

      mmx = mx
      mmy = my
      
      dx    =(grd_Ix(ni/2)-grd_Ix(ni/2-1))*111111.
     .                   *COS(grd_Iy(nj/2)*degrad)
      dy    =(grd_Iy(nj/2)-grd_Iy(nj/2-1))*111111.
      dist_I=SQRT(dx*dx+dy*dy)

      icent1=MAX(1,mx/2)
      icent2=MAX(1,mx/2-1)
      jcent1=MAX(1,my/2)
      jcent2=MAX(1,my/2-1)
      IF (mmx.eq.2) icent1=2
      IF (mmy.eq.2) jcent1=2
      
      AUXlo1=grd_Ox(icent1,jcent1)
CWARNINGXla1=grd_Oy(icent1,icent1)
      AUXla1=grd_Oy(icent1,jcent1)
      AUXlo2=grd_Ox(icent2,jcent2)
      AUXla2=grd_Oy(icent2,jcent2)

C +        ******
      CALL SPHERC (SPHgrd,AUXlo1,AUXla1)
      CALL SPHERC (SPHgrd,AUXlo2,AUXla2)
C +        ******

      dx    =(AUXlo1-AUXlo2)*111111.*COS(AUXla1*degrad)
      IF (mmx.le.1) dx = 1000.
      dy    =(AUXla1-AUXla2)*111111.
      IF (mmy.le.1) dy = 1000.
      dist_O=SQRT(dx*dx+dy*dy)

      IF (dist_I.lt.dist_O) THEN
       bilNb=sm
      ELSE
       bilNb=1
      ENDIF

C +---Coordinates indexes inversion (if necessary)
C +   ============================================

C +---Storage in temporary arrays
C +   ---------------------------

      DO ii=1,ni
                          tmp_Ix(ii)=grd_Ix(ii)
       if(grd_Ix(ii)>180) grd_Ix(ii)=grd_Ix(ii)-360.
      ENDDO

      DO jj=1,nj
       tmp_Iy(jj)=grd_Iy(jj)
      ENDDO

C +---Revert grd_Ix (1) <--> grd_Ix (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Ix(ni).lt.grd_Ix(1)) THEN     
       DO ii=1,ni   
        tmp_Ix(ii)=grd_Ix(ni-ii+1) 
       ENDDO
      ENDIF

C +---Revert grd_Iy (1) <--> grd_Iy (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Iy(nj).lt.grd_Iy(1)) THEN     
       DO jj=1,nj   
        tmp_Iy(jj)=grd_Iy(nj-jj+1)
       ENDDO
      ENDIF

C +---Extended coordinates in longitude and latitude
C +   ==============================================

C +---Check validity of longitude
C +   ---------------------------

      IF (SPHgrd) THEN
       IF ((tmp_Ix(1).lt.(-180.)).or.(tmp_Ix(ni).gt.180.)) THEN
        WRITE(6,*) 'Longitudes of data fields are not between'
        WRITE(6,*) '-180 and +180 deg. (as required by bilSet)'
        WRITE(6,*) 'but rather between : '
        WRITE(6,*) tmp_Ix(1),tmp_Ix(ni)
        WRITE(6,*) '--- STOP in bilSet ---'
        STOP
       ENDIF
      ENDIF

C +---Extended left/right boundaries (longitude if SPHgrd)
C +   ----------------------------------------------------

      tmp_Ix(0)   =2.*tmp_Ix( 1)-tmp_Ix(2)
      tmp_Ix(ni+1)=2.*tmp_Ix(ni)-tmp_Ix(ni-1) 


C +---Extended bottom/top boundaries (latitude if SPHgrd)
C +   ---------------------------------------------------

      tmp_Iy(0)   =2.*tmp_Iy( 1)-tmp_Iy(2)
      tmp_Iy(nj+1)=2.*tmp_Iy(nj)-tmp_Iy(nj-1)


C +---Define extra lower and upper boundaries (latitude)
C +   --------------------------------------------------

      IF (SPHgrd) THEN     ! Stereographic coordinates

       IF (tmp_Iy(0).lt.(-90.))
     .  tmp_Iy(0)=MIN(-90.,tmp_Iy(1)-epsi)

       IF (tmp_Iy(nj+1).gt.90.)      
     .  tmp_Iy(nj+1)=MAX(90.,tmp_Iy(nj)+epsi)

      ENDIF


C +---Check the extension of the sub-domain to be read
C     ================================================

      AUXlon = grd_Ox(1,1)
      AUXlat = grd_Oy(1,1)
C +        ******
      CALL SPHERC (SPHgrd,AUXlon,AUXlat)
C +        ******
      MINlon = AUXlon
      MAXlon = AUXlon
      MINlat = AUXlat
      MAXlat = AUXlat

      DO j=1,my
      DO i=1,mx
       AUXlon = grd_Ox(i,j)
       AUXlat = grd_Oy(i,j)
C +         ******
       CALL SPHERC (SPHgrd,AUXlon,AUXlat)
C +         ******

       MINlon = min(AUXlon,MINlon)
       MAXlon = max(AUXlon,MAXlon)
       MINlat = min(AUXlat,MINlat)
       MAXlat = max(AUXlat,MAXlat)
      ENDDO
      ENDDO

      IF ((tmp_Ix(   0).gt.MINlon) .or.
     .    (tmp_Ix(ni+1).lt.MAXlon) .or.
     .    (tmp_Iy(   0).gt.MINlat) .or.
     .    (tmp_Iy(nj+1).lt.MAXlat)) THEN
       WRITE(6,*) 'Output domain is not (fully) included in'
       WRITE(6,*) 'the input domain.'
       WRITE(6,*) 'Input  domain :'
       WRITE(6,*) tmp_Ix(0),tmp_Ix(ni+1),tmp_Iy(0),tmp_Iy(nj+1)
       WRITE(6,*) 'Output domain :'
       WRITE(6,*) MINlon,MAXlon,MINlat,MAXlat
       WRITE(6,*) '--- STOP in bilSet ---'
      ENDIF

C +---Bi-linear interpolation
C +   =======================

C +---Some initialisations
C +   --------------------

      p=0
      q=0

      i1(1)=-1  ; j1(1)=+1
      i1(2)=+1  ; j1(2)=-1
      i1(3)=-1  ; j1(3)=-1
      i1(4)=+1  ; j1(4)=+1

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      DO i=1,mx   ! LOOP on output grid-points : BEGIN
      DO j=1,my  
          
C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

C +---Define sampling point positions
C +   -------------------------------

       IF (i.ne.1.and.i.ne.mx.and.j.ne.1.and.j.ne.my) THEN

        samOx(1)= grd_Ox(i,j)
        samOy(1)= grd_Oy(i,j)

        do i2=1,sm-1
         samOx(i2+1)=0.6*grd_Ox(i,j)+0.4*grd_Ox(i+i1(i2),j+j1(i2))
         samOy(i2+1)=0.6*grd_Oy(i,j)+0.4*grd_Oy(i+i1(i2),j+j1(i2))

         if(sign(1.,grd_Ox(i,j))              .ne.
     .      sign(1.,grd_Ox(i+i1(i2),j+j1(i2))).and.
     .       abs(   grd_Ox(i,j))              .ge. 170.0 ) then
          samOx(i2+1)=grd_Ox(i,j)
         endif

        enddo

       ELSE
        DO is=1,sm   ! Boundaries : no sampling
         samOx(is)=grd_Ox(i,j)
         samOy(is)=grd_Oy(i,j)
        ENDDO
       ENDIF
       
       DO is=1,bilNb ! Loop on the sampling points: BEGIN

        x=samOx(is)
        y=samOy(is)


C +---Check the range of latitude and longitude
C +   -----------------------------------------

C +          ******
        CALL SPHERC (SPHgrd,x,y)
C +          ******

C +---Search for the bottom-left corner of the surrounding mesh
C +   ---------------------------------------------------------

C +...This simple method accounts for the fact that two successive
C +...requests usually refer to neighbour points

C +---Search for dimension 1 value
C +   ----------------------------

        IF (tmp_Ix(p).le.x) THEN   ! Search upwards
         DO WHILE (tmp_Ix(p+1).lt.x)
          p=p+1
         ENDDO
        ELSE                       ! Search downwards
         DO WHILE (tmp_Ix(p).gt.x)
          p=p-1
         ENDDO
        ENDIF

C +---Search for dimension 2 value
C +   ----------------------------

        IF (tmp_Iy(q).le.y) THEN  ! Search upwards
         DO WHILE (tmp_Iy(q+1).lt.y)
          q=q+1
         ENDDO
        ELSE                      ! Search downwards
         DO WHILE (tmp_Iy(q).gt.y)
          q=q-1
         ENDDO
        ENDIF

C +---Check the validity of p/q indexes
C +   ---------------------------------

        IF ((p.lt.     0).or.(q.lt.     0).or.
     .      (p.gt.(ni+1)).or.(q.gt.(nj+1))) THEN
         WRITE (6,*) 'Inconsistency between input and output'
         WRITE (6,*) 'domains.'
         WRITE (6,*) 'p and q = ',p,q
         WRITE (6,*) '--- STOP in bilSet ---'
         STOP
        ENDIF

C +---Storing values for the linear interpolation
C +   -------------------------------------------

        bilPix(i,j,is,1) = p ! (Bottom-left) X index from input array
        bilPix(i,j,is,2) = q ! (Bottom-left) Y index from input array
        
        bilDat(i,j,is,1) = x ! Longitude for sampling point in O(i,j)
        bilDat(i,j,is,2) = tmp_Ix(p) ! x0
        bilDat(i,j,is,3) = tmp_Ix(p+1) ! x1
        bilDat(i,j,is,4) = y ! Latitude for sampling point in O(i,j)
        bilDat(i,j,is,5) = tmp_Iy(q) ! y0
        bilDat(i,j,is,6) = tmp_Iy(q+1) ! y1

       ENDDO          ! LOOP on the sampling points: END

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      ENDDO
      ENDDO           ! Loop on output grid-points : END

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      RETURN
      END

C   +-------------------------------------------------------------------+
C   |  Subroutine bilDo                                 30-03-2022  JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Performs the bilinear interpolation with the data that has been   | 
C   | previously prepared by the bicSet subroutine.                     |
C   |                                                                   |
C   | Input : grd_Ix (ni)     : Input grid points position x(i)         |
C   | ^^^^^^^ grd_Iy (nj)     :   "     "    "       "     y(j)         |
C   |         var_I  (ni, nj) : Input field values                      |
C   |         SPHgrd (T/F)    : If true, spherical coordinates for      |
C   |                           input fields                            |
C   |         bilNb : Number of useful sampling points (output from     |
C   |                 bilSet; integer)                                  |
C   |         bilPix (mx, my, 5, 2) : Indexes of sampling points        |
C   |                                 selected by bilSet                |
C   |         bilDat (mx, my, 5, 6) : Values precomputed by bilSet      |
C   |         The last three are provided by intBil.inc.                |
C   |                                                                   |
C   | Output: var_O  (mx, my) : Output field values                     |
C   | ^^^^^^^                                                           |
C   +-------------------------------------------------------------------+

      SUBROUTINE bilDo (grd_Ix, grd_Iy, var_I, SPHgrd, var_O)

      IMPLICIT NONE

      INCLUDE 'NSTdim.inc' ! Dimensions of LSC and NST domains
      INCLUDE 'intBil.inc'

C +---General and local variables
C +   ---------------------------

      INTEGER    i,j,ii,jj,p,q,is,ind0,ind1,LocDim,sm

      PARAMETER (sm=5) ! "Samples maximum" (number of)
      PARAMETER (LocDim=21601) ! Dim. of local 1D arrays
      
      REAL x,y,tmp,tmp2,x0,x1,y0,y1,epsi,AUXlon,MINlon,MAXlon,
     .     AUXlat,MINlat,MAXlat

      REAL grd_Ix(ni),grd_Iy(nj),tmp_in(0:ni+1,0:nj+1),
     .     tmp_Ix(0:LocDim+1),tmp_Iy(0:LocDim+1),var_I(ni,nj),
     .     var_O(mx,my)

      LOGICAL SPHgrd

C +---Data
C +   ----

      DATA epsi   / 1.d-4          /

C +---Check dimensions of temporary arrays
C +   ====================================

      IF (ni.gt.LocDim .or. nj.gt.LocDim) THEN
        WRITE(6,*) 'bilDo - fatal error: dimension',LocDim
        WRITE(6,*) 'Please change LocDim   -   STOP'
        STOP
      ENDIF

C +---Coordinates indexes inversion (if necessary)
C +   ============================================

C +---Storage in temporary arrays
C +   ---------------------------

      DO jj=1,nj
      DO ii=1,ni 
       tmp_in(ii,jj)=var_I(ii,jj)
      ENDDO
      ENDDO

C +---Revert grd_Ix (1) <--> grd_Ix (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Ix(ni).lt.grd_Ix(1)) THEN     
       DO ii=1,ni   
        DO jj=1,nj                       
         tmp_in(ii,jj)=var_I(ni-ii+1, jj)
        ENDDO
        tmp_Ix(ii)=grd_Ix(ni-ii+1) 
       ENDDO
      ENDIF

C +---Revert grd_Iy (1) <--> grd_Iy (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Iy(nj).lt.grd_Iy(1)) THEN     
       DO jj=1,nj   
        DO ii=1,ni                       
         tmp_in(ii,jj)=var_I(ii,nj-jj+1)
        ENDDO
        tmp_Iy(jj)=grd_Iy(nj-jj+1)
       ENDDO
      ENDIF

C +---Extended coordinates in longitude and latitude
C +   ==============================================

C +---Extended left/right boundaries (longitude if SPHgrd)
C +   ----------------------------------------------------

      tmp_Ix(0)   =2.*tmp_Ix( 1)-tmp_Ix(2)
      tmp_Ix(ni+1)=2.*tmp_Ix(ni)-tmp_Ix(ni-1) 

C +---Extended bottom/top boundaries (latitude if SPHgrd)
C +   ---------------------------------------------------

      tmp_Iy(0)   =2.*tmp_Iy( 1)-tmp_Iy(2)
      tmp_Iy(nj+1)=2.*tmp_Iy(nj)-tmp_Iy(nj-1)

C +---Define the cyclic field in longitude
C +   ------------------------------------

      IF (SPHgrd) THEN     ! Stereographic coordinates

       ind0=-1
       ind1=-1
 
       AUXlon=tmp_Ix(0)+360.
       DO i=1,ni
        IF (ABS(AUXlon-tmp_Ix(i)).lt.epsi) ind0=i
       ENDDO
 
       AUXlon=tmp_Ix(ni+1)-360.
       DO i=1,ni
        IF (ABS(AUXlon-tmp_Ix(i)).lt.epsi) ind1=i
       ENDDO
 
       IF (NOT(ind0.gt.(-1).and.ind1.gt.(-1))) THEN
        ind0=ni
        ind1= 1
       ENDIF

      ELSE                 ! Non spherical coordinates

       ind0=ni
       ind1= 1

      ENDIF

      DO j=1,nj
       tmp_in(   0,j)=tmp_in(ind0,j)
       tmp_in(ni+1,j)=tmp_in(ind1,j)
      ENDDO

C +---Define extra lower and upper boundaries (latitude)
C +   --------------------------------------------------

      DO i=0,ni+1
       tmp_in(i,   0)=tmp_in(i, 1)
       tmp_in(i,nj+1)=tmp_in(i,nj)
      ENDDO

C +---Bi-linear interpolation (prepared by bilSet)
C +   ============================================

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      DO i=1,mx   ! LOOP on output grid-points : BEGIN
      DO j=1,my  

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

       tmp2=0.0       ! Initialisation of sum of sampled values

       DO is=1,bilNb ! Loop on the sampling points: BEGIN

        p = bilPix(i,j,is,1)
        q = bilPix(i,j,is,2)
        
        x = bilDat(i,j,is,1)
        y = bilDat(i,j,is,4)
        x0 = bilDat(i,j,is,2)
        x1 = bilDat(i,j,is,3)
        y0 = bilDat(i,j,is,5)
        y1 = bilDat(i,j,is,6)

        tmp=(x-x0)*((y-y0)*tmp_in(p+1,q+1)
     .             +(y1-y)*tmp_in(p+1,q  ))
     .     +(x1-x)*((y-y0)*tmp_in(p  ,q+1)
     .             +(y1-y)*tmp_in(p  ,q  ))
        tmp2=tmp2+tmp/((x1-x0)*(y1-y0))

       ENDDO          ! LOOP on the sampling points: END

C +---Output value given by the average of the samplings
C +   --------------------------------------------------

       var_O(i,j)=tmp2/REAL(bilNb)

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      ENDDO
      ENDDO           ! Loop on output grid-points : END

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      RETURN
      END
