C   +-------------------------------------------------------------------+
C   |  Subroutine bicSet                                10-05-2022  JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Pre-computes the data for bicubic interpolation, i.e., finds for  |
C   | each NST cell the closest 4x4 square found in the LSC domain.     |
C   |                                                                   |
C   | Input : with dimensions provided by NSTdim.inc:                   |
C   | ^^^^^^^ grd_Ix (ni)     : Input grid points position x(i)         |
C   |         grd_Iy (nj)     :   "     "    "       "     y(j)         |
C   |         grd_Ox (mx, my) : Output grid positions x(i,j)            |
C   |         grd_Oy (mx, my) : Output grid positions y(i,j)            |
C   |                                                                   |
C   | Output: stored via intBic.inc:                                    |
C   | ^^^^^^^ bicNb : Number of useful sampling points for O(i,j) (in   |
C   |                 practice, 1 or 16, 1 being used if LSC grid is    |
C   |                 4x4 or smaller)                                   |
C   |         bicSqr (mx, my, 2) : For each NST point, stores the       |
C   |                              indexes in the LSC grid of the top   |
C   |                              left cell of the sampling square     |
C   |                              (4x4 region in the LSC grid). If     |
C   |                              bicNb is set to 1, this will be the  |
C   |                              sole cell used by the interpolation. |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE bicSet (grd_Ix, grd_Iy, grd_Ox, grd_Oy)

      IMPLICIT NONE

      INCLUDE 'NSTdim.inc' ! Dimensions of LSC and NST domains
      INCLUDE 'intBic.inc' ! To save interpolation data
      
      INTEGER i,j,k,l,imin,jmin,sqimin,sqimax,sqjmin,sqjmax,corner
      REAL grd_Ix(ni),grd_Iy(nj),grd_Ox(mx,my),grd_Oy(mx,my),
     .     tmp_Ix(ni),tmp_Iy(nj),dist,curDst,minDst,dists(4),crnDst
      LOGICAL border
      
      border = .false.
      
      IF ((ni.le.4).or.(nj.le.4)) THEN
       bicNb = 1
      ELSE
       bicNb = 16
      ENDIF
      
      ! For each NST pixel
      DO i=1,mx; DO j=1,my
      
       ! Finds closest LSC pixel
       minDst = 10000.
       DO k=1,ni; DO l=1,nj
       
        curDst = dist(grd_Ix(k),grd_Iy(l),grd_Ox(i,j),grd_Oy(i,j))
        IF (curDst.lt.minDst) THEN
        
         minDst = curDst
         imin = k
         jmin = l
         
        END IF
        
       ENDDO; ENDDO
       
       ! Extreme case: very small grid (unlikely), uses closest pixel
       IF (bicNb.eq.1) THEN
       
        bicSqr(i,j,1) = imin
        bicSqr(i,j,2) = jmin
        
       ELSE
       
        ! Important remark for the next instructions: the closest 
        ! pixel is by default considered to be the bottom right pixel 
        ! of the top left 2x2. I.e. (- = pixel, x = closest pixel):
        ! 
        ! - - - -
        ! - x - -
        ! - - - -
        ! - - - -

        ! Evaluates if encompassing 4x4 crosses borders of LSC grid.
        sqimin = imin-1
        sqimax = imin+2
        sqjmin = jmin-1
        sqjmax = jmin+2
        
        IF (sqimin.lt.1.or.sqimax.gt.ni.or.
     .      sqjmin.lt.1.or.sqjmax.gt.nj) THEN
         border = .true.
        ENDIF
        
        IF (sqimin.lt.1) THEN
         sqimin = sqimin+1
        ELSE IF (sqimax.gt.ni) THEN
         sqimin = sqimin-(sqimax-ni)
        ENDIF
        
        IF (sqjmin.lt.1) THEN
         sqjmin = sqjmin+1
        ELSE IF (sqjmax.gt.nj) THEN
         sqjmin = sqjmin-(sqjmax-nj)
        ENDIF
        
        ! If 4x4 initially crossed borders or is right next to them
        IF (border.or.sqimin.eq.1.or.sqjmin.eq.1) THEN
        
         ! Saves top left corner of the resulting 4x4 square
         bicSqr(i,j,1) = sqimin
         bicSqr(i,j,2) = sqjmin
        
        ! Else, finds the closest corner of the encompassing 5x5
        ELSE

         ! 1 = top left, 2 = top right, 3 = bot right, 4 = bot left
         dists(1) = dist(grd_Ix(sqimin-1),grd_Iy(sqjmin-1),
     .              grd_Ox(i,j),grd_Oy(i,j))
         dists(2) = dist(grd_Ix(sqimax),grd_Iy(sqjmin-1),
     .              grd_Ox(i,j),grd_Oy(i,j))
         dists(3) = dist(grd_Ix(sqimax),grd_Iy(sqjmax),
     .              grd_Ox(i,j),grd_Oy(i,j))
         dists(4) = dist(grd_Ix(sqimin-1),grd_Iy(sqjmax),
     .              grd_Ox(i,j),grd_Oy(i,j))

         ! Finds closest corner to adjust sqimin,sqjmin
         corner = 1
         crnDst = dists(1)

         DO k=2,4

          IF (dists(k).lt.crnDst) THEN

           corner = k
           crnDst = dists(k)

          ENDIF

         ENDDO
         
         ! Adjusts sqimin, sqjmin based on the closest corner
         IF (corner.eq.1) THEN
         
          sqimin = sqimin-1
          sqjmin = sqjmin-1
         
         ELSE IF (corner.eq.2) THEN
          sqjmin = sqjmin-1
         ELSE IF (corner.eq.4) THEN
          sqimin = sqimin-1
         ENDIF
         
         ! Saves top left corner of the final 4x4 square
         bicSqr(i,j,1) = sqimin
         bicSqr(i,j,2) = sqjmin
        
        ENDIF
        
       ENDIF
        
      ENDDO; ENDDO
      
      RETURN
      END

C   +-------------------------------------------------------------------+
C   |  Subroutine bicDo                                 10-05-2022  JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Performs the bicubic interpolation with the 4x4 sampling square   |
C   | that has previously been precomputed for each O(i,j) by bicSet.   |
C   |                                                                   |
C   | Input : with dimensions provided by NSTdim.inc:                   |
C   | ^^^^^^^ grd_Ix (ni)     : Input grid points position x(i)         |
C   |         grd_Iy (nj)     :   "     "    "       "     y(j)         |
C   |         var_I  (ni, nj) : Input field values                      |
C   |         grd_Ox (mx, my) : Output grid positions x(i,j)            |
C   |         grd_Oy (mx, my) : Output grid positions y(i,j)            |
C   |         bicNb : Number of sampling points (from bicSet; integer)  |
C   |         bicSqr (mx, my, 2) : Indexes of the top left corner of    |
C   |                              the 4x4 sampling square (in LSC      |
C   |                              grid) as selected by bicSet          |
C   |         The last two are provided by intBic.inc.                  |
C   |                                                                   |
C   | Output: var_O  (mx, my) : Output field values                     |
C   | ^^^^^^^                                                           |
C   +-------------------------------------------------------------------+
      
      SUBROUTINE bicDo (grd_Ix, grd_Iy, var_I, grd_Ox, grd_Oy, var_O)
      
      IMPLICIT NONE

      INCLUDE 'NSTdim.inc' ! Dimensions of LSC and NST domains
      INCLUDE 'intBic.inc' ! Buffered interpolation data
      
      INTEGER i,j,k,l
      REAL grd_Ix(ni),grd_Iy(nj),var_I(ni,nj),
     .     grd_Ox(mx,my),grd_Oy(mx,my),var_O(mx,my),
     .     axLon(4), axLat(4), square(4,4), coeffs(4,4)
      
      ! For each NST pixel
      DO i=1,mx; DO j=1,my
      
       IF (bicNb.eq.1) THEN
        var_O(i,j) = var_I(bicSqr(i,j,1),bicSqr(i,j,2))
       ELSE
       
        ! Buffers longitudes, latitudes, values
        DO k=1,4
         axLon(k) = grd_Ix(bicSqr(i,j,1)+k-1)
        ENDDO
        
        DO k=1,4
         axLat(k) = grd_Iy(bicSqr(i,j,2)+k-1)
        ENDDO
        
        DO k=1,4; DO l=1,4
         square(k,l) = var_I(bicSqr(i,j,1)+k-1,bicSqr(i,j,2)+l-1)
        ENDDO; ENDDO
        
        ! Inverts data to comply with Numerical Recipes requirements.
        ! I.e., x(1) < x(2) < ... => revert if necessary.

C +---Revert axLon (1) <--> axLon (4) ?
C +   ---------------------------------

        IF (axLon(4).lt.axLon(1)) THEN
         DO k=1,4
          DO l=1,4
           square(k,l)=square(4-k+1,l)
          ENDDO
          axLon(k)=axLon(4-k+1)
         ENDDO
        ENDIF

C +---Revert axLat (1) <--> axLat (4) ?
C +   ---------------------------------

        IF (axLat(4).lt.axLat(1)) THEN
         DO l=1,4
          DO k=1,4
           square(k,l)=square(k,4-l+1)
          ENDDO
          axLat(l)=axLat(4-l+1)
         ENDDO
        ENDIF
        
        ! Performs the bicubic interpolation on the 4x4 sampling square
        CALL SPLIE2(axLon,axLat,square,4,4,coeffs)
        CALL SPLIN2(axLon,axLat,square,coeffs,4,4,
     .              grd_Ox(i,j),grd_Oy(i,j),var_O(i,j))
       
       ENDIF
      
      ENDDO; ENDDO
      
      RETURN
      END
      
C     +--------------------------------------------------------------+
C     |     * From numerical recipes (H. Press et al., 1992)         |
C     +--------------------------------------------------------------+

      SUBROUTINE SPLIE2(X1A,X2A,YA,M,N,Y2A)
 
      PARAMETER (NN=4)
C +...NN = max value allowed for N (for 1D arrays only -> overdim.)
 
      DIMENSION X1A(M),X2A(N),YA(M,N),Y2A(M,N)
      DIMENSION YTMP(NN),Y2TMP(NN)
 
      DO 13 J=1,M
 
        DO 11 K=1,N
          YTMP(K)=YA(J,K)
11      CONTINUE
 
C +          ******
        CALL SPLINE(X2A,YTMP,N,1.E30,1.E30,Y2TMP)
C +          ******
C +...  NB : 1.E30 = switch value to select "natural" bicub spline
 
        DO 12 K=1,N
          Y2A(J,K)=Y2TMP(K)
12      CONTINUE
 
13    CONTINUE
 
      RETURN
      END
 
C     +--------------------------------------------------------------+
C     |     * From numerical recipes (H. Press et al., 1992)         |
C     +--------------------------------------------------------------+

      SUBROUTINE SPLIN2(X1A,X2A,YA,Y2A,M,N,X1,X2,Y)
 
      PARAMETER (NN=4)
C +   NN = max value allowed for N (for 1D arrays only -> overdim.)
 
      DIMENSION X1A(M),X2A(N),YA(M,N),Y2A(M,N),YTMP(NN),Y2TMP(NN)
      DIMENSION YYTMP(NN)

      DO 12 J=1,M

       DO 11 K=1,N

        YTMP(K)=YA(J,K)
        Y2TMP(K)=Y2A(J,K)

11     CONTINUE

C +         ******
       CALL SPLINT(X2A,YTMP,Y2TMP,N,X2,YYTMP(j))
C +         ******
       
12    CONTINUE

C +        ******
      CALL SPLINE(X1A,YYTMP,M,1.E30,1.E30,Y2TMP)
C +        ******
      CALL SPLINT(X1A,YYTMP,Y2TMP,M,X1,Y)
C +        ******

      RETURN
      END
