C   +-------------------------------------------------------------------+
C   |  Subroutine XCPvgd                                 12-04-2022 JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Vertical grid of the ECMWF/NCEP models. Tailored for a 2D grid.   |
C   | There were initially separate routines for both ECP and NCP modes |
C   | (see calls in LSCvgd.f), but due to the code being nearly         |
C   | identical, one logical parameter was added to distinguish both    |
C   | modes when coming across the set of instructions that differ      |
C   | between the original routines. The XCPvgd name refers to how both |
C   | modes share the same suffix -CP.                                  |
C   |                                                                   |
C   | Input : - fID    : identificator of the Netcdf data file          |
C   | ^^^^^^^ - nk     : number of vertical levels                      |
C   |         - baseI  : minimum X index of the relevant LSC sub-region |
C   |         - baseJ  : minimum Y index of the relevant LSC sub-region |
C   |         - maxI   : maximum X index of the relevant LSC sub-region |
C   |         - maxJ   : maximum Y index of the relevant LSC sub-region |
C   |         - klev   : if specified, the level at which pressure and  |
C   |                    hybrid coordinate has to be computed           |
C   |         - isNCP  : true if using the NCEP model                   |
C   |         - XCP_sp(ni,nj) : surface pressure                        |
C   |                                                                   |
C   | Output: Vertical grid of the ECMWF model :                        |
C   | ^^^^^^^ - XCP__p(ni,nj,nk+1) : pressure at each level  [kPa]      |
C   |         - XCP__z(ni,nj,nk+1) : hybrid coordinates                 |
C   |                                                                   |
C   | Remarks on optimization via sub-region selection (29/05/2022):    |
C   | -to compute the vertical grid for the full LSC domain, use        |
C   |  baseI=1, baseJ=1, maxI=ni, maxJ=nj.                              |
C   | -code assumes that the user will use 1 <= baseI <= maxI <= ni and |
C   |  1 <= baseJ <= maxJ <= nj.                                        |
C   | -if the variables baseI, baseJ, maxI and maxJ are set to delimit  |
C   |  a sub-region of the LSC grid, only this sub-region will be       |
C   |  completed in the output grids.                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE XCPvgd(fID,ni,nj,nk,baseI,baseJ,maxI,maxJ,klev,isNCP,
     .                  XCP_sp,XCP__p,XCP__z)

      IMPLICIT NONE

C +---Local variables
C +   ---------------

      INTEGER fID,ni,nj,baseI,baseJ,maxI,maxJ,nk,klev,i,j,k,k1,k2
 
      REAL pp,ppm,pps,ppf,pp1,dpsl,hh,empty0(nk+1),empty1(1),
     .     XCP_sp(ni,nj),XCP__p(ni,nj,nk+1),XCP__z(ni,nj,nk+1),
     .     plevel(nk)

      CHARACTER*10 var_units

      LOGICAL isNCP 

C +---Atmospheric levels: pressure levels
C +   -----------------------------------

C +        ******
C +---J.-F. Grailet: in the original code, the vector used as the 
C +---tenth argument was not used in any meaningful way, hence why 
C +---it was replaced with empty0.
C +---     ******
      CALL UNread (fID,'level',0,0,0,0,nk,1,1,empty0,empty1,empty1,
     .             var_units,plevel)
C +        ******


C +---Computation for a given level or all levels ?
C +   ---------------------------------------------

      IF ((klev.le.0).or.(klev.gt.nk)) THEN
       k1=1
       k2=nk
      ELSE
       k1=1
       k2=klev
      ENDIF
      
      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

C +---For each i,j pixel (start of grid traversal)
C +   --------------------------------------------
C +   29/05/2022: added a small optimization; grid traversal now only 
C +   takes account of the sub-region of the LSC domain which includes 
C +   the NST domain.

      DO i=baseI,maxI ! i=1,ni
      DO j=baseJ,maxJ ! j=1,nj

C +---Compute pressure at each levels
C +   -------------------------------

       DO k=k1,k2
        XCP__p(i,j,k)=plevel(k)/10.  ! (kPa)
c       IF (XCP__p(i,j,k).gt.XCP_sp(i,j)) THEN
c        IF (isNCP) THEN
c         XCP__p(i,j,k)=XCP_sp(i,j)-REAL(k)*0.1
c        ELSE
c         XCP__p(i,j,k)=XCP_sp(i,j)-REAL(k)*0.01
c        ENDIF
c       ENDIF
       ENDDO

       IF (isNCP) THEN
        ! WARNING (X. Fettweis)
        ! The NCEP and MERRA2 files are built with the surface 
        ! variable at the first pressure level
        ! Q2,U10,V10,T2
        XCP__p(i,j,nk)=XCP_sp(i,j)
        XCP__p(i,j,nk+1)=XCP_sp(i,j)
       ELSE
        XCP__p(i,j,nk+1)=105.
       ENDIF

C +---Compute hybrid coordinates (required by nesting procedure)
C +   --------------------------
C +...Local hybrid coordinate: set parameters:

       pps = XCP_sp(i,j)
       ppm = pps - dpsl
       DO k = k1,k2+1
        pp = XCP__p(i,j,k)
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        XCP__z(i,j,k) = LOG(hh)
       ENDDO
      
      END DO; END DO ! End of grid traversal

      RETURN
      END
