C   +-------------------------------------------------------------------+
C   |  Subroutine MRLvgd                                 13-04-2022 JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Creation of the vertical grid of the MAR model (as LSC source).   |
C   | MRL stands for "MaR Lsc". This code was initially part of the     |
C   | MARvgd original routine, but was separated in two routines        |
C   | (MRLvgd and MRNvgd) for convenience and readability.              |
C   |                                                                   |
C   | Input : - fID    : identificator of the Netcdf data file          |
C   | ^^^^^^^ - nk     : number of vertical levels                      |
C   |         - baseI  : minimum X index of the relevant LSC sub-region |
C   |         - baseJ  : minimum Y index of the relevant LSC sub-region |
C   |         - maxI   : maximum X index of the relevant LSC sub-region |
C   |         - maxJ   : maximum Y index of the relevant LSC sub-region |
C   |         - klev   : if specified, the level at which pressure and  |
C   |                    hybrid coordinate has to be computed           |
C   |         - VGD_sp(ni,nj) : surface pressure (kPa)                  |
C   |                                                                   |
C   | Output: Vertical grid of the MAR model in hybrid coordinates :    |
C   | ^^^^^^^ - VGD__p(ni,nj,nk+1) : pressure coordinates (kPa)         |
C   |         - VGD_hp(ni,nj,nk+1) : local hybrid coord. for vertical   |
C   |                                interpolation                      |
C   |                                                                   |
C   | Remarks on optimization via sub-region selection (29/05/2022):    |
C   | -to compute the vertical grid for the full LSC domain, use        |
C   |  baseI=1, baseJ=1, maxI=ni, maxJ=nj.                              |
C   | -code assumes that the user will use 1 <= baseI <= maxI <= ni and |
C   |  1 <= baseJ <= maxJ <= nj.                                        |
C   | -if the variables baseI, baseJ, maxI and maxJ are set to delimit  |
C   |  a sub-region of the LSC grid, only this sub-region will be       |
C   |  completed in the output grids.                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE MRLvgd (fID,ni,nj,nk,baseI,baseJ,maxI,maxJ,klev,
     .                   VGD_sp,VGD__p,VGD_hp)

      IMPLICIT NONE

C +---Local variables
C +   ---------------

      INTEGER fID,ni,nj,baseI,baseJ,maxI,maxJ,nk,klev,i,j,k,k1,k2

      REAL pp1,pps,ppm,dpsl,pp,hh,ppf,ptopDY,empty1(1)
     
      REAL VGD_sp(ni,nj),VGD_hp(ni,nj,nk+1),VGD__p(ni,nj,nk+1),sigma(nk)

      CHARACTER*10 var_units
      
C +---CREATION OF SIGMA MAR GRID USING LSC NetCDF FILE
C +   ================================================

C +---Read SIGMA in NetCDF file
C +   -------------------------

      CALL UNsread (fID,'level',0,0,0,0,nk,1,1,var_units,sigma)

C +---HYBRID AND PRESSURE COORDINATES (required by the nesting code)
C +   ===============================

C +---Selection of vertical levels
C +   ----------------------------

      IF ((klev.le.0).or.(klev.gt.nk)) THEN
       k1=1
       k2=nk
      ELSE
       k1=1
       k2=klev
      ENDIF

C +---Reference levels for hybrid coordinates
C +   ---------------------------------------

      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

      ptopDY = 0.01

C +---For each i,j pixel (start of grid traversal)
C +   --------------------------------------------
C +   29/05/2022: added a small optimization; grid traversal now only 
C +   takes account of the sub-region of the LSC domain which includes 
C +   the NST domain.

      DO i=baseI,maxI ! i=1,ni
      DO j=baseJ,maxJ ! j=1,nj

C +---Computation of hybrid coordinates used in vertic. interp.
C +   ---------------------------------------------------------
      
       pps = VGD_sp(i,j)
       ppm = pps - dpsl
       DO k = k1,k2+1
        IF (k.eq.(nk+1)) THEN
         pp = VGD_sp(i,j)
        ELSE
         pp = sigma(k)*(VGD_sp(i,j)-ptopDY) + ptopDY
        ENDIF
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        ENDIF
        VGD_hp(i,j,k) = LOG(hh)
        VGD__p(i,j,k) = pp
       ENDDO
      
      END DO; END DO ! End of grid traversal

      RETURN
      END
