C   +-------------------------------------------------------------------+
C   |  File contents:                                                   |
C   |     intHor                                                        |
C   |     bilSim                                                        |
C   |     intLin                                                        |
C   |     SPHERC                                                        |
C   |     SEARCH                                                        |
C   |     SPLINE, SPLINT (from Numerical Recipies)                      |
C   |     dist (computes a distance between two lon,lat points)         |
C   |  Additional files complementing this one:                         |
C   |     intBil.f (subroutines for bilinear interpolation)             |
C   |     intBic.f (subroutines for bicubic interpolation)              |
C   |     intMAR.f (subroutine for MAR on MAR forcing)                  |
C   +-------------------------------------------------------------------+

C   +-------------------------------------------------------------------+
C   |  Subroutine intHor                               Dec. 95  NESTING |
C   |                                                  (Rev 2002 may)   |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Horizontal interpolation from LSC grid to NST grid distribute     |
C   | tasks to bicubic, linear... routines according to the "intype"    |
C   | variable (1=bilinear, 3=bicubic).                                 |
C   | Note that this routine uses the dimensions specified in NSTdim.inc|
C   | The bilinear interpolation is able to treat cyclic domains, or    |
C   | domains including the South/North pole.                           |
C   |                                                                   |
C   | Input : intype         : requested interpolation type             |
C   | ^^^^^^^ grd_Ix (ni,nj) : input grid points position x(i,j)        |
C   |         grd_Iy (ni,nj) : input grid points position y(i,j)        |
C   |         var_I  (ni,nj) : input field values                       |
C   |         SPHgrd (T/F)   : if true, spherical coordinates  for      |
C   |                          input fields                             |
C   |         grd_Ox (mx,my) : output grid positions x(i,j)             |
C   |         grd_Oy (mx,my) : output grid positions y(i,j)             |
C   |         REGgrd (T/F)   : if true, means the input grid is regular |
C   |                                                                   |
C   | Output: var_O  (mx,my) : output field values                      |
C   | ^^^^^^^ pos_Ox (mx,my) : retained posit.for non-regular grid(long)|
C   |         pos_Oy (mx,my) : retained posit.for non-regular grid (lat)|
C   |                                                                   |
C   | J.-F. Grailet note (27/04/2022): this routine now takes advantage |
C   | of the new intBil.f and intBic.f libraries to store interpolation |
C   | data at the first interpolation to speed up the subsequent ones.  |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE intHor (intype,grd_Ix,grd_Iy,var_I,
     .                   SPHgrd,grd_Ox,grd_Oy,var_O,
     .                   REGgrd,pos_Ox,pos_Oy)
 
C +---LSC and NST domain dimensions
C +   -----------------------------

      include 'NSTdim.inc'

C +---Local variables
C +   ---------------

      INTEGER intype,i,j

      INTEGER pos_Ox(mx,my),pos_Oy(mx,my)

      REAL grd_Ix(ni,nj),grd_Iy(ni,nj),var_I(ni,nj),
     .     grd_Ox(mx,my),grd_Oy(mx,my),var_O(mx,my)

      LOGICAL SPHgrd,REGgrd

C +---Temporary arrays
C +   ----------------

      REAL tmp_I2a(ni,nj),tmp1in(ni,nj), ! tmp2in(0:ni+1,0:nj+1),
     .     grd1Ix(ni),grd1Iy(nj)

C +---Logical to know if interpolation data has been buffered
C +   -------------------------------------------------------
C +   Addition made by J.-F. Grailet on 27/04/2022, based on the same 
C +   mechanism as "lfirst" in MARvgd in the unmodified NESTOR.

      LOGICAL ifirst
      SAVE    ifirst
      DATA ifirst/.true./

C +---Interpolation
C +   -------------

      IF (REGgrd) THEN            ! Regular input grid

       DO i=1,ni
        grd1Ix(i)=grd_Ix(i,1)
        if(grd1Ix(i)>180) grd1Ix(i)=grd1Ix(i)-360.
       ENDDO

       DO j=1,nj
        grd1Iy(j)=grd_Iy(1,j)
       ENDDO

       ! Bilinear interpolation
       IF (intype.EQ.1) THEN

         IF (ifirst) THEN
          CALL bilSet(grd1Ix, grd1Iy, SPHgrd, grd_Ox, grd_Oy)
          ifirst=.false.
         ENDIF
         CALL bilDo(grd1Ix, grd1Iy, var_I, SPHgrd, var_O)
       
       ! Bicubic interpolation
       ELSE IF (intype.EQ.3) THEN 

         IF (ifirst) THEN
          CALL bicSet(grd1Ix, grd1Iy, grd_Ox, grd_Oy)
          ifirst=.false.
         ENDIF
         CALL bicDo(grd1Ix, grd1Iy, var_I, grd_Ox, grd_Oy, var_O)

       ENDIF

      ! Non-regular input grid (MAR forced by MAR in practice)
      ELSE

       CALL intMAR (grd_Ix,grd_Iy,var_I,grd_Ox,grd_Oy,var_O,
     .              pos_Ox,pos_Oy)

      ENDIF

      RETURN
      END

C   +-------------------------------------------------------------------+
C   |  Subroutine bilSim                            01-07-2004  NESTING |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | This routine is a bilinear interpolation of a 2D scalar fields.   |
C   | If the output resolution is lower than input, an average of 5     |
C   | bilinear interpolations is performed, considering 5 sampling      |
C   | points located around the selected point in the output mesh.      |
C   | Note that a specific treatment of latitudes/longitudes is         |
C   | included for input grids using spherical coordinates.             |
C   |                                                                   |
C   | J.-F. Grailet note (09/05/2022): the "sim" is for "simple" as     |
C   | this specific routine is subtly different from calling            |
C   | successively bilSet and bilDo. Moreover, it is meant to work with |
C   | varying dimensions in practice, hence why its parameters still    |
C   | include them rather than including NSTdim.inc. In the original    |
C   | code, this routine was called INTsimple and located in ETOPO1.f.  |
C   | Since it's also called by the ICEmsk routine, it was moved here.  |
C   |                                                                   |
C   | It should be noted that this specific routine, beyond a few minor |
C   | changes (like the name), is untouched compared to the unmodified  |
C   | NESTOR. This is because ETOPO1 and ICEmsk both only call it once, |
C   | at the very start of NESTOR, meaning no optimization is needed.   |
C   | Moreover, in the original NESTOR, this routine is called by       |
C   | ETOPO1 and ICEmsk even if the user selected bicubic interpolation |
C   | via the control file.                                             |
C   |                                                                   |
C   | Input : grd_Ix (ni)     : Input grid points position x(i)         |
C   | ^^^^^^^ grd_Iy (nj)     :   "     "    "       "     y(j)         |
C   |         var_I  (ni, nj) : Input field values                      |
C   |         grd_Ox (mx, my) : Output grid positions x(i,j)            |
C   |         grd_Oy (mx, my) : Output grid positions y(i,j)            |
C   |         SPHgrd (T/F)    : If true, spherical coordinates for      |
C   |                           input fields                            |
C   |                                                                   |
C   | Output: var_O  (mx, my) : Output field values                     |
C   | ^^^^^^^                                                           |
C   +-------------------------------------------------------------------+

      SUBROUTINE bilSim (ni,nj,grd_Ix,grd_Iy,var_I,SPHgrd,
     .                   mx,my,grd_Ox,grd_Oy,var_O,tmp_in)

      IMPLICIT NONE

C +---General and local variables
C +   ---------------------------

      INTEGER    ns,i,j,ii,jj,p,q,is,ind0,ind1,nsamp,LocDim,
     .           ni,nj,mx,my,mmx,mmy,icent1,jcent1,icent2,jcent2

      PARAMETER (ns = 5) ! Number of sampling points
      PARAMETER (LocDim=21601) ! Dim. of local 1D arrays

      REAL x,y,tmp,tmp2,x0,x1,y0,y1,epsi,AUXlon,MINlon,MAXlon,
     .     AUXlat,MINlat,MAXlat,dist_O,dist_I,AUXlo1,AUXlo2,
     .     AUXla1,AUXla2,dx,dy,degrad,ns2,tmp3

      REAL grd_Ix(ni),grd_Iy(nj),grd_Ox(mx,my),grd_Oy(mx,my),
     .     tmp_in(0:ni+1,0:nj+1),tmp_Ix(0:LocDim+1),samOx(ns),
     .     samOy(ns),tmp_Iy(0:LocDim+1),var_I(ni,nj),var_O(mx,my)

      ! JFG (09/05/2022): cyclic, npole and spole were only used for 
      ! display purposes (display instructions were commented).
      LOGICAL SPHgrd ! ,cyclic,npole,spole

C +---Data
C +   ----

      DATA epsi   / 1.d-4          /
      DATA degrad / 1.745329252d-2 /

C +---Check dimensions of temporary arrays
C +   ====================================

      IF (ni.gt.LocDim .or. nj.gt.LocDim) THEN
        WRITE(6,*) 'bilSim - fatal error: dimension',LocDim
        WRITE(6,*) 'Please change LocDim   -   STOP'
        STOP
      ENDIF

C +---Check if the sampling technique is required
C +   ===========================================

      mmx = mx
      mmy = my
      
      dx    =(grd_Ix(ni/2)-grd_Ix(ni/2-1))*111111.
     .                   *COS(grd_Iy(nj/2)*degrad)
      dy    =(grd_Iy(nj/2)-grd_Iy(nj/2-1))*111111.
      dist_I=max(dx,dy)

      icent1=MAX(1,mx/2)
      icent2=MAX(1,mx/2-1)
      jcent1=MAX(1,my/2)
      jcent2=MAX(1,my/2-1)
      IF (mmx.eq.2) icent1=2
      IF (mmy.eq.2) jcent1=2
      
      AUXlo1=grd_Ox(icent1,jcent1)
CWARNINGXla1=grd_Oy(icent1,icent1)
      AUXla1=grd_Oy(icent1,jcent1)
      AUXlo2=grd_Ox(icent2,jcent2)
      AUXla2=grd_Oy(icent2,jcent2)

C +        ******
      CALL SPHERC (SPHgrd,AUXlo1,AUXla1)
      CALL SPHERC (SPHgrd,AUXlo2,AUXla2)
C +        ******

      dx    =(AUXlo1-AUXlo2)*111111.*COS(AUXla1*degrad)
      IF (mmx.le.1) dx = 1000.
      dy    =(AUXla1-AUXla2)*111111.
      IF (mmy.le.1) dy = 1000.
      dist_O=max(dx,dy)

      nsamp=1
      ns2  =  max(2.,(dist_O/dist_I))

      if(ns2==1) then
       print *,"WARNING: in bilSim dist_O < dist_I!!"
      endif
       
C +---Coordinates indexes inversion (if necessary)
C +   ============================================

C +---Storage in temporary arrays
C +   ---------------------------

      DO jj=1,nj
      DO ii=1,ni 
       tmp_in(ii,jj)=var_I(ii,jj)
      ENDDO
      ENDDO

      DO ii=1,ni
       tmp_Ix(ii)=grd_Ix(ii)
      ENDDO

      DO jj=1,nj
       tmp_Iy(jj)=grd_Iy(jj)
      ENDDO

C +---Revert grd_Ix (1) <--> grd_Ix (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Ix(ni).lt.grd_Ix(1)) THEN     
       DO ii=1,ni   
        DO jj=1,nj                       
         tmp_in(ii,jj)=var_I(ni-ii+1, jj)
        ENDDO
        tmp_Ix(ii)=grd_Ix(ni-ii+1) 
       ENDDO
      ENDIF

C +---Revert grd_Iy (1) <--> grd_Iy (n), ... ?
C +   ----------------------------------------
      
      IF (grd_Iy(nj).lt.grd_Iy(1)) THEN     
       DO jj=1,nj   
        DO ii=1,ni                       
         tmp_in(ii,jj)=var_I(ii,nj-jj+1)
        ENDDO
        tmp_Iy(jj)=grd_Iy(nj-jj+1)
       ENDDO
      ENDIF

C +---Extended coordinates in longitude and latitude
C +   ==============================================

C +---Check validity of longitude
C +   ---------------------------

      IF (SPHgrd) THEN
       IF ((tmp_Ix(1).lt.(-180.)).or.(tmp_Ix(ni).gt.180.)) THEN
        WRITE(6,*) 'Longitudes of data fields are not between'
        WRITE(6,*) '-180 and +180 deg. (as required by bilSim)'
        WRITE(6,*) 'but rather between : '
        WRITE(6,*) tmp_Ix(1),tmp_Ix(ni)
        WRITE(6,*) '--- STOP in bilSim ---'
        STOP
       ENDIF
      ENDIF

C +---Extended left/right boundaries (longitude if SPHgrd)
C +   ----------------------------------------------------

      tmp_Ix(0)   =2.*tmp_Ix( 1)-tmp_Ix(2)
      tmp_Ix(ni+1)=2.*tmp_Ix(ni)-tmp_Ix(ni-1) 

C +---Extended bottom/top boundaries (latitude if SPHgrd)
C +   ---------------------------------------------------

      tmp_Iy(0)   =2.*tmp_Iy( 1)-tmp_Iy(2)
      tmp_Iy(nj+1)=2.*tmp_Iy(nj)-tmp_Iy(nj-1)

C +---Define the cyclic field in longitude
C +   ------------------------------------

      IF (SPHgrd) THEN     ! Stereographic coordinates

       ind0=-1
       ind1=-1
 
       AUXlon=tmp_Ix(0)+360.
       DO i=1,ni
        IF (ABS(AUXlon-tmp_Ix(i)).lt.epsi) ind0=i
       ENDDO
 
       AUXlon=tmp_Ix(ni+1)-360.
       DO i=1,ni
        IF (ABS(AUXlon-tmp_Ix(i)).lt.epsi) ind1=i
       ENDDO
 
       ! .not.(ind0.gt.(-1).and.ind1.gt.(-1))
       IF (ind0.lt.(0).or.ind1.lt.(0)) THEN
        ! cyclic=.false.
        ind0=ni
        ind1= 1
       ! ELSE
       !  cyclic=.true.
       ENDIF
 
       IF (ABS(tmp_Ix(ni+1)-180.).lt.epsi) tmp_Ix(ni+1)=180.+epsi

      ELSE                 ! Non spherical coordinates

       ind0=ni
       ind1= 1

      ENDIF

      DO j=1,nj
       tmp_in(   0,j)=tmp_in(ind0,j)
       tmp_in(ni+1,j)=tmp_in(ind1,j)
      ENDDO
      
C +---Define extra lower and upper boundaries (latitude)
C +   --------------------------------------------------

      IF (SPHgrd) THEN     ! Stereographic coordinates

       IF (tmp_Iy(0).lt.(-90.))
     .  tmp_Iy(0)=MIN(-90.,tmp_Iy(1)-epsi)

       IF (tmp_Iy(nj+1).gt.90.)      
     .  tmp_Iy(nj+1)=MAX(90.,tmp_Iy(nj)+epsi)

       !spole=.false.
       !npole=.false.

       !IF (tmp_Iy(0).le.(-90.)) spole=.true.
       !IF (tmp_Iy(nj+1).ge.90.) npole=.true.

      ENDIF

      DO i=0,ni+1
       tmp_in(i,   0)=tmp_in(i, 1)
       tmp_in(i,nj+1)=tmp_in(i,nj)
      ENDDO

C +---Check the extension of the sub-domain to be read
C     ================================================

      AUXlon = grd_Ox(1,1)
      AUXlat = grd_Oy(1,1)
C +        ******
      CALL SPHERC (SPHgrd,AUXlon,AUXlat)
C +        ******
      MINlon = AUXlon
      MAXlon = AUXlon
      MINlat = AUXlat
      MAXlat = AUXlat

      DO j=1,my
      DO i=1,mx
       AUXlon = grd_Ox(i,j)
       AUXlat = grd_Oy(i,j)
C +         ******
       CALL SPHERC (SPHgrd,AUXlon,AUXlat)
C +         ******
   
       MINlon = min(AUXlon,MINlon)
       MAXlon = max(AUXlon,MAXlon)
       MINlat = min(AUXlat,MINlat)
       MAXlat = max(AUXlat,MAXlat)
      ENDDO
      ENDDO

      IF ((tmp_Ix(   0).gt.MINlon) .or.
     .    (tmp_Ix(ni+1).lt.MAXlon) .or.
     .    (tmp_Iy(   0).gt.MINlat) .or.
     .    (tmp_Iy(nj+1).lt.MAXlat)) THEN
       WRITE(6,*) 'Output domain is not (fully) included in'
       WRITE(6,*) 'the input domain.'
       WRITE(6,*) 'Input  domain :'
       WRITE(6,*) tmp_Ix(0),tmp_Ix(ni+1),tmp_Iy(0),tmp_Iy(nj+1)
       WRITE(6,*) 'Output domain :'
       WRITE(6,*) MINlon,MAXlon,MINlat,MAXlat
       WRITE(6,*) '--- STOP in bilSim ---'
      ENDIF

C +---Bi-linear interpolation
C +   =======================

C +---Some initialisations
C +   --------------------

      p=0
      q=0
      
C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      DO i=1,mx   ! LOOP on output grid-points : BEGIN
      DO j=1,my  
          
C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

C +---Define sampling point positions
C +   -------------------------------

       DO is=1,nsamp   ! Boundaries : no sampling
        samOx(is)=grd_Ox(i,j)
        samOy(is)=grd_Oy(i,j)
       ENDDO

       tmp2=0.0       ! Initialisation of sum of sampled values

       DO is=1,nsamp  ! Loop on the sampling points: BEGIN

        x=samOx(is)
        y=samOy(is)

C +---Check the range of latitude and longitude
C +   -----------------------------------------

C +          ******
        CALL SPHERC (SPHgrd,x,y)
C +          ******

C +---Search for the bottom-left corner of the surrounding mesh
C +   ---------------------------------------------------------

C +...This simple method accounts for the fact that two successive
C +...requests usually refer to neighbour points.

C +---Search for dimension 1 value
C +   ----------------------------

        IF (tmp_Ix(p).le.x) THEN   ! Search upwards
         DO WHILE (tmp_Ix(p+1).lt.x)
          p=p+1
         ENDDO
        ELSE                       ! Search downwards
         DO WHILE (tmp_Ix(p).gt.x)
          p=p-1
         ENDDO
        ENDIF

C +---Search for dimension 2 value
C +   ----------------------------

        IF (tmp_Iy(q).le.y) THEN  ! Search upwards
         DO WHILE (tmp_Iy(q+1).lt.y)
          q=q+1
         ENDDO
        ELSE                      ! Search downwards
         DO WHILE (tmp_Iy(q).gt.y)
          q=q-1
         ENDDO
        ENDIF

C +---Check the validity of p/q indexes
C +   ---------------------------------

        IF ((p.lt.     0).or.(q.lt.     0).or.
     .      (p.gt.(ni+1)).or.(q.gt.(nj+1))) THEN
         WRITE (6,*) 'Inconsistency between input and output'
         WRITE (6,*) 'domains.'
         WRITE (6,*) 'p and q = ',p,q
         WRITE (6,*) '--- STOP in intSim ---'
         STOP
        ENDIF

C +---Linear interpolation
C +   --------------------

        tmp2=0 ; tmp3=0

        do ii=nint(-1*ns2/2.),nint(ns2/2.) 
        do jj=nint(-1*ns2/2.),nint(ns2/2.)  

         x0=min(ni,max(1,p+ii))  
         y0=min(nj,max(1,q+jj))  

                                           tmp = 1.
        !if(max(abs(ii),abs(jj))>= ns2/2.) tmp = 2/3.

         tmp2 = tmp2 + tmp_in(x0,y0) * tmp
         tmp3 = tmp3 + tmp  
        enddo
        enddo

       ENDDO          ! LOOP on the sampling points: END

C +---Output value given by the average of the samplings
C +   --------------------------------------------------

       var_O(i,j)=tmp2/tmp3

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      ENDDO
      ENDDO           ! Loop on output grid-points : END

C +   +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      ! IF (cyclic) WRITE(6,*) 'bilSim info: cyclic boundaries'
      ! IF (npole ) WRITE(6,*) 'bilSim info: North pole included'
      ! IF (spole ) WRITE(6,*) 'bilSim info: South pole included'

      RETURN
      END

C     +--------------------------------------------------------------+
      SUBROUTINE intLin (xx,vv,ni,xreq,outvar)   ! Last modif. : 04/99
C     +--------------------------------------------------------------+

      REAL xx(ni), vv(ni)
      REAL xreq, outvar, fdis
      INTEGER ind, KLO, KHI,ni,k

      KLO=1
      KHI=ni
 1    IF (KHI-KLO.GT.1) THEN
        K=(KHI+KLO)/2
        IF(xx(K).GT.xreq)THEN
          KHI=K
        ELSE
          KLO=K
        ENDIF
      GOTO 1
      ENDIF
      ind=KLO

      fdis = xx(ind+1)-xx(ind)
      outvar= vv(ind)*((xx(ind+1)-xreq)/fdis)                 
     .    + vv(ind+1)*((xreq-xx(ind  ))/fdis)                 

      IF (xreq.LT.xx(ind  )) outvar=vv(ind  )
      IF (xreq.GT.xx(ind+1)) outvar=vv(ind+1)

      RETURN
      END
      
C   +-------------------------------------------------------------------+
C   |  Subroutine SPHERC                               July 99  NESTING |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | This routine sets longitude between -180 and +180, and latitude   |
C   | between -90 and +90, as required by some interpolation sub-       |
C   | routines.                                                         |
C   |                                                                   |
C   | Input : SPHgrd : If true, LONval and LATval really are spherical  |
C   | ^^^^^^^          coordinates                                      |
C   |         LONval : longitude   (degree)                             |
C   |         LATval : latitude    (degree)                             |
C   |                                                                   |
C   | Output: LONval : longitude   (degree)                             |
C   | ^^^^^^^ LATval : latitude    (degree)                             |
C   |                                                                   |
C   | J.-F. Grailet remark (03/05/2022): SPHERC subroutine is also      |
C   | called outside interpolation operations from time to time.        |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE SPHERC (SPHgrd,LONval,LATval)

      IMPLICIT NONE

      REAL    LONval,LATval
      LOGICAL SPHgrd

      IF (SPHgrd) THEN

C +---Longitude defined between -180 and +180 degree
C +   ----------------------------------------------

       IF (LONval.ge.  180. ) LONval=LONval-360.
       IF (LONval.lt.(-180.)) LONval=LONval+360.

C +---Latitude defined between -90 and +90 degree
C +   -------------------------------------------

       IF (LATval.gt.   90.1 ) LATval=LATval-180.
       IF (LATval.lt. (-90.1)) LATval=LATval+180.

      ENDIF

      RETURN
      END

C   +-------------------------------------------------------------------+
C   |  Subroutine SEARCH                             June 03  NESTING ? |
C   +-------------------------------------------------------------------+
C   |                                                                   |      
C   | J.-F. Grailet remark (09/05/2022): this routine was originally    |
C   | in ETOPOg.f in older NESTOR versions, but given that it's called  |
C   | by routines that also call bilSim, it was moved here.             |
C   |                                                                   |
C   | This routine was originally not documented, but given that it was |
C   | declared in the (now deprecated) ETOPOg.f source file, it may     |
C   | have been written roughly around the same time or before (2003).  |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE SEARCH (xx,ni,xreq,KLO,KHI)

      REAL xx(ni)

      KLO=1
      KHI=ni
 1    IF (KHI-KLO.GT.1) THEN
        K=(KHI+KLO)/2
        IF(xx(K).GT.xreq)THEN
          KHI=K
        ELSE
          KLO=K
        ENDIF
      GOTO 1
      ENDIF

      RETURN
      END

C     +--------------------------------------------------------------+
C     |     * From numerical recipes (H. Press et al., 1992)         |
C     +--------------------------------------------------------------+

      SUBROUTINE SPLINE(X,Y,N,YP1,YPN,Y2)
 
      PARAMETER (NMAX=500)
      DIMENSION X(N),Y(N),Y2(N),U(NMAX)
 
      IF (YP1.GT..99E30) THEN
        Y2(1)=0.
        U(1)=0.
      ELSE
        Y2(1)=-0.5
        U(1)=(3./(X(2)-X(1)))*((Y(2)-Y(1))/(X(2)-X(1))-YP1)
      ENDIF
      DO 11 I=2,N-1
        SIG=(X(I)-X(I-1))/(X(I+1)-X(I-1))
        P=SIG*Y2(I-1)+2.
        Y2(I)=(SIG-1.)/P
        U(I)=(6.*((Y(I+1)-Y(I))/(X(I+1)-X(I))-(Y(I)-Y(I-1))
     *      /(X(I)-X(I-1)))/(X(I+1)-X(I-1))-SIG*U(I-1))/P
11    CONTINUE
      IF (YPN.GT..99E30) THEN
        QN=0.
        UN=0.
      ELSE
        QN=0.5
        UN=(3./(X(N)-X(N-1)))*(YPN-(Y(N)-Y(N-1))/(X(N)-X(N-1)))
      ENDIF
      Y2(N)=(UN-QN*U(N-1))/(QN*Y2(N-1)+1.)
      DO 12 K=N-1,1,-1
        Y2(K)=Y2(K)*Y2(K+1)+U(K)
12    CONTINUE
      RETURN
      END
 
C     +--------------------------------------------------------------+
C     |     * From numerical recipes (H. Press et al., 1992)         |
C     +--------------------------------------------------------------+

      SUBROUTINE SPLINT(XA,YA,Y2A,N,X,Y)
 
      DIMENSION XA(N),YA(N),Y2A(N)
      KLO=1
      KHI=N
1     IF (KHI-KLO.GT.1) THEN
        K=(KHI+KLO)/2
        IF(XA(K).GT.X)THEN
          KHI=K
        ELSE
          KLO=K
        ENDIF
      GOTO 1
      ENDIF
      H=XA(KHI)-XA(KLO)
      IF (H.EQ.0.) PAUSE 'Bad XA input.'
 
      A=(XA(KHI)-X)/H
      B=(X-XA(KLO))/H
      Y=A*YA(KLO)+B*YA(KHI)+
     *      ((A**3-A)*Y2A(KLO)+(B**3-B)*Y2A(KHI))*(H**2)/6.
 
      RETURN
      END
      
C   +-------------------------------------------------------------------+
C   |  Function dist                             31/08/2004   NESTING ? |
C   +-------------------------------------------------------------------+
C   |                                                                   |      
C   | J.-F. Grailet remark (09/05/2022): this function was originally   |
C   | not documented, but given that it was declared in the original    |
C   | INTnrg2.f source file (in practical equivalent to this file), it  |
C   | may have been written roughly around the same time or before      |
C   | (2004). The name and purpose are self-explanatory.                |
C   |                                                                   |
C   | The function was moved here rather than staying in intMAR.f since |
C   | it use is now a bit more general: it is now called by both intMar |
C   | and bicSet (new bicubic interpolation subroutine).                |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      function dist(lon2o,lat2o,lon1o,lat1o)

      implicit none
      real,parameter :: pi   = 3.141592
      real,parameter :: R    = 6371.

      real :: lon1,lat1
      real :: lon2,lat2
      real :: dlat,dlon,a,c,dist
      real :: lon2o,lat2o,lon1o,lat1o

      lon1=lon1o * Pi/180.
      lon2=lon2o * Pi/180.
      lat1=lat1o * Pi/180.
      lat2=lat2o * Pi/180.

      dlat = (lat2-lat1)
      dlon = (lon2-lon1)
         a =  sin(dLat/2.) * sin(dLat/2.) + cos(lat1) * cos(lat2) 
     .     *  sin(dLon/2.) * sin(dLon/2.)
         c =  2. * atan2(sqrt(a), sqrt(1.-a))
      dist =  R * c

      end function
