C   +-------------------------------------------------------------------+
C   |  Subroutine GRAvgd                                 13-04-2022 JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Vertical grid for GRADS output analysis. Tailored for 2D grids in |
C   | April 2022 to improve performance.                                |
C   |                                                                   |
C   | Input : - nz     : number of vertical levels (N.B.: nz rather     |
C   | ^^^^^^^            than nk because nk already used in NSTdim.inc) |
C   |         - klev   : if specified, the level at which pressure and  |
C   |                    hybrid coordinate has to be computed           |
C   |         - GRA_sp(mx,my) : surface pressure                        |
C   |         - dimensions from NSTdim.inc (e.g. mx, my)                |
C   |                                                                   |
C   | Output: Vertical grid of the ECMWF model :                        |
C   | ^^^^^^^ - GRA__p(mx,my,nz+1) : pressure at each level  [kPa]      |
C   |         - GRA_hp(mx,my,nz+1) : hybrid coordinates                 |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE GRAvgd (nz,klev,GRA_sp,GRA__p,GRA_hp)
 
      IMPLICIT NONE

      INCLUDE 'NSTdim.inc' ! Provides mx, my

C +---Local variables
C +   ---------------

      INTEGER nz,klev,i,j,k,k1,k2
 
      REAL pp,ppm,pps,ppf,pp1,dpsl,hh,plevel(nz),
     .     GRA_sp(mx,my),GRA__p(mx,my,nz+1),GRA_hp(mx,my,nz+1)

      CHARACTER*10 var_units

C +---Atmospheric levels: pressure levels
C +   -----------------------------------

      IF (nz.ne.12) THEN
       write(6,*)
       write(6,*) 'GRADS output grid is valid only with 12 vertical'
       write(6,*) 'levels. Please set mz=12 in NSTdim.inc'
       write(6,*)
       write(6,*) '--> STOP in GRAvgd.f'
       write(6,*)
       STOP
      ENDIF

      plevel( 1) = 100.
      plevel( 2) = 150.
      plevel( 3) = 200.
      plevel( 4) = 250.
      plevel( 5) = 300.
      plevel( 6) = 400.
      plevel( 7) = 500.
      plevel( 8) = 600.
      plevel( 9) = 700.
      plevel(10) = 850.
      plevel(11) = 925.
      plevel(12) = 1000.

C +---Computation for a given level or all levels ?
C +   ---------------------------------------------

      IF ((klev.le.0).or.(klev.gt.nz)) THEN
       k1=1
       k2=nz
      ELSE
       k1=1
       k2=klev
      ENDIF
      
      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

C +---For each i,j pixel (start of grid traversal)
C +   --------------------------------------------

      DO i=1,mx
      DO j=1,my

C +---Compute pressure at each levels
C +   -------------------------------

       DO k=k1,k2
        GRA__p(i,j,k)=plevel(k)/10.  ! (kPa)
ccccc   IF (GRA__p(i,j,k).gt.GRA_sp)
ccccc.   GRA__p(i,j,k)=GRA_sp(i,j)-REAL(k)*0.1
       ENDDO

       GRA__p(i,j,nz+1)=GRA_sp(i,j)

C +---Compute hybrid coordinates (required by nesting procedure)
C +   --------------------------
C +...Local hybrid coordinate: set parameters:

       pps = GRA_sp(i,j)
       ppm = pps - dpsl
       DO k = k1,k2+1
        pp = GRA__p(i,j,k)
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        GRA_hp(i,j,k) = LOG(hh)
       ENDDO

      END DO; END DO ! End of grid traversal

      RETURN
      END
