C   +-------------------------------------------------------------------+
C   |  Subroutine ETOP1_2O22 (version 2022 of ETOPO1)           NESTING |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Input : Grid : NST__x and NST__y (longitude and latitude, degrees)|
C   | ^^^^^^^ ETOPO data set, resolution: 1 minutes                     |
C   |                                                                   |
C   | Output: NST_sh: surface elevation                                 |
C   | ^^^^^^^ NSTsol: land (4) / sea (1) mask                           |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE ETOPO22

      use netcdf        

      IMPLICIT NONE
      
C +---Netcdf specifications
C +   ---------------------

      INCLUDE 'libUN.inc'

C +---NST variables
C +   -------------

      INCLUDE 'NSTdim.inc'
      INCLUDE 'NSTvar.inc'
      INCLUDE 'NESTOR.inc'
      INCLUDE 'LOCfil.inc'
      INCLUDE 'MARvar.inc'
 
C +---Local variables
C +   ---------------

      INTEGER      i,j,mlon,mlat,elat,midlon,mlon1,mlat1,
     .             nbchar,TOPOmx,TOPOmy

      PARAMETER   (mlon  =  21600)
      PARAMETER   (mlat  =  10800)
      PARAMETER   (mlon1 =mlon+1)
      PARAMETER   (mlat1 =mlat+1)
C +...Size of full ETOPO file

      PARAMETER   (elat  =  4000)
      PARAMETER   (TOPOmx=  mlon)
      PARAMETER   (TOPOmy=  elat)
C +...Size of sub-domain (latitude only)

      INTEGER     DIMS(1),TOPO_ID,LAT_ID,LON_ID,sol,start(3),
     .            count(3),i1lon,i2lon,i1lat,i2lat,imlon,imlat,
     .            irien,ncid,Rcode,i1,j1,flag

      real*4   , allocatable :: etopo(:,:)
      real*4   , allocatable :: TOPsh(:,:)
      real*4   , allocatable :: tmpTOP(:,:)
      real*4   , allocatable :: tmp_in(:,:)

      REAL        topo_lon(mlon),topo_lat(mlat),size_lon,
     .            TOPlon(TOPOmx),TOPlat(TOPOmy),size_lat,
     .            MINlon,MINlat,MAXlon,MAXlat,AUXlon,AUXlat

      LOGICAL     Vtrue

      CHARACTER*80   ETOPOfile
      CHARACTER*32   var_units
C +---Data
C +   ----

      DATA start  / 1,1,1/
      DATA Vtrue  /.true./
      integer           :: ncid1,status,XXvarid

      allocate   (etopo(mlon,elat))
      allocate   (TOPsh(TOPOmx,TOPOmy))
      allocate   (tmpTOP(TOPOmx,TOPOmy))
      allocate   (tmp_in(0:TOPOmx+1,0:TOPOmy+1))
C +---Opening and reading of ETOPO data file
C +   ======================================

      write(6,*) 'Topography : ETOPO1_2022 data set (1 minutes)'
      write(6,*) '~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
      write(6,*)

C +---Open NetCDF file
C +   ----------------

      nbchar=1

      DO i=1,60
       IF (ETOPO_dir(i:i).ne.' ') nbchar=i
      ENDDO

      ETOPOfile = ETOPO_dir(1:nbchar-1) // '1/etopo1_2022.nc'
      INQUIRE(FILE=trim(ETOPOfile), EXIST=Vtrue)
      if(.not.Vtrue) then
       print *,"new "//trim(ETOPOfile)//" file not found (20/05/2025)"
       stop
      endif        

C +---Read latitudes and longitudes of ETOPO
C +   --------------------------------------

C +...! etopo_lon and _lat are real*8 !

      status = nf90_open(TRIM(ETOPOfile), nf90_NoWrite, ncid1)
      start(1)=1
      start(2)=1
      start(3)=1
      count(1)=mlon
      count(2)=1
      count(3)=1

      status = nf90_open(TRIM(ETOPOfile), nf90_NoWrite, ncid1)
      status = nf90_inq_varid(ncid1,"lon", XXvarid)
      status = nf90_get_var(ncid1, XXvarid, topo_lon,start,count)

      count(1)=mlat
      status = nf90_inq_varid(ncid1,"lat", XXvarid)
      status = nf90_get_var(ncid1, XXvarid, topo_lat,start,count)

C +---Compute the extension of the sub-domain to be read
C     --------------------------------------------------

      AUXlon = NST__x(1,1)
      AUXlat = NST__y(1,1)
C +        ******
      CALL SPHERC (Vtrue,AUXlon,AUXlat)
C +        ******
      MINlon = AUXlon
      MAXlon = AUXlon
      MINlat = AUXlat
      MAXlat = AUXlat
      DO j=1,my
      DO i=1,mx
       AUXlon = NST__x(i,j)
       AUXlat = NST__y(i,j)
C +         ****** 
       CALL SPHERC (Vtrue,AUXlon,AUXlat)
C +         ******
       MINlon = min(AUXlon,MINlon)
       MAXlon = max(AUXlon,MAXlon)
       MINlat = min(AUXlat,MINlat)
       MAXlat = max(AUXlat,MAXlat)
      ENDDO
      ENDDO

C +---Define extraction zone
C +   ----------------------

C +        ******
      CALL SEARCH (topo_lon,mlon,MINlon,i1lon,irien)
      CALL SEARCH (topo_lon,mlon,MAXlon,irien,i2lon)
C +        ******
      imlon = i2lon - i1lon + 1
C +        ******
      CALL SEARCH (topo_lat,mlat,MINlat,i1lat,irien)
      CALL SEARCH (topo_lat,mlat,MAXlat,irien,i2lat)
C +        ******
      imlat = i2lat - i1lat + 1

      IF (imlat.ge.elat) THEN
       write(*,*) 'Extent of the simulation domain in latitude'
       write(*,*) 'is too large. Please choose a larger value '
       write(*,*) 'for the elat parameter.             - STOP '
       STOP
      ENDIF

      i1lat = i1lat + (i2lat-i1lat)/2 - elat/2
      i1lat = MAX(1,i1lat)
      i2lat = i1lat + elat - 1
      IF (i2lat.gt.mlat) THEN
       i2lat= mlat
       i1lat= i2lat - elat + 1
      ENDIF

C +---Read values of the variables for the sub-domain
C +   -----------------------------------------------

      start(1)=1
      start(2)=max(1,i1lat-1)
      count(1)=mlon
      count(2)=elat

      status = nf90_inq_varid(ncid1,"z", XXvarid) 
      status = nf90_get_var(ncid1, XXvarid, etopo,start,count)
      status = nf90_close(ncid1)

      DO i=1,mlon
        DO j=1,elat
        TOPsh(i,j) = etopo(i,j)
        END DO
      END DO

      DO i=1,TOPOmx
       TOPlon(i)=topo_lon(i)
      ENDDO

      DO j=1,TOPOmy
       TOPlat(j)=topo_lat(i1lat-1+j)
      ENDDO

C +---Interpolation of topography to the NST grid
C +   -------------------------------------------

C +        ******
      CALL bilSim (TOPOmx,TOPOmy,TOPlon,TOPlat,TOPsh ,Vtrue ,
     .             mx    ,my    ,NST__x,NST__y,NST_sh,tmp_in)
C +        ******

C +---Distinction between land and sea (further refined)
C +   --------------------------------

      DO j=1,my
      DO i=1,mx

       IF (NST_sh(i,j)<=-1) THEN
        NSTsol(i,j)=1
        NSTice(i,j)=0
       ELSE
        NSTsol(i,j)=4
       ENDIF

      ENDDO
      ENDDO

C +---Special topo for Greenland Simulation
C +   -------------------------------------

      IF (region.eq."GRD") call USRgrd ('ETOPOg')
      IF (region.eq."EUR") call USReur ('ETOPO1')

C +---No atmosphere below sea level...
C +   --------------------------------

      DO j=1,my
      DO i=1,mx

       IF (NST_sh(i,j)<=0.0.and.NSTsol(i,j)==1) THEN
        NST_sh(i,j)= 0.0
       ENDIF

      ENDDO
      ENDDO

      deallocate   (etopo)
      deallocate   (TOPsh)
      deallocate   (tmpTOP)
      deallocate   (tmp_in)

      RETURN
      END
