C   +-------------------------------------------------------------------+
C   |  Subroutine ECMvgd                                 11-04-2022 JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Vertical grid of the ECMWF model. Tailored for a 2D grid. One     |
C   | additional logical parameter is used to also re-use this routine  |
C   | for the hadCM3/ECHAM5/CanESM2/NorESM1 model (same method, only    |
C   | the UNsread calls change).                                        |
C   |                                                                   |
C   | Input : - fID    : identificator of the Netcdf data file          |
C   | ^^^^^^^ - nk     : number of vertical levels                      |
C   |         - baseI  : minimum X index of the relevant LSC sub-region |
C   |         - baseJ  : minimum Y index of the relevant LSC sub-region |
C   |         - maxI   : maximum X index of the relevant LSC sub-region |
C   |         - maxJ   : maximum Y index of the relevant LSC sub-region |
C   |         - klev   : if specified, the level at which pressure and  |
C   |                    hybrid coordinate has to be computed           |
C   |         - isCM3  : true if hadCM3/ECHAM5/CanESM2/NorESM1 model    |
C   |         - ECM_sp(ni,nj) : surface pressure                        |
C   |                                                                   |
C   | Output: Vertical grid of the ECMWF model :                        |
C   | ^^^^^^^ - ECM__p(ni,nj,nk+1) : pressure at each level  [kPa]      |
C   |         - ECM_hp(ni,nj,nk+1) : local hybrid coord. for vertical   |
C   |                                interpolation.                     |
C   |                                                                   |
C   | Remarks on optimization via sub-region selection (29/05/2022):    |
C   | -to compute the vertical grid for the full LSC domain, use        |
C   |  baseI=1, baseJ=1, maxI=ni, maxJ=nj.                              |
C   | -code assumes that the user will use 1 <= baseI <= maxI <= ni and |
C   |  1 <= baseJ <= maxJ <= nj.                                        |
C   | -if the variables baseI, baseJ, maxI and maxJ are set to delimit  |
C   |  a sub-region of the LSC grid, only this sub-region will be       |
C   |  completed in the output grids.                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE ECMvgd(fID,ni,nj,nk,baseI,baseJ,maxI,maxJ,klev,isCM3,
     .                  ECM_sp,ECM__p,ECM_hp)

      IMPLICIT NONE

C +---Local variables
C +   ---------------
      INTEGER fID,ni,nj,baseI,baseJ,maxI,maxJ,nk,klev,i,j,k,k1,k2,k21

      REAL pp,ppm,pps,ppf,pp1,dpsl,hh,empty1(1),CSTp(nk),SIGp(nk),
     .     ECM_sp(ni,nj),ECM__p(ni,nj,nk+1),ECM_hp(ni,nj,nk+1) 

      CHARACTER*10 var_units
      
      LOGICAL isCM3
      
C +---Atmospheric levels: pressure levels
C +   -----------------------------------
C +   Initially, there was a separate CM3vgd routine, but it was 
C +   identical to ECMvgd except for the subsequent calls, hence the 
C +   additional logical argument and the following lines.

      IF (isCM3) THEN
       CALL UNsread (fID,'CSTp',1,1,1,1,nk,1,1,var_units,CSTp)
       CALL UNsread (fID,'SIGp',1,1,1,1,nk,1,1,var_units,SIGp)
      ELSE
       CALL UNsread (fID,'CSTp',0,0,0,0,nk,1,1,var_units,CSTp)
       CALL UNsread (fID,'SIGp',0,0,0,0,nk,1,1,var_units,SIGp)
      ENDIF

C +---Adapt units
C +   -----------

      DO k=1,nk
       CSTp(k) = CSTp(k) * 1.E-3  !(Pa-->KPa)
      ENDDO

C +---Computation for a given level or all levels ?
C +   ---------------------------------------------

      IF ((klev.le.0).or.(klev.gt.(nk+1))) THEN
       k1 =1
       k2 =nk
       k21=nk+1
      ELSE
       k1 =klev
       k2 =klev
       k21=klev
      ENDIF

      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

C +---For each i,j pixel (start of grid traversal)
C +   --------------------------------------------
C +   29/05/2022: added a small optimization; grid traversal now only 
C +   takes account of the sub-region of the LSC domain which includes 
C +   the NST domain.

      DO i=baseI,maxI ! i=1,ni
      DO j=baseJ,maxJ ! j=1,nj

C +---Compute pressure at each levels
C +   -------------------------------

C +...Pressure in LS atmosphere is such that :
C +...p(level) = CSTp(level) + SIGp(level) * Surf_pressure

       IF (klev.ne.(nk+1)) THEN
         DO k=k1,k2
           ECM__p(i,j,k)=CSTp(k)+SIGp(k)*ECM_sp(i,j) ! (kPa)
         ENDDO
       ENDIF

       ECM__p(i,j,nk+1)=ECM_sp(i,j)


C +---Compute hybrid coordinates (required by nesting procedure)
C +   --------------------------
C +...Local hybrid coordinate: set parameters

       pps = ECM_sp(i,j)
       ppm = pps - dpsl
       DO k = k1,k21
        pp = ECM__p(i,j,k)
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        ECM_hp(i,j,k) = LOG(hh)
       ENDDO

      END DO; END DO ! End of grid traversal

      RETURN
      END
