C   +-------------------------------------------------------------------+
C   |  Subroutine CPLvgd                                 15-04-2022 JFG |
C   +-------------------------------------------------------------------+
C   |                                                                   |
C   | Creation of the vertical grid of the MAR model (as NST output)    |
C   | (SVAT coupling).                                                  |
C   |                                                                   |
C   | Input : - nz     : number of vertical levels (N.B.: nz rather     |
C   | ^^^^^^^            than nk because nk already used in NSTdim.inc) |
C   |         - klev   : if specified, the level at which pressure and  |
C   |                    hybrid coordinate has to be computed           |
C   |         - VGD_sp(mx,my) : surface pressure (kPa)                  |
C   |         - parameters from CPLgrd.ctr                              |
C   |         - dimensions from NSTdim.inc (e.g. mx, my)                |
C   |                                                                   |
C   | Output: Vertical MAR grid given in hybrid coordinates :           |
C   | ^^^^^^^ - VGD__p(mx,my,nz+1) : pressure coordinates (kPa)         |
C   |         - VGD_hp(mx,my,nz+1) : local hybrid coord. for vertical   |
C   |                                interpolation                      |
C   |         - VGDgdz(nz  ) : model coordinates (sigma)                |
C   |                                                                   |
C   | J.-F. Grailet remark: contrary to LSCvgd and its associated       |
C   | routines, VGDgdz has been kept because it appears to be actually  | 
C   | used in a meaningful way by NESTOR (could be checked thoroughly). |
C   | Only change here is the position of the parameter in the list.    |
C   |                                                                   |
C   +-------------------------------------------------------------------+

      SUBROUTINE CPLvgd (nz,klev,VGD_sp,VGD__p,VGD_hp,VGDgdz)

      IMPLICIT NONE

      INCLUDE 'NSTdim.inc'

C +---Local variables
C +   ---------------
      
      INTEGER nz,klev,i,j,k,k1,k2

      ! J.-F. Grailet remark: GElat0, GElon0 + others seem unused.
      REAL pp1,pps,ppm,dpsl,pp,hh,ppf,GElat0,GElon0,dx,GEddxx,
     .     ptopDY,zmin,aavu,bbvu,ccvu,sst_SL,TUkhmx,long1,lati1,
     .     long2,lati2,long3,lati3,long4,lati4

      REAL VGD_sp(mx,my),VGD__p(mx,my,nz+1),VGD_hp(mx,my,nz+1),
     .     VGDgdz(nz),sigma(nz)

      LOGICAL vertic

      CHARACTER*10 var_units

C +---CREATION OF SIGMA MAR GRID USING PARAMETERS IN CPLgrd.ctr
C +   =========================================================

C +---Read grid parameters in CPLgrd.ctr
C +   ----------------------------------

      OPEN (unit=51,status='old',file='CPLgrd.ctr')

       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) long1
       read (51,*) lati1
       read (51,*) long2
       read (51,*) lati2
       read (51,*) long3
       read (51,*) lati3
       read (51,*) long4
       read (51,*) lati4
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) ptopDY
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) zmin
       read (51,*) aavu
       read (51,*) bbvu
       read (51,*) ccvu
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,'(l4)') vertic
       read (51,*) !- - - - - - - - - - - - - - - - - -
       read (51,*) sst_SL
       read (51,*) !- - - - - - - - - - - - - - - - - -

      CLOSE(unit=51)

C +---Sets the standard values of vertical grid parameters
C +   ----------------------------------------------------

      CALL SETsig (nz,zmin,aavu,bbvu,ccvu,ptopDY)

C +---Computation of vertical grid
C +   ----------------------------

      CALL GRDsig (nz,zmin,aavu,bbvu,ccvu,vertic,sst_SL,TUkhmx,sigma)

C +---Print the characteristics of the vertical grid
C +   ----------------------------------------------

      write(6,*) 'Vertical CPL grid parameters'
      write(6,*) '~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
      write(6,300) nz,ptopDY
300   format(' Number of grid points : ',i4,/,
     .       ' Pressure at the top   : ',f9.4,' kPa.')
      write(6,310) zmin, aavu, bbvu, ccvu
310   format(' First level height    : ', f6.1,/,
     .       ' aavu, bbvu, ccvu      : ',(f6.1,', ',f6.1,', ',f6.1),/)

C +---Sigma coordinates
C +   -----------------

      DO k=1,nz
       VGDgdz(k)=sigma(k)
      ENDDO

C +---HYBRID AND PRESSURE COORDINATES (required by the nesting code)
C +   ===============================

C +---Reference levels for hybrid coordinates
C +   ---------------------------------------

      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

C +---Selection of vertical levels
C +   ----------------------------

      IF ((klev.le.0).or.(klev.gt.nz)) THEN
       k1=1
       k2=nz
      ELSE
       k1=1
       k2=klev
      ENDIF

C +---For each i,j pixel (start of grid traversal)
C +   --------------------------------------------

      DO i=1,mx
      DO j=1,my

C +---Computation of hybrid coordinates used in vertic. interp.
C +   ---------------------------------------------------------

       pps = VGD_sp(i,j)
       ppm = pps - dpsl
       DO k = k1,k2+1
        IF (k.eq.(nz+1)) THEN
         pp = VGD_sp(i,j)
        ELSE
         pp = VGDgdz(k)*(VGD_sp(i,j)-ptopDY) + ptopDY
        ENDIF
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        VGD_hp(i,j,k) = LOG(hh)
        VGD__p(i,j,k) = pp
       ENDDO

      END DO; END DO ! End of grid traversal

      RETURN
      END

