C +----------------------------------------------------------------------+
C | External Nesting System from MAR to MAR                     08/1997  |
C |                                                                      |
C | MARget                                                               |
C |  interpolates and adapt Large Scale atmospheric data to MAR grid.    |
C +----------------------------------------------------------------------+
C |                                                                      |
C | INPUT  : Ctime : time for which the data is requested                |
C | ^^^^^^^^                     (unit = DAYS (fractional), all times    |
C |                               values are relative to LSfile begin.)  |
C |          dx, sigma, sh                                               |
C |                                                                      |
C | INPUT FILE: Output File porduced by MAR (NetCDF file)                |
C | ^^^^^^^^^^^                                                          |
C |   *file name = Netcdf file produced during MAR execution.            |
C |   *file contents:                                                    |
C |                                                                      |
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |                     variable    |No |Unit                            |
C |                  in atm.| 10m   |   |                                |
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |     Wind          |U    |uairDY |1  |m/s                             |
C |       "           |V    |vairDY |2  |m/s                             |
C |     Specif. humid.|Q    |qvDY   |3  |Kg/Kg + all hydrometeores       |
C |     Temperature   |T    |tairDY |4  |K                               |
C |                   |     |       |   |                                |
C |     Pressure      |     |pstar  |-  |hPa                             |
C |     Soil Height   |-    |sh     |-  |m (and not geopotential !)      |
C |                   |     |       |   |                                |
C |                   |     |       |   |                                |
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |                                                                      |
C | OUTPUT : Rtime : time corresponding to the input data                |
C | ^^^^^^^^           Rtime is the last time step                       |
C |                    for which  Rtime .le. Ctime                       |
C |          Ftime : the next time available                             |
C |          (Ftime > Rtime, = -1 at end-of-data)                        |
C |                                                                      |
C |   varext (mx,my,mz,{iv}): data valid at Rtime,                       |
C |                                                                      |
C |                           contains  u, v, qv, pkt                    |
C |                           with iv = 1, 2,  3,   4 [=ivnum]           |
C |                                                                      |
C |   pstDY (mx,my)         : model pressure tickness (ps-pt)            |
C |                                                                      |
C |   sst   (mx,my)         : sea or soil surface temperature            |
C |                                                                      |
C +----------------------------------------------------------------------+

      subroutine MARget (LSfile,Ctime,Rtime,Ftime,dx,sigma,
     .                   GElat0,GElon0,ptop,sh,varext,pstDY,
     .                   sst2D,gplvDY,TOPObnd)


C     Variables
C     =========

      include 'MARdim.inc'
      include 'LSMARIN.inc'

      INTEGER FILEid,time,level,i_dbeg,j_dbeg,i,j,k,l,imez,jmez,n88,
     .        ierror

      REAL    cap,ra,grav,exxpo,gamTz,dxLS,dyLS,aux,eps,GElat0,GElon0,
     .        ptop,Ctime,Rtime,Ftime,dx,Li,Lv,cp,aux1,aux2,pps,dpsl,
     .        ppf,hh,pp1,ppm

      REAL    uairLS(mx,my,mz),vairLS(mx,my,mz),tairLS(mx,my,mz),
     .        qvLS  (mx,my,mz),pktaLS(mx,my,mz),pstLS (mx,my)   ,
     .        qiLS  (mx,my,mz),qwLS  (mx,my,mz),qrLS  (mx,my,mz),
     .        qsLS  (mx,my,mz),sstLS (mx,my)   ,shLS  (mx,my)   ,
     .        psLS  (mx,my)   ,latLS (mx,my)   ,lonLS (mx,my)   ,
     .        uairDY(mx,my,mz),vairDY(mx,my,mz),pktaDY(mx,my,mz),
     .        qvDY  (mx,my,mz),qwDY  (mx,my,mz),qiDY  (mx,my,mz),
     .        qrDY  (mx,my,mz),qsDY  (mx,my,mz),pstDY (mx,my)   ,
     .        sh    (mx,my)   ,sst2D (mx,my)   ,varext(mx,my,mz,4),
     .        utmpLS(mx,my)   ,vtmpLS(mx,my)   ,ttmpLS(mx,my)   ,
     .        qtmpLS(mx,my)   ,utmpDY(mx,my)   ,vtmpDY(mx,my)   ,
     .        ttmpDY(mx,my)   ,qtmpDY(mx,my)   ,wk1_LS(mx,my)   ,
     .        wk2_LS(mx,my)   ,LSax  (mx)      ,LSay  (my)      ,
     .        LSaz  (mz)      ,MARax (mx,my)   ,MARay (mx,my)   ,
     .        sigma (mz)      ,sh_asLS(mx,my)  ,ps_asLS(mx,my)  ,
     .        LSvert(mzz)     ,MAvert(mz)      ,MAR1D(mz)       ,
     .        LS1D  (mzz)     ,gplvDY(mx,my,mzz),
     .        halfway(mx,my,mzz,4)

      CHARACTER*10 var_units
      CHARACTER*60 LSfile
      CHARACTER*80 FILEtit

      LOGICAL TOPObnd,vert_int


C     Data
C     ====

      DATA cap   /   0.28586e0/
      DATA cp    /1005.     e0/
      DATA ra    / 287.     e0/
      DATA grav  /   9.81   e0/
      DATA eps   /   1.    e-3/
      DATA Lv    /   2.5    e6/
      DATA Li    /   2.8    e6/

      DATA vert_int / .false. /


C     Read Large-Scale Data
C     =====================

      CALL UNropen (LSfile,FILEid,FILEtit)

      Ctime=Ctime*24.   !  Conversion : day --> hour

      CALL UNgtime (FILEid,Ctime,Rtime,Ftime,time)
      
      Ctime=Ctime/24.
      Rtime=Rtime/24.   !  Conversion : hour --> day
      Ftime=Ftime/24.

      CALL UNread (FILEid,'sh'    ,  1 ,1,1,1,mx,my, 1,
     .             LSax,LSay,LSaz,var_units,shLS  )

      CALL UNread (FILEid,'lat'   ,  1 ,1,1,1,mx,my, 1,
     .             LSax,LSay,LSaz,var_units,latLS )

      CALL UNread (FILEid,'lon'   ,  1 ,1,1,1,mx,my, 1,
     .             LSax,LSay,LSaz,var_units,lonLS )

      CALL UNread (FILEid,'pstar' ,  1 ,1,1,1,mx,my, 1,
     .             LSax,LSay,LSaz,var_units,pstLS )

      CALL UNread (FILEid,'tsrfSL',time,1,1,1,mx,my, 1,
     .             LSax,LSay,LSaz,var_units,sstLS )

      CALL UNread (FILEid,'uairDY',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,uairLS)

      CALL UNread (FILEid,'vairDY',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,vairLS)

      CALL UNread (FILEid,'tairDY',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,tairLS)

      CALL UNread (FILEid,'qvDY  ',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,qvLS  )

      CALL UNread (FILEid,'qwHY  ',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,qwLS  )

      CALL UNread (FILEid,'qiHY  ',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,qiLS  )

      CALL UNread (FILEid,'qrHY  ',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,qrLS  )

      CALL UNread (FILEid,'qsHY  ',time,0,1,1,mx,my,mz,
     .             LSax,LSay,LSaz,var_units,qsLS  )

      CALL NCCLOS (FILEid,ierror)


C     Check Vertical Grid Compatibility
C     =================================

      DO k=1,mz
       aux=LSaz(k)/sigma(k)
       IF (aux.gt.(1.+eps) .or. aux.lt.(1.-eps)) THEN
        write(*,*) 'Specified vertical grid is not compatible with'
        write(*,*) 'that of large-scale data.            S T O P !'
        STOP
       ENDIF
      ENDDO


C     Horizontal Resolution and Compatibility
C     =======================================

      dxLS=LSax(2)-LSax(1)
      dyLS=LSay(2)-LSay(1)

      IF (dxLS.ne.dyLS) THEN
       write(*,*) 'Warning : dx and dy in the large-scale file ',
     .            'are not equal.'
      ENDIF

      IF (dx.gt.dxLS*1000.) THEN
       write(*,*) 'Specified horizontal resolution is greater (or'
       write(*,*) 'equal) than that of large-scale data.  S T O P !'
       STOP
      ENDIF


C     Center Domain Compatibility
C     ===========================

      imez=mx/2
      jmez=my/2

      aux1=latLS(imez,jmez)/GElat0
      aux2=lonLS(imez,jmez)/GElon0

      IF (aux1.gt.(1.+eps) .or. aux1.lt.(1.-eps) .or.
     .    aux2.gt.(1.+eps) .or. aux2.lt.(1.-eps)) THEN
       write(*,*) 'Specified center domain is not compatible'
       write(*,*) 'with that of large-scale data.      STOP !'
       STOP
      ENDIF
      

C     Evaporation of Hydrometeores
C     ============================

      DO k=1,mz
      DO j=1,my
      DO i=1,mx
       qvLS(i,j,k)=qvLS(i,j,k)+qwLS(i,j,k)+qiLS(i,j,k)
     .                        +qrLS(i,j,k)+qsLS(i,j,k)
       tairLS(i,j,k)=tairLS(i,j,k)
     .              -Lv/cp*(qwLS(i,j,k)+qrLS(i,j,k))
     .              -Li/cp*(qiLS(i,j,k)+qsLS(i,j,k))
      ENDDO
      ENDDO
      ENDDO


C     New Nested Grid
C     ===============

      DO j=1,my
      DO i=1,mx
       MARax(i,j)=LSax(imez)+real(i-imez)*dx/1000.
       MARay(i,j)=LSay(jmez)+real(j-jmez)*dx/1000.
      ENDDO
      ENDDO


C     Surface Pressure
C     ================

      DO j=1,my
      DO i=1,mx
       psLS(i,j)=pstLS(i,j)+ptop
      ENDDO
      ENDDO


C     Potential Temperature
C     =====================

      DO k=1,mz
      DO j=1,my
      DO i=1,mx
       pktaLS(i,j,k)=tairLS(i,j,k)
     .              /((psLS(i,j)*LSaz(k)+ptop)**cap)
      ENDDO
      ENDDO
      ENDDO


C     Horizontal Interpolation
C     ========================

      WRITE(*,*)
      WRITE(*,'(A,$)') ' LS Level:'

      DO k=1,mz

       WRITE(*,'(I3,$)') k

       DO j=1,my
       DO i=1,mx
        utmpLS(i,j)=uairLS(i,j,k)
        vtmpLS(i,j)=vairLS(i,j,k)
        ttmpLS(i,j)=pktaLS(i,j,k)
        qtmpLS(i,j)=qvLS  (i,j,k)
       ENDDO
       ENDDO

       CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,utmpLS,
     .                            mx,my,MARax,MARay,utmpDY)

       CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,vtmpLS,
     .                            mx,my,MARax,MARay,vtmpDY)

       CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,ttmpLS,
     .                            mx,my,MARax,MARay,ttmpDY)

       CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,qtmpLS,
     .                            mx,my,MARax,MARay,qtmpDY)

       DO j=1,my
       DO i=1,mx
        uairDY(i,j,k)=utmpDY(i,j)
        vairDY(i,j,k)=vtmpDY(i,j)
        pktaDY(i,j,k)=ttmpDY(i,j)
        qvDY  (i,j,k)=qtmpDY(i,j)
       ENDDO
       ENDDO

      ENDDO

      CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,psLS,
     .                           mx,my,MARax,MARay,ps_asLS)

      CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,shLS,
     .                           mx,my,MARax,MARay,sh_asLS)

      CALL INTbic (wk1_LS,wk2_LS,mx,my,LSax ,LSay ,sstLS,
     .                           mx,my,MARax,MARay,sst2D)


C     Compute Geopotential at the Surface
C     ===================================

      DO j=1,my
      DO i=1,mx
       gplvDY(i,j,mzz) = sh(i,j) * grav
      ENDDO
      ENDDO


C     Computation of surface pressure for MAR.
C     ========================================

C     Computation of a surface pressure adapted to MAR topography,
C     using 2 simple assumptions : - constant T gradient = gamTz
C                                  - basic T = 1st level near surface
C     Constants:
      gamTz = - 6.5E-3
      exxpo = - grav / (gamTz * ra)

C     Compute ps for MAR
      DO j=1,my
      DO i=1,mx
        tsurf     =pktaDY(i,j,mz)
     .            *(((ps_asLS(i,j)-ptop)*sigma(mz)+ptop)**cap)
        pstDY(i,j)=-ptop+ps_asLS(i,j)
     .            *(1.+gamTz*(sh(i,j)-sh_asLS(i,j))/tsurf) ** exxpo
      ENDDO
      ENDDO


C     Correction of soil surface according to topography change
C     =========================================================

      DO i=1,mx
      DO j=1,my
        sst2D(i,j)=sst2D(i,j)+gamTz*(sh(i,j)-sh_asLS(i,j))
      ENDDO
      ENDDO


C     ******************
      IF (vert_int) THEN   ! Vertical Interpolation
C     ******************

C     Create halfway Variable
C     =======================

      DO j=1,my
      DO i=1,mx
       DO k=1,mz
        halfway(i,j,k,1)=uairDY(i,j,k)
        halfway(i,j,k,2)=vairDY(i,j,k)
        halfway(i,j,k,3)=qvDY  (i,j,k)
        halfway(i,j,k,4)=pktaDY(i,j,k)
       ENDDO
       k=mzz
       halfway(i,j,k,1)=0.
       halfway(i,j,k,2)=0.
       halfway(i,j,k,3)=qvDY (i,j,mz)
       halfway(i,j,k,4)=sst2D(i,j)/(ps_asLS(i,j)**cap)
      ENDDO
      ENDDO


C     Vertical Interpolation
C     ======================

C +...Local hybrid coordinate: set parameters:
      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

      DO j = 1,my      !*BEGIN LOOP on horizontal grid pts.
      DO i = 1,mx

C +----Set local hybrid coordinate for vertical interpolation:
C +    -------------------------------------------------------
 
C +----...for Large Scale data :
C +    ~~~~~~~~~~~~~~~~~~~~~~~~~
       pps = ps_asLS(i,j)
       ppm = pps-dpsl
       DO k = 1,mzz
        IF (k.eq.mzz) THEN
         pp = ps_asLS(i,j)
        ELSE
         pp = ptop+(ps_asLS(i,j)-ptop)*LSaz(k)
        ENDIF
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        LSvert(k)= log(hh)
       END DO

C +----...for MAR: 
C +    ~~~~~~~~~~~
       pps = pstDY(i,j)+ptop
       ppm = pps-dpsl
       DO k = 1,mz
        pp = sigma(k)*pstDY(i,j)+ptop
        hh = pp/pp1
        IF (pp.gt.ppm) THEN
         ppf= (pp-ppm)/(pps-ppm)
         hh = hh + (pp1-pps)/pp1 * ppf * ppf
        END IF
        MAvert(k) = log(hh)
       END DO


C +----Set mass conservation coefficient (for u,v):
C +    --------------------------------------------
       pspsn=(ps_asLS(i,j)-ptop)/pstDY(i,j)


C +----Vertical loop -> various operations
C +    -----------------------------------
       DO iv = 1,ivnum    !*BEGIN LOOP on iv (No variables)
C +..   *get vertical 1D variables values from 4D "halfway":
        DO k=1,mzz
         LS1D(k) = halfway (i,j,k,iv)
        END DO

C +-----Interpolate:
C +     ~~~~~~~~~~~~
C +..   (1) Natural cubic spline:
C #VS   CALL S4LINE(LSvert, LS1D, LSnk2, 1.E30, 1.E30, work1D)
C #VS   DO k=1,mz
C #VS     CALL S4LINT(LSvert, LS1D, work1D, mz,
C #VS.               MAvert(k), MAR1D(k))
C #VS   END DO
C +...  (2) Linear:
        DO k=1,mz
         CALL INTlin(LSvert,LS1D,mzz,MAvert(k),MAR1D(k))
        END DO

C +-----put 1D vertic into 4D "varext",
C +     and correct speed to keep mass conservation
C +     despite the different ps: ~~~~~~~~~~~~~~~~~
C +     ~~~~~~~~~~~~~~~~~~~~~~~~~

        IF (iv.le.2) THEN
         DO k=1,mz
          varext(i,j,k,iv)=MAR1D(k)*pspsn
         END DO
        ELSE
         DO k=1,mz
          varext(i,j,k,iv)=MAR1D(k)
         END DO
        END IF
 
       END DO        !*END LOOP on iv (No variables)

      END DO
      END DO         !*END LOOP  on horizontal grid pts

C     ****
      ELSE   ! No Vertical Interpolation
C     ****

      write(6,*) 'MARget : No Vertical Interpolation !'

C     Correct wind velocities to keep mass conservation
C     =================================================

      DO i=1,mx
      DO j=1,my
       pspsn=(ps_asLS(i,j)-ptop)/pstDY(i,j)
       DO k=1,mz
        uairDY(i,j,k)=uairDY(i,j,k)*pspsn
        vairDY(i,j,k)=vairDY(i,j,k)*pspsn
       ENDDO
      ENDDO
      ENDDO

C     Create varext Variable
C     ======================

      DO k=1,mz
      DO j=1,my
      DO i=1,mx
       varext(i,j,k,1)=uairDY(i,j,k)
       varext(i,j,k,2)=vairDY(i,j,k)
       varext(i,j,k,3)=qvDY  (i,j,k)
       varext(i,j,k,4)=pktaDY(i,j,k)
      ENDDO
      ENDDO
      ENDDO

C     *****
      ENDIF  
C     *****



      RETURN
      END
