C +----------------------------------------------------------------------+
C | External Nesting System for MAR                             08/1997  |
C |                                                   warning: test 10.97|
C | LSget                                                                |
C |  interpolates and adapt Large Scale atmospheric data to MAR grid.    |
C +----------------------------------------------------------------------+
C |                                                                      |
C | REFERENCE:                                                           |
C | ^^^^^^^^^^                                                           |
C |   - No progress report available yet; see MARONH/=nestinfo           |
C |                                                                      |
C | INPUT  : Ctime : time for which the data is requested                |
C | ^^^^^^^^                     (unit = DAYS (fractional), all times    |
C |                               values are relative to LSfile begin.)  |
C |          intype: horizontal interp. type  (1= bilin, 3= bicub)       |
C |          alaso1, aloso1                                              |
C |          dx                                                          |
C |          sigma                                                       |
C |          sh                                                          |
C |                                                                      |
C | INPUT FILE: a LSRD - Large Scale Raw Data file (NetCDF)              |
C | ^^^^^^^^^^^                                                          |
C |   *file name = {LSmodel}.YYMMDD.nC                                   |
C |                 where YYMMDD = Year, Month, and Day of file begin.   |
C |   *'time' variable = Universal time from 0 hour the YYMMDD day       |
C |                      (unit = DAYS)                                   |
C |   *file contents:                                                    |
C |                                                                      |
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |                     variable    |No |Unit                            |
C |                  in atm.| 10m   |   |                                | 
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |     Wind          |U    |U10    |1  |m/s                             |
C |       "           |V    |V10    |2  |m/s                             |
C |     Specif. humid.|Q    |Q10    |3  |Kg/Kg                           |
C |     Temperature   |T    |T10    |4  |K                               |
C |                   |     |       |   |                                |
C |     Pressure      |     |PS     |-  |hPa                             |
C |     Soil Height   |-    |SH     |-  |m (and not geopotential !)      |
C |                   |     |       |   |                                |
C |                   |     |       |   |                                |
C |     - - - - - - - + - - + - - - + - + - - - - - - - - - - - - -      |
C |                                                                      | 
C | OUTPUT : Rtime : time corresponding to the input data                |
C | ^^^^^^^^           Rtime is the last time step                       |
C |                    for which  Rtime .le. Ctime                       |
C |          Ftime : the next time available                             |
C |          (Ftime > Rtime, = -1 at end-of-data)                        |
C |                                                                      |
C |   varext (mx,my,mz,{iv}): data valid at Rtime,                       |
C |                                                                      |
C |                           contains  u, v, qv, pkt                    |
C |                           with iv = 1, 2,  3,   4 [=ivnum]           |
C |                                                                      | 
C |   pstDY (mx,my)         : model pressure tickness (ps-ptop)          |
C |                                                                      |
C |   sst   (mx,my)         : sea or soil surface temperature            |
C |   t2_SL (mx,my)         : deep soil temperature                      |
C |                                                                      | 
C | OPTIONS:                                                             |
C | ^^^^^^^^                                                             |
C |                                                                      | 
C +----------------------------------------------------------------------+
      subroutine LSget (LSfile, Ctime,Rtime,Ftime, intype, 
     .                  alaso1, aloso1, dx, sigma, ptop, sh,
     .                  ps_asLS,  sh_asLS, CSTp, SIGp, halfway, 
     .                  varext, pstDY, sst, t2_SL,lsp_EXT, cp_EXT)
 
      IMPLICIT NONE
C +
C     LS and MAR domain dimensions :
C     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +...* Large Scale input data dimension + share 'ivnum' :
      include 'LSMARIN.inc'

C +...* MAR dimensions :
      include 'MARdim_def.inc'
      include 'MARdim.inc'

C +   INPUT
C +   ~~~~~
      REAL Ctime,Rtime,Ftime
      REAL alaso1(mx,my), aloso1(mx,my)     
      REAL dx, sigma(mz), ptop, sh(mx,my)
      INTEGER intype

C +   OUTPUT
C +   ~~~~~~
      REAL varext(mx,my,mz,ivnum)
      REAL pstDY(mx,my), sst(mx,my), t2_SL(mx,my)

      REAL lsp_EXT(mx,my), cp_EXT(mx,my)
C +..._EXT variables : temporaire, serait mis ds routine a part

      REAL sh_asLS(mx,my), ps_asLS(mx,my)
C +..._asLS variables are interpolated LS values (.ne. MAR)

      REAL halfway   (mx,my, 1:LSnk1, ivnum)
C +...Defined on LS vertical and MAR horizontal grid:

C +   LOCAL
C +   ~~~~~
      INTEGER Ltit, ipass
      PARAMETER (Ltit = 90)
      CHARACTER *(Ltit) LStitle
C +
      CHARACTER*13 var_name(ivnum), SV_name(ivnum)
      CHARACTER*13 ps_name, p_name, topo_name
      INTEGER i,j,k,iv,it, kl
 
      INTEGER LSfID, t_index, ierror
      CHARACTER *60 LSfile
      CHARACTER *10 var_units
      CHARACTER *10 LSvc_name
 
      REAL unitfact(ivnum)
      REAL p_unitfact
      REAL exxpo, gamTz, pspsn, rexxpo, fcort
      REAL pp, ppm, pps, ppf, pp1, dpsl, hh
      REAL empty1(1)
 
C +...*Defined on Large Scal grid:
C +...Note: LSnk1 = nk vertical levels + 10m OR surface values

      include 'LSM_WK3.inc'
C +...LS grid work area

      REAL LSlon(LSni), LSlat(LSnj) 
      REAL CSTp(LSnk1), SIGp(LSnk1), LSlev(LSnk)
      REAL LSvert(LSnk1), LS1D(LSnk1), work1D(LSnk1)
      REAL ps_LS (LSni, LSnj)
      REAL pkt_LS (LSni, LSnj), qv_LS (LSni, LSnj), sh_LS (LSni, LSnj)
      REAL pk1_LS (LSni, LSnj), px1_LS(LSni, LSnj), Zp_LS(LSni, LSnj)
      LOGICAL iZp_LS (LSni, LSnj) 
      LOGICAL iZterm

C +...*Defined on MAR grid:
C +...Note: this is the only section which may be replaced 
C +...      by MAR work arrays
C +...         ---------------
C +...      _asLS variables are interpolated LS values (.ne. MAR)
      REAL MARlat(mx,my), MARlon(mx,my)
      REAL MARva1 (mx,my), MARva2 (mx,my)
      REAL T1_asLS(mx,my), Zp_asLS(mx,my)
      REAL pk1_ma (mx,my), px1_ma(mx,my), Zp_ma(mx,my)
      REAL zpscor (mx,my)
      REAL MAvert(mz), MAR1D(mz+1), sigmar(mz+1), ptpmar(mz+1)
      LOGICAL iZp_ma(mx,my)

C +...* Physical constants
      REAL cap, ra, pi, rad, grav

C +...* Control output selector:
      INTEGER icheck

C +...* Physical constants
      data ra    / 287.     d0/
C +...     ra    : Perfect Gas Law  Constant (J/kg/K)
      data cap   /   0.28586d0/
      data grav  /   9.81   d0/
      pi  = ACOS(-1.)
      rad = pi/180. 

      icheck=0 
      ipass =0

C +--If special check output are requested, open a specific file :
C +  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      IF(icheck.ge.2)THEN 
        OPEN (unit=71,status='unknown',file='LSget.chk')
        WRITE(71,'(A)') 'LSget: begin'  
      ENDIF
C +
C +--Various data preparation
C +  ========================

C +--Convert lat,lon coord. to degrees.
C +  ----------------------------------
      DO j=1,my
       DO i=1,mx
          MARlon(i,j) =  aloso1(i,j) * 15.d0
C ++...   ^^->  Conversion: Hour->degrees
          MARlat(i,j) =  alaso1(i,j) / rad
C ++...   ^^->  Conversion: rad ->degrees
       ENDDO
      ENDDO
 
C
C +--prepare reading of LS data.
C +  ---------------------------
 
C +...Open NetCDF Large Scale Raw Data file and
C +...inquire about last available time step before Ctime (it):
C +        *******
      CALL UNropen (LSfile, LSfID, LStitle)
      CALL UNgtime (LSfID, Ctime, Rtime, Ftime, it)
C +        *******


      IF(icheck.ge.1) THEN
         WRITE(71,*)'LSget-info: reading new large scale data,'
	 WRITE(71,*)'Req. date : ', Ctime
         WRITE(71,*)'Read date : ', Rtime
         WRITE(71,*)'Next date : ', Ftime
      END IF

C +
C +--Reading and horizontal interpolation of surface fields.
C +  =======================================================
C +
C +--Surface Pressure:    
C +  -----------------
C
C +        ******
      CALL UNread
     &   (LSfID    , 'SP '  ,  it ,  1, LSbi, LSbj,
     &    LSni     ,LSnj    , 1     ,
     &    LSlon    ,LSlat  , empty1,
     &    var_units ,ps_LS   )
C +...     ****** 
C +...^Reads a level from 2D or 3D field of "var_name"
C +...   found in the NetCDF file (at time index = it)
C +...   -> put the result in 'ps_LS'
C +...   See inside routine UNread for details.
C +...   (LSbi,j LSni,j: see LSMARIN.inc)

C +--Check extension of the domain
C    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      IF(icheck.ge.1) THEN
        WRITE(71,*)'Large Scale domain read inside'
        WRITE(71,*)'Longitude:', LSlon(1), LSlon(LSni) 
        WRITE(71,*)'Latitude :', LSlat(1), LSlat(LSnj)
      END IF

C +...*Adapt units to MAR:
      CALL LSuCHG (ps_LS, 1.E-3) !(Pa-->KPa)

C +...*Interpolate horizontaly (here = bicubic spline):
      CALL INThor (intype,
     &   LSlon  , LSlat  , ps_LS,
     &   MARlon, MARlat, ps_asLS)

C +--Topography:
C +  ----------- 
 
      CALL UNread
     &   (LSfID    ,'SH '  ,  it ,  1, LSbi, LSbj, 
     &    LSni     ,LSnj   , 1     ,
     &    LSlon    ,LSlat  , empty1,
     &    var_units , sh_LS  )

      CALL INThor (intype,
     &   LSlon  , LSlat  , sh_LS,
     &   MARlon, MARlat, sh_asLS)

C +--Soil or Sea surface temperature
C +  -------------------------------

      CALL UNread
     &   (LSfID    ,'STL1'  ,  it ,  1, LSbi, LSbj, 
     &    LSni      ,LSnj   , 1   ,
     &    LSlon    ,LSlat   , empty1,
     &    var_units ,wkV_LS  )

      CALL INThor (intype,
     &   LSlon  , LSlat, wkV_LS,
     &   MARlon, MARlat, sst)

C +--Deep soil temperature (for simple soil initialisation only)
C +  -----------------------------------------------------------

      CALL UNread
     &   (LSfID    ,'STL2'  ,  it ,  1, LSbi, LSbj,
     &    LSni      ,LSnj   , 1   ,
     &    LSlon    ,LSlat   , empty1,
     &    var_units ,wkV_LS  )

      CALL INThor (intype,
     &   LSlon  , LSlat, wkV_LS,
     &   MARlon, MARlat, t2_SL)

C +--Precipitation: !temporary: faire une subrout independante?
C                   = probleme d'output-comparaisons.
C +  -----------------------------------------------------------

      CALL UNread
     &   (LSfID    ,'LSP-'  ,  it ,  1, LSbi, LSbj,
     &    LSni      ,LSnj   , 1   ,
     &    LSlon    ,LSlat   , empty1,
     &    var_units , wkV_LS)

      CALL INThor (1,
     &   LSlon  , LSlat, wkV_LS,
     &   MARlon, MARlat, lsp_EXT)
C +.. Interpol. type 1 = linear.

      CALL UNread
     &   (LSfID    ,'CP'  ,  it ,  1, LSbi, LSbj,
     &    LSni     ,LSnj   , 1   ,
     &    LSlon    ,LSlat   , empty1,
     &    var_units ,wkV_LS  )

      CALL INThor (1,
     &   LSlon  , LSlat, wkV_LS,
     &   MARlon, MARlat, cp_EXT)

C +--Atmospheric levels: read hybrid or sigma-p coord. of LS data.
C +  -------------------------------------------------------------

C +...Pressure in LS atmosphere is such that
C +... p(level) = CSTp(level) + SIGp(level) * ps

      CALL UNread
     &   (LSfID    ,'CSTp ', 0, 0, 0, 0,
     &    LSnk     ,1         , 1     ,
     &    LSlev    ,empty1    , empty1,
     &    var_units, CSTp)

C +...*Adapt units to MAR:
      DO kl=1,LSnk
        CSTp(kl) = CSTp(kl) * 1.E-3  !(Pa-->KPa)
      ENDDO

      CALL UNread
     &   (LSfID    ,'SIGp ', 0, 0, 0, 0,
     &    LSnk     ,1         , 1     ,
     &    LSlev    ,empty1    , empty1,
     &    var_units, SIGp)

C +...*Additional levels: surface
      CSTp(LSnk1) = 0.0
      SIGp(LSnk1) = 1.0

C +--Atmospheric variables at surface:
C +  (bottom boundary for vertic. interpolation)
C +  -------------------------------------------       
C +...  *10m or Surface value for atmosph. variables:
        SV_name (1) = '10U        '
        SV_name (2) = '10V        '
        SV_name (3) = 'QL1        '
        SV_name (4) = 'TL1        '
        unitfact(3) = 1.           !(Kg/Kg)

C +...  Wind vector:
C +...  ~~~~~~~~~~~~

C +...  *Reading, u component:
        CALL UNread
     &        (LSfID    ,SV_name(1), it ,1, LSbi, LSbj, 
     &         LSni     ,LSnj     , 1     ,
     &         LSlon    ,LSlat    , empty1,
     &         var_units, wkV_LS)

C +...  *Interpolation LS -> MAR grid :
C +...   (Uses LS coordinates read for the variable
C +...     -> staggered horizontal grid is allowed)
        CALL INThor (intype,
     &     LSlon  , LSlat  , wkV_LS,
     &     MARlon, MARlat, MARva1 )

C +...  *Reading, v component:
        CALL UNread
     &     (LSfID    ,SV_name(2), it ,1, LSbi, LSbj, 
     &      LSni     ,LSnj   , 1     ,
     &      LSlon    ,LSlat  , empty1,
     &      var_units, wkV_LS)

C +...  *Interpolation LS -> MAR grid :
        CALL INThor (intype,
     &     LSlon , LSlat , wkV_LS,
     &     MARlon, MARlat, MARva2 )

C +...  *Wind vector rotation (according to projection)
        CALL VecRotP (
     &     MARlon, MARlat, dx,
     &     MARva1, MARva2 )

C +...  *Store in "halfway" for further treatments:
C +...  (10m level = possible extension)
C +...  (surface level:) 
        CALL put2D4M (MARva1,LSnk1,1,halfway)
        CALL put2D4M (MARva2,LSnk1,2,halfway)

C +... Water vapour qv (iv = 3)
C +... ~~~~~~~~~~~~~~~~~~~~~~~~
         CALL UNread
     &        (LSfID    ,SV_name(3), it , 1, LSbi, LSbj, 
     &         LSni     ,LSnj   , 1     ,
     &         LSlon    ,LSlat  , empty1,
     &         var_units, wkV_LS)

C +...   *Adapt units to MAR:
         CALL LSuCHG (wkV_LS, unitfact(3))

C +...   *Interpolation LS -> MAR grid :
C +...    (MARva1 is interpolated from wkV_LS)
         CALL INThor (intype,
     &       LSlon , LSlat , wkV_LS,
     &       MARlon, MARlat, MARva1 )

C +...   *Store at the (LSnk) level in halfway
C +...     for (later) vertical interpolation:
C +...     (!ASSUMES that altitude decrease with vertical index)
         CALL put2D4M (MARva1,LSnk1,3,halfway)
 
C +... Potential temperature pkt (iv = 4)
C +... ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
         CALL UNread
     &        (LSfID    ,SV_name(4), it , 1, LSbi, LSbj,
     &         LSni     ,LSnj   , 1     ,
     &         LSlon    ,LSlat  , empty1,
     &         var_units, wkV_LS)

C +...   *Compute potential temp 'pkt':
C +...    (pkt = potential temp divided by 100.[kPa]**(R/Cp))
         DO j = 1,LSnj
         DO i = 1,LSni
           wkV_LS(i,j) = wkV_LS(i,j) / (ps_LS(i,j))**cap
C +...     =pkt          =real T        =surf pres
         END DO
         END DO

C +...   *Interpolation LS -> MAR grid :
C +...    (MARva1 is interpolated from wkV_LS)
         CALL INThor (intype,
     &       LSlon , LSlat , wkV_LS,
     &       MARlon, MARlat, MARva1 )

C +...   *Store at the (LSnk) level in halfway
C +...     for (later) vertical interpolation:
C +...     (!ASSUMES that altitude decrease with vertical index)
         CALL put2D4M (MARva1,LSnk1,4,halfway)

C +--Reading and horizontal interpolation (+rotation).
C +   (for each atm. prognostic variable and each level)
C +  -----------------------------------------------------
C +...*Atmospheric variables:
      var_name(1) = 'U          '
      var_name(2) = 'V          '
      var_name(3) = 'Q          '
      var_name(4) = 'T          '
      unitfact(3) = 1.           !(Kg/Kg)

      WRITE(*,*)
      WRITE(*,'(A,$)') ' LS Level:' 
      DO kl = LSnk, 1, -1   !*BEGIN LOOP on vertical levels
        WRITE(*,'(I3,$)') kl
 
C +...  Wind vector:
C +...  ~~~~~~~~~~~~
C +...  *Reading, u component:
        CALL UNread
     &        (LSfID    ,var_name(1), it ,  kl, LSbi, LSbj, 
     &         LSni     ,LSnj  , 1     ,
     &         LSlon    ,LSlat , empty1,
     &         var_units, wkV_LS)

C +...  *Interpolation LS -> MAR grid :
C +...   (Uses LS coordinates read for the variable
C +...     -> staggered horizontal grid is allowed)
         CALL INThor (intype,
     &     LSlon  , LSlat  , wkV_LS,
     &     MARlon, MARlat, MARva1 )

C +...  *Reading, v component:
         CALL UNread
     &     (LSfID    ,var_name(2), it ,  kl, LSbi, LSbj, 
     &      LSni     ,LSnj   , 1     ,
     &      LSlon    ,LSlat  , empty1,
     &      var_units, wkV_LS)
 
C +...  *Interpolation LS -> MAR grid :
         CALL INThor (intype,
     &     LSlon  , LSlat  , wkV_LS,
     &     MARlon, MARlat, MARva2 )
 
C +...  *Wind vector rotation (according to projection)
         CALL VecRotP (
     &     MARlon, MARlat, dx,
     &     MARva1, MARva2 )
 
C +...  *Store in "halfway" for further treatments:
         CALL put2D4M (MARva1,kl,1,halfway)
         CALL put2D4M (MARva2,kl,2,halfway)

C +... Water vapour qv (iv = 3)
C +... ~~~~~~~~~~~~~~~~~~~~~~~~
 
          CALL UNread
     &      (LSfID    ,var_name(3), it ,  kl, LSbi, LSbj, 
     &       LSni     ,LSnj   , 1     ,
     &       LSlon    ,LSlat  , empty1,
     &       var_units, qv_LS)

C +...    *Adapt units to MAR (for qv):
          CALL LSuCHG (qv_LS, unitfact(3))

C +...    *Interpolation LS -> MAR grid :

          CALL INThor (intype,
     &      LSlon  , LSlat  , qv_LS,
     &      MARlon, MARlat, MARva1 )
 
C +...   *Store in "halfway" for further treatments:
         CALL put2D4M (MARva1,kl,3,halfway)
 
C +... Potential temperature pkt (iv = 4)
C +... ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          CALL UNread
     &      (LSfID    ,var_name(4), it ,  kl, LSbi, LSbj,
     &       LSni     ,LSnj   , 1     ,
     &       LSlon    ,LSlat  , empty1,
     &       var_units, wkV_LS)

C +...    *Store interpolated temp. T1 (near surface)
          IF (kl.eq.LSnk) THEN
            CALL INThor (intype,
     &            LSlon ,LSlat ,wkV_LS,
     &            MARlon,MARlat,T1_asLS)
          ENDIF
C +...    *Compute potential temp 'pkt':
C +...       (pkt = potential temp divided by 100.[kPa]**(R/Cp))
          DO j = 1,LSnj
          DO i = 1,LSni
            pkt_LS(i,j) = wkV_LS(i,j)
     &                  / (CSTp(kl)+ps_LS(i,j)*SIGp(kl))**cap
          END DO
          END DO

C +...    *Interpolation LS -> MAR grid :

          CALL INThor (intype,
     &      LSlon  , LSlat, pkt_LS,
     &      MARlon, MARlat, MARva1 )

C +...    *Store in "halfway" for further treatments:
          CALL put2D4M (MARva1,kl,4,halfway)

C +... Geopotential of 600 hPa level (for later use).
C +... ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +...    *Compute LS 600 hPa geopotential (iteration) :
          CALL Zplev(pkt_LS, qv_LS, sh_LS, ps_LS, CSTp, SIGp,
     .                 kl, LSni, LSnj, LSnk,
     .                 pk1_LS, px1_LS, iZp_LS, iZterm, Zp_LS)

      END DO         !*END LOOP on kl (vertical levels)
      WRITE(*,*)

C +... Interpolate 600 hPa geopot. -> MAR grid (later use).
C +... ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          CALL INThor (intype,
     &      LSlon  , LSlat, Zp_LS,
     &      MARlon, MARlat, Zp_asLS)

      IF(icheck.ge.1)
     .   WRITE(91,*) Zp_asLS

C +  Close the NetCDF file :
C +  -----------------------
      CALL NCCLOS(LSfID, ierror)

      IF(icheck.ge.1) THEN
        WRITE (71,*) 'Large Scale data read from: '
     &            // LSfile
        WRITE (71,*) '                containing: '
     &            // LStitle
      END IF

C
C +--Computation of surface pressure for MAR.
C +  ========================================      
 
C +... Computation of a surface pressure adapted to MAR topography,
C +... using 2 simple assumptions : - constant T gradient = gamTz
C +...                              - basic T = 1st level near surface
C +... Constants:
      gamTz = - 6.5E-3
      exxpo = - grav / (gamTz * ra)

C +... Compute ps for MAR
      DO j = 1,my
      DO i = 1,mx
        pstDY(i,j)= -ptop +  ps_asLS(i,j) * ( 1. 
     .            + gamTz* (sh(i,j)-sh_asLS(i,j)) /T1_asLS(i,j) )
     .            ** exxpo
c       pstDY(i,j)= -ptop
c    .            + ps_asLS(i,j) * exp ( -9.81 / ra / T1_asLS(i,j)
c    .                                  * (sh(i,j)-sh_asLS(i,j)) )
      END DO
      END DO

      IF (icheck.ge.3) THEN 
      WRITE (71,*) 'Computation of ps: (pn, shMAR-shLS, shMAR, T1)'
      DO j = 1,my,10
      DO i = 1,mx,10
        WRITE (71,*) pstDY(i,j), (sh(i,j)-sh_asLS(i,j)),
     .               sh(i,j), T1_asLS(i,j) 
      END DO
      END DO
      END IF
      IF (icheck.ge.2) WRITE(71,*) 'pn(center) =', pstDY(mx/2,my/2)

C +--Correction of soil temperature accord. to topography change.
C +  ============================================================

      DO j = 1,my
      DO i = 1,mx
        fcort = gamTz * (sh(i,j)-sh_asLS(i,j))
        sst(i,j)  = sst(i,j)  + fcort
        t2_SL(i,j)= t2_SL(i,j)+ fcort
      END DO
      END DO
C
C +--Vertical interpolation.
C +  =======================      
C
      IF (icheck.ge.2) WRITE(71,*) 'MAR sigma levels', sigma

C +...Local hybrid coordinate: set parameters:
      pp1  = 105.       ! Reference pressure (KPa)
      dpsl = 20.        ! "> boundary layer" (KPa)

      DO j = 1,my      !*BEGIN LOOP on horizontal grid pts.
      DO i = 1,mx

C +-----Set local hybrid coordinate for vertical interpolation:
C +     -------------------------------------------------------
 
C +-----...for Large Scale data :
C +     ~~~~~~~~~~~~~~~~~~~~~~~~~
        pps = ps_asLS(i,j)
        ppm = pps - dpsl
        DO kl = 1, LSnk1
          pp = CSTp(kl) + SIGp(kl) * pps
          hh = pp/pp1
          IF (pp.gt.ppm) THEN
            ppf= (pp-ppm)/(pps-ppm)
            hh = hh + (pp1-pps)/pp1 * ppf * ppf
          END IF
          LSvert(kl)= log(hh)
        END DO

        IF (icheck.ge.2.and.ipass.lt.2) THEN
         WRITE(71,*) 'Sample LS  internal hybrid coord.:'
         WRITE(71,*) (exp(LSvert(kl)),kl=1,LSnk1) 
        ENDIF

C +-----...for MAR: 
C +     ~~~~~~~~~~~
        pps = pstDY(i,j)+ptop
        ppm = pps - dpsl
        DO k = 1, mz
          pp = sigma(k) * pstDY(i,j) + ptop
          hh = pp/pp1
          IF (pp.gt.ppm) THEN
            ppf= (pp-ppm)/(pps-ppm)
            hh = hh + (pp1-pps)/pp1 * ppf * ppf
          END IF
          MAvert(k) = log(hh)
        END DO

        IF (icheck.ge.2.and.ipass.lt.2) THEN 
         WRITE(71,*) 'Sample MAR internal hybrid coord.:'
         WRITE(71,*) (exp(MAvert(kl)),kl=1,mz) 
        ENDIF


C +-----Set mass conservation coefficient (for u,v):
C +     --------------------------------------------
        pspsn = (ps_asLS(i,j)-ptop) / pstDY(i,j)


C +-----Various operations on all variables
C +     -----------------------------------
        DO iv = 1,ivnum    !*BEGIN LOOP on iv (No variables)
C +...    *get vertical 1D variables values from 4D "halfway":
          DO kl = 1, LSnk1
           LS1D(kl) = halfway (i,j,kl,iv)
          END DO

C +-------Interpolate:
C +       ~~~~~~~~~~~~
C +...    (1) Natural cubic spline:
C #VS     CALL S4LINE(LSvert, LS1D, LSnk1, 1.E30, 1.E30, work1D)
C #VS     DO k = 1, mz
C #VS       CALL S4LINT(LSvert, LS1D, work1D, LSnk,
C #VS.                 MAvert(k), MAR1D(k))
C #VS     END DO
C +...    (2) Linear:
          DO k = 1, mz
            CALL INTlin(LSvert, LS1D, LSnk1,
     .                  MAvert(k), MAR1D(k))
          END DO

          IF (icheck.ge.2.and.ipass.lt.4) THEN
            WRITE(71,*) 'Interpolation ok for var',iv             
            ipass = ipass + 1
          ENDIF

C +-------put 1D vertic into 4D "varext",
C +       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
          DO k = 1, mz
            varext(i,j,k,iv) = MAR1D(k)
          END DO
 
         END DO        !*END LOOP on iv (No variables)
       END DO
      END DO           !*END LOOP  on horizontal grid pts
C
C

C +---Correct surface pressure <==> Z600 ext = Z600 int.    
C +   ==================================================

C +... Geopotential of 600 hPa level from the MAR-adapted data.
C +... ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      DO k= 1,mz
        sigmar(k)= sigma(k) 
        ptpmar(k)= ptop
C +..   *Because pure sigma coordinate in MAR: "CSTp"=ptop
      ENDDO
      sigmar(mz+1)= 1.0
      ptpmar(mz+1)= ptop

      DO k = mz,1,-1 !(begin integration at surface in Zplev)
        DO j = 1,my
        DO i = 1,mx
           MARva1(i,j)=varext(i,j,k,3)  !qv
           MARva2(i,j)=varext(i,j,k,4)  !pkt
        END DO
        END DO

        CALL Zplev(MARva2, MARva1, sh, pstDY, ptpmar, sigmar,
     .               k, mx, my, mz,
     .               pk1_ma, px1_ma, iZp_ma, iZterm, Zp_ma)
      ENDDO

      IF(icheck.ge.1) THEN
        WRITE(91,*) pstDY
        WRITE(91,*) Zp_ma
      ENDIF

C +... Correct surface pressure
C +... ~~~~~~~~~~~~~~~~~~~~~~~~
      DO j = 1,my
      DO i = 1,mx
        pstDY (i,j)= pstDY(i,j) * (1.0 +    
     .             (Zp_asLS(i,j)-Zp_ma(i,j)) * grav
     .              / (ra*varext(i,j,mz,4)* 
     .                 (ptop+pstDY(i,j)*sigma(mz))**cap ) )

C +..    NB: "1e approx", cor_ps = - ps / (R*Tv(surf)) * err_phi

      END DO
      END DO

C +... TEST eventuel: Recalcul de Z 600 hPa
C +... ~ ~ ~ ~ ~ ~ ~ ~ 

c     IF(icheck.ge.1) THEN
c       DO k = mz,1,-1 !(begin integration at surface in Zplev)
c         DO j = 1,my
c         DO i = 1,mx
c          MARva1(i,j)=varext(i,j,k,3)  !qv
c          MARva2(i,j)=varext(i,j,k,4)  !pkt
c         END DO
c         END DO
c         CALL Zplev(MARva2, MARva1, sh, pstDY, ptpmar, sigmar,
c    .               k, mx, my, mz,
c    .               pk1_ma, px1_ma, iZp_ma, iZterm, Zp_ma)
c       ENDDO
c     ENDIF 

C +---Remove over_saturation and convective instabilities.
C +   ====================================================


C    DO j = 1,my      !*BEGIN LOOP on horizontal grid pts.
C     DO i = 1,mx
C       DO k = 1, mz  

C       tGP    = varext(i,j,k,4)
C       qvGP   = varext(i,j,k,3)
C       esatGP =0.611*exp(17.27*(tGP(k)-273.16)/(tGP(k)-35.86))
C       qsatGP =0.622*esatGP/((pn*s(k)+ptop)-0.378*esatGP))

C       END DO
C     END DO
C    END DO           !*END LOOP   on horizontal grid pts

C +--Close special check file if open.  
C +  - - - - - - - - - - - - - - - - -
      IF(icheck.ge.2)THEN
        CLOSE(unit=71)
      ENDIF


      RETURN
      END


C     +--------------------------------------------------------------+
      subroutine put2D4M (var2D,lev,numvar,var4M)
C     +--------------------------------------------------------------+
 
C +   ** MAR dimensions :
      include 'MARdim.inc'
C +   ** Large Scale input data dimension :
      include 'LSMARIN.inc'
 
      REAL var2D (mx,my)
      REAL var4M (mx,my,LSnk+1,ivnum)
C     **   ^^^^^ ! "Mixed" grid var: MAR horiz. & LS vertic.
 
      DO j = 1,my
       DO i = 1,mx
        IF (ABS(var2D(i,j)).gt.1.E+29) THEN
C ...     *Look for strange values : wrong input file ?
C ...      (side effect of this routine)
          WRITE(*,*) 'LSget - put2D4M control :'
          WRITE(*,*) 'Strange value at i,j,lev,iv ='
          WRITE(*,*) ' ',i,j,lev,numvar                
          WRITE(*,*) 'Value = ',var2D (i,j)
          STOP
        ENDIF
        var4M (i,j,lev,numvar) = var2D (i,j)
       END DO
      END DO
 
      RETURN
      END

C     +--------------------------------------------------------------+
      subroutine LSuCHG (var2D,unitfact)
C     +--------------------------------------------------------------+
 
C +   ** MAR dimensions :
      include 'LSMARIN.inc'
 
      REAL var2D (LSni,LSnj)
      REAL unitfact
 
      DO j = 1,LSnj
       DO i = 1,LSni
        var2D (i,j) = var2D (i,j) * unitfact
       END DO
      END DO
 
      RETURN
      END
