C +------------------------------------------------------------------------+
C |   LSMout.f contents:                                                   |
C |   ------------------                                                   |
C |                                                                        |
C |      OUTgra     Full output of the "adapted to MAR" LS fields          |
C |                    purpose: evaluate initial & boundary conditions     |
C |      OUTils     Interplated Large Scale : horizontal interp &          |
C |                    wind rotation according to projection only          |
C |                    purpose: compare ECMWF input to MAR output.         |
C |      ILSwrite   Subroutine called by OUTils                            |
C |                                                                        |
C |                                                                        |
C |                                                                        |
C |      OUTsta     Time-statistics: output                                |
C |                                                                        |
C |      STAgen     Time-statistics: computation                           |
C |                                                                        |
C |      INTver     Time-statistics: vertical interpolation to p-leveis    |
C |                                                                        |
C +------------------------------------------------------------------------+
C +
C +
C +

      subroutine OUTgra(labexp,ipr_nc,npr_nc,jdh_LB,OUTdir)
C +------------------------------------------------------------------------+
C | LSMARIN output                                                         |
C |   SubRoutine OUTgra is similar to OUTidl (from MAR)                    |
C |   It writes interpolated LS forcing to a NetCDF file                   |
C +------------------------------------------------------------------------+
C |                                                                        |
C |   INPUT: ipr_nc: Current time step    number                           |
C |   ^^^^^^         (starting from ipr_nc=1, which => new file creation)  |
C |          npr_nc: Total  'time slices' number (max value of ipr_nc)     |
C |          jdh_LB: # hours between outputs                               |
C |                                                                        |
C |   OUTPUT: NetCDF File adapted to IDL Graphic Software                  |
C |   ^^^^^^                                                               |
C |                                                                        |
C |   OPTION: #HY  Explicit Cloud Microphysics                             |
C |   ^^^^^^^                                                              |
C |                                                                        |
C |   CAUTION: 1) This Routine requires the usual NetCDF library,          |
C |   ^^^^^^^^    and a complementary access library called 'libUN.a'      |
C |                                                                        |
C +------------------------------------------------------------------------+
C +
C +
C +--General Variables
C +  =================
C +
      include 'MARphy.inc'
C +
      include 'MARdim.inc'
      include 'MARgrd.inc'
      include 'MAR_GE.inc'
C +
      include 'MAR_DY.inc'
      include 'MAR_SL.inc'
      include 'MAR_TU.inc'
C +
C +
c #PO include 'MAR_PO.inc'
c #TV include 'MARveg.inc'
C +
      include 'LSM_WK.inc'
C +
C +
C +--Local   Variables
C +  =================
      common/IDLloc/ fnamNC
C +...               fnamNC: To retain file name.
C +
      PARAMETER (Lfnam= 80, Ltit= 90, Luni= 16, Lnam= 13, Llnam=50)
C +...Length of char strings 
C +
      PARAMETER (NdimNC = 3)
C +...Number of defined spatial dimensions (exact)
C +
      PARAMETER (MXdim = 200)
C +...Maximum Number of all dims: recorded Time Steps
C +   and also maximum of spatial grid points for each direction. 
C +
      PARAMETER (MX_var = 80)
C +...Maximum Number of Variables 
C +
      PARAMETER (NattNC = 2)
C +...Number of real attributes given to all variables
C +
      DIMENSION         moisNC(MXdim)
      DIMENSION         jourNC(MXdim)
      DIMENSION         dateNC(MXdim)
      DIMENSION         timeNC(MXdim)
      DIMENSION         nDFdim(      0:NdimNC)
      DIMENSION         VALdim(MXdim,0:NdimNC)
      DIMENSION         NvatNC(NattNC)
      CHARACTER*3       labexp
      CHARACTER*(Lnam)  NAMdim(      0:NdimNC)
      CHARACTER*(Luni)  UNIdim(      0:NdimNC)
      CHARACTER*(Lnam)  SdimNC(4,MX_var)       
      CHARACTER*(Luni)  unitNC(MX_var)
      CHARACTER*(Lnam)  nameNC(MX_var)
      CHARACTER*(Llnam) lnamNC(MX_var)
      CHARACTER*(Lfnam) fnamNC
      CHARACTER*(Ltit ) tit_NC
      CHARACTER*(Lnam)  NAMrat(NattNC)
      CHARACTER*4   coment
      CHARACTER*60  OUTdir
      CHARACTER*120 tmpINP
      INTEGER VARSIZE, ichrsz
      EXTERNAL VARSIZE
C +
C +
C +--NetCDF File Initialization
C +  ============================
C +
      IF (ipr_nc.eq.1) THEN
C +
      m10   = 1 +     mmarGE/10
      m1    = 1 + mod(mmarGE,10)
      jd10  = 1 +     jdarGE/10
      jd1   = 1 + mod(jdarGE,10)
C +
C +
C +--Output directory
C +  ----------------
C +
      ichrsz = VARSIZE (OUTdir)
C +.. (from libUN)
C +
C +
C +--Output File Label
C +  -----------------
C +
       fnamNC = OUTdir(1:ichrsz)//'LSM.'
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
     .        // '.' // labexp
     .        // '.nc    '
C +
C +
C +--Output Title
C +  ------------
C +
       tit_NC = 'LSMARIN output'
     .        // ' - Exp: ' // labexp
     .        // ' - '
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
C +
C +
C +--Create File / Write Constants
C +  -----------------------------
       MMXstp = MXdim
C +...To check array bounds... silently
C +
C +--Time Variable (hour)
C +  ~~~~~~~~~~~~~~~~~~~~
C +
C +...  To define a NetCDF dimension (size, name, unit):
        nDFdim(0)= npr_nc
        NAMdim(0)= 'time'
        UNIdim(0)= 'days'
C +
C +...  Check temporary arrays: large enough ?
        IF (npr_nc.gt.MMXstp)
     &  STOP '*** OUTgra - ERROR : MXdim to low ***'
C +
              starti     = jhurGE + minuGE/60.d0 + jsecGE/3600.d0
C +...        starti :     Starting Time (= current time in the day)
C +
        DO it = 1,npr_nc
              timeNC(it) = starti + (it-1) * jdh_LB 
C +...                                       jdh_LB: #hours between output
C +
              VALdim(it,0) = timeNC(it)
C +...        VALdim(  ,0) : values of the dimension # 0 (time) 

C +--Time Variable (date)
C +  ~~~~~~~~~~~~~~~~~~~~
              dateNC(it) =          timeNC(it)
              jourNC(it) = jdarGE + timeNC(it) / 24.d0
        END DO
                                   mois = mma0GE
        DO it = 1,npr_nc
          if (jourNC(it).gt.njmoGE(mois))then
           DO iu=it,npr_nc
              jourNC(iu) = jourNC(iu) - njmoGE(mois)
           END DO
                                   mois = mois +  1
             if (mois.gt.12)       mois =         1
          end if
              moisNC(it) =         mois
C +
          if (dateNC(it).gt.24.d0-epsi) then
           DO iu=it,npr_nc
              dateNC(iu) = dateNC(iu) -  24.d0
           END DO
          end if
        END DO
C +
        DO it = 1,npr_nc
              dateNC(it) = dateNC(it)
     .             + 1.d2 *jourNC(it) + 1.d4 *moisNC(it)
        END DO
C +

C     Define horizontal spatial dimensions :    
C +   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +...Check temporary arrays: large enough ?
      IF (    mx .gt.MMXstp.or.my.gt.MMXstp
     &    .or.mzz.gt.MMXstp.or.mw.gt.MMXstp)
     &  STOP '*** OUTgra - ERROR : MXdim to low ***'
C +
C +...To define NetCDF dimensions (size, name, unit):
C +
      DO i = 1, mx
        VALdim(i,1) = xxkm(i)
      END DO
      nDFdim(1)= mx
      NAMdim(1)= 'x'
      UNIdim(1)= 'km'
C +
      DO j = 1, my
        VALdim(j,2) = yykm(j)
      END DO
      nDFdim(2)= my
      NAMdim(2)= 'y'
      UNIdim(2)= 'km'
C +
      do k = 1, mz
        VALdim(k,3) = sigma(k)
      enddo
      nDFdim(3)= mz
      NAMdim(3)= 'level'
      UNIdim(3)= '[sigma]'
C +... For levels k
C +
C +--Variable's Choice (Table LSMvou.dat)
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
        OPEN(unit=15,status='unknown',file='LSMvou.dat')
C +
        itotNC = 0
 980    CONTINUE
          READ (15,'(A120)',end=990) tmpINP
          IF (tmpINP(1:4).eq.'    ') THEN 
            itotNC = itotNC + 1
            READ (tmpINP,'(4x,5A9,A12,A50)')
     &          nameNC(itotNC)  ,SdimNC(1,itotNC),SdimNC(2,itotNC),
     &          SdimNC(3,itotNC),SdimNC(4,itotNC),
     &          unitNC(itotNC)  ,lnamNC(itotNC)
C +...          nameNC: Name
C +             SdimNC: Names of Selected Dimensions (max.4/variable) 
C +             unitNC: Units
C +             lnamNC: Long_name, a description of the variable
C +
          ENDIF
        GOTO 980
 990    CONTINUE
C +
        NtotNC = itotNC 
C +...  NtotNC : Total number of variables writen in NetCDF file.
C +
C +--List of NetCDF attributes given to all variables:
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +... The "actual_range" is the (min,max)
C +    of all data for each variable:
      NAMrat(1) = 'actual_range'
      NvatNC(1) = 2

C +... The "[var]_range" is NOT of attribute type,
C +    it is a true variable containing the (min,max) for
C +    each level, for 4D (space+time) variables only
C +    (automatically handled by UN library;
C +     must be the LAST attribute)
      NAMrat(NattNC) = '[var]_range'
      NvatNC(NattNC) = 2
C +
C +--Automatic Generation of the NetCDF File Structure
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +     **************
        CALL UNscreate (fnamNC,tit_NC,
     &                  NdimNC, nDFdim, MXdim , NAMdim, UNIdim, VALdim,
     &                  MX_var, NtotNC, nameNC, SdimNC, unitNC, lnamNC,
     &                  NattNC, NAMrat, NvatNC,
     &                  ID__nc) 
C +     **************
C +
C +
C +--Write Time - Constants
C +  ~~~~~~~~~~~~~~~~~~~~~~
        DO j=1,my
        DO i=1,mx
          Wkxy1(i,j) =  GElonh(i,j) * 15.d0
C +...    Conversion: Hour->degrees
C +
          WKxy2(i,j) =  GElatr(i,j) / degrad
C +...    Conversion: rad ->degrees
C +
          WKxy3(i,j) =  isolSL(i,j)
C +...    Conversion to REAL type (integer not allowed)
C +
        END DO
        END DO
C +
C +     ************
        CALL UNwrite (ID__nc, 'date  ', 1  , npr_nc, 1 , 1 , dateNC)
        CALL UNwrite (ID__nc, 'lon   ', 1  , mx    , my, 1 , Wkxy1)
        CALL UNwrite (ID__nc, 'lat   ', 1  , mx    , my, 1 , Wkxy2)
        CALL UNwrite (ID__nc, 'sh    ', 1  , mx    , my, 1 , sh)
        CALL UNwrite (ID__nc, 'isol  ', 1  , mx    , my, 1 , Wkxy3)
C +     ************
C +
C +--Re-Open file if already created.
C +  ================================
C +
C +
      ELSE
C +
C +   ************
      CALL UNwopen (fnamNC,ID__nc)
C +   ************
C +
      END IF
C +
C +
C +--Write Time-dependent variables:
C +  ===============================
C +
      do k=1,mz
      do j=1,my
      do i=1,mx
        WKxyz1(i,j,k)=gplvDY(i,j,k)/gravit
      enddo
      enddo
      enddo
C +
C +   ************
      CALL UNwrite (ID__nc, 'uairDY ', ipr_nc, mx, my, mz, uairDY)
      CALL UNwrite (ID__nc, 'vairDY ', ipr_nc, mx, my, mz, vairDY)
      CALL UNwrite (ID__nc, 'tairDY ', ipr_nc, mx, my, mz, tairDY)
      CALL UNwrite (ID__nc, 'qvDY   ', ipr_nc, mx, my, mz, qvDY  )
      CALL UNwrite (ID__nc, 'zzDY   ', ipr_nc, mx, my, mz, WKxyz1)
      CALL UNwrite (ID__nc, 'pstar  ', ipr_nc, mx, my, 1 , pstDY )
      CALL UNwrite (ID__nc, 'hmelSL ', ipr_nc, mx, my, 1 , hmelSL)
      CALL UNwrite (ID__nc, 'tairSL ', ipr_nc, mx, my, 1 , tairSL)
      CALL UNwrite (ID__nc, 'tsrfSL ', ipr_nc, mx, my, 1 , tsrfSL)
C +   ************
C +
C +   ************
c #PO CALL UNwrite (ID__nc, 'hatmPO ', ipr_nc, mx, my, 1 , hatmPO)
c #PO CALL UNwrite (ID__nc, 'hfraPO ', ipr_nc, mx, my, 1 , hfraPO)
c #PO CALL UNwrite (ID__nc, 'aicePO ', ipr_nc, mx, my, 1 , aicePO)
c #PO CALL UNwrite (ID__nc, 'hicePO ', ipr_nc, mx, my, 1 , hicePO)
c #PO CALL UNwrite (ID__nc, 'hiavPO ', ipr_nc, mx, my, 1 , hiavPO)
C +   ************
C +
C +
C +--That 's all, folks: NetCDF File Closure
C +  =======================================
C +
C +   ***********
      CALL NCCLOS (ID__nc, Rcode)
C +   ***********
C +
C +
C +--Work Arrays Reset
C +  =================
C +
      do j=1,my
      do i=1,mx
        WKxy1(i,j)   =0.0
        WKxy2(i,j)   =0.0
        WKxy3(i,j)   =0.0
      enddo
      enddo
C +
      do k=1,mz
      do j=1,my
      do i=1,mx
        WKxyz1(i,j,k)=0.0
      enddo
      enddo
      enddo
C +
      return
      end
C + 
C +
      subroutine OUTils(labexp,ipr_nc,npr_nc,jdh_LB,OUTdir,
     .                  ps_asLS,  sh_asLS, CSTp, SIGp, halfway,
     .                  lsp_EXT, cp_EXT)
C +
C +------------------------------------------------------------------------+
C | LSMARIN output                                                         |
C |   SubRoutine OUTils is similar to OUTidl (from MAR)                    |
C |              It writes "Interplated Large Scale" (ILS) file :          |
C |              horizontal interp & wind rotation according to projection |
C |              purpose: compare ECMWF input to MAR output.               |
C +------------------------------------------------------------------------+
C |                                                                        |
C |   INPUT: ipr_nc: Current time step    number                           |
C |   ^^^^^^         (starting from ipr_nc=1, which => new file creation)  |
C |          npr_nc: Total  'time slices' number (max value of ipr_nc)     |
C |          jdh_LB: # hours between outputs                               |
C |          ps_asLS,  sh_asLS, CSTp, SIGp, halfway:                       |
C |                  LS data horizontaly interpolated to MAR grid.         |
C |                  (cf. LSget subroutine)                                |
C |                                                                        |
C |   OUTPUT: NetCDF File adapted to IDL Graphic Software (ILS...)         |
C |   ^^^^^^                                                               |
C |                                                                        |
C |   OPTION: #HY  Explicit Cloud Microphysics                             |
C |   ^^^^^^^                                                              |
C |                                                                        |
C |   CAUTION: 1) This Routine requires the usual NetCDF library,          |
C |   ^^^^^^^^    and a complementary access library called 'libUN.a'      |
C |                                                                        |
C +------------------------------------------------------------------------+
C +
C +
C +--General Variables
C +  =================
C +
      include 'MARphy.inc'
C +
      include 'MARdim.inc'
      include 'MARgrd.inc'
      include 'MAR_GE.inc'
C +
      include 'MAR_DY.inc'
      include 'MAR_SL.inc'
      include 'MAR_TU.inc'
C +
      include 'LSMARIN.inc'
      include 'LSM_WK.inc'
C +

C +--Inputs 
C +  ======
      REAL sh_asLS(mx,my), ps_asLS(mx,my)
      REAL halfway   (mx,my, 1:LSnk1, ivnum)
      REAL lsp_EXT(mx,my), cp_EXT(mx,my)
      REAL CSTp(LSnk1), SIGp(LSnk1)

C +
C +--Local   Variables
C +  =================
      common/IDLils/ namiNC
C +...               namiNC: To retain file name.
C +
      PARAMETER (Lfnam= 80, Ltit= 90, Luni= 16, Lnam= 13, Llnam=50)
C +...Length of char strings 
C +
      PARAMETER (NdimNC = 3)
C +...Number of defined spatial dimensions (exact)
C +
      PARAMETER (MXdim = 200)
C +...Maximum Number of all dims: recorded Time Steps
C +   and also maximum of spatial grid points for each direction. 
C +
      PARAMETER (MX_var = 80)
C +...Maximum Number of Variables 
C +
      PARAMETER (NattNC = 2)
C +...Number of real attributes given to all variables
C +
      DIMENSION         moisNC(MXdim)
      DIMENSION         jourNC(MXdim)
      DIMENSION         dateNC(MXdim)
      DIMENSION         timeNC(MXdim)
      DIMENSION         nDFdim(      0:NdimNC)
      DIMENSION         VALdim(MXdim,0:NdimNC)
      DIMENSION         NvatNC(NattNC)
      CHARACTER*3       labexp
      CHARACTER*(Lnam)  NAMdim(      0:NdimNC)
      CHARACTER*(Luni)  UNIdim(      0:NdimNC)
      CHARACTER*(Lnam)  SdimNC(4,MX_var)       
      CHARACTER*(Luni)  unitNC(MX_var)
      CHARACTER*(Lnam)  nameNC(MX_var)
      CHARACTER*(Llnam) lnamNC(MX_var)
      CHARACTER*(Lfnam) namiNC
      CHARACTER*(Ltit ) tit_NC
      CHARACTER*(Lnam)  NAMrat(NattNC)
      CHARACTER*4   coment
      CHARACTER*60  OUTdir
      CHARACTER*120 tmpINP
      INTEGER VARSIZE,ichrsz
      EXTERNAL VARSIZE

      LOGICAL logadd

C +
C +
C +--NetCDF File Initialization
C +  ============================
C +
      IF (ipr_nc.eq.1) THEN
C +
      m10   = 1 +     mmarGE/10
      m1    = 1 + mod(mmarGE,10)
      jd10  = 1 +     jdarGE/10
      jd1   = 1 + mod(jdarGE,10)
C +
C +
C +--Output File Label
C +  -----------------
C +
      ichrsz = VARSIZE (OUTdir)
C +.. for output directory (from libUN)
C +
       namiNC = OUTdir(1:ichrsz)//'ILS.'
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
     .        // '.' // labexp
     .        // '.nc    '

C +
C +
C +--Output Title
C +  ------------
C +
       tit_NC = 'ILS - LSMARIN output'
     .        // ' - Exp: ' // labexp
     .        // ' - '
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
C +
C +
C +--Create File / Write Constants
C +  -----------------------------
       MMXstp = MXdim
C +...To check array bounds... silently
C +
C +--Time Variable (hour)
C +  ~~~~~~~~~~~~~~~~~~~~
C +
C +...  To define a NetCDF dimension (size, name, unit):
        nDFdim(0)= npr_nc
        NAMdim(0)= 'time'
        UNIdim(0)= 'days'
C +
C +...  Check temporary arrays: large enough ?
        IF (npr_nc.gt.MMXstp)
     &  STOP '*** OUTgra - ERROR : MXdim to low ***'
C +
              starti     = jhurGE + minuGE/60.d0 + jsecGE/3600.d0
C +...        starti :     Starting Time (= current time in the day)
C +
        DO it = 1,npr_nc
              timeNC(it) = starti + (it-1) * jdh_LB 
C +...                                       jdh_LB: #hours between output
C +
              VALdim(it,0) = timeNC(it)
C +...        VALdim(  ,0) : values of the dimension # 0 (time) 

C +--Time Variable (date)
C +  ~~~~~~~~~~~~~~~~~~~~
              dateNC(it) =          timeNC(it)
              jourNC(it) = jdarGE + timeNC(it) / 24.d0
        END DO
                                   mois = mma0GE
        DO it = 1,npr_nc
          if (jourNC(it).gt.njmoGE(mois))then
           DO iu=it,npr_nc
              jourNC(iu) = jourNC(iu) - njmoGE(mois)
           END DO
                                   mois = mois +  1
             if (mois.gt.12)       mois =         1
          end if
              moisNC(it) =         mois
C +
          if (dateNC(it).gt.24.d0-epsi) then
           DO iu=it,npr_nc
              dateNC(iu) = dateNC(iu) -  24.d0
           END DO
          end if
        END DO
C +
        DO it = 1,npr_nc
              dateNC(it) = dateNC(it)
     .             + 1.d2 *jourNC(it) + 1.d4 *moisNC(it)
        END DO
C +
C     Define horizontal spatial dimensions :    
C +   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +...Check temporary arrays: large enough ?
      IF (    mx  .gt.MMXstp.or.my.gt.MMXstp
     &    .or.LSnk.gt.MMXstp.or.mw.gt.MMXstp)
     &  STOP '*** OUTgra - ERROR : MXdim to low ***'
C +
C +...To define NetCDF dimensions (size, name, unit):
C +
      DO i = 1, mx
        VALdim(i,1) = xxkm(i)
      END DO
      nDFdim(1)= mx
      NAMdim(1)= 'x'
      UNIdim(1)= 'km'
C +
      DO j = 1, my
        VALdim(j,2) = yykm(j)
      END DO
      nDFdim(2)= my
      NAMdim(2)= 'y'
      UNIdim(2)= 'km'
C +
      do k = 1, LSnk 
        VALdim(k,3) = float(k)
      enddo
      nDFdim(3)= LSnk
      NAMdim(3)= 'level'
      UNIdim(3)= 'No'
C +... For levels k
C +
C +--Variable's Choice (Table ILSvou.dat)
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
        OPEN(unit=15,status='unknown',file='ILSvou.dat')
C +
        itotNC = 0
 980    CONTINUE
          READ (15,'(A120)',end=990) tmpINP
          IF (tmpINP(1:4).eq.'    ') THEN 
            itotNC = itotNC + 1
            READ (tmpINP,'(4x,5A9,A12,A50)')
     &          nameNC(itotNC)  ,SdimNC(1,itotNC),SdimNC(2,itotNC),
     &          SdimNC(3,itotNC),SdimNC(4,itotNC),
     &          unitNC(itotNC)  ,lnamNC(itotNC)
C +...          nameNC: Name
C +             SdimNC: Names of Selected Dimensions (max.4/variable) 
C +             unitNC: Units
C +             lnamNC: Long_name, a description of the variable
C +
          ENDIF
        GOTO 980
 990    CONTINUE
C +
        NtotNC = itotNC 
C +...  NtotNC : Total number of variables writen in NetCDF file.
C +
C +--List of NetCDF attributes given to all variables:
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +... The "actual_range" is the (min,max)
C +    of all data for each variable:
      NAMrat(1) = 'actual_range'
      NvatNC(1) = 2

C +... The "[var]_range" is NOT of attribute type,
C +    it is a true variable containing the (min,max) for
C +    each level, for 4D (space+time) variables only
C +    (automatically handled by UN library;
C +     must be the LAST attribute)
      NAMrat(NattNC) = '[var]_range'
      NvatNC(NattNC) = 2
C +
C +--Automatic Generation of the NetCDF File Structure
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +     **************
        CALL UNscreate (namiNC,tit_NC,
     &                  NdimNC, nDFdim, MXdim , NAMdim, UNIdim, VALdim,
     &                  MX_var, NtotNC, nameNC, SdimNC, unitNC, lnamNC,
     &                  NattNC, NAMrat, NvatNC,
     &                  ID__nc) 
C +     **************
C +
C +
C +--Write Time - Constants
C +  ~~~~~~~~~~~~~~~~~~~~~~
        DO j=1,my
        DO i=1,mx
          Wkxy1(i,j) =  GElonh(i,j) * 15.d0
C +...    Conversion: Hour->degrees
C +
          WKxy2(i,j) =  GElatr(i,j) / degrad
C +...    Conversion: rad ->degrees
C +
          WKxy3(i,j) =  isolSL(i,j)
C +...    Conversion to REAL type (integer not allowed)
C +
        END DO
        END DO
C +
C +     ************
        CALL UNwrite (ID__nc, 'date  ', 1, npr_nc, 1 , 1  , dateNC)
        CALL UNwrite (ID__nc, 'lon   ', 1, mx    , my, 1  , Wkxy1)
        CALL UNwrite (ID__nc, 'lat   ', 1, mx    , my, 1  , Wkxy2)
        CALL UNwrite (ID__nc, 'CSTp  ', 1, LSnk  , 1 , 1  , CSTp )
        CALL UNwrite (ID__nc, 'SIGp  ', 1, LSnk  , 1 , 1  , SIGp )
        CALL UNwrite (ID__nc, 'sh    ', 1, mx    , my, 1  , sh_asLS)
        CALL UNwrite (ID__nc, 'isol  ', 1, mx    , my, 1  , Wkxy3)
C +     ************
C +
        logadd = .FALSE.
C +...  For "cumulated" variable such as precip: initialise.
C +
C +--Re-Open file if already created.
C +  ================================
C +
C +
      ELSE
C +
C +   ************
      CALL UNwopen (namiNC,ID__nc)
C +   ************
C +
        logadd = .TRUE.
C +...  For "cumulated" variable such as precip: add 
C +
      END IF
C +
C +
C +--Write Time-dependent variables:
C +  ===============================
C +
C +   *************
      CALL ILSwrite(ID__nc, 'uairDY', ipr_nc, 1, halfway)
      CALL ILSwrite(ID__nc, 'vairDY', ipr_nc, 2, halfway)
      CALL ILSwrite(ID__nc, 'qvDY'  , ipr_nc, 3, halfway)
      CALL ILSwrite(ID__nc, 'pktDY' , ipr_nc, 4, halfway)

      CALL UNwrite (ID__nc, 'ps    ', ipr_nc, mx, my, 1 , ps_asLS)

      do j=1,my
      do i=1,mx
        WKxy1(i,j)= lsp_EXT(i,j) + cp_EXT(i,j)
      enddo
      enddo

      CALL ILSadd  (ID__nc,'lsp_EXT',ipr_nc,logadd,lsp_EXT,WKxy2)
      CALL ILSadd  (ID__nc,'tp_EXT' ,ipr_nc,logadd,WKxy1  ,WKxy2)

C +
C +
C +--That 's all, folks: NetCDF File Closure
C +  =======================================
C +
C +   ***********
      CALL NCCLOS (ID__nc, Rcode)
C +   ***********
C +
C +
C +--Work Arrays Reset
C +  =================
C +
      do j=1,my
      do i=1,mx
        WKxy1(i,j)   =0.0
        WKxy2(i,j)   =0.0
        WKxy3(i,j)   =0.0
      enddo
      enddo
C +
      return
      end

C +--------------------------------------------------------------+
      subroutine ILSwrite(ID__nc, varname, ipr_nc, iv, halfway)
C +--------------------------------------------------------------+
C +
      INCLUDE 'MARdim.inc'
      INCLUDE 'LSMARIN.inc'
      INCLUDE 'LSM_WK2.inc'

C +--Inputs
C +  ------
      INTEGER ID__nc, ipr_nc, iv
      CHARACTER*(*) varname
      REAL halfway   (mx,my, 1:LSnk1, ivnum)

C +--Extract to 3D variable
C +  ----------------------
      DO lsk= 1, LSnk
       DO j = 1,my
       DO i = 1,mx
         WKxyls(i,j,lsk) = halfway (i,j,lsk,iv) 
       END DO
       END DO
      END DO

C +--Write in file.
C +  --------------
      CALL UNwrite (ID__nc, varname, ipr_nc, mx, my, LSnk, WKxyls)

      end

C +--------------------------------------------------------------+
      subroutine ILSadd(ID__nc,varname,ipr_nc,logadd,var,WKxy)
C +--------------------------------------------------------------+
C +
      INCLUDE 'MARdim.inc'
      INCLUDE 'LSMARIN.inc'

C +--Inputs
C +  ------
      INTEGER ID__nc, ipr_nc
      LOGICAL logadd
      CHARACTER*(*) varname
      REAL var(mx,my), WKxy(mx,my)
C +--Local
C +  -----
      REAL cx(mx), REAL cy(my), REAL cz(1)
      REAL valerr
      CHARACTER *30 vrunits
      INTEGER ipprec
      LOGICAL logerr

      IF (logadd) THEN
C +--Read previous values in file.
C +  ------------------------------
        ipprec = ipr_nc-1
        CALL UNread
     &      (ID__nc ,varname, ipprec,  0, 0, 0,
     &       mx , my, 1,
     &       cx , cy, cz,
     &       vrunits, WKxy)

        logerr = .FALSE. 
        do j=1,my
        do i=1,mx
          IF (var(i,j).LT.2.) THEN
             WKxy(i,j)= WKxy(i,j) + var(i,j)
          ELSE
             logerr = .TRUE.
             valerr = var(i,j)
          ENDIF
        enddo
        enddo

        IF (logerr) THEN
           write(*,*) 'Valeur bizarre', valerr
        ENDIF

      ELSE   
C +--Set initial values in file.
C +  ---------------------------
        do j=1,my
        do i=1,mx
          IF (var(i,j).LT.2.) THEN
             WKxy(i,j)= var(i,j)
          ELSE
             WKxy(i,j)= 0.0     
          ENDIF
        enddo
        enddo

      ENDIF

C +--Write to file.
C +  --------------
      CALL UNwrite (ID__nc, varname, ipr_nc, mx, my, 1, WKxy) 

      end

      subroutine OUTsta(iostat,OUTdir)
C +
C +------------------------------------------------------------------------+
C | MAR OUTPUT                                             xx-03-1997  MAR |
C |   SubRoutine OUTsta is used to compute time STATISTICS and             |
C |                                WRITE or UPDATE the output NetCDF file  |
C +------------------------------------------------------------------------+
C |                                                                        |
C |   INPUT:                                                               |
C |   ^^^^^^  iostat : "selector" for statistics outputs:                  |
C |                1 = reset everything (forget all prev. val.= new file)  |
C |                2 = reset only this run section,                        |
C |                3 = reset nothing (i.e. update all values)              |
C |                                                                        |
C |   OUTPUT: NetCDF File (adapted to IDL Graphic Software)                |
C |   ^^^^^^                                                               |
C |                                                                        |
C |   OPTION: #HY  Explicit Cloud Microphysics                             |
C |   ^^^^^^^                                                              |
C |                                                                        |
C |   CAUTION: 1) This Routine requires the usual NetCDF library,          |
C |   ^^^^^^^^    and a complementary access library called 'libUN.a'      |
C |                                                                        |
C +------------------------------------------------------------------------+
C +
C +
C +--General Variables
C +  =================
C +
      include 'MARphy.inc'
C +
      include 'MARdim.inc'
      include 'MARgrd.inc'
      include 'MAR_GE.inc'
      include 'MAR_IO.inc'
C +
      include 'MAR_DY.inc'
      include 'MAR_SL.inc'
      include 'MAR_TU.inc'
C +
C +
c #PO include 'MAR_PO.inc'
c #TV include 'MARveg.inc'
C +
      include 'LSM_WK.inc'
C +
C +
C +--Local   Variables
C +  =================
      common/STAloc/ namSTA
C +...               namSTA: To retain file name.
C +
      PARAMETER (Lfnam= 80, Ltit= 90, Luni= 16, Lnam= 13, Llnam=50)
C +...Length of char strings
C +
      PARAMETER (NdimNC = 4)
C +...Number of defined spatial dimensions (exact)
C +
      PARAMETER (MXdim = 300)
C +...Maximum Number of all dims: recorded Time Steps
C +   and also maximum of spatial grid points for each direction.
C +
      PARAMETER (MX_var = 80)
C +...Maximum Number of Variables
C +
      PARAMETER (NattNC = 1)
C +...Number of real attributes given to all variables
C +
      PARAMETER(nplevs=5)
C +...Number of pressure levels used in statistics outputs.
C +

      DIMENSION         moisNC(MXdim)
      DIMENSION         jourNC(MXdim)
      DIMENSION         dateNC(MXdim)
      DIMENSION         timeNC(MXdim)
      DIMENSION         nDFdim(      0:NdimNC)
      DIMENSION         VALdim(MXdim,0:NdimNC)
      DIMENSION         NvatNC(1     )
      DIMENSION         WKstat(mx,my,nplevs)
C +...         (interpolated values,...) 
C +...         Takes memory, but usefull for clear & portable code
      DIMENSION         plevs(nplevs)  
      CHARACTER*(Lnam)  NAMdim(      0:NdimNC)
      CHARACTER*(Luni)  UNIdim(      0:NdimNC)
      CHARACTER*(Lnam)  SdimNC(4,MX_var)
      CHARACTER*(Luni)  unitNC(MX_var)
      CHARACTER*(Lnam)  nameNC(MX_var)
      CHARACTER*(Llnam) lnamNC(MX_var)
      CHARACTER*(Lfnam) namSTA
      CHARACTER*(Ltit ) tit_NC
      CHARACTER*(Lnam)  NAMrat(NattNC)
      CHARACTER*4   coment
      CHARACTER*60  OUTdir
      CHARACTER*120 tmpINP
      INTEGER VARSIZE, ichrsz
      EXTERNAL VARSIZE

      DATA plevs / 850., 800., 700., 500., 200. /
C +... Output p-levels             
C +
C +
C +--NetCDF File Initialization
C +  ============================
C +
      IF (iostat.EQ.1) THEN
C +
      m10   = 1 +     mmarGE/10
      m1    = 1 + mod(mmarGE,10)
      jd10  = 1 +     jdarGE/10
      jd1   = 1 + mod(jdarGE,10)
C +
C +
C +--Output File Label
C +  -----------------
C +
      ichrsz = VARSIZE (OUTdir)
C +.. for output directory (from libUN)

       namSTA = OUTdir(1:ichrsz) // 'STA.'
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
     .        // '.' // explIO
     .        // '.nc    '
C +
C +
C +--Output Title
C +  ------------
C +
       tit_NC = 'MAR - Statistics for'
     .        // ' Exp: ' // explIO
     .        // ' - '
     .        // labnum( m10) // labnum( m1)
     .        // labnum(jd10) // labnum(jd1)
C +
C +
C +
C +--Create File / Write Constants
C +  -----------------------------
       MMXstp = MXdim
C +...To check array bounds... silently
C +
C +--Time Variable 
C +  ~~~~~~~~~~~~~
C +
C +...  Define a NetCDF dimension (size, name, unit) 
C +...    for an "index" to the 6 types of informations
C +...    included in the variable  : 
C +...    statindex = 1 > time average, = 2 > Sx**2, ...

        nDFdim(0)= 6 
        NAMdim(0)= 'time'
        UNIdim(0)= '-'
        DO i = 1, nDFdim(0)
          VALdim(i,0) = i
        END DO
C +
C +
 
C     Define horizontal spatial dimensions :
C +   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +...Check temporary arrays: large enough ?
      IF (    mx .gt.MMXstp.or.my.gt.MMXstp
     &    .or.mzz.gt.MMXstp.or.mw.gt.MMXstp)
     &  STOP '*** OUTidl - ERROR : MXdim to low ***'
C +
C +...To define NetCDF dimensions (size, name, unit):
C +
      DO i = 1, mx
        VALdim(i,1) = xxkm(i)
      END DO
      nDFdim(1)= mx
      NAMdim(1)= 'x'
      UNIdim(1)= 'km'
C +
      DO j = 1, my
        VALdim(j,2) = yykm(j)
      END DO
      nDFdim(2)= my
      NAMdim(2)= 'y'
      UNIdim(2)= 'km'
C +
      do k = 1, nplevs
        VALdim(k,3) = plevs(k)
      enddo
      nDFdim(3)= nplevs
      NAMdim(3)= 'p-level'
      UNIdim(3)= 'hPa'
C +... For p interpolated levels 
C +
      do k = 1, mw
        VALdim(k,4) = k
      enddo
      nDFdim(4)= mw
      NAMdim(4)= 'sector'
      UNIdim(4)= '[index]'
C +... For Surface Sectors
C +
C +--Variable's Choice (Table MARsta.dat)
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
        OPEN(unit=15,status='unknown',file='MARsta.dat')
C +
        itotNC = 0
 980    CONTINUE
          READ (15,'(A120)',end=990) tmpINP
          IF (tmpINP(1:4).eq.'    ') THEN
            itotNC = itotNC + 1
            READ (tmpINP,'(4x,5A9,A12,A50)')
     &          nameNC(itotNC)  ,SdimNC(1,itotNC),SdimNC(2,itotNC),
     &          SdimNC(3,itotNC),SdimNC(4,itotNC),
     &          unitNC(itotNC)  ,lnamNC(itotNC)
C +...          nameNC: Name
C +             SdimNC: Names of Selected Dimensions (max.4/variable)
C +             unitNC: Units
C +             lnamNC: Long_name, a description of the variable
C +
          ENDIF
        GOTO 980
 990    CONTINUE
C +
        NtotNC = itotNC
C +...  NtotNC : Total number of variables writen in NetCDF file.
C +
C +--List of NetCDF attributes given to all variables:
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +... No attributes are set here (range is unusefull: statistic)
      NAMrat(1) = ' ' !NAMrat MUST be defined, but-> do nothing
      NvatNC(1) = 1
C +
C +--Automatic Generation of the NetCDF File Structure
C +  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
C +
C +     **************
        CALL UNscreate (namSTA,tit_NC,
     &                  NdimNC, nDFdim, MXdim , NAMdim, UNIdim, VALdim,
     &                  MX_var, NtotNC, nameNC, SdimNC, unitNC, lnamNC,
     &                  NattNC, NAMrat, NvatNC,
     &                  ID_sta)
C +     **************

C +
C +
C +--Write Time - Constants
C +  ~~~~~~~~~~~~~~~~~~~~~~
        DO j=1,my
        DO i=1,mx
          Wkxy1(i,j) =  GElonh(i,j) * 15.d0
C +...    Conversion: Hour->degrees
C +
          WKxy2(i,j) =  GElatr(i,j) / degrad
C +...    Conversion: rad ->degrees
C +
          WKxy3(i,j) =  isolSL(i,j)
C +...    Conversion to REAL type (integer not allowed)
C +
        END DO
        END DO
C +
C +     ************
        CALL UNwrite (ID_sta, 'lon   ', 1  , mx    , my, 1 , Wkxy1)
        CALL UNwrite (ID_sta, 'lat   ', 1  , mx    , my, 1 , Wkxy2)
        CALL UNwrite (ID_sta, 'sh    ', 1  , mx    , my, 1 , sh)
        CALL UNwrite (ID_sta, 'isol  ', 1  , mx    , my, 1 , Wkxy3)
C +     ************
C +
C +--Re-Open file if already created.
C +  ================================
C +
C +
      ELSE
C +
C +   ************
      CALL UNwopen (namSTA,ID_sta)
C +   ************
C +
C +
      END IF
C +
C +--Compute Output Specific Variables (interpolate...): 
C +--Update Time Statistics:                  
C +  ===================================================
C +
      do k=1,mz
      do j=1,my
      do i=1,mx
        WKxyz1(i,j,k)=gplvDY(i,j,k)/gravit
C +...  Geopotential height

        WKxyz2(i,j,k)= (sigma(k) * pstDY(i,j) + ptopDY) * 10.
C +...  Pressure 
      enddo
      enddo
      enddo

C +--Pressure-Levels Interpolated variables:
C +  ---------------------------------------

C +   ***********
      CALL INTver (WKxyz2,uairDY,plevs,nplevs,WKstat)
      CALL STAgen (ID_sta, 'uairDY',nplevs,iostat,WKstat,
     .             WKxyz6,WKxyz7,WKxyz8)

      CALL INTver (WKxyz2,vairDY,plevs,nplevs,WKstat)
      CALL STAgen (ID_sta, 'vairDY',nplevs,iostat,WKstat,
     .             WKxyz6,WKxyz7,WKxyz8)

      CALL INTver (WKxyz2,WKxyz1,plevs,nplevs,WKstat)
      CALL STAgen (ID_sta, 'zzDY'  ,nplevs,iostat,WKstat,
     .             WKxyz6,WKxyz7,WKxyz8)

      CALL INTver (WKxyz2,tairDY,plevs,nplevs,WKstat)
      CALL STAgen (ID_sta, 'tairDY',nplevs,iostat,WKstat,
     .             WKxyz6,WKxyz7,WKxyz8)

      CALL INTver (WKxyz2,qvDY  ,plevs,nplevs,WKstat)
      CALL STAgen (ID_sta, 'qvDY'  ,nplevs,iostat,WKstat,
     .             WKxyz6,WKxyz7,WKxyz8)
C +   ***********
C +

C +--Mean Sea Level Pressure:
C +  ------------------------
C +
      izslp =  mz - 4
C +...  (mslp is computed using the 4th level above surface)

      gamTz = - 6.5E-3
      exxpo = - gravit / (gamTz * ra)
      DO j=1,my
      DO i=1,mx

        WKxy4(i,j)= 10.*(ptopDY+pstDY(i,j)*sigma(izslp))
     .    *(1.-gamTz*WKxyz1(i,j,izslp)/tairDY(i,j,izslp))**exxpo
C +...                zzDY
C +...  (mslp is computed using the first level close to surface)

      ENDDO
      ENDDO

C +   ***********
      CALL STAgen (ID_sta, 'pmslDY',1,iostat,WKxy4,
     .             WKxy1,WKxy2,WKxy3)
C +   ***********

C +--Surface variables:
C +  ------------------

C +   ***********
      CALL STAgen (ID_sta, 'tsrfSL',mw,iostat,tsrfSL,
     .             WKxyz6,WKxyz7,WKxyz8)
C +   ***********

      gamTz = - 6.5E-3
      exxpo = - gravit / (gamTz * ra)

C +--That 's all, folks: NetCDF File Closure
C +  =======================================
C +
C +   ***********
      CALL NCCLOS (ID_sta, Rcode)
C +   ***********
C +
C +
C +--Work Arrays Reset
C +  =================
C +
      do j=1,my
      do i=1,mx
        WKxy1(i,j)   =0.0
        WKxy2(i,j)   =0.0
        WKxy3(i,j)   =0.0
        WKxy4(i,j)   =0.0
      enddo
      enddo
C +
      do k=1,mz
      do j=1,my
      do i=1,mx
        WKxyz1(i,j,k)=0.0
        WKxyz2(i,j,k)=0.0

        WKxyz6(i,j,k)=0.0
        WKxyz7(i,j,k)=0.0
        WKxyz8(i,j,k)=0.0
      enddo
      enddo
      enddo
C +
      return
      end


C +------------------------------------------------------------------------+
C | MAR (output)                                           xx-03-1997  MAR |
C |   SubRoutine STAgen compute time statistics and updates ouput files    |
C +------------------------------------------------------------------------+
C |     iostat = Updating Level for statistics:                            |
C |                1 = reset everything (forget all prev. val.= new file)  |
C |                2 = reset only this run section,                        |
C |                3 = reset nothing (i.e. update all values)              | 
C |                                                                        |
C |     NOTE: The routine accounts for 'missing values' in the input:      |
C |           Missing values are ignored, valid values are counted (validn)|
C |           *validn is a REAL because UN library does not handle integers|
C +------------------------------------------------------------------------+
      SUBROUTINE STAgen (ID_sta, varname,nplevs,iostat,WKstat,
     &                   vamean, sumxsq, validn)

      INCLUDE 'MARdim.inc'

      INTEGER ID_sta
      CHARACTER    varname*(*)
      CHARACTER*13 vrunits
      DIMENSION  WKstat(mx,my,nplevs)
      DIMENSION  vamean(mx,my,nplevs), sumxsq(mx,my,nplevs)
      DIMENSION  validn(mx,my,nplevs)  
      DIMENSION  cx(mx), cy(my), plevs(20)

C +--Loop over the 2 statistical periods
C +  -----------------------------------
      DO iperio= 1,2     !(*iperiod loop*) 
C +... iperio=1 -> runs ensemble
C +... iperio=2 -> this run section
      idxper = iperio*3 -2

      IF (iperio.LT.iostat) THEN
C +--Read 'old' data to update
C +  =========================

C +...Read old vamean values...
        CALL UNread	
     &      (ID_sta    ,varname, idxper  ,  0, 0, 0, 
     &       mx , my, nplevs,
     &       cx , cy, plevs,
     &       vrunits, vamean)

C +...Read old sumxsq (sum of data**2)
        CALL UNread
     &      (ID_sta    ,varname, idxper+1,  0, 0, 0, 
     &       mx , my, nplevs,
     &       cx , cy, plevs,
     &       vrunits, sumxsq)

C +...Read old valid data number
        CALL UNread
     &      (ID_sta    ,varname, idxper+2,  0, 0, 0, 
     &       mx , my, nplevs,
     &       cx , cy, plevs,
     &       vrunits, validn)

      ELSE
C +--Reset statistics
C +  ================

        DO lv= 1, nplevs
         DO j= 1, my
         DO i= 1, mx
             vamean(i,j,lv)= 1.E30   
             sumxsq(i,j,lv)= 1.E30 
C +...                  Missing value
             validn(i,j,lv)= 0.0
         ENDDO
         ENDDO
        ENDDO

      ENDIF 

C +--Update statistics for each valid data
C +  =====================================
      DO lv= 1, nplevs
       DO j= 1, my
       DO i= 1, mx
         IF (WKstat(i,j,lv).LT.1.E15) THEN

           IF (validn(i,j,lv).le.0.1) THEN
C +--Initialise statistics at that point (first valid data)
C    ------------------------------------------------------
             vamean(i,j,lv)= WKstat(i,j,lv) 
             sumxsq(i,j,lv)= WKstat(i,j,lv)*WKstat(i,j,lv)
             validn(i,j,lv)= 1.0

           ELSE
C +--Update statistics at that point
C    -------------------------------
             vamean(i,j,lv)= ( vamean(i,j,lv)*validn(i,j,lv) 
     &         + WKstat(i,j,lv) ) / (validn(i,j,lv)+1.0)
             sumxsq(i,j,lv)= sumxsq(i,j,lv)
     &         + WKstat(i,j,lv)*WKstat(i,j,lv)
             validn(i,j,lv)= validn(i,j,lv)+1.0
           
           ENDIF
         ELSE
           IF (WKstat(i,j,lv).LT.1.E29)
     .     write(*,*) 'Surprise:', WKstat(i,j,lv), lv
         ENDIF

       ENDDO
       ENDDO
      ENDDO

C +--Write updated data
C +  ==================

      CALL UNwrite (ID_sta, varname, idxper  , mx, my, nplevs, vamean)
      CALL UNwrite (ID_sta, varname, idxper+1, mx, my, nplevs, sumxsq)
      CALL UNwrite (ID_sta, varname, idxper+2, mx, my, nplevs, validn)

C +--End loop over statistical time periods
C +  --------------------------------------
      ENDDO                 !(*iperiod loop*)

      RETURN
      END



C +------------------------------------------------------------------------+
C | MAR (output)                                           xx-03-1997  MAR |
C |   SubRoutine INTver handles linear interpolation along the vertical    |
C |                     (e.g. to pressure levels)                          |
C +------------------------------------------------------------------------+
C |                                                                        |
C |       xin(mx,my,mz)    = input vertical coordinate (ascending order)   |
C |       vin(mx,my,mz)    = input value                                   |
C |       alvout(nlv)      = output coordinates (e.g. pressure level)      |
C |       nlv              = the true number of output coordinates (alvout)|
C |       valout(mx,my,nlv)= output values                                 |
C |                                                                        |
C |   NOTES:                                                               |
C |      -The 'MISSING VALUE' = 1.0E30 is returned when the requested      |
C |                             alvout is out of xin range                 |
C |      -The routine is best for sequential processing, not vectorial.    |
C +------------------------------------------------------------------------+
      SUBROUTINE INTver (xin,vin,alvout,nlv,valout)

      INCLUDE 'MARdim.inc'

      INTEGER nlv
      REAL xin(mx,my,mz),vin(mx,my,mz)
      REAL alvout(nlv),valout(mx,my,nlv)
      INTEGER ind, KLO, KHI
      REAL fdis

      DO lv= 1, nlv
       DO j= 1, my
       DO i= 1, mx

C +---  Search for the appropriate level in the input values:
        KLO=1
        KHI=mz
 1      IF (KHI-KLO.GT.1) THEN
          K=(KHI+KLO)/2
          IF(xin(i,j,K).GT.alvout(lv))THEN
            KHI=K
          ELSE
            KLO=K
          ENDIF
        GOTO 1
        ENDIF
        ind=KLO

        IF (alvout(lv).LE.xin(i,j,mz) 
     .    .AND. alvout(1).GE.xin(i,j,1)) THEN

C +---    Linearly interpolate:
          fdis  = xin(i,j,ind+1)-xin(i,j,ind)
          valout(i,j,lv)
     .        = vin(i,j,ind)* ((xin(i,j,ind+1)-alvout(lv)) /fdis)
     .      + vin(i,j,ind+1)* ((alvout(lv)-xin(i,j,ind  )) /fdis)

        ELSE
C +---    Set a missing value:
          valout(i,j,lv)= 1.0E30

        ENDIF

       ENDDO
       ENDDO
      ENDDO

      RETURN
      END

